/* Copyright (C) 2000/2001 sgop@users.sourceforge.net
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <ctype.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <unistd.h>
#include <stdio.h>
#include <strings.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <time.h>
#include <netdb.h>
#include <sys/utsname.h>

#include "lopster.h"
#include "connection.h"
#include "global.h"
#include "search.h"
#include "transfer.h"
#include "resume.h"
#include "interface.h"
#include "support.h"
#include "callbacks.h"
#include "browse.h"
#include "share.h"
#include "hotlist.h"
#include "commands.h"
#include "chat.h"
#include "dirselect.h"
#include "scheme.h"
#include "handler.h"
#include "resume.h"
#include "server.h"
#include "preferences.h"
#include "log.h"
#include "exec.h"
#include "wizard.h"
#include "whois.h"
#include "whiteboard.h"

char *FolderNames(int no)
{
  switch (no) {
  case 0:
    return _("Chat Commands");
  case 1:
    return _("User Modification Commands");
  case 2:
    return _("Infomation Commands");
  case 3:
    return _("Misc Commands");
  case 4:
    return _("Client Commands");
  case 5:
    return _("Channel Operator Commands");
  case 6:
    return _("Moderator Commands");
  case 7:
    return _("Admin Commands");
  default:
    return _("Unspecified command");
  }
}

int command_new(int id, char *iname, char *syntax, char *ides,
		user_level_t ilevel, folder_t ifolder,
		COMMAND((*ihandler)))
{
  command_t *new_comm;

  if (!iname) {
    g_warning(_("no command name given"));
    return 0;
  }
  if (command_is_command(iname)) {
    g_warning(_("chat command already exists [%s]"), iname);
    return 0;
  }
  new_comm = (command_t *) l_malloc(sizeof(command_t));
  new_comm->id = id;
  new_comm->name = l_strdup(iname);
  new_comm->syntax = l_strdup(syntax);
  new_comm->description = l_strdup(ides);
  new_comm->is_alias = 0;
  new_comm->level = ilevel;
  new_comm->folder = ifolder;
  new_comm->handler = ihandler;

  global.commands = g_list_append(global.commands, new_comm);

  return 1;
}

int command_rename(int iid, char *iname)
{
  command_t *comm;

  (void)iid;
  comm = command_is_command(iname);

  if (!comm) {
    g_warning(_("chat command does not exists"));
    return 0;
  }

  l_free(comm->name);
  comm->name = l_strdup(iname);

  return 1;
}

GList *command_search(char *prefix)
{
  GList *result = NULL;
  command_t *comm;
  int len = strlen(prefix);
  GList* dlist;

  for (dlist = global.commands; dlist; dlist = dlist->next) {
    comm = dlist->data;
    if (!strncasecmp(prefix, comm->name, len))
      result = g_list_append(result, comm);
  }

  return result;
}

command_t *command_is_command(char *iname)
{
  command_t *comm;
  char *name;
  GList* dlist;

  if (!iname)
    return NULL;
  if (*iname == '/')
    name = iname + 1;
  else
    name = iname;

  for (dlist = global.commands; dlist; dlist = dlist->next) {
    comm = dlist->data;
    if (!l_strcasecmp(name, comm->name))
      return comm;
  }

  return NULL;
}

char *command_get_alias_string(int id)
{
  command_t *comm;
  static char result[1024];
  GList* dlist;

  result[0] = 0;
  for (dlist = global.commands; dlist; dlist = dlist->next) {
    comm = dlist->data;
    if ((comm->id == id) && (comm->is_alias)) {
      if (strlen(result) > 0)
	strcat(result, " ");
      strcat(result, comm->name);
    }
  }

  return result;
}

int command_make_alias(char *iname, char *alias)
{
  command_t *comm;
  command_t *new_alias;
  char temp[1024];

  if (!iname) {
    g_warning(_("no command name given"));
    return 0;
  }
  comm = command_is_command(alias);
  if (comm) {
    client_message(_("Error"), _("Command does already exists"));
    return 0;
  }
  if (!alias) {
    client_message(_("Error"), _("No alias name given"));
    return 0;
  }
  comm = command_is_command(iname);
  if (!comm) {
    client_message(_("Error"), _("Command does not exists"));
    return 0;
  }

  new_alias = (command_t *) l_malloc(sizeof(command_t));
  new_alias->id = comm->id;
  new_alias->name = l_strdup(alias);
  new_alias->syntax = l_strdup_printf("see %s", iname);
  strcpy(temp, _("This is an alias for <"));
  strcat(temp, iname);
  strcat(temp, ">");
  new_alias->description = l_strdup(temp);
  new_alias->is_alias = 1;
  new_alias->level = comm->level;
  new_alias->folder = comm->folder;
  new_alias->handler = comm->handler;
  global.commands = g_list_append(global.commands, new_alias);

  strcpy(temp, _("New alias: "));
  strcat(temp, alias);
  strcat(temp, " := ");
  strcat(temp, iname);
  client_message(_("Message"), temp);

  return 1;
}

void command_delete_alias_for(int id)
{
  command_t *comm;
  int i1;

  i1 = 0;
  while (i1 < (int)g_list_length(global.commands))
  {
    comm = (command_t *) (g_list_nth(global.commands, i1)->data);
    if ((comm->id == id) && (comm->is_alias)) {
      global.commands = g_list_remove(global.commands, comm);
    } else {
      i1++;
    }
  }
}

void command_delete_all()
{
  command_t *comm;
  GList* dlist;

  for (dlist = global.commands; dlist; dlist = dlist->next) {
    comm = (command_t *) (dlist->data);
    l_free(comm->name);
    l_free(comm->syntax);
    l_free(comm->description);
    l_free(comm);
  }
  g_list_free(global.commands);
  global.commands = NULL;
}

void handle_lopster_command(char *data)
{
  char *pos;
  command_t *comm;

  if (global.commands == NULL) {
    g_warning("no commands declared!");
    return;
  }
  if (data[0] != '/') {
    return;
  } else
    data++;

  pos = arg(data, 0);
  if (pos) {
    comm = command_is_command(pos);
    if (!comm) {
      client_message(_("Error"),
		     _
		     ("Unknown command! Type /help for a list of commands."));
      return;
    }
    if ((comm->id >= 100) && (!opennap_version(0, 0))) {
      opennap_notify();
      return;
    }
    pos = arg(NULL, 1);
    comm->handler(pos);
  }
}

COMMAND(lopster_none)
{
  (void)data;
  g_warning("command with id C_NONE");
}

COMMAND(lopster_whisper)
{
  char *nick;
  char *message;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("You have to specify a nickname!"));
    return;
  }
  message = arg(NULL, 1);
  if (!message) {
    client_message(_("Error"), _("Whisper what?"));
    return;
  }
  send_private(nick, message, 1);
}

COMMAND(lopster_whois)
{
  char *nick;

  if (global.status.connection < 2)
    return;

  nick = arg(data, 0);
  if (!nick) {
    if (global.reply)
      whois_request(global.reply, WHOIS_SHOW);
    else
      whois_request(SERVER->nick, WHOIS_SHOW);
  } else {
    if (!strncasecmp(nick, "-text", 5)) {
      nick = arg(NULL, 0);
      if (!nick) {
	if (global.reply)
	  whois_request(global.reply, WHOIS_TEXT);
	else
	  whois_request(SERVER->nick, WHOIS_TEXT);
      } else {
	whois_request(nick, WHOIS_TEXT);
      }
    } else {
      whois_request(nick, WHOIS_SHOW);
    }
  }
}

COMMAND(lopster_emote)
{
  char *message;

  message = arg(data, 1);
  if (!message) {
    client_message(_("Error"), _("Please specify an emote message!"));
    return;
  }
  if (in_channel()) {
    send_emote(message);
  } else {
    chat_print(_("Error"), _("/emote not supported this page\n"));
  }
}

COMMAND(lopster_join)
{
  char *channel;

  channel = arg(data, 0);
  if (!channel) {
    client_message(_("Error"), _("Join which channel?"));
    return;
  }
  join_channel(channel);
}

COMMAND(lopster_raw)
{
  int type;
  char *args;

  args = arg(data, 0);
  if (!args) {
    client_message(_("Error"), _("You have to specify a command code!"));
    return;
  }
  type = atoi(args);
  if (type < 0 || type > 10302) {
    client_message(_("Error"), _("Command code is out of bounds!"));
    return;
  }
  
  args = arg(NULL, 1);
  if (!args)
    send_command(type, "");
  else
    send_command(type, args);
}

void help_print_folder(folder_t folder, int all) {
  char* temp_str;
  GList* dlist;
  command_t* command;
  int cnt = 0;
  char str[1024];
  int i1;
  user_level_t level;
  
  level = (global.user.level > L_LEECH)?global.user.level:L_USER;

  chat_print("message", " ");
  temp_str = l_strdup(FolderNames(folder));
  chat_print("error", temp_str);
  l_free(temp_str);
  cnt = 0;
  for (dlist = global.commands; dlist; dlist = dlist->next) {
    command = dlist->data;
    if (command->folder != folder) continue;
    if (!all && (command->level > level)) continue;

    if (cnt % 5 == 0)
      chat_print("NULL", "\n      ");
    else
      chat_print("NULL", " ");
    //      chat_print("message", "/");
    sprintf(str, "%%K[%%n%s", command->name);
    for (i1 = strlen(command->name); i1 < 10; i1++) strcat(str, " ");
    strcat(str, "%K]");
    temp_str = cparse(str);
    chat_print("message", temp_str);
    cnt++;
  }
  chat_print("NULL", "\n");
}

COMMAND(lopster_help)
{
  char *com;
  int i2;
  command_t *command;
  int all;

  com = arg(data, 0);
  if (!com || !strcasecmp(com, "all")) {
    all = (com != NULL);
    chat_print("message", _("[Available Commands]\n"));
    for (i2 = 0; i2 < NO_FOLDERS; i2++) {
      if (!all) {
	if ((i2 == F_OP) && !global.string_list[LIST_OPCHANNEL] &&
	    ((int)global.user.level < (int)L_MOD))
	  continue;
	if ((i2 == F_MOD) && ((int) global.user.level < (int)L_MOD))
	  continue;
	if ((i2 == F_ADMIN) && ((int) global.user.level < (int)L_ADMIN))
	  continue;
      }
      help_print_folder(i2, all);
    }
    chat_print("text", _("Type "));
    chat_print("message", "/help <command>");
    chat_print("text", _(" for more information!\n"));
  } else {
    command = command_is_command(com);
    if (command) {
      client_message(_("Help for command"), command->name);
      client_message(_("Syntax"), command->syntax);
      client_message(_("Description"), command->description);
      if ((int) command->level > L_ADMIN)
	if (global.user.level > L_ADMIN)
	  client_message(_("Access"), _("Elite (and you are!)"));
	else
	  client_message(_("Access"), _("Elite (not you!)"));
      else if ((int) command->level > L_MOD)
	if (global.user.level > L_MOD)
	  client_message(_("Access"), _("Admin or higher (and you are!)"));
	else
	  client_message(_("Access"), _("Admin or higher (not you!)"));
      else if ((int) command->level > L_USER)
	if (global.user.level > L_USER)
	  client_message(_("Access"),
			 _("Moderator or higher (and you are!)"));
	else
	  client_message(_("Access"), _("Moderator or higher (not you!)"));
      else if ((int) command->folder == F_OP)
	client_message(_("Access"), _("All channel operators"));
      else
	client_message(_("Access"), _("All users"));
    } else {
      client_message(_("Error"), _("This is not a command!"));
    }
  }
}

unsigned long pop_time(char** string) {
  char* pos;
  unsigned long result = 0;

  if (!string || !(*string)) return 0;
  if (!isdigit(**string)) return 0;

  pos = *string;
  while (isdigit(*pos)) {
    result *= 10;
    result += *pos -'0';
    pos++;
  }

  switch (*pos) {
  case 'm': 
    result *= 60;
    break;
  case 'h': 
    result *= 60*60;
    break;
  case 'd': 
    result *= 60*60*24;
    break;
  case ' ': 
  case 's': 
    break;
  default:
    return 0;
  }
  
  /* skip until space reached */
  while (*pos && (*pos != ' ')) pos++;
  /* skip spaces */
  while (*pos == ' ') pos++;

  if (*pos) *string = pos;
  else *string = NULL;

  return result;
}

COMMAND(lopster_ban)
{
  char *nick;
  char *message;
  char *command;
  unsigned long timeout;

  nick = arg(data, 0);
  if (!nick) {
    show_bans(NULL);
    return;
  }

  message = arg(NULL, 1);
  if (!message) {
    send_command(CMD_CLIENT_BAN, nick);
  } else {
    timeout = pop_time(&message);
    if (message) {
      if (timeout > 0)
	command = l_strdup_printf("%s \"%s\" %lu", nick, message, timeout);
      else
	command = l_strdup_printf("%s \"%s\"", nick, message);
    } else {
      client_message(_("Error"), _("Timeout specified, but no message!"));
      return;
    }
    send_command(CMD_CLIENT_BAN, command);
    l_free(command);
  }
}

COMMAND(lopster_chlevel)
{
  char *arg1;
  char *arg2;
  char *command;

  arg1 = arg(data, 0);
  if (!arg1) {
    if (in_channel())
      send_command(CMD_CLIENT_SET_CHAN_LEVEL, global.current_page->name);
    else
      client_message(_("Error"),
		     _
		     ("Please specify a channel or switch to a channel and specify a level!"));
    return;
  }
  arg2 = arg(NULL, 0);
  if (!arg2) {
    if (level2int(arg1) >= 0) {
      if (in_channel()) {
	command = l_strdup_printf("%s %s", global.current_page->name,
				  arg1);
	send_command(CMD_CLIENT_SET_CHAN_LEVEL, command);
	l_free(command);
      } else
	client_message(_("Error"),
		       _
		       ("Please specify a channel or switch to a channel!"));
    } else {
      send_command(CMD_CLIENT_SET_CHAN_LEVEL, arg1);
    }
  } else {
    command = l_strdup_printf("%s %s", arg1, arg2);
    send_command(CMD_CLIENT_SET_CHAN_LEVEL, command);
    l_free(command);
  }
}

COMMAND(lopster_clear)
{
  char *where;
  chat_page_t *page;

  page = global.current_page;
  if (page->type == P_FIX)
    return;

  where = arg(data, 0);
  if (where) {
    if (!l_strcasecmp(where, "TOP") && in_channel()) {
      chat_page_clear(page, 0, 1);
    } else if (!l_strcasecmp(where, "BOTTOM")) {
      chat_page_clear(page, 1, 0);
    } else {
      client_message(_("Error"), _("Clear what?"));
    }
  } else {
    chat_page_clear(page, 1, 1);
  }
}

COMMAND(lopster_cloak)
{
  (void)data;
  send_command(CMD_CLIENT_CLOAK, "");
  if (global.status.connection > 1) {
    whois_request(SERVER->nick, WHOIS_NONE);
  }
}

COMMAND(lopster_conn)
{
  GtkWidget *temp;

  (void)data;
  if (global.napster && (global.napster->timeout >= 0)) {
    gtk_timeout_remove(global.napster->timeout);
    global.napster->timeout = -1;
  }
  if (!global.connect_win) {
    on_connect_activate(NULL, NULL);
    temp = lookup_widget(global.connect_win, "combo_entry12");
    gtk_entry_set_text(GTK_ENTRY(temp), _("Last Server"));
  }
  on_connection_ok_clicked(NULL, NULL);
}

COMMAND(lopster_dataport)
{
  char *nick;
  char *port;
  char *command;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"),
		   _("Missing argument in command /dataport !"));
    return;
  }

  port = arg(NULL, 0);
  if (!port) {
    create_upload_port(atoi(nick), TRUE);
  } else {
    command = l_strdup_printf("%s %s", nick, port);
    send_command(CMD_CLIENT_ALTER_PORT, command);
    l_free(command);
  }
}

COMMAND(lopster_disc)
{
  (void)data;
  on_disconnect_activate(NULL, NULL);
}

COMMAND(lopster_email)
{
  char *email;

  email = arg(data, 0);
  if (!email) {
    client_message(_("Error"),
		   _("You have to specify an e-mail address!"));
    return;
  }
  send_command(CMD_CLIENT_CHANGE_EMAIL, email);
}

COMMAND(lopster_kick)
{
  char *arg1;
  char *arg2;
  char *arg3;
  char *command;

  arg1 = arg(data, 0);

  if (!arg1) {
    client_message(_("Error"),
		   _
		   ("Please specify a channel or a user or switch to a channel"));
    return;
  }

  if (user_search_full(arg1)) {
    arg2 = arg(NULL, 1);
    if (arg2)
      command =
	  l_strdup_printf("%s %s \"%s\"", global.current_page->name, arg1,
			  arg2);
    else
      command = l_strdup_printf("%s %s", global.current_page->name, arg1);
    send_command(CMD_CLIENT_KICK, command);
    l_free(command);
  } else {
    arg2 = arg(NULL, 0);
    arg3 = arg(NULL, 1);
    if (!arg2) {
      client_message(_("Error"), _("You have to specify a user"));
      return;
    }
    if (arg3)
      command = l_strdup_printf("%s %s \"%s\"", arg1, arg2, arg3);
    else
      command = l_strdup_printf("%s %s", arg1, arg2);
    send_command(CMD_CLIENT_KICK, command);
    l_free(command);
  }

}

COMMAND(lopster_kill)
{
  char *nick;
  char *reason;
  char *command;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("Who should be killed?"));
    return;
  }
  reason = arg(NULL, 1);
  if (!reason)
    send_command(CMD_CLIENT_KILL, nick);
  else {
    command = l_strdup_printf("%s \"%s\"", nick, reason);
    send_command(CMD_CLIENT_KILL, command);
    l_free(command);
  }
}

COMMAND(lopster_level)
{
  char *nick;
  char *level;
  char *command;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("Please specify a user!"));
    return;
  }
  level = arg(NULL, 0);
  if (!level) {
    if (level2int(nick) >= 0) {
      client_message(_("Error"),
		     _("If you are sure that you want change your own user level, then use: /raw 606 <yournick> <level>"));
    } else {
      client_message(_("Error"), _("Please sepcify a user level!"));
    }
  } else {
    command = l_strdup_printf("%s %s", nick, level);
    send_command(CMD_CLIENT_SETUSERLEVEL, command);
    l_free(command);
  }
}

COMMAND(lopster_chlist)
{
  (void)data;
  on_join_chatrooms_activate(NULL, NULL);
}

COMMAND(lopster_motd)
{
  (void)data;
  send_command(CMD_SERVER_MOTD, "");
}

COMMAND(lopster_muzzle)
{
  char *nick;
  char *reason;
  char *command;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("Who should be muzzled?"));
    return;
  }
  reason = arg(NULL, 1);
  if (!reason) {
    send_command(CMD_CLIENT_MUZZLE, nick);
  } else {
    command = l_strdup_printf("%s \"%s\"", nick, reason);
    send_command(CMD_CLIENT_MUZZLE, command);
    l_free(command);
  }
}

COMMAND(lopster_wallop)
{
  char *message;

  message = arg(data, 1);
  if (!message) {
    client_message(_("Error"),
		   _("Do you want to send an operator message or NOT?"));
    return;
  }
  send_wallop(message);
}

COMMAND(lopster_part)
{
  (void)data;
  chat_page_leave(global.current_page);
}

COMMAND(lopster_password)
{
  char *nick;
  char *passwd;
  char *command;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("Please specify a nickname!"));
    return;
  }
  passwd = arg(NULL, 0);
  if (!passwd) {
    send_command(CMD_CLIENT_CHANGE_PASS, nick);
    command =
	l_strdup_printf(_("Your password was changed to [%s]"), nick);
    client_message(_("Message"), command);
    l_free(command);
  } else {
    command = l_strdup_printf("%s %s", nick, passwd);
    send_command(CMD_CLIENT_ALTER_PASS, command);
    l_free(command);
  }
}

COMMAND(lopster_ping)
{
  char *nick;
  user_timestamp_t *stamp;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("Please specify a nickname!"));
    return;
  }

  stamp = timestamp_search(global.userstamp, nick);
  if (!stamp) {
    stamp = timestamp_new(nick);
    global.userstamp = g_list_append(global.userstamp, stamp);
  } else
    timestamp_touch(stamp);

  client_message(_("Message"), _("Pinging %s"), nick);
  send_command(CMD_CLIENT_PING, nick);
}

COMMAND(lopster_quit)
{
  (void)data;
  on_exit_activate(NULL, NULL);
}

COMMAND(lopster_speed)
{
  char *nick;
  char *speed;
  char *command;

  nick = arg(data, 0);

  if (!nick) {
    client_message(_("Error"), _("Please specify a nickname!"));
    return;
  }
  speed = arg(NULL, 0);
  if (!speed) {
    if (isdigit(*nick)) {
      send_command(CMD_CLIENT_CHANGE_SPEED, nick);
      client_message(_("LineSpeed"),
		     _("Your linespeed was changed to %s!"), nick);
      global.user.linespeed = atoi(nick);
    } else
      speed_request(nick, SPEED_SHOW);
  } else {
    command = l_strdup_printf("%s %s", nick, speed);
    send_command(CMD_CLIENT_ALTER_SPEED, command);
    l_free(command);
  }
}

COMMAND(lopster_stats)
{
  (void)data;
  send_command(CMD_SERVER_STATS, "");
}

COMMAND(lopster_topic)
{
  char *channel;
  char *topic;
  char *command;

  channel = arg(data, 0);
  if (!channel) {
    if (in_channel())
      send_command(CMD_SERVER_TOPIC, global.current_page->name);
    else
      client_message(_("Error"), _("You have to specify a channel!"));
    return;
  }

  topic = arg(NULL, 1);
  if (!topic) {
    if (!strcmp(channel, "."))
      send_command(CMD_SERVER_TOPIC, global.current_page->name);
    else
      send_command(CMD_SERVER_TOPIC, channel);
  } else {
    if (global.options.parse_color)
      topic = cparse(topic);
    if (!strcmp(channel, "."))
      command = l_strdup_printf("%s %s", global.current_page->name, topic);
    else
      command = l_strdup_printf("%s %s", channel, topic);
    send_command(CMD_SERVER_TOPIC, command);
    l_free(command);
  }
}

COMMAND(lopster_unban)
{
  char *nick;
  char *message;
  char *command;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("Who should be unbanned?"));
    return;
  }
  message = arg(NULL, 1);
  if (!message) {
    send_command(CMD_CLIENT_UNBAN, nick);
  } else {
    command = l_strdup_printf("%s \"%s\"", nick, message);
    send_command(CMD_CLIENT_UNBAN, command);
    l_free(command);
  }
}

COMMAND(lopster_unmuzzle)
{
  char *nick;
  char *reason;
  char *command;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("Who should be unmuzzled?"));
    return;
  }
  reason = arg(NULL, 1);
  if (!reason) {
    send_command(CMD_CLIENT_UNMUZZLE, nick);
  } else {
    command = l_strdup_printf("%s \"%s\"", nick, reason);
    send_command(CMD_CLIENT_UNMUZZLE, command);
    l_free(command);
  }
}

COMMAND(lopster_version)
{
  (void)data;
  send_command(CMD_CLIENT_SERVER_VERSION, "");
}

COMMAND(lopster_scheme)
{
  char *scheme;
  char *command;
  scheme_t *scheme1;

  scheme = arg(data, 1);
  if (!scheme) {
    command =
	l_strdup_printf(_("Current scheme is [%s]"), global.scheme->name);
    client_message(_("Message"), command);
    l_free(command);
  } else {
    scheme1 = scheme_load(scheme);
    if (scheme1) {
      if (global.scheme)
	scheme_destroy(global.scheme);
      global.scheme = scheme1;
      client_message(_("Message"), _("Scheme [%s] loaded!"),
		     global.scheme->name);
    } else {
      client_message(_("Message"), _("Loading scheme [%s] failed!"),
		     scheme);
    }
  }
}

COMMAND(lopster_alias)
{
  char *alias;
  char *command;

  alias = arg(data, 0);
  if (!alias) {
    client_message(_("Error"), _("Please give an alias name!"));
    return;
  }
  command = arg(NULL, 0);
  if (!command) {
    client_message(_("Error"), _("Please give an command name!"));
    return;
  }
  command_make_alias(command, alias);
}

COMMAND(lopster_afk)
{
  char *message;

  message = arg(data, 1);
  if (!message)
    set_afk(NULL);
  else
    set_afk(message);
}

COMMAND(lopster_public)
{
  char *message;

  message = arg(data, 1);
  if (!message) {
    client_message(_("Error"), _("Want to send _nothing_?"));
    return;
  }
  if (!in_channel()) {
    client_message(_("Message"), _("/public not supported in this page"));
    return;
  }
  send_public(global.current_page->name, message);
}

COMMAND(lopster_reply)
{
  char *message;

  if (global.reply == NULL) {
    client_message(_("Message"), _("Noone to reply!"));
    return;
  }
  message = arg(data, 1);
  if (!message) {
    client_message(_("Error"), _("Want to reply _nothing_?"));
    return;
  }
  send_private(global.reply, message, 1);
}

COMMAND(lopster_private)
{
  char *nick;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("Yes, private mode, but to whom?"));
    return;
  }
  create_private_page(nick);
}

COMMAND(lopster_ignore)
{
  char *nick;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("Missing nickname!"));
  } else {
    send_command(CMD_CLIENT_IGNORE_USER, nick);
  }
}

COMMAND(lopster_chclear)
{
  char *channel;

  channel = arg(data, 0);
  if (!channel) {
    if (!in_channel()) {
      client_message(_("Message"),
		     _
		     ("/chclear is not supported without argument in this page!"));
    } else {
      send_command(CMD_CLIENT_CLEAR_CHANNEL, global.current_page->name);
    }
  } else {
    send_command(CMD_CLIENT_CLEAR_CHANNEL, channel);
  }
}

COMMAND(lopster_unignore)
{
  char *nick;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("Who should be unignored?"));
  } else {
    send_command(CMD_CLIENT_UNIGNORE_USER, nick);
  }
}

COMMAND(lopster_chban)
{
  char *nick;
  char *channel;
  char *reason;
  char *command;
  GtkCList *clist;
  int user;

  channel = arg(data, 0);

  if (!channel) {
    if (in_channel()) {
      show_bans(global.current_page->name);
    } else {
      client_message(_("Error"),
		     _
		     ("Trying to ban a user from a channel? Better type /help chban !"));
    }
    return;
  }

  if (in_channel()) {
    clist = GTK_CLIST(global.current_page->online);
    user = search_user_in_list(clist, channel);
    if (user >= 0)
      nick = channel;
    else
      nick = arg(NULL, 0);
  } else {
    nick = arg(NULL, 0);
    user = -1;
  }

  if (!nick) {
    client_message(_("Error"), _("Missing Nickname!"));
    return;
  }

  reason = arg(NULL, 1);

  if (user >= 0) {		// user found
    if (reason)
      command = l_strdup_printf("%s %s \"%s\"", global.current_page->name,
				nick, reason);
    else
      command = l_strdup_printf("%s %s", global.current_page->name, nick);
  } else {
    if (reason)
      command = l_strdup_printf("%s %s \"%s\"", channel, nick, reason);
    else
      command = l_strdup_printf("%s %s", channel, nick);
  }

  send_command(CMD_CLIENT_CHANNEL_BAN, command);
  l_free(command);
}

COMMAND(lopster_chunban)
{
  char *nick;
  char *channel;
  char *reason;
  char *command;

  channel = arg(data, 0);

  if (!channel) {
    client_message(_("Error"),
		   _
		   ("Trying to unban a user from a channel? Better type /help chunban !"));
    return;
  }

  nick = arg(NULL, 0);
  if (!nick) {
    client_message(_("Error"), _("Missing Nickname!"));
    return;
  }

  reason = arg(NULL, 1);

  if (reason)
    if (!strcmp(channel, "."))
      command = l_strdup_printf("%s %s \"%s\"", global.current_page->name,
				nick, reason);
    else
      command = l_strdup_printf("%s %s \"%s\"", channel, nick, reason);
  else if (!strcmp(channel, "."))
    command = l_strdup_printf("%s %s", global.current_page->name, nick);
  else
    command = l_strdup_printf("%s %s", channel, nick);

  send_command(CMD_CLIENT_CHANNEL_UNBAN, command);
  l_free(command);
}

COMMAND(lopster_nuke)
{
  char *nick;

  nick = arg(data, 0);

  if (!nick) {
    client_message(_("Error"), _("Specify a user!"));
    return;
  }
  send_command(CMD_CLIENT_NUKE, nick);
}

COMMAND(lopster_unnuke)
{
  char *nick;

  nick = arg(data, 0);

  if (!nick) {
    client_message(_("Error"), _("Specify a user!"));
    return;
  }
  send_command(CMD_CLIENT_UNNUKE, nick);
}

COMMAND(lopster_global)
{
  char *message;

  message = arg(data, 1);
  if (!message) {
    client_message(_("Error"), _("Want to send _nothing_?"));
    return;
  }
  send_global(message);
}

#define INTERN_MICRO_VERSION   0

COMMAND(lopster_cversion)
{
  char* t2 = NULL;
  struct utsname uts;

  (void)data;
  
  if (uname(&uts) < 0) {
    if (INTERN_MICRO_VERSION)
      t2 = l_strdup_printf("Lopster %s.%d", VERSION, INTERN_MICRO_VERSION);
    else
      t2 = l_strdup_printf("Lopster %s", VERSION);
  } else {
    if (INTERN_MICRO_VERSION)
      t2 = l_strdup_printf("Lopster %s.%d / %s %s", VERSION, INTERN_MICRO_VERSION,
			   uts.sysname, uts.release);
    else
      t2 = l_strdup_printf("Lopster %s / %s %s", VERSION,
			   uts.sysname, uts.release);
  }

  if (global.current_page->type == P_PUBLIC) {
    send_public(global.current_page->name, t2);
  } else if (global.current_page->type == P_PRIVATE) {
    send_private(global.current_page->name, t2, 1);
  } else if (!strcmp("Wallop", global.current_page->name)) {
    send_wallop(t2);
  } else if (!strcmp("Global", global.current_page->name)) {
    send_wallop(t2);
  } else {
    client_message(_("Message"), t2);
  }
  l_free(t2);
}

COMMAND(lopster_echo)
{
  char *message;
  char *text2;

  message = arg(data, 1);
  if (!message) {
    client_message(_("Error"), _("Echo what?"));
  } else {
    if (global.options.parse_color)
      text2 = cparse(message);
    else
      text2 = message;
    chat_print_ln("text", text2);
  }
}

COMMAND(lopster_chlimit)
{
  char *channel;
  char *limit;
  char *command;

  channel = arg(data, 0);
  if (!channel) {
    if (in_channel())
      send_command(CMD_CLIENT_CHANNEL_LIMIT, global.current_page->name);
    else
      client_message(_("Error"),
		     _("Please specify a channel or switch to a channel"));
    return;
  }
  limit = arg(NULL, 0);
  if (!limit) {
    if (isdigit(*channel)) {
      if (in_channel()) {
	command = l_strdup_printf("%s %s",
				  global.current_page->name, channel);
	send_command(CMD_CLIENT_CHANNEL_LIMIT, command);
	l_free(command);
      } else
	client_message(_("Error"),
		       _
		       ("Please specify a channel or switch to a channel"));
    } else {
      send_command(CMD_CLIENT_CHANNEL_LIMIT, channel);
    }
  } else {
    command = l_strdup_printf("%s %s", channel, limit);
    send_command(CMD_CLIENT_CHANNEL_LIMIT, command);
    l_free(command);
  }
}

COMMAND(lopster_sping)
{
  char *server;
  user_timestamp_t *stamp;

  server = arg(data, 0);
  if (!server) {
    client_message(_("Error"), _("Pinging, yes, but what server?"));
    return;
  }

  stamp = timestamp_search(global.userstamp, server);
  if (!stamp) {
    stamp = timestamp_new(server);
    global.userstamp = g_list_append(global.userstamp, stamp);
  } else
    timestamp_touch(stamp);

  client_message(_("Message"), _("Pinging server %s"), server);
  send_command(CMD_CLIENT_PING_SERVER, server);
}

COMMAND(lopster_friend)
{
  char *nick;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("You have to specify a user!"));
    return;
  }
  toggle_friend(nick);
}

COMMAND(lopster_enemy)
{
  char *nick;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("You have to specify a user!"));
    return;
  }
  toggle_enemy(nick);
}

COMMAND(lopster_hotadd)
{
  char *nick;
  char *des;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("You have to specify a user!"));
    return;
  }

  des = arg(NULL, 1);

  hotlist_add(nick, des, 1);
}

COMMAND(lopster_hotrem)
{
  char *nick;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("You have to specify a user!"));
    return;
  }
  hotlist_remove(nick, 1);
}

COMMAND(lopster_clientinfo)
{
  char *info;

  info = arg(data, 1);
  if (info) {
    if (global.clientinfo)
      l_free(global.clientinfo);
    if (!l_strcasecmp(info, "NONE"))
      global.clientinfo = NULL;
    else
      global.clientinfo = l_strdup(info);
  }
  if (global.clientinfo) {
    client_message(_("Client Info"), "%s", global.clientinfo);
  } else {
    client_message(_("Client Info"), _("No alternative client info"));
  }
}

COMMAND(lopster_guser)
{
  char *message;

  message = arg(data, 1);
  if (global.user_win)
    gtk_widget_show(global.user_win);
  if (!message) {
    show_global_users("", TRUE);
  } else {
    show_global_users(message, TRUE);
  }
}

COMMAND(lopster_sconfig)
{
  char *var;
  char *val;
  char *command;

  var = arg(data, 0);
  if (!var) {
    send_command(CMD_CLIENT_SERVER_CONFIG, "");
    return;
  }
  val = arg(NULL, 0);
  if (!val) {
    send_command(CMD_CLIENT_SERVER_CONFIG, var);
  } else {
    command = l_strdup_printf("%s %s", var, val);
    send_command(CMD_CLIENT_SERVER_CONFIG, command);
    l_free(command);
  }
}

COMMAND(lopster_clones)
{
  GList *dlist;
  GList *dlist2;
  char *pos;
  int cnt, cnt2, cnt3;
  clone_t *clone;
  char t[1024];
  char *which;
  int number;

  which = arg(data, 0);
  if (!which) {
    client_message(_("Error"), _("Print what clones?"));
    return;
  }
  pos = arg(NULL, 0);
  if (pos)
    number = atoi(pos);
  else
    number = 2;

  if (!strcmp(which, "all")) {
    cnt = cnt2 = 0;
    sprintf(t, _("Printing IPs with at least %d nicks"), number);
    client_message(_("Clones"), t);
    for (dlist = global.clones; dlist; dlist = dlist->next) {
      clone = dlist->data;
      if (g_list_length(clone->nicks) < (unsigned)number)
	continue;
      sprintf(t, _("IP with %d clones"), g_list_length(clone->nicks));
      client_message(t, clone->ip);
      for (dlist2 = clone->nicks; dlist2; dlist2 = dlist2->next) {
	pos = (char *) (dlist2->data);
	client_message("--->", "<%s>", pos);
	cnt++;
      }
      cnt2++;
    }
    sprintf(t, _("[%d clones found] (%d IPs)"), cnt, cnt2);
    client_message("--Sum-->", t);
  } else if (!strcmp(which, ".")) {
    if (!in_channel()) {
      client_message(_("Error"), _("Have to switch to a channel"));
      return;
    }
    sprintf(t, _("Printing IPs with at least %d nicks"), number);
    client_message(_("Clones"), t);
    cnt = cnt2 = cnt3 = 0;
    for (dlist = global.clones; dlist; dlist = dlist->next) {
      clone = (clone_t *) (dlist->data);
      if (g_list_length(clone->nicks) < (unsigned)number)
	continue;
      for (dlist2 = clone->nicks; dlist2; dlist2 = dlist2->next) {
	pos = (char *) (dlist2->data);
	if (user_search_full(pos))
	  break;
      }
      if (dlist2) {
	sprintf(t, _("IP with %d clones"), g_list_length(clone->nicks));
	client_message(t, clone->ip);
	cnt++;
      } else
	continue;
      for (dlist2 = clone->nicks; dlist2; dlist2 = dlist2->next) {
	pos = (char *) (dlist2->data);
	if (user_search_full(pos)) {
	  client_message("--->", "(C) <%s>", pos);
	  cnt3++;
	} else {
	  client_message("--->", "( ) <%s>", pos);
	}
	cnt2++;
      }
    }
    sprintf(t, _("[%d clones found in channel %s] (%d IPs / %d clones)"),
	    cnt3, global.current_page->name, cnt, cnt2);
    client_message("--Sum-->", t);
  } else {
    sprintf(t, _("Printing nicks for IP %s"), which);
    client_message(_("Clones"), t);
    for (dlist = global.clones; dlist; dlist = dlist->next) {
      clone = (clone_t *) (dlist->data);
      if (strcmp(clone->ip, which))
	continue;
      sprintf(t, _("IP with %d clones"), g_list_length(clone->nicks));
      client_message(t, clone->ip);
      for (dlist2 = clone->nicks; dlist2; dlist2 = dlist2->next) {
	pos = (char *) (dlist2->data);
	client_message("--->", "<%s>", pos);
      }
      break;
    }
  }
}

COMMAND(lopster_op)
{
  char *nick;
  char *channel;
  char *command;

  channel = arg(data, 0);
  if (!channel) {
    if (in_channel())
      send_command(CMD_CLIENT_OP, global.current_page->name);
    else
      client_message(_("Error"),
		     _
		     ("Please specify a channel or a user switch to a channel"));
    return;
  }
  nick = arg(NULL, 0);
  if (!nick) {
    if (user_search_full(channel)) {
      command = l_strdup_printf("%s %s", global.current_page->name,
				channel);
      send_command(CMD_CLIENT_OP, command);
      l_free(command);
    } else if (!strcmp(channel, "."))
      send_command(CMD_CLIENT_OP, global.current_page->name);
    else
      send_command(CMD_CLIENT_OP, channel);
  } else {
    if (!strcmp(channel, "."))
      command = l_strdup_printf("%s %s", global.current_page->name, nick);
    else
      command = l_strdup_printf("%s %s", channel, nick);
    send_command(CMD_CLIENT_OP, command);
    l_free(command);
  }
}

COMMAND(lopster_deop)
{
  char *nick;
  char *channel;
  char *command;

  channel = arg(data, 0);
  if (!channel) {
    client_message(_("Error"),
		   _
		   ("Please specify a channel or a user switch to a channel"));
    return;
  }
  nick = arg(NULL, 0);
  if (!nick) {
    if (user_search_full(channel)) {
      command = l_strdup_printf("%s %s", global.current_page->name,
				channel);
      send_command(CMD_CLIENT_DEOP, command);
      l_free(command);
    } else
      client_message(_("Error"),
		     _
		     ("Please specify a channel or a user switch to a channel"));
  } else {
    if (!strcmp(channel, "."))
      command = l_strdup_printf("%s %s", global.current_page->name, nick);
    else
      command = l_strdup_printf("%s %s", channel, nick);
    send_command(CMD_CLIENT_DEOP, command);
    l_free(command);
  }
}

COMMAND(lopster_chvoice)
{
  char *nick;
  char *channel;
  char *command;

  channel = arg(data, 0);
  if (!channel) {
    if (in_channel())
      send_command(CMD_CLIENT_CHANNEL_VOICE, global.current_page->name);
    else
      client_message(_("Error"),
		     _
		     ("Please specify a channel or a user switch to a channel"));
    return;
  }
  nick = arg(NULL, 0);
  if (!nick) {
    if (user_search_full(channel)) {
      command = l_strdup_printf("%s %s", global.current_page->name,
				channel);
      send_command(CMD_CLIENT_CHANNEL_VOICE, command);
      l_free(command);
    } else if (!strcmp(channel, "."))
      send_command(CMD_CLIENT_CHANNEL_VOICE, global.current_page->name);
    else
      send_command(CMD_CLIENT_CHANNEL_VOICE, channel);
  } else {
    if (!strcmp(channel, "."))
      command = l_strdup_printf("%s %s", global.current_page->name, nick);
    else
      command = l_strdup_printf("%s %s", channel, nick);
    send_command(CMD_CLIENT_CHANNEL_VOICE, command);
    l_free(command);
  }
}

COMMAND(lopster_chunvoice)
{
  char *nick;
  char *channel;
  char *command;

  channel = arg(data, 0);
  if (!channel) {
    client_message(_("Error"),
		   _
		   ("Please specify a channel or a user switch to a channel"));
    return;
  }
  nick = arg(NULL, 0);
  if (!nick) {
    if (user_search_full(channel)) {
      command = l_strdup_printf("%s %s", global.current_page->name,
				channel);
      send_command(CMD_CLIENT_CHANNEL_UNVOICE, command);
      l_free(command);
    } else
      client_message(_("Error"),
		     _
		     ("Please specify a channel or a user switch to a channel"));
  } else {
    if (!strcmp(channel, "."))
      command = l_strdup_printf("%s %s", global.current_page->name, nick);
    else
      command = l_strdup_printf("%s %s", channel, nick);
    send_command(CMD_CLIENT_CHANNEL_UNVOICE, command);
    l_free(command);
  }
}

COMMAND(lopster_links)
{
  char *action = NULL;

  if (data)
    action = arg(data, 0);

  if (!action) {
    if (!global.links || !data) {
      client_message(_("Message"), _("Evaluating linked server list"));
      if (!global.links) {
	free_links(global.links);
	global.links = NULL;
      }
      send_command(CMD_SERVER_LINKS, "");
    } else {
      print_links(global.links, 0);
    }
  } else if (!l_strcasecmp(action, "GET")) {
    client_message(_("Message"), _("Evaluating linked server list"));
    if (!global.links) {
      free_links(global.links);
      global.links = NULL;
    }
    send_command(CMD_SERVER_LINKS, "");
  } else if (!l_strcasecmp(action, "PRINT")) {
    print_links(global.links, 0);
  } else {
    client_message(_("Error"), _("Unknown action for /links"));
  }
}

COMMAND(lopster_invite)
{
  char *nick;
  char *channel;
  char *command;

  channel = arg(data, 0);
  if (!channel) {
    client_message(_("Error"), _("You have to specify at least a user!"));
    return;
  }

  nick = arg(NULL, 0);
  if (!nick) {
    if (in_channel()) {
      command =
	  l_strdup_printf("%s %s", global.current_page->name, channel);
      send_command(CMD_CLIENT_CHANNEL_INVITE, command);
      l_free(command);
    } else
      client_message(_("Error"),
		     _
		     ("Please specify a channel or switch to a channel!"));
  } else {
    if (!strcmp(channel, "."))
      command = l_strdup_printf("%s %s", global.current_page->name, nick);
    else
      command = l_strdup_printf("%s %s", channel, nick);
    send_command(CMD_CLIENT_CHANNEL_INVITE, command);
    l_free(command);
  }
}

COMMAND(lopster_chmuzzle)
{
  char *command;
  char *nick;
  char *reason;

  nick = arg(data, 0);
  if (!nick) {
    if (in_channel())
      send_command(CMD_CLIENT_CHANNEL_MUZZLE, global.current_page->name);
    else
      client_message(_("Error"),
		     _("Please specify a user switch to a channel"));
    return;
  }
  if (!in_channel()) {
    client_message(_("Error"),
		   _("Please specify a user switch to a channel"));
    return;
  }

  reason = arg(NULL, 1);
  if (!reason) {
    command = l_strdup_printf("%s %s", global.current_page->name, nick);
    send_command(CMD_CLIENT_CHANNEL_MUZZLE, command);
  } else {
    command =
	l_strdup_printf("%s %s \"%s\"", global.current_page->name, nick,
			reason);
    send_command(CMD_CLIENT_CHANNEL_MUZZLE, command);
  }
  l_free(command);
}

COMMAND(lopster_chunmuzzle)
{
  char *command;
  char *arg1;
  char *arg2;
  char *arg3;

  arg1 = arg(data, 0);
  arg2 = arg(NULL, 0);
  arg3 = arg(NULL, 1);

  if (!arg1) {
    client_message(_("Error"),
		   _
		   ("Please specify a channel or a user switch to a channel"));
    return;
  }
  if (!arg2) {
    command = l_strdup_printf("%s %s", global.current_page->name, arg1);
    send_command(CMD_CLIENT_CHANNEL_UNMUZZLE, command);
    l_free(command);
    return;
  }
  if (!arg3) {
    if (user_search_full(arg1)) {
      command = l_strdup_printf("%s %s \"%s\"",
				global.current_page->name, arg1, arg2);
    } else if (!strcmp(arg1, ".")) {
      command = l_strdup_printf("%s %s", global.current_page->name, arg2);
    } else {
      command = l_strdup_printf("%s %s", arg1, arg2);
    }
    send_command(CMD_CLIENT_CHANNEL_UNMUZZLE, command);
    l_free(command);
    return;
  } else {
    if (!strcmp(arg1, "."))
      command = l_strdup_printf("%s %s \"%s\"", global.current_page->name,
				arg2, arg3);
    else
      command = l_strdup_printf("%s %s \"%s\"", arg1, arg2, arg3);
    send_command(CMD_CLIENT_CHANNEL_UNMUZZLE, command);
    l_free(command);
  }
}


COMMAND(lopster_usermode)
{
  char *message;

  message = arg(data, 1);
  if (!message) {
    send_command(CMD_CLIENT_USER_MODE, "");
  } else {
    send_command(CMD_CLIENT_USER_MODE, message);
  }
}

COMMAND(lopster_sstats)
{
  char *server;

  if (data) server = arg(data, 0);
  else server = NULL;

  if (!server) {
    client_message(NULL, _("..requesting stats for local server"));
    send_command(CMD_CLIENT_USAGE_STATS, "");
  } else {
    client_message(NULL, _("..requesting stats for server [%s]"), server);
    send_command(CMD_CLIENT_USAGE_STATS, server);
  }
}

COMMAND(lopster_chmode)
{
  char *channel;
  char *mode;
  char *command;

  channel = arg(data, 0);
  if (!channel) {
    if (in_channel())
      send_command(CMD_CLIENT_CHANNEL_MODE, global.current_page->name);
    else
      client_message(_("Error"),
		     _
		     ("Please specify a channel or switch to a channel!"));
    return;
  }
  mode = arg(NULL, 0);
  if (!mode) {
    if ((*channel == '+') || (*channel == '-')) {
      if (in_channel()) {
	command = l_strdup_printf("%s %s", global.current_page->name,
				  channel);
	send_command(CMD_CLIENT_CHANNEL_MODE, command);
	l_free(command);
      } else
	client_message(_("Error"),
		       _
		       ("Please specify a channel or switch to a channel!"));
    } else {
      if (!strcmp(channel, "."))
	send_command(CMD_CLIENT_CHANNEL_MODE, global.current_page->name);
      else
	send_command(CMD_CLIENT_CHANNEL_MODE, channel);
    }
  } else {
    if (!strcmp(channel, "."))
      command = l_strdup_printf("%s %s", global.current_page->name, mode);
    else
      command = l_strdup_printf("%s %s", channel, mode);
    send_command(CMD_CLIENT_CHANNEL_MODE, command);
    l_free(command);
  }
}

COMMAND(lopster_chwallop)
{
  char *message;

  message = arg(data, 1);
  if (!message) {
    client_message(_("Error"), _("You want say something, but what?"));
    return;
  }
  if (!in_channel()) {
    client_message(_("Error"), _("Switch to a channel page!"));
    return;
  }
  send_chwallop(global.current_page->name, message);
}

COMMAND(lopster_killghost)
{
  char *ghost;
  char *passwd;
  char *command;

  ghost = arg(data, 0);
  if (!ghost) {
    client_message(_("Error"), _("You have to specify the ghost!"));
    return;
  }
  passwd = arg(NULL, 0);
  if (!passwd) {
    command = l_strdup_printf("ghost %s %s", ghost, global.user.password);
  } else {
    command = l_strdup_printf("ghost %s %s", ghost, passwd);
  }

  send_private("NickServ", command, 1);
  l_free(command);
}

COMMAND(lopster_locate)
{
  char *nick;

  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("You have to specify a user!"));
    return;
  }
  send_command(CMD_CLIENT_WHICH_SERVER, nick);
}

COMMAND(lopster_clients)
{
  (void)data;
  show_clients();
}

// /exec by R00t
////////////////////////////////

COMMAND(lopster_exec)
{
  exec_command(data);
}

COMMAND(lopster_whowas)
{
  char *nick;

  if (!opennap_version(0, 38)) {
    client_message(_("Error"),
		   _
		   ("Not supported on this server (need to be opennap 0.38)"));
    return;
  }
  nick = arg(data, 0);
  if (!nick) {
    client_message(_("Error"), _("Whowas who?"));
    return;
  }
  send_command(CMD_CLIENT_WHO_WAS, nick);
}

COMMAND(lopster_rcolors)
{
  int i1;
  char str[1024];
  int pos = 0;
  char *codes;

  codes = arg(data, 0);

  if (!codes) {
  } else if (!l_strcasecmp(codes, "-reset")) {
    l_free(global.rainbow_colors);
    global.rainbow_colors = l_strdup("RrpPBbcCgGYy");
  } else {
    for (i1 = 0; (i1 < (int)strlen(data)) && (i1 < 1023); i1++) {
      if (strchr("krgybmpcwKRGYBMPCWn", data[i1]))
	str[pos++] = data[i1];
    }
    str[pos] = 0;
    if (pos == 0)
      global.rainbow_colors = l_strdup("RrpPBbcCgGYy");
    else
    global.rainbow_colors = l_strdup(str);
  }
  client_message("Rainbow colors", global.rainbow_colors);
}

COMMAND(lopster_rainbow)
{
  char *size;
  int isize;
  char *sep = NULL;
  char *message;
  char command[2048];
  int pos;
  int cpos;
  int cnt;
  int i1;

  if (!in_channel() && !in_private()) {
    client_message(_("Error"), _("Not allowed in this page!"));
    return;
  }
  size = arg(data, 0);
  if (!size) {
    client_message(_("Error"), _("Wrong format in command /rainbow!"));
    return;
  }
  if (!isdigit(*size)) {
    sep = size;
    size = arg(NULL, 0);
  }
  if (!size) {
    client_message(_("Error"), _("Wrong format in command /rainbow!"));
    return;
  }
  message = arg(NULL, 1);
  if (!message) {
    client_message(_("Error"), _("Wrong format in command /rainbow!"));
    return;
  }

  isize = atoi(size);
  if (isize < 1)
    isize = 1;
  pos = cnt = 0;
  cpos = strlen(global.rainbow_colors) - 1;
  for (i1 = 0; i1 < (int)strlen(message); i1++) {
    if (message[i1] == '\n') {
      command[pos++] = message[i1];
      command[pos++] = '%';
      command[pos++] = global.rainbow_colors[cpos];
    } else {
      if ((cnt % isize) == 0) {
	if (sep) {
	  int temp = 0;
	  command[pos++] = '%';
	  command[pos++] = 'n';
	  while (sep[temp] != 0) {
	    command[pos++] = sep[temp];
	    temp++;
	  }
	}

	cpos++;
	if (cpos == (int)strlen(global.rainbow_colors))
	   cpos = 0;
	command[pos++] = '%';
	command[pos++] = global.rainbow_colors[cpos];
      }
      cnt++;
      command[pos++] = message[i1];
    }
  }
  command[pos++] = 0;

  if (global.current_page->type == P_PUBLIC) {
    lopster_public(command);
  } else if (global.current_page->type == P_PRIVATE) {
    send_private(global.current_page->name, command, 1);
  } else if (!strcmp("Wallop", global.current_page->name)) {
    send_wallop(command);
  } else if (!strcmp("Global", global.current_page->name)) {
    send_global(command);
  }
}

COMMAND(lopster_browse)
{
  char *nick;
  GtkWidget *temp;
  char *max;

  nick = arg(data, 0);
  max = arg(NULL, 0);

  if (!nick) {
    client_message(_("Error"), _("Browse who?"));
    return;
  }
  temp = lookup_widget(global.win, "entry79");
  gtk_entry_set_text(GTK_ENTRY(temp), nick);
  if (max) {
    temp = lookup_widget(global.win, "checkbutton36");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(temp), TRUE);
    temp = lookup_widget(global.win, "spinbutton33");
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp), atoi(max));
  }
  on_entry79_activate(NULL, NULL);
}

COMMAND(lopster_send)
{
  char *nick;
  char *file;

  nick = arg(data, 0);
  file = arg(NULL, 1);

  if (!nick) {
    client_message(_("Error"), _("Missing nickname!"));
    return;
  }
  if (file)
    dcc_send_file(nick, file);
  else
    dcc_select_file(nick);
}

COMMAND(lopster_slink)
{
  char *server;
  char *remote;
  char *command;

  server = arg(data, 0);
  remote = arg(NULL, 0);

  if (!server) {
    client_message(_("Error"), _("Link a server or what?"));
    return;
  }

  if (remote) {
    command = l_strdup_printf("%s %s", server, remote);
    send_command(CMD_CLIENT_CONNECT, command);
    l_free(command);
  } else {
    send_command(CMD_CLIENT_CONNECT, data);
  }
}

COMMAND(lopster_sdelink)
{
  char *server;
  char *reason;
  char *command;

  server = arg(data, 0);
  reason = arg(NULL, 1);
  if (!server) {
    client_message(_("Error"), _("Delink a server or what?"));
    return;
  }
  if (reason)
    command = l_strdup_printf("%s \"%s\"", server, reason);
  else
    command = l_strdup_printf("%s", server);

  send_command(CMD_CLIENT_DISCONNECT, command);
  l_free(command);
}

COMMAND(lopster_skill)
{
  char *server;
  char *reason;
  char *command;

  server = arg(data, 0);
  reason = arg(NULL, 1);
  if (!server) {
    client_message(_("Error"), _("Kill what server?"));
    return;
  }

  if (reason)
    command = l_strdup_printf("%s \"%s\"", server, reason);
  else
    command = l_strdup_printf("%s", server);

  send_command(CMD_CLIENT_KILL_SERVER, command);
  l_free(command);
}

COMMAND(lopster_sremove)
{
  char *server;
  char *reason;
  char *command;

  server = arg(data, 0);
  reason = arg(NULL, 1);
  if (!server) {
    client_message(_("Error"), _("Remove what server?"));
    return;
  }

  if (reason)
    command = l_strdup_printf("%s \"%s\"", server, reason);
  else
    command = l_strdup_printf("%s", server);

  send_command(CMD_CLIENT_REMOVE_SERVER, command);
  l_free(command);
}

COMMAND(lopster_sreload)
{
  if (data)
    send_command(CMD_CLIENT_REHASH, data);
  else
    send_command(CMD_CLIENT_REHASH, "");
}

COMMAND(lopster_masskill)
{
  char *ip;
  char *reason;
  char *command;

  ip = arg(data, 0);
  reason = arg(NULL, 1);
  if (!ip) {
    client_message(_("Error"), _("Kill what ip?"));
    return;
  }
  if (reason) {
    command = l_strdup_printf("%s \"%s\"", ip, reason);
    send_command(CMD_CLIENT_MASS_KILL, command);
    l_free(command);
  } else {
    send_command(CMD_CLIENT_MASS_KILL, ip);
  }
}

COMMAND(lopster_shistogram)
{
  (void)data;
  client_message(NULL, _("Requesting server histogram..."));
  send_command(CMD_CLIENT_HISTOGRAM, "");
}

COMMAND(lopster_register)
{
  char *user;
  char *passwd;
  char *email;
  char *level;
  char *command;

  if (global.status.connection < 2)
    return;
  user = arg(data, 0);
  passwd = arg(NULL, 0);
  email = arg(NULL, 0);
  level = arg(NULL, 0);

  if (!user) {
    command = l_strdup_printf("%s %s anon@anon",
			      SERVER->nick, SERVER->passwd);
    send_command(CMD_CLIENT_REGISTER_USER, command);
    l_free(command);
    return;
  }
  if (!passwd || !email) {
    client_message(_("Error"), _("Specify password and E-mail."));
    return;
  }
  if (level)
    command = l_strdup_printf("%s %s %s %s", user, passwd, email, level);
  else
    command = l_strdup_printf("%s %s %s", user, passwd, email);
  send_command(CMD_CLIENT_REGISTER_USER, command);
  l_free(command);
}

COMMAND(lopster_redirect)
{
  char *user;
  char *server;
  char *port;
  char *command;

  user = arg(data, 0);
  server = arg(NULL, 0);
  port = arg(NULL, 0);

  if (!user || !server || !port) {
    client_message(_("Error"), _("Wrong number of arguments."));
    return;
  }
  command = l_strdup_printf("%s %s %s", user, server, port);
  send_command(CMD_CLIENT_REDIRECT, command);
  l_free(command);
}

COMMAND(lopster_cycle)
{
  char *user;
  char *server;
  char *command;

  user = arg(data, 0);
  server = arg(NULL, 0);

  if (!user || !server) {
    client_message(_("Error"), _("Wrong number of arguments."));
    return;
  }
  command = l_strdup_printf("%s %s", user, server);
  send_command(CMD_CLIENT_CYCLE, command);
  l_free(command);
}

COMMAND(lopster_limit)
{
  char *tag;
  char *ip;
  int add;

  tag = arg(data, 0);
  if (!tag || !l_strcasecmp(tag, "list")) {
    client_message(_("Message"), _("Requesting limits"));
    add = 2;
  } else if (!l_strcasecmp(tag, "add")) {
    add = 0;
  } else if (!l_strcasecmp(tag, "remove")) {
    add = 1;
  } else {
    client_message(_("Error"), _("Unknown action"));
    return;
  }

  ip = arg(NULL, 1);
  if (!ip && (add != 2)) {
    client_message(_("Error"), _("No ip specified"));
    return;
  }

  if (ip)
    send_command(CMD_CLIENT_CLASS_ADD + add, ip);
  else
    send_command(CMD_CLIENT_CLASS_ADD + add, "");
}

COMMAND(lopster_iline)
{
  char *tag;
  char *ip;
  int add;

  tag = arg(data, 0);
  if (!tag || !l_strcasecmp(tag, "list")) {
    client_message(_("Message"), _("Requesting ilines"));
    add = 2;
  } else if (!l_strcasecmp(tag, "add")) {
    add = 0;
  } else if (!l_strcasecmp(tag, "remove")) {
    add = 1;
  } else {
    client_message(_("Error"), _("Unknown action"));
    return;
  }

  ip = arg(NULL, 1);
  if (!ip && (add != 2)) {
    client_message(_("Error"), _("No ip specified"));
    return;
  }

  if (ip)
    send_command(CMD_CLIENT_ILINE_ADD + add, ip);
  else
    send_command(CMD_CLIENT_ILINE_ADD + add, "");
}

COMMAND(lopster_dline)
{
  char *tag;
  char *ip;
  int add;

  tag = arg(data, 0);
  if (!tag || !l_strcasecmp(tag, "list")) {
    client_message(_("Message"), _("Requesting dlines"));
    add = 2;
  } else if (!l_strcasecmp(tag, "add")) {
    add = 0;
  } else if (!l_strcasecmp(tag, "remove")) {
    add = 1;
  } else {
    client_message(_("Error"), _("Unknown action"));
    return;
  }

  ip = arg(NULL, 1);
  if (!ip && (add != 2)) {
    client_message(_("Error"), _("No ip specified"));
    return;
  }

  if (ip)
    send_command(CMD_CLIENT_DLINE_ADD + add, ip);
  else
    send_command(CMD_CLIENT_DLINE_ADD + add, "");
}

COMMAND(lopster_eline)
{
  char *tag;
  char *ip;
  int add;

  tag = arg(data, 0);
  if (!tag || !l_strcasecmp(tag, "list")) {
    client_message(_("Message"), _("Requesting elines"));
    add = 2;
  } else if (!l_strcasecmp(tag, "add")) {
    add = 0;
  } else if (!l_strcasecmp(tag, "remove")) {
    add = 1;
  } else {
    client_message(_("Error"), _("Unknown action"));
    return;
  }

  ip = arg(NULL, 1);
  if (!ip && (add != 2)) {
    client_message(_("Error"), _("No ip specified"));
    return;
  }

  if (ip)
    send_command(CMD_CLIENT_ELINE_ADD + add, ip);
  else
    send_command(CMD_CLIENT_ELINE_ADD + add, "");
}

COMMAND(lopster_sreconfig)
{
  send_command(CMD_CLIENT_SERVER_RECONFIG, data);
}

COMMAND(lopster_pretend)
{
  char* code;
  char* message;

  code = arg(data, 0);
  message = arg(NULL, 1);
  if (!code) {
    client_message(_("Error"), _("Pretend what?"));
    return;
  }
  handle_command(atoi(code), message);
}

/*
COMMAND(lopster_search)
{
  char* string;
  search_pattern_t *pattern;
  search_pattern_t *pattern2;
  search_t *search;

  string = arg(data, 0);
  if (!string) {
    client_message(_("Error"), _("Search for what?"));
    return;
  }
  
  pattern = search_pattern_new();
  pattern->include = l_strdup(string);
  pattern->name = l_strdup(string);
  pattern->media_type = l_strdup("any");
  pattern->level = PATTERN_TEMPORARY;
  
  pattern2 = search_pattern_find_same(pattern);
  if (pattern2) {
    search_pattern_destroy(pattern);
    search_pattern_search(pattern2);
    return;
  }
  search_pattern_insert(search_pattern_copy(pattern), 1);

  //////
  search = search_new();
  search->pattern = pattern;
  search->local = 0;
  search_create_page(search);
  /////

  search_queue(search);
}
*/

#ifdef TRAFFIC_DEBUG
COMMAND(lopster_traffic)
{
  char str[1024];
  int i1;
  traffic_t t;

  (void)data;
  t.cnt = 0;
  t.bytes = 0;
  client_message(NULL, _("Incoming Traffic"));
  for (i1 = 0; i1 < 10302 + 1; i1++) {
    if (global.statistic.traffic[0][i1].cnt) {
      t.cnt += global.statistic.traffic[0][i1].cnt;
      t.bytes += global.statistic.traffic[0][i1].bytes;
      sprintf(str, "%5d", i1);
      client_message(str, "%6d : %10lu Bytes",
		     global.statistic.traffic[0][i1].cnt,
		     global.statistic.traffic[0][i1].bytes);
    }
  }
  client_message(" Sum ", "%6d : %10lu Bytes", t.cnt, t.bytes);
  t.cnt = 0;
  t.bytes = 0;
  client_message(NULL, _("Outgoing Traffic"));
  for (i1 = 0; i1 < 10302 + 1; i1++) {
    if (global.statistic.traffic[1][i1].cnt) {
      t.cnt += global.statistic.traffic[1][i1].cnt;
      t.bytes += global.statistic.traffic[1][i1].bytes;
      sprintf(str, "%5d", i1);
      client_message(str, "%6d : %10lu Bytes",
		     global.statistic.traffic[1][i1].cnt,
		     global.statistic.traffic[1][i1].bytes);
    }
  }
  client_message(" Sum ", "%6d : %10lu Bytes", t.cnt, t.bytes);
}
#endif

#ifdef PROTOCOL_DEBUG
COMMAND(lopster_test)
{
  (void)data;
  //  access_convert();
  /*
#ifdef ENABLE_WHITE_BOARD
  whiteboard_t* wb;

  wb = whiteboard_new("New Whiteboard");
  whiteboard_invite_user(wb, data, WHITEBOARD_READ);
#endif
  */
  return;
}
#endif
