/* LogJam, a GTK LiveJournal client.
 * Copyright (C) 2000,2001 Evan Martin <evan@livejournal.com>
 * vim:ts=4:sw=4:
 *
 * $Id: security.c,v 1.2 2001/07/14 05:56:17 martine Exp $
 */

#include "config.h"

#include <gtk/gtk.h>
#include <string.h>
#include <stdlib.h>
#include "util.h"
#include "security.h"
#include "dotconf.h"

#include "../pixmaps/lock.xpm"

static void secmgr_init(SecMgr *sm);

static void realized_cb(GtkWidget *w, gpointer d);
static void popup_menu_cb(SecMgr *sm, GdkEventButton *eb);
static guint32 custom_security_dlg_run(GtkWidget *parent, guint32 mask);

/* gtk stuff */
guint
secmgr_get_type() {
	static guint sm_type = 0;
	if (!sm_type) {
		GtkTypeInfo sm_info = {
			"SecMgr",
			sizeof (SecMgr),
			sizeof (SecMgrClass),
			(GtkClassInitFunc) NULL,
			(GtkObjectInitFunc) secmgr_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		sm_type = gtk_type_unique(gtk_button_get_type(), &sm_info);
	}
	return sm_type;
}

static void
secmgr_init(SecMgr *sm) {
	sm->security = NULL;
	gtk_signal_connect_after(GTK_OBJECT(sm), "realize",
			GTK_SIGNAL_FUNC(realized_cb), NULL);
	gtk_signal_connect(GTK_OBJECT(sm), "button-press-event",
			GTK_SIGNAL_FUNC(popup_menu_cb), NULL);
}

GtkWidget*
secmgr_new(security_data *security) {
	SecMgr *sm;
	sm = SECMGR(gtk_type_new(secmgr_get_type()));

	secmgr_set_data(sm, security);
	return GTK_WIDGET(sm);
}

void secmgr_set_data(SecMgr *sm, security_data *security) {
	sm->security = security;
}

static void
realized_cb(GtkWidget *w, gpointer d) {
	GdkPixmap *pix = NULL;
	GdkBitmap *bit;
	GtkWidget *pixmap;
	pix = gdk_pixmap_create_from_xpm_d(w->window, &bit, 
			NULL, lock_xpm);

	pixmap = gtk_pixmap_new(pix, bit);
	gtk_widget_show(pixmap);

	gtk_container_add(GTK_CONTAINER(w), pixmap);
}

static void
security_toggled(GtkWidget *w, SecMgr *sm, SecurityType type) {
	if (GTK_CHECK_MENU_ITEM(w)->active)
		sm->security->type = type;
}
static void
security_public_toggled(GtkWidget *w, SecMgr *sm) {
	security_toggled(w, sm, SECURITY_PUBLIC);
}
static void
security_private_toggled(GtkWidget *w, SecMgr *sm) {
	security_toggled(w, sm, SECURITY_PRIVATE);
}
static void
security_friends_toggled(GtkWidget *w, SecMgr *sm) {
	security_toggled(w, sm, SECURITY_FRIENDS);
}
static void
security_custom_toggled(GtkWidget *w, SecMgr *sm) {
	GtkWidget *win = 
		gtk_widget_get_ancestor(GTK_WIDGET(sm), gtk_window_get_type());

	if (GTK_CHECK_MENU_ITEM(w)->active) {
		sm->security->allowmask = custom_security_dlg_run(win, sm->security->allowmask);
	} else {
		sm->security->allowmask = 0;
	}
	security_toggled(w, sm, SECURITY_CUSTOM);
}

static void
popup_menu_cb(SecMgr *sm, GdkEventButton *eb) {
	GtkWidget *menu, *item;

	if (eb->button != 1) return;
	menu = gtk_menu_new(); {
		GSList *group = NULL;

		item = gtk_radio_menu_item_new_with_label(group, "Public");
		if (sm->security->type == SECURITY_PUBLIC)
			gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE);
		group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(item));
		gtk_signal_connect(GTK_OBJECT(item), "activate",
				GTK_SIGNAL_FUNC(security_public_toggled), sm);
		gtk_widget_show(item);
		gtk_menu_append(GTK_MENU(menu), item);

		item = gtk_radio_menu_item_new_with_label(group, "Private");
		if (sm->security->type == SECURITY_PRIVATE)
			gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE);
		group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(item));
		gtk_signal_connect(GTK_OBJECT(item), "activate",
				GTK_SIGNAL_FUNC(security_private_toggled), sm);
		gtk_widget_show(item);
		gtk_menu_append(GTK_MENU(menu), item);

		item = gtk_radio_menu_item_new_with_label(group, "Friends-Only");
		if (sm->security->type == SECURITY_FRIENDS)
			gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE);
		group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(item));
		gtk_signal_connect(GTK_OBJECT(item), "activate",
				GTK_SIGNAL_FUNC(security_friends_toggled), sm);
		gtk_widget_show(item);
		gtk_menu_append(GTK_MENU(menu), item);
	
		item = gtk_radio_menu_item_new_with_label(group, "Custom...");
		if (sm->security->type == SECURITY_CUSTOM)
			gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE);
		gtk_signal_connect(GTK_OBJECT(item), "activate",
				GTK_SIGNAL_FUNC(security_custom_toggled), sm);
		gtk_widget_show(item);
		gtk_menu_append(GTK_MENU(menu), item);
	}
	gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 
			eb->button, eb->time);
}

void 
security_append_to_request(security_data *security, GHashTable *request) {
	char *text = NULL;

	switch (security->type) {
		case SECURITY_PUBLIC:
			text = "public"; break;
		case SECURITY_PRIVATE:
			text = "private"; break;
		case SECURITY_FRIENDS:
		case SECURITY_CUSTOM:
			text = "usemask"; break;
	}
	g_hash_table_insert(request, g_strdup("security"),
			g_strdup(text));

	if (security->type == SECURITY_FRIENDS) {
		g_hash_table_insert(request, g_strdup("allowmask"), g_strdup("1"));
	} else if (security->type == SECURITY_CUSTOM) {
		g_hash_table_insert(request, g_strdup("allowmask"), 
				g_strdup_printf("%u", security->allowmask));
	}
}

void
security_load(security_data *security, char *sectext, char *allowmask) {
	if (strcasecmp(sectext, "public") == 0) {
		security->type = SECURITY_PUBLIC;
	} else if (strcasecmp(sectext, "private") == 0) {
		security->type = SECURITY_PRIVATE;
	} else if (strcasecmp(sectext, "usemask") == 0) {
		unsigned int am;
		am = atoi(allowmask);
		if (am == 1) {
			security->type = SECURITY_FRIENDS;
		} else {
			security->type = SECURITY_CUSTOM;
			security->allowmask = am;
		}
	} else {
		g_warning("security: '%s' unhandled", sectext);
	}
}

void 
security_load_from_request(security_data *security, GHashTable *request) {
	char *sectext, *allowmask;

	sectext = g_hash_table_lookup(request, "security");
	if (sectext) {
		allowmask = g_hash_table_lookup(request, "allowmask");
		security_load(security, sectext, allowmask);
	}
}

static void
ok_cb(GtkWidget *w, gboolean *ok) {
	*ok = TRUE;
}

static void
rowsel_cb(GtkCList *list, gint row, gint col, 
		GdkEventButton *e, guint32 *mask) {
	friendgroup *fg = gtk_clist_get_row_data(GTK_CLIST(list), row);
	
	*mask |= (1L << fg->id);
}

static void
rowunsel_cb(GtkCList *list, gint row, gint col, 
		GdkEventButton *e, guint32 *mask) {
	friendgroup *fg = gtk_clist_get_row_data(GTK_CLIST(list), row);
	
	*mask &= ~(1L << fg->id);
}

guint32
custom_security_dlg_run(GtkWidget *parent, guint32 origmask) {
	GtkWidget *dlg, *vbox, *list, *bok;
	gboolean ok = FALSE;
	char *text[1];
	GList *l;
	friendgroup *fg;
	int row;
	guint32 mask = origmask;
	
	dlg = lj_dialog_new(parent, "Select Friend Groups", 250, -1);
	gtk_signal_connect(GTK_OBJECT(dlg), "destroy",
			GTK_SIGNAL_FUNC(gtk_main_quit), NULL);

	vbox = gtk_vbox_new(FALSE, 5); {
		
		gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);

		text[0] = "Friend Groups";
		list = gtk_clist_new_with_titles(1, text);
		gtk_clist_set_selection_mode(GTK_CLIST(list), GTK_SELECTION_MULTIPLE);
		gtk_signal_connect(GTK_OBJECT(list), "select-row", 
				GTK_SIGNAL_FUNC(rowsel_cb), &mask);
		gtk_signal_connect(GTK_OBJECT(list), "unselect-row", 
				GTK_SIGNAL_FUNC(rowunsel_cb), &mask);

		for (l = conf.friendgroups; l != NULL; l = l->next) {
			fg = l->data;
			text[0] = fg->name;
			row = gtk_clist_append(GTK_CLIST(list), text);
			gtk_clist_set_row_data(GTK_CLIST(list), row, fg);
			if ((1L << fg->id) & mask) 
				gtk_clist_select_row(GTK_CLIST(list), row, 0);
		}

		gtk_box_pack_start(GTK_BOX(vbox), scroll_wrap(list), TRUE, TRUE, 0);
	}
	lj_dialog_set_contents(dlg, vbox);
	bok = lj_dialog_add_okcancel(dlg, NULL, ok_cb, &ok);
	gtk_signal_connect_object(GTK_OBJECT(bok), "clicked",
			GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dlg));

	gtk_widget_show(dlg);

	gtk_main();

	if (ok) {
		return mask;
	} else {
		return origmask;
	}
}

