/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */


 /* (C) Marcin Kwadrans <quar@vitea.pl> */

#include "include/support.h"
#include "include/row.h"

/*! \brief Inicjalizowanie wiersza

	Inicjalizuje obiekt, metoda prywatna używana wewnętrznie przez konstruktor
	
	\param board Plansza do której będzie należał wiersz 
*/
void LWRow::init (LWBoard *a_board)
{
	board = a_board;
	
	widget = gtk_hbox_new (FALSE, a_board->isGridEnabled() == TRUE ? 2 : 0);
	gtk_widget_show (widget);
	
	if (LW_TYPE_PROGRAM == board->getType()) {
		dummy_piece = new LWPiece (this);

		gtk_box_pack_end (GTK_BOX (widget), dummy_piece->getWidget(), TRUE, TRUE, 0);
	}
}

/*! \brief Konstruktor wiersza

	Konstruje wiersz, który następnie będzie można przyłączyć do planszy
	
	\param board Plansza do której będzie należał wiersz 
*/
LWRow::LWRow (LWBoard *a_board): list_piece(NULL), dummy_piece(NULL)
{
	init (a_board);
}

/*! \brief Konstruktor kopiujący

	Kopiuje wiersz wraz z należącymi do niego klockami, skopiowany
	wiersz będzie można przyłączyć do planszy board
	
	\param board Plansza do której będzie należał wiersz 
*/
LWRow::LWRow (const LWRow *row, LWBoard *a_board): list_piece(NULL), dummy_piece(NULL)
{
	init (a_board);
	
	for (GSList *l=row->list_piece; l != NULL; l = l->next) {
		LWPiece *piece = (LWPiece *) l->data;
		LWPiece *newpiece = new LWPiece (piece, this);
	 	addPiece (newpiece);
	}
}

/*! \brief Destruktor wiersza

	Niszczy wiersz oraz należące do niego klocki 
*/
LWRow::~LWRow ()
{
	clear();
	
	if (NULL != dummy_piece)
		delete dummy_piece;
	
	gtk_widget_destroy (widget);
}

LWBoard *LWRow::getBoard()
{
	return board;
}

/*!	\brief Rekonstrukcja elementów wiersza na podstawie opisu XML

	Rekonstrukcja wiersza na podstawie opisu XML zawartego w węźle 
	drzewa XML. Poprzednia zawartość wiersza zostaje usunięta.

	\param node Węzeł drzewa XML na postawie którego dokonana zostanie
				rekonstrukcja
*/
void LWRow::restoreFromXML (xmlNode *node)
{
	g_return_if_fail (!xmlStrcasecmp (node->name, (xmlChar *) "Row"));

	clear();
	
	for (xmlNode *n=node->children; n != NULL; n=n->next) {
		LWPiece *piece = new LWPiece (this);
		piece->restoreFromXML (n);
		addPiece (piece);
	}
}

/*! \brief Zapis elementów wiersza w pastaci opisu XML
 
	Zapis wiersza do postaci węzła drewa XML, utworzony węzeł zostanie dodany
	do struktury rodzica.

	Dokonywane jest sprawdzenie czy wiersz węzłem nadrzędnym jest plansza,
	jeśli tak nie jest zwrócone jest ostrzeżenie

	\param node Węzeł drzewa XML na postawie którego dokonanie zostanie
				rekonstrukcja
*/
void LWRow::storeToXML (xmlNode *parent_node)
{
	g_return_if_fail (!xmlStrcasecmp (parent_node->name, (xmlChar *) "Board"));
	
	xmlNode *node = xmlNewChild(parent_node, NULL, (xmlChar *) "Row", NULL);

	for (GSList *l = list_piece; l != NULL; l = l->next) {
		LWPiece *piece = (LWPiece *) l->data;
		piece->storeToXML (node);
	}
}

/*! \brief Dodanie klocka do wiersza
 
	Dodanie klocka do wiersza, dadany klocek będzie znajdował się na końcu 
	wiersza. 

	\param piece Dodawany klocek
*/
void LWRow::addPiece (LWPiece *piece)
{
	g_return_if_fail (piece != NULL);
	g_return_if_fail (piece->row == this);
	
	gtk_box_pack_start (GTK_BOX (widget), piece->getWidget(), FALSE, FALSE, 0);
	list_piece = g_slist_append (list_piece, (gpointer) piece);
}

/*! \brief Dodanie klocka do wiersza przed innym klockiem
 
	Dodanie klocka piece do wiersza, dadany klocek będzie znajdował się na lewo
	od klocka sibiling, jeśli sibiling jest NULL wtedy klocek zostanie dodany na
	początku wiersza

	Metoda szczególnie przydatna w mechaniźmie Przeciągnij i Upuść

	\param piece Dodawany klocek
	\param sibiling Klocek przed którym zostanie dodany element
*/
void LWRow::insertPieceBefore (LWPiece *piece, LWPiece *sibiling)
{
gint a=0;
	
	g_return_if_fail (piece != NULL);
	g_return_if_fail (piece->row == this);
	g_return_if_fail (sibiling != NULL);
	g_return_if_fail (sibiling->row == this);
	
	/* Niezbyt ładny sposób na dodanie elementu w wierszu poprzednim,
	 * jeśli wiersz jest dodatkowy */
	if (TRUE == board->isRowDummy(this)) {
		LWRow *row = new LWRow (board);
		board->addRow (row);
		piece->row = row; 
		row->addPiece (piece);
		return;
	}

	if (dummy_piece == sibiling) {
		addPiece (piece);
		return;
	}

	list_piece = g_slist_insert_before (list_piece,
			g_slist_find (list_piece, sibiling),	(gpointer) piece);

	gtk_box_pack_start (GTK_BOX (widget), piece->getWidget(), FALSE, FALSE, 0);

	/* 
	Funckja gtk_box_reorder_child wstawia kontrolke na danej pozycji,
	jednak pozycja na ktorej jest wstawiana kontrolka zalezy
	od pozycji w wewętrznej liście już wstawionych kontrolek,
	lista jest posortowana w kolejnosci wywoływania przez gtk_box_pack_start,
	i gtk_box_pack_end. Co oznacza, że zawrtość może sie różnić od faktycznej
	kolejności kontrolek w przypadku mieszania tych funkcji. Aby obejść
	to ograniczenie sprawdzana jest obecność dodatkowego klocka, który
	był dodany przez gtk_box_pack_end. Ponieważ jest to kontrolka, która
	zawsze jest dodawana jako perwsza więc w liście widnieje też jako pierwsza,
	choć fizycznie w wierszu znajduje się jako ostatnia, z tego powodu
	w funkcji gtk_box_reorder_child pozycja jest zwiększana o 1 w
	przypadku istnienia dodatkowego klocka.	
	*/
	
	if (dummy_piece != NULL) a = 1;	

	gtk_box_reorder_child (GTK_BOX (widget), piece->getWidget(), 
		g_slist_index (list_piece, (gpointer) piece)+a);
}

/*! \brief Usuwanie klocka
	
	Usuwanie klocka z wiersza. Po usunięciu ze struktury wiersza, wywoływany
	jest destruktor wiersza

	\param piece Usuwany klocek
*/
void LWRow::removePiece (LWPiece *piece)
{
	g_return_if_fail (piece != NULL);
	g_return_if_fail (piece->row == this);

	if (NULL == piece->row->getPieceNth (1)) {
		piece->row->board->removeRow (piece->row); return;
		
/*
		LWRow *del_row = NULL;
		gint n = piece->row->board->getRowIndex (piece->row);
		
		if (n == 0) 
			del_row = piece->row;
		else	
		if (NULL == piece->row->board->getRowNth(n-1)->getPieceNth(0))
			del_row = piece->row;
		else
		if (NULL == piece->row->board->getRowNth(n+1))
			del_row = piece->row;
		else
		if (NULL == piece->row->board->getRowNth(n+1)->getPieceNth(0))
			del_row = piece->row;
		
		if (del_row != NULL) {
			piece->row->board->removeRow (del_row); return;				
		}
*/
	}
	
	list_piece = g_slist_remove (list_piece, (gpointer) piece);
	delete piece;
}

/*! \brief Dzieli wiersz

	Dzieli wiersz na dwa wiersze, elementy za piece są przenoszone
	do nowego wiersza.

	\param piece Łam wiersz przed piece
*/
void LWRow::splitBefore (LWPiece *piece)
{
	g_return_if_fail (piece != NULL);
	g_return_if_fail (piece->row == this);

	LWRow *new_row = new LWRow (board);
	board->insertRowBefore (new_row, this);
	
	for (GSList *l = list_piece; l != NULL && l->data != piece;) {
		LWPiece *orig_piece = (LWPiece *) l->data;
		l = l->next;
		
		LWPiece *new_piece = new LWPiece (orig_piece, new_row);
		new_row->addPiece (new_piece);
		removePiece (orig_piece);
	}
}

/*! \brief Pozycja wiersza w planszy

	Zwraca pozycję klocka strukturze wiersza.

	\param row Pozycja klocka w wierszu liczona od 0. Jeśli klocek nie należy
	do wiersza zwrócone zostanie -1.
*/
gint LWRow::getPieceIndex (LWPiece *piece)
{
	g_return_val_if_fail (piece != NULL, -1);
	g_return_val_if_fail (piece->row == this, -1);

	return g_slist_index (list_piece, (gpointer) piece);
}

/*! \brief Pozycja klocka w wierszu

	Zwraca pozycję klocka w strukturze wiersza.

	\param row Pozycja klocka na planszy liczona od 0. Jeśli klocek nie należy
	do wiersza zwrócone zostanie -1.
*/
LWPiece *LWRow::getPieceNth (guint n)
{
	return (LWPiece *) g_slist_nth_data (list_piece, n);
}	

/*! \brief Lista klocków w wierszu

	Zwraca listę klocków w wierszu. 

	\return Zwrócona kopia listy klocków. Listę należy zwolnić
	przez g_slist_free.
*/
GSList *LWRow::getPieceList ()
{
	return g_slist_copy (list_piece);	
}

/*! \brief Ustawienie rozmiaru wiersza 

	Ustawia długość wiersza w klockach

	\param length Ustawiana długość 
*/
void LWRow::setWidth (guint length)
{
GSList *l;
	
	/* Zmniejsz liczbe elementow jeśli potrzeba */
	while (NULL != (l = g_slist_nth (list_piece, length))) {
		LWPiece *piece = (LWPiece *) l->data;
		g_slist_delete_link (list_piece, l);
		delete piece;
	}

	/* Zwiększ liczbę elementów jeśli potrzeba */
	for (guint i = g_slist_length (list_piece); i < length; i++)
		addPiece (new LWPiece (this));
	
}

guint LWRow::getWidth ()
{
	return g_slist_length (list_piece);
}

void LWRow::updatePieceSize ()
{
	for (GSList *l = list_piece; l != NULL; l = l->next) {
		LWPiece *piece = (LWPiece *) l->data;
		piece->updateImage ();
	}
}

void LWRow::updateGrid ()
{
	gtk_box_set_spacing (GTK_BOX (widget),
		(board->isGridEnabled() == TRUE) ? 2 : 0);
}

/*! \brief Czyszczenie zawarości wiersza

	Usuwa wszystkie klocki w wierszu.
	W przypadku gdy typem planszy do której należy wiersz jest LW_TYPE_PROGRAM
	pozostaje tylko dodatkowy klocek na końcu wiersza przeznaczony
	do upuszczania nowych klocków przy pomocy przeciągnij i upuść.
	Metoda nie przyjmuje argumentów i nie zwraca wyniku
*/
void LWRow::clear ()
{
	for (GSList *l=list_piece; l != NULL; l = l->next) {
		LWPiece *piece = (LWPiece*) l->data;
		delete piece;
	}

	g_slist_free (list_piece);
	list_piece = NULL;	
}

/*! \brief Pobieranie kontrolki
 
	Pobiera kontrolkę (ang. widget) zawierającą wiersz

	\return Zwrócona kontrolka
*/
GtkWidget *LWRow::getWidget ()
{
	return widget;
}

/*! \brief Sprawdzanie czy element jest dodatkowy

	Sprawdza czy elemet jest dodadkowym elementem, 
	Metoda przeznaczona wyłącznie do obsługi mechanizmu 
	Przeciągnij i Upuść.
	
	\return Rezultat testu. TRUE jeśli wiersz jest dodatkowy,
	w przeciwnym wypdaku FALSE
*/
gboolean LWRow::isPieceDummy (LWPiece *piece)
{
	return (piece == dummy_piece) ? TRUE : FALSE;
}
