package Lire::WWW::UserAgent;

use strict;

use vars qw/$REVISION $VERSION/;

use Lire::WWW::UserAgent::Robot;
use Lire::WWW::UserAgent::Browser;
use Lire::WWW::UserAgent::OS;
use Lire::WWW::UserAgent::Language;

use Carp;

BEGIN {
    $REVISION  = '$Id: UserAgent.pm,v 1.8 2002/01/19 18:20:05 vanbaal Exp $';
    $VERSION   = "0.2.0";
}

my %customLanguageHash = ();
my %compiledLanguageHash = ();
my %notInLanguageHash = ();

my %customOSHash = ();
my %compiledOSHash = ();
my %notInOSHash = ();

my %customRobotHash = ();
my %compiledRobotHash = ();
my %notInRobotHash = ();

my %customBrowserHash = ();
my %compiledBrowserHash = ();
my %notInBrowserHash = ();

# -----------------------------------------------------------------------------
# Functions
# -----------------------------------------------------------------------------

sub new {
     my ($class) = @_;

     my $self = {};
     bless $self, $class;

     return $self;
}

sub setUserAgent {
    my ($self, $ua) = @_;

    if (defined $ua) {
        $self->{user_agent} = lc($ua);
    }
}


# If the 'browser' is a robot, it returns "Web Robot". Details
# can be obtained with getRobot();
sub getBrowser {
    my ($self) = @_;

    my $user_agent = $self->{user_agent};
    if (defined $user_agent) {
        return "Unknown" if ($notInBrowserHash{$user_agent});

        if (defined Lire::WWW::UserAgent::Robot::getRobot($user_agent)) {
            # browser is a robot. Do not specify which one.
            return "Web Robot";
        } elsif (defined (my $browser = 
                 Lire::WWW::UserAgent::Browser::getBrowser($user_agent))) {
            return $browser;
        } else {
            # check custom hash
            foreach my $key (keys %customBrowserHash) {
		my $re = $compiledBrowserHash{$key};
                if ($user_agent =~ /$re/) {
                    return $customBrowserHash{$key};
                }
            }
            $notInBrowserHash{$user_agent} = "1";
            return "Unknown";
        }
    }
    return undef;
}

sub addBrowser {
    my ($self, $code, $value) = @_;

    croak "Must supply both code and value!\n"
      unless $code eq "" || $value eq "";

    $code = lc $code;
    $customBrowserHash{$code} = $value;
    $compiledBrowserHash{$code} = qr/$code/;
}

sub addBrowserHash {
    my ($self, %bhash) = @_;

    foreach my $code (keys %bhash) {
        $self->addBrowser($code, $bhash{$code});
    }
}

sub getRobot {
    my ($self) = @_;

    my $user_agent = $self->{user_agent};
    if (defined $user_agent) {
        return "Unknown/No Robot" if ($notInRobotHash{$user_agent});

        if (defined (my $robot = Lire::WWW::UserAgent::Robot::getRobot($user_agent))) {
            return $robot;
        } else {
            # check custom hash
            foreach my $key (keys %customRobotHash) {
		my $re = $compiledRobotHash{$key};
                if ($user_agent =~ /$re/) {
                    return $customRobotHash{$key};
                }
            }
            $notInRobotHash{$user_agent} = "1";
            return "Unknown/No Robot";
        }
    }
    return undef;
}

sub addRobot {
    my ($self, $code, $value) = @_;

    croak "Must supply both code and value!\n"
      unless $code eq "" || $value eq "";

    $code = lc $code;
    $customRobotHash{$code} = $value;
    $compiledRobotHash{$code} = qr/$code/;
}

sub addRobotHash {
    my ($self, %rhash) = @_;

    foreach my $code (keys %rhash) {
        $self->addRobot($code, $rhash{$code});
    }
}

sub getOS {
    my ($self) = @_;

    my $user_agent = $self->{user_agent};
    if (defined $user_agent) {
        return "Unknown" if ($notInOSHash{$user_agent});

        if (defined (my $os = Lire::WWW::UserAgent::OS::getOS($user_agent))) {
            return "$os";
        } else {
            # check custom hash
            foreach my $key (keys %customOSHash) {
		my $re = $compiledOSHash{$key};
                if ($user_agent =~ /$re/) {
                    return $customOSHash{$key};
                }
            }
            $notInOSHash{$user_agent} = "1";
            return "Unknown";
        }
    }
    return undef;
}

sub addOS {
    my ($self, $code, $value) = @_;

    croak "Must supply both code and value!\n"
      unless $code eq "" || $value eq "";

    $code = lc $code;
    $customOSHash{$code} = $value;
    $compiledOSHash{$code} = qr/$code/;
}

sub addOSHash {
    my ($self, %ohash) = @_;

    foreach my $code (keys %ohash) {
        $self->addOS($code, $ohash{$code});
    }
}

sub getLanguage {
    my ($self) = @_;

    my $user_agent = $self->{user_agent};
    if (defined $user_agent) {
        return "Unknown" if ($notInLanguageHash{$user_agent});

        if (defined (my $lang = Lire::WWW::UserAgent::Language::getLanguage($user_agent))) {
            return "$lang";
        } else {
            # check custom hash
            foreach my $key (keys %customLanguageHash) {
		my $re = $compiledLanguageHash{$key};
                if ($user_agent =~ /$re/) {
                    return $customLanguageHash{$key};
                }
            }
            $notInLanguageHash{$user_agent} = "1";
            return "Unknown";
        }
    }
    return undef;
}

sub addLanguage {
    my ($self, $code, $value) = @_;

    croak "Must supply both code and value!\n"
      unless $code eq "" || $value eq "";

    $code = lc $code;
    $customLanguageHash{$code} = $value;
    $compiledLanguageHash{$code} = qr/$code\;/;
}

sub addLanguageHash {
    my ($self, %langhash) = @_;
    
    foreach my $code (keys %langhash) {
        $self->addLanguage($code, $langhash{$code});
    }
}

1;

__END__

=pod

=head1 NAME

WWW::UserAgent - detect browser, operating system and more from the user agent string

=head1 SYNOPSIS

 use Lire::WWW::UserAgent;

 my $test_string = "Mozilla/5.0 (compatible; Konqueror/2.1.2; X11)";

 my $analyzer = new Lire::WWW::UserAgent();
 $analyzer->setUserAgent($test_string);

 print " Operating System      : " . $analyzer->getOS() . "\n";
 print " Browser               : " . $analyzer->getBrowser() . "\n";
 print "   using GUI language  : " . $analyzer->getLanguage() . "\n";
 print "   (opt) Web Robot name: " . $analyzer->getRobot() . "\n";

 %customBrowserHash = ("CustomWebBrowser" => "Custom Web Browser 2005");
 $analyzer->addBrowserHash(%testhash);

 $analyzer->addBrowser("LastMinuteBrowser", "Lastest Released LM Browser");

=head1 DESCRIPTION

This module offers an interface to databases with known browsers, operating
systems and other information useful in analyzing user agent strings as found
in Apache combined logfiles.

Optionally, the user can add custom information to the database with
the addX() and addXHash() functions, where X is the type of database.

This module is based on ideas as seen in AWStats.

=head1 VERSION

$Id: UserAgent.pm,v 1.8 2002/01/19 18:20:05 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2001  Stichting LogReport Foundation LogReport@LogReport.org

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHORS

Egon Willighagen <egonw@logreport.org> and
Francis Lacoste <flacoste@logreport.org>

=cut

