package Lire::ReportParser::AsciiWriter;

use strict;

use vars qw/ $VERSION @ISA /;

use Lire::Config;
use Lire::ReportParser;

use XML::Parser;
use POSIX qw/strftime/;
use Lire::Program qw/ :msg /;

use Text::Wrap qw/ wrap /;

use constant MAX_LINE_LENGTH => 64;
use constant COLUMNS	     => 72;

BEGIN {
    ($VERSION)	= '$Revision: 1.16 $' =~ m!Revision: ([.\d]+)!;
    @ISA = qw/ Lire::ReportParser /;
}

sub new {
    my $proto = shift;
    my $class = ref $proto || $proto;
    my $self  = $class->SUPER::new( @_ );

    my %args = @_;
    $self->{'target-user'}  = $args{'target-user'} || 'sysadmin';
    $self->{userlevel}	    = $args{userlevel} || 'normal';

    $self;
}

sub report_start {
    my ( $self, $expat, $name, %attr ) = @_;

    $self->SUPER::report_start( $expat, $name, %attr );
    print "Report generated: ", $attr{date}, "\n";

}

sub handle_timespan {
    my ( $self, $expat, $timespan ) = @_;

    print "Reporting on period:\n";
    print $timespan, "\n\n";
}

sub handle_title {
    my ( $self, $expat, $title ) = @_;

    # Trim the title
    $title =~ s/^\s*//;
    $title =~ s/\s*$//;
    if ($self->in_section_intro ) {
	# Section's title
	my $len = length $title;
	if ( $len > COLUMNS - 12 ) {
	    local $Text::Wrap::columns = COLUMNS - 12;
	    print wrap( "    ", "    ", $title ), "\n";
	    print "    ", "-" x (COLUMNS - 12), "\n\n";
	} else {
	    my $pad = int( (COLUMNS - $len) / 2);
	    print ' ' x $pad, $title, "\n", ' ' x $pad, '-' x $len, "\n\n";
	}
    } else {
	local $Text::Wrap::columns = COLUMNS - 12;
	print wrap( "  ", "  ", $title ), "\n\n";
    }
}

sub section_end {
    my ( $self, $expat, $name ) = @_;

    print "  No subreports were generated for this section.\n\n"
      if ( ! $self->current_section_subreport_count);

    $self->SUPER::section_end( $expat, $name );
}

sub subreport_start {
    my ( $self, $expat, $name, %attr ) = @_;

    $self->SUPER::subreport_start( $expat, $name, %attr );
    $self->{printed_description} = 0;
}

sub subreport_end {
    my ( $self, $expat, $name ) = @_;

    $self->SUPER::subreport_end( $expat, $name );
    print "\n";
}

sub table_end {
    my ( $self, $expat, $name ) = @_;

    print "  No content in report.\n" if ( ! $self->current_table_entry_count);

    $self->SUPER::table_end( $expat, $name );

    # Reset
    $self->{extra_nl} = 0;
}

sub group_start {
    my ( $self, $expat, $name ) = @_;

    $self->SUPER::group_start( $expat, $name );

    # Unnecessary when a subgroup is started
    $self->{extra_nl} = 0;
}

sub entry_start {
    my ( $self, $expat, $name ) = @_;
    # Output newline when multiple names were printed
    # in the last entry
    print "\n" if $self->{extra_nl};
    $self->SUPER::entry_start( $expat, $name );
    $self->{last_name} = [];
    $self->{extra_nl} = 0;
}

sub handle_name {
    my ( $self, $expat, $name ) = @_;

    push @{$self->{last_name}}, $name;
}

# FIXME: Doesn't handle multiple values.
sub handle_value {
    my ( $self, $expat, $value ) = @_;

    my $names = $self->{last_name};
    my $name_len = 0;
    foreach my $name ( @$names ) {
	$name_len = length $name
	  if length $name > $name_len;
    }
    # Make sure that all name are the same length
    foreach my $name ( @$names ) {
	$name = sprintf "%-${name_len}s", $name;
    }
    my $level = $self->current_group_level();

    my $pre = "  " x ($level + 1);
    if ( length($pre) + $name_len + length($value) + 3 > MAX_LINE_LENGTH) {
	my $max_name_len = MAX_LINE_LENGTH - (2 + length($pre) +
					      length($value) );
	foreach my $name ( @$names ) {
	    $name = substr $name, 0, $max_name_len;
	}
	print $pre, " ", $names->[0], " ", $value, "\n";
	if ( @$names > 1 ) {
	    foreach my $name (@{$names}[1..$#$names]) {
		print $pre, " ", $name, "\n";
	    }
	    $self->{extra_nl} = 1;
	}
    } else {
	my $dots = ($level > 0 || length $value == 0 ) ? " " : ".";
	$dots x= MAX_LINE_LENGTH - ( $name_len + length($value) +
				     length($pre) + 3 );

	print $pre, " ", $names->[0], " ",  $dots, " ",  $value, "\n";
	if ( @$names > 1 ) {
	    foreach my $name (@{$names}[1..$#$names]) {
		print $pre, " ", $name, "\n";
	    }
	    $self->{extra_nl} = 1;
	}
    }

}

sub description_start {
    my ( $self, $expat, $name, %attr )= @_;

    if ( exists $attr{'target-user'} && 
	 $attr{'target-user'} ne $self->{'target-user'})
    {
	# Skip this description if there is a target-user attribute on it
	# and it isn't the same as the requested target-user.
	$self->skip( $expat );
    }

    $self->init_dbk;
}

########################################################################
#		       DOCBOOK FORMATTING
########################################################################

use vars qw/ @dbk_elmnts @auto_pcdata @dbk_pcdata /;

BEGIN {
    @dbk_elmnts = qw/listitem orderedlist itemizedlist 
		     variablelist varlistentry caution note tip warning
		     important /;

    @dbk_pcdata = qw/para term ulink title quote/;

    @auto_pcdata = qw/abbrev acronym emphasis phrase trademark wordasword 
		      action application classname command computeroutput 
		      database email envar errorcode errorname errortype 
		      filename function hardware
		      interface keycap keycode keycombo keysym
		      literal constant markup
		      option optional parameter prompt property 
		      replaceable returnvalue sgmltag structfield structname
		      symbol systemitem token type userinput varname anchor
		      author authorinitials corpauthor modespec othercredit
		      productname productnumber subscript superscript 
		     /;

    foreach my $elmnt ( @auto_pcdata ) {
	no strict 'refs';
	my $sub_pfx = __PACKAGE__ . '::dbk_' . $elmnt;
	*{$sub_pfx . '_start'} = sub {};
	*{$sub_pfx . '_end'} = sub {};
	*{$sub_pfx . '_char'} = sub { 
	    my $self = shift;
	    $self->inline_char( @_ );
	};
    }
}

sub known_dbk_elements {
    return [ @dbk_elmnts, @dbk_pcdata, @auto_pcdata ];
}

sub known_dbk_pcdata_elements {
    return [ @dbk_pcdata, @auto_pcdata ];
}

sub init_dbk {
    my ( $self ) = @_;

    $self->{text_blocks}    = [];
    $self->{lists}	    = [];
    $self->{left_margin}    = 4;
    $self->{right_margin}   = 8;
}

sub element_start {
    my ( $self, $expat, $name, %attr ) = @_;

    # Skip element which have a higher userlevel 
    # than requested
    return 0 unless exists $attr{userlevel};
    return 0 if $self->{userlevel} eq 'advanced';
    return 0 if $attr{userlevel} eq $self->{userlevel};

    # Requested userlevel is 'normal' and element isn't
    $self->skip( $expat );

    return 1;
}

sub parent_block {
    return undef unless @{$_[0]{text_blocks}} > 1;
    return $_[0]{text_blocks}[$#{$_[0]{text_blocks}} - 1];
}

sub current_block {
    return undef unless @{$_[0]{text_blocks}};
    return $_[0]{text_blocks}[$#{$_[0]{text_blocks}}];
}

sub print_block {
    my ( $self ) = @_;

    my $block = $self->current_block;
    return unless $block;
    return unless length $block->{text};
    
    my $margin = ' ' x $self->{left_margin};
    my $initial = $margin . $block->{initial_indent};
    my $next	= $margin . $block->{indent};

    # Squash space and trim the string.
    $block->{text} =~ tr/\n\t / /s;
    $block->{text} =~ s/^\s*//;
    $block->{text} =~ s/\s*$//;
    return if $block->{text} =~ /^\s*$/;

    local $Text::Wrap::columns = COLUMNS - $self->{right_margin};
    print wrap( $initial, $next, $block->{text} );

    if ( $block->{skip_line} ) {
	print "\n\n";
    } else {
	print "\n";
    }

    # Flush text buffer
    $block->{text} = "";
}

sub inline_char {
    my ( $self, $expat, $str ) = @_;
    my $block = $self->current_block;
    $block->{text} .= $str if $block;
}

sub start_block {
    my ( $self, $left_margin_indent, $right_margin_indent )= @_;

    $left_margin_indent ||= 0;
    $right_margin_indent ||= 0;

    # Flush the current block, if there is one
    $self->print_block;

    $self->{left_margin}  += $left_margin_indent;
    $self->{right_margin} += $right_margin_indent;
    push @{$self->{text_blocks}}, { text	    => "",
				    initial_indent  => '',
				    indent	    => '',
				    left_margin_indent	=> $left_margin_indent,
				    right_margin_indent => $right_margin_indent,
				    skip_line	    => 1,
				    children	    => 0,
				  };

    my $parent = $self->{parent_block};
    $parent->{children}++ if $parent;

    return $self->current_block;
}

sub end_block {
    my ( $self ) = @_;

    # Flush the current block, if there is one
    $self->print_block;

    my $block = $self->current_block;

    $self->{left_margin}  -= $block->{left_margin_indent};
    $self->{right_margin} -= $block->{right_margin_indent};
    pop @{$self->{text_blocks}};
}

sub current_list {
    return undef unless @{$_[0]{lists}};
    return $_[0]{lists}[$#{$_[0]{lists}}];
}

sub start_list {
    my ( $self, $type, %attr )= @_;

    my $block = $self->start_block( 2 );
    
    push @{$self->{lists}}, { type => $type,
			      %attr,
			    };

    return $self->current_list;
}

sub end_list {
    my ( $self ) = @_;

    $self->end_block;
    pop @{$self->{lists}};
}

sub dbk_para_start {
    my ( $self, $expat, $name, %attr )= @_;

    my $block = $self->start_block;

    if ( $expat->in_element( "listitem" ) ) {
	my $parent = $self->parent_block;

	my $list = $self->current_list;
	$block->{skip_line} = 0 if $list->{spacing} eq 'compact';
	
	# Copy listitem indent and initial_indent attribute
	if ( $parent->{children} == 1 ) {
	    $block->{initial_indent} = $parent->{initial_indent};
	} else {
	    # Add extra space before the paragraph if it wasn't the first
	    # and the list is compact
	    print "\n" 
	      if $parent->{children} > 1 && $list->{spacing} eq 'compact';

	    # Put mark only on first para
	    $block->{initial_indent} = $parent->{indent};
	}
	$block->{indent} = $parent->{indent};
    }
}

sub dbk_para_end {
    my ( $self, $expat, $name )= @_;

    $self->end_block;
}

sub dbk_para_char {
    my $self = shift;
    $self->inline_char( @_ );
}

sub dbk_itemizedlist_start {
    my ( $self, $expat, $name, %attr )= @_;

    $self->start_list( 'itemized', 
		       mark     => '-',
		       spacing  => 'normal',
		       %attr,
		     );
}

sub dbk_itemizedlist_end {
    my ( $self, $expat, $name )= @_;

    $self->end_list;
}

sub dbk_orderedlist_start {
    my ( $self, $expat, $name, %attr )= @_;

    $self->start_list( 'ordered', 
		       spacing   => 'normal',
		       %attr,
		       item_count => 0,
		     );
}

sub dbk_orderedlist_end {
    my ( $self, $expat, $name )= @_;

    $self->end_list;
}

sub dbk_variablelist_start {
    my ( $self, $expat, $name, %attr )= @_;

    $self->start_list( 'variable', 
		       spacing => 'normal',
		       %attr,
		     );
}

sub dbk_variablelist_end {
    my ( $self, $expat, $name )= @_;

    $self->end_list;
}

sub dbk_varlistentry_start {}

sub dbk_varlistentry_end {}

sub dbk_term_start {
    my ( $self, $expat, $name, %attr )= @_;

    my $block = $self->start_block;
    $block->{skip_line} = 0;
}

sub dbk_term_end {
    my ( $self, $expat, $name )= @_;

    $self->end_block;
}

sub dbk_term_char {
    my $self = shift;
    $self->inline_char( @_ );
}

sub dbk_listitem_start {
    my ( $self, $expat, $name, %attr )= @_;

    my $list = $self->current_list;
    my $block = $self->start_block;
    if ( $list->{type} eq 'itemized' ) {
	my $mark = $attr{override} || $list->{mark};

	$block->{initial_indent} = $mark . ' ';
	$block->{indent} = ' ' x length $block->{initial_indent};
    } elsif ( $list->{type} eq 'ordered' ) {
	$list->{item_count}++;

	$block->{initial_indent} = $list->{item_count} . '. ';
	$block->{initial_indent} .= ' '
	  if length $block->{initial_indent} < 4 ;
	$block->{indent} = ' ' x length $block->{initial_indent};
    } elsif ( $list->{type} eq 'variable' ) {
	$block->{initial_indent} = ' ' x 4;
	$block->{indent} = ' ' x 4;
    } else {
	lr_warn( "unknown list type: $list->{type}" );
    }

    $block->{skip_line} = 0 if $list->{spacing} eq 'compact';
}

sub dbk_listitem_end {
    my ( $self, $expat, $name )= @_;

    $self->end_block;
}

sub dbk_title_start {
    my ( $self, $expat, $name, %attr )= @_;

    $self->start_block( 0, 4 );
}

sub dbk_title_end {
    my ( $self, $expat, $name )= @_;

    $self->end_block();
}

sub dbk_title_char {
    my $self = shift;
    $self->inline_char( @_ );
}

sub dbk_ulink_start {
    my ( $self, $expat, $name, %attr )= @_;

    $self->{curr_url_attr} = $attr{url} || "";
    $self->{curr_url} = "";
}

sub dbk_ulink_end {
    my ( $self, $expat, $name )= @_;

    $self->inline_char( $expat, ' (' . $self->{curr_url_attr} . ')' )
      if ( $self->{curr_url_attr} ne $self->{curr_url} );
    delete $self->{curr_url_attr};
    delete $self->{curr_url};
}

sub dbk_ulink_char {
    my ( $self, $expat, $str )= @_;
    $self->inline_char( $expat, $str );
    $self->{curr_url} .= $str;
}

sub dbk_quote_start {
    my ( $self, $expat, $name, %attr )= @_;

    $self->inline_char( $expat, '"' );
}

sub dbk_quote_end {
    my ( $self, $expat, $name )= @_;

    $self->inline_char( $expat, '"' );
}

sub dbk_quote_char {
    my $self = shift;
    $self->inline_char( @_ );
}

sub admonition_start {
    my ( $self, $expat, $name, %attr ) = @_;

    my $block = $self->start_block;
    $block->{skip_line} = 0;
    $self->inline_char( $expat, ucfirst $name . ":" );
    $self->end_block;
    $self->start_block( 2 );
}

sub admonition_end {
    my ( $self, $expat, $name ) = @_;
    $self->end_block;
}

sub dbk_note_start {
    my $self = shift;
    $self->admonition_start( @_ );
}

sub dbk_note_end {
    my $self = shift;
    $self->admonition_end( @_ );
}

sub dbk_tip_start {
    my $self = shift;
    $self->admonition_start( @_ );
}

sub dbk_tip_end {
    my $self = shift;
    $self->admonition_end( @_ );
}

sub dbk_important_start {
    my $self = shift;
    $self->admonition_start( @_ );
}

sub dbk_important_end {
    my $self = shift;
    $self->admonition_end( @_ );
}

sub dbk_caution_start {
    my $self = shift;
    $self->admonition_start( @_ );
}

sub dbk_caution_end {
    my $self = shift;
    $self->admonition_end( @_ );
}

sub dbk_warning_start {
    my $self = shift;
    $self->admonition_start( @_ );
}

sub dbk_warning_end {
    my $self = shift;
    $self->admonition_end( @_ );
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::ReportParser -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: AsciiWriter.pm,v 1.16 2002/02/05 18:27:20 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
