/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <cmath>
using namespace std;

#include "SFFT.h"
#include "Inline.h"

#include <wx/object.h>		// Not sure why !
#include <wx/debug.h>

const double STABCOEFF = 0.9999;

CSFFT::CSFFT(int len, int first, int last) :
m_fftLen(len),
m_first(first),
m_last(last),
m_ptr(0)
{
	wxASSERT(m_fftLen > 0);
	wxASSERT(m_first > 0);
	wxASSERT(m_last > first);

	m_cos         = new double[m_fftLen];
	m_sin         = new double[m_fftLen];
	m_historyReal = new double[m_fftLen];
	m_historyImag = new double[m_fftLen];
	m_binsReal    = new double[m_fftLen];
	m_binsImag    = new double[m_fftLen];
	m_bins        = new double[m_fftLen];

	for (int i = 0; i < m_fftLen; i++) {
		double val = double(i) * 2.0 * M_PI / double(m_fftLen);

		m_cos[i] = ::cos(val) * STABCOEFF;
		m_sin[i] = ::sin(val) * STABCOEFF;

		m_historyReal[i] = 0.0;
		m_historyImag[i] = 0.0;
		m_binsReal[i]    = 0.0;
		m_binsImag[i]    = 0.0;
		m_bins[i]        = 0.0;
	}

	m_corr = ::pow(STABCOEFF, double(m_fftLen));
}

CSFFT::~CSFFT()
{
	delete[] m_cos;
	delete[] m_sin;
	delete[] m_historyReal;
	delete[] m_historyImag;
	delete[] m_binsReal;
	delete[] m_binsImag;
	delete[] m_bins;
}

/*
 * Sliding FFT
 */
double* CSFFT::process(double in)
{
	/* restore the sample fftlen samples back */
	double oldReal = m_historyReal[m_ptr] * m_corr;
	double oldImag = m_historyImag[m_ptr] * m_corr;

	/* save the new sample */
	m_historyReal[m_ptr] = in;
	m_historyImag[m_ptr] = 0.0;

	/* advance the history pointer */
	m_ptr = (m_ptr + 1) % m_fftLen;

	/* calculate the wanted bins */
	for (int i = m_first; i < m_last; i++) {
		double zReal = m_binsReal[i];
		double zImag = m_binsImag[i];

		zReal -= oldReal;
		zImag -= oldImag;

		zReal += in;

		m_binsReal[i] = zReal * m_cos[i] - zImag * m_sin[i];
		m_binsImag[i] = zReal * m_sin[i] + zImag * m_cos[i];

		m_bins[i] = ::CABS(m_binsReal[i], m_binsImag[i]);
	}

	return m_bins;
}
