#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <net/if.h>
#include <unistd.h>
#include <string.h>
#include "netconf.h"

static int ifconfig_ioctl(
	int fd,
	const char *ifname,
	int cmd,
	struct ifreq &ifr)
{
	strcpy(ifr.ifr_name, ifname);
	return ioctl(fd, cmd,&ifr);
}


static void ifconfig_format (struct sockaddr &adr, char *buf)
{
	struct sockaddr_in *sin = (struct sockaddr_in*)&adr;
	ipnum_ip2a (htonl(sin->sin_addr.s_addr),buf);
}

/*
	Fetch the inteface configuration from the kernel.
	Assume the device is already available in the kernel
	Return -1 if any error.
*/
int ifconfig_getinfo_nocheck (
	const char *ifname,
	IFCONFIG_INFO &info)
{
	int ret = -1;
	memset (&info,0,sizeof(IFCONFIG_INFO));
	int skfd = socket(AF_INET, SOCK_DGRAM, 0);
	if (skfd != -1){
		struct ifreq ifr;
		if (ifconfig_ioctl(skfd, ifname, SIOCGIFFLAGS, ifr) >= 0){
			info.flags = ifr.ifr_flags;
			ret = 0;
			// In linux 2.1, unconfigured interfaces fail for the
			// following calls, while with 2.0 and below, they
			// return 0 values.
			if (ifconfig_ioctl(skfd,ifname,SIOCGIFADDR, ifr) >= 0){
				ifconfig_format (ifr.ifr_addr,info.ip_addr);
				if (ifconfig_ioctl(skfd,ifname, SIOCGIFDSTADDR, ifr) >= 0) {
					ifconfig_format (ifr.ifr_dstaddr,info.dst_addr);
					if (ifconfig_ioctl(skfd, ifname,SIOCGIFNETMASK, ifr) >= 0) {
						ifconfig_format (ifr.ifr_netmask,info.netmask);
						if (ifconfig_ioctl(skfd, ifname,SIOCGIFBRDADDR, ifr) >= 0) {
							ifconfig_format (ifr.ifr_broadaddr,info.bcast);
						}
					}
				}
			} else {
			// return zero values in case of failure,
			// to be symmetrical with the ifconfig command.
				ipnum_ip2a((unsigned long)0,info.ip_addr);
				strcpy(info.dst_addr, info.ip_addr);
				strcpy(info.netmask, info.ip_addr);
				strcpy(info.bcast, info.ip_addr);
			}
		}
		close (skfd);
	}
	return ret;
}

/*
	Fetch the inteface configuration from the kernel.
	Check if the device is already available in the kernel before
	attempting to probe it. This avoid the on-demand loading of the device
	Return -1 if any error.
*/
EXPORT int ifconfig_getinfo (
	const char *ifname,
	IFCONFIG_INFO &info)
{
	int ret = -1;
	if (devlist_devexist(ifname)){
		ret = ifconfig_getinfo_nocheck(ifname,info);
	}
	return ret;
}
#ifdef TEST

int main (int, char *argv[])
{
	IFCONFIG_INFO info;
	if (ifconfig_getinfo(argv[1],info) == -1){
		fprintf (stderr,"No info on network device %s\n",argv[1]);
	}else{
		printf ("Network devices %s\n",argv[1]);
		printf ("flags    : %04x\n",info.flags);
		printf ("addr     : %s\n",info.ip_addr);
		printf ("dst addr : %s\n",info.dst_addr);
		printf ("netmask  : %s\n",info.netmask);
	}
	return 0;
}

#endif


