/*
 * $Id: attrib.c,v 1.16 2001/06/13 00:40:49 antona Exp $
 *
 * attrib.c - Attribute handling code. Part of the Linux-NTFS project.
 *
 * Copyright (c) 2000,2001 Anton Altaparmakov.
 *
 * This program/include file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program/include file is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS 
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stddef.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>

#include "types.h"
#include "attrib.h"
#include "unistr.h"
#include "endians.h"
#include "disk_io.h"
#include "support.h"
#include "mft.h"

int write_non_resident_attr(attr *a)
{
	return -ENOTSUP;
}

int ntfs_flush_attr(attr *a)
{
	int err = 0;

	if (!AttrDirty(a) || !AttrMapped(a))
		return 0;
	if (!AttrNonResident(a)) {
		/* Resident, just flush the mft entry. */
		if (err = ntfs_flush_mft_entry(a->a_m_entry))
			return err;
		ClearAttrDirty(a);
	} else	/* Non-resident, write out to disk. */
		err = write_non_resident_attr(a);
	ntfs_flush();
	return err;
}

BOOL __set_attr_dirty(attr *a)
{
	ntfs_volume *v;
	
	/* If resident just mark mft entry as dirty. */
	if (!AttrNonResident(a)) {
		ntfs_set_mft_entry_dirty(a->a_m_entry);
		return;
	}
	/* It is non-resident so add to dirty list. */
	v = a->a_vol;
	list_add_tail(&a->a_dirty_list, &v->dirty_attrs);
	v->nr_dirty_attrs++;
	return TRUE;
}

__inline__ attr *__allocate_attr(void)
{
	attr *a = (attr *)malloc(sizeof(attr));
	if (!a)
		return NULL;
	memset(a, 0, sizeof(attr));
	INIT_LIST_HEAD(&a->a_dirty_list);
	return a;
}

__inline__ int __free_attr(attr *a)
{
	if (a->a_count || AttrDirty(a) || AttrMapped(a))
		return -EBUSY;
	if (a->a_run_list)
		free(a->a_run_list);
	free(a);
	return 0;
}

int __map_attr_value(attr *a)
{
	if (!AttrNonResident(a)) {
	}
	return -ENOTSUP;
}

int __unmap_attr_value(attr *a)
{
	if (!AttrMapped(a))
		return 0;
	return -ENOTSUP;
}
 
int insert_attr(mft_entry *me, attr **a, ATTR_RECORD *arec, attr_val *aval,
		const BOOL dirty, const BOOL nonresident, run_list *arl,
		const BOOL inattrlist, ATTR_LIST_ENTRY *alist,
		ATTR_LIST_ENTRY *alistentry, const BOOL inextrecord)
{
	attr *aent;
	
	if (!*a && !(*a = __allocate_attr()))
		return -errno;
	aent = *a;
	aent->a_rec = arec;
	if (aval) {
		if (nonresident) {
			aent->a_run_list = arl;
			SetAttrNonResident(aent);
		}
		aent->a_val = aval;
		
	} else {
		if (__map_attr_value(aent)) {
			// error
		}
	}
	
	return -ENOTSUP;
}

int remove_attr(attr *a)
{
	return -ENOTSUP;
}

/**
 * find_attr - find (next) attribute in mft record
 * @type:	attribute type to find
 * @name:	attribute name to find (optional, i.e. NULL means don't care)
 * @name_len:	attribute name length (only needed if @name present)
 * @ic:		IGNORE_CASE or CASE_SENSITIVE (ignored if @name not present)
 * @upcase:	unicode upcase table
 * @ucpase_len:	length of upcase table in unicode characters
 * @val:	attribute value to find (optional, resident attributes only)
 * @val_len:	attribute value length
 * @ctx:	search context with mft record and attribute to search from
 *
 * find_attr() takes a search context @ctx as parameter and searches the mft
 * record specified by @ctx->mrec, beginning at @ctx->attr, for an attribute of
 * @type, optionally @name and @val. If found, find_attr() returns TRUE and
 * @ctx->attr will point to the found attribute. If not found, find_attr()
 * returns FALSE and @ctx->attr is undefined (i.e. do not rely on it not
 * changing).
 *
 * If @ctx->is_first is TRUE, the search begins with @ctx->attr itself. If it
 * is FALSE, the search begins after @ctx->attr.
 *
 * If @ic is IGNORE_CASE, the @name comparisson is not case sensitive and if it
 * is CASE_SENSITIVE, the comparison is case sensitive. When @name is present,
 * @name_len is the @name length in Unicode characters.
 *
 * If @name is not present (NULL), we assume that the unnamed attribute is
 * being searched for.
 *
 * Finally, the resident attribute value @val is looked for, if present. If @val
 * is not present (NULL), @val_len is ignored.
 *
 * find_attr() only searches the specified mft record and it ignores the
 * presence of an attribute list attribute (unless it is the one being searched
 * for, obviously). If you need to take attribute lists into consideration, use
 * lookup_attr() instead (see below). This also means that you cannot use
 * find_attr() to search for extent records of non-resident attributes, as
 * extents with lowest_vcn != 0 are usually described by the attribute list
 * attribute only. - Note that it is possible that the first extent is only in
 * the attribute list while the last extent is in the base mft record, so don't
 * rely on being able to find the first extent in the base mft record.
 *
 * Warning: Never use @val when looking for attribute types which can be
 *	    non-resident as this most likely will result in a crash!
 */
BOOL find_attr(const ATTR_TYPES type, const uchar_t *name, const __u32 name_len,
		const IGNORE_CASE_BOOL ic, const uchar_t *upcase,
		const __u32 upcase_len, const __u8 *val, const __u32 val_len,
		attr_search_context *ctx)
{
	ATTR_RECORD *a;
	
#ifdef DEBUG
	if (!ctx || !ctx->mrec || !ctx->attr) {
		fprintf(stderr, "find_attr() received NULL pointer!\n");
		return FALSE;
	}
#endif	
	/*
	 * Iterate over attributes in mft record starting at @ctx->attr, or the
	 * attribute following that, if @ctx->is_first is TRUE.
	 */
	if (ctx->is_first) {
		a = ctx->attr;
		ctx->is_first = FALSE;
	} else
		a = (ATTR_RECORD*)((char*)ctx->attr +
						le32_to_cpu(ctx->attr->length));
	for (;;	a = (ATTR_RECORD*)((char*)a + le32_to_cpu(a->length))) {
		if (p2n(a) < p2n(ctx->mrec) || (char*)a > (char*)ctx->mrec +
					le32_to_cpu(ctx->mrec->bytes_allocated))
			break;
		ctx->attr = a;
		/* We catch $END with this more general check, too... */
		if (le32_to_cpu(a->type) > le32_to_cpu(type))
			return FALSE;
		if (!a->length)
			break;
		if (a->type != type)
			continue;
		/* 
		 * If @name is present, compare the two names. If @name is
		 * missing, assume we want an unnamed attribute.
		 */
		if (!name) {
			/* The search failed if the found attribute is named. */
			if (a->name_length)
				return FALSE;
		} else if (!ntfs_are_names_equal(name, name_len,
			    (uchar_t*)((char*)a + le16_to_cpu(a->name_offset)),
			    a->name_length, ic, upcase, upcase_len)) {
			register int rc;
			
			rc = ntfs_collate_names(name, name_len,
					(uchar_t*)((char*)a +
						le16_to_cpu(a->name_offset)),
					a->name_length, 1, IGNORE_CASE,
					upcase, upcase_len);
			/*
			 * If @name collates before a->name, there is no
			 * matching attribute.
			 */
			if (rc == -1)
				return FALSE;
			/* If the strings are not equal, continue search. */
			if (rc)
	 			continue;
			rc = ntfs_collate_names(name, name_len,
					(uchar_t*)((char*)a +
						le16_to_cpu(a->name_offset)),
					a->name_length, 1, CASE_SENSITIVE,
					upcase, upcase_len);
			if (rc == -1)
				return FALSE;
			if (rc)
				continue;
		}
		/*
		 * The names match or @name not present and attribute is
		 * unnamed. If no @val specified, we have found the attribute
		 * and are done.
		 */
		if (!val)
			return TRUE;
		/* @val is present; compare values. */
		else {
			register int rc;
			
			rc = memcmp(val, (char*)a +le16_to_cpu(a->value_offset),
				min(val_len, le32_to_cpu(a->value_length)));
			/*
			 * If @val collates before the current attribute's
			 * value, there is no matching attribute.
			 */
			if (!rc) {
				register __u32 avl;
				avl = le32_to_cpu(a->value_length);
				if (val_len == avl)
					return TRUE;
				if (val_len < avl)
					return FALSE;
			} else if (rc < 0)
				return FALSE;
		}
	}
#ifdef DEBUG
	fprintf(stderr, "find_attr(): File is corrupt. Run chkdsk.\n");
#endif
	return FALSE;
}

/**
 * find_first_attr - find first attribute in mft record
 * @type:	attribute type to find
 * @name:	attribute name to find (optional, i.e. NULL means don't care)
 * @name_len:	attribute name length (only needed if @name present)
 * @ic:		IGNORE_CASE or CASE_SENSITIVE (ignored if @name not present)
 * @upcase:	unicode upcase table
 * @ucpase_len:	length of upcase table in unicode characters
 * @val:	attribute value to find (optional, resident attributes only)
 * @val_len:	attribute value length
 * @ctx:	search context with mft record and attribute to search from
 *
 * find_first_attr() is just an inlined wrapper for find_attr(). For details
 * see description of find_attr() above.
 */
__inline__ BOOL find_first_attr(const ATTR_TYPES type, const uchar_t *name,
		const __u32 name_len, const IGNORE_CASE_BOOL ic,
		const uchar_t *upcase, const __u32 upcase_len,
		const __u8 *val, const __u32 val_len, attr_search_context *ctx)
{
	MFT_RECORD *m;
	ATTR_RECORD *a;
	
#ifdef DEBUG
	if (!ctx || !ctx->mrec) {
		fprintf(stderr, "find_first_attr() received NULL pointer!\n");
		return FALSE;
	}
	if (ctx->attr)
		fprintf(stderr, "find_first_attr(): received non-NULL "
				"attribute pointer.\nThis will be overwritten "
				"resulting in possible memory leakage.\n");
#endif
	m = ctx->mrec;
	ctx->is_first = TRUE;
	ctx->attr = a = (ATTR_RECORD*)((char*)m + le16_to_cpu(m->attrs_offset));
	if (p2n(a) >= p2n(m) && (char*)a <= (char*)m +
						le32_to_cpu(m->bytes_allocated))
		return find_attr(type, name, name_len, ic, upcase, upcase_len,
				val, val_len, ctx);
#ifdef DEBUG
	fprintf(stderr, "find_first_attr(): file is corrupt.");
#endif
	return FALSE;
}

/* FIXME: Need to write the new flags to disk. */
int set_ntfs_volume_flags(ntfs_volume *v, MFT_RECORD *b,
			  const __u16 flags)
{
	ATTR_RECORD *r;
	VOLUME_INFORMATION *c;
	attr_search_context ctx;

	/* Sanity checks. */
	if (!b || !is_mft_recordp(b))
                return 0;
	/* Get a pointer to the volume information attribute. */
	memset(&ctx, 0, sizeof(attr_search_context));
	ctx.mrec = b;
	if (!find_first_attr($VOLUME_INFORMATION, NULL, 0, 0, NULL, 0, NULL, 0,
			     &ctx)) {
		fprintf(stderr, "Error: Attribute $VOLUME_INFORMATION was " \
				"not found in $Volume!\n");
		return 0;
	}
	r = ctx.attr;
	/* Sanity check. */
	if (r->non_resident) {
		fprintf(stderr, "Error: Attribute $VOLUME_INFORMATION must " \
				"be resident (and it isn't)!\n");
		return 0;
	}
	/* Get a pointer to the value of the attribute. */
	c = (VOLUME_INFORMATION*)(le16_to_cpu(r->value_offset) + (char*)r);
	/* Sanity checks. */
	if ((char*)c + le32_to_cpu(r->value_length) > 
				le16_to_cpu(b->bytes_in_use) + (char*)b ||
	    le16_to_cpu(r->value_offset) + le32_to_cpu(r->value_length) >
				le32_to_cpu(r->length)) {
		fprintf(stderr, "Error: Attribute $VOLUME_INFORMATION in " \
				"$Volume is corrupt!\n");
		return 0;
	}
	/* Set the volume flags. */
	c->flags = cpu_to_le16(flags);
	/* Success! */
	return 1;
}

__s64 get_attribute_value_length(const ATTR_RECORD *a)
{
	if (!a) {
		errno = EINVAL;
		return 0;
	}
	errno = 0;
	if (a->non_resident)
		return sle64_to_cpu(a->data_size);
	else
		return (__s64)le32_to_cpu(a->value_length);
	errno = EINVAL;
	return 0;
}

__s64 get_attribute_value(const ntfs_volume *vol, const MFT_RECORD *m,
			  const ATTR_RECORD *a, __u8 *b)
{
	/* Sanity checks. */
	if (!vol || !m || !a || !b) {
		errno = EINVAL;
		return 0;
	}
	/* Complex attribute? */
	if (a->flags) {
		puts("Enountered non-zero attribute flags. Cannot handle this "
		     "yet.");
		errno = ENOTSUP;
		return 0;
	}
	if (!a->non_resident) {		/* Attribute is resident. */
		/* Sanity check. */
		if (le32_to_cpu(a->value_length) +
		    le16_to_cpu(a->value_offset) > le32_to_cpu(a->length)) {
			return 0;
		}
		memcpy(b, (char*)a + le16_to_cpu(a->value_offset),
			  		le32_to_cpu(a->value_length));
		errno = 0;
		return (__s64)le32_to_cpu(a->value_length);
	} else {			/* Attribute is not resident. */
		run_list *rl;
		__s64 total;
		int r, i;

		/* If no data, return 0. */
		if (!(a->data_size)) {
			errno = 0;
			return 0;
		}
		/*
		 * FIXME: What about attribute lists?!? (AIA)
		 */
		/* Decompress the mapping pairs array into a run list. */
		rl = decompress_run_list(a);
		if (!rl) {
			errno = EINVAL;
			return 0;
		}
		/*
		 * FIXED: We were overflowing here in a nasty fashion when we
		 * reach the last cluster in the run list as the buffer will
		 * only be big enough to hold data_size bytes while we are 
		 * reading in allocated_size bytes which is usually larger
		 * than data_size, since the actual data is unlikely to have a
		 * size equal to a multiple of the cluster size!
		 */
		/* Now load all clusters in the run list into b. */
		for (i = 0, total = 0; rl[i].length; i++) {
			if (!rl[i+1].length) {
				unsigned char *intbuf = NULL;
				/*
				 * We have reached the last run so we were
				 * going to overflow when executing the
				 * ntfs_pread() which is BAAAAAAAD!
				 * Temporary fix:
				 * 	Allocate a new buffer with size:
				 * 	rl[i].length << vol->cluster_size_bits,
				 * 	do the read into our buffer, then
				 * 	memcpy the correct amount of data into
				 * 	the caller supplied buffer, free our
				 * 	buffer, and continue.
				 */
				intbuf = malloc(rl[i].length << 
							vol->cluster_size_bits);
				if (!intbuf) {
					int eo = errno;
					perror("Couldn't allocate memory for "
							"internal buffer.\n");
					free(rl);
					errno = eo;
					return 0;
				}
				/*
				 * FIXME: If compressed file: Only read if 
				 * lcn != -1. Otherwise, we are dealing with a 
				 * sparse run and we just memset the user buffer
				 * to 0 for the length of the run, which should
				 * be 16 (= compression unit size). 
				 * FIXME: Really only when file is compressed,
				 * or can we have sparse runs in uncompressed
				 * files as well?
				 */
				r = ntfs_pread(vol->fd, intbuf,
					       vol->cluster_size, 
					       rl[i].length,
					       rl[i].lcn << 
							vol->cluster_size_bits);
				if (r != rl[i].length) {
#define ESTR "Error reading attribute value"
					if (r == -1) {
						int eo = errno;
						perror(ESTR);
						errno = eo;
					} else if (r < rl[i].length) {
						fprintf(stderr, ESTR ": Ran "
							"out of input data.\n");
						errno = EIO;
					} else {
			                        fprintf(stderr, ESTR ": "
							   "unknown error\n");
						errno = EIO;
					}
#undef ESTR
					free(rl);
					return 0;
				}
				memcpy(b + total, intbuf, 
				       sle64_to_cpu(a->data_size) - total);
				free(intbuf);
				total = sle64_to_cpu(a->data_size);
			} else {
				/*
				 * FIXME: If compressed file: Only read if 
				 * lcn != -1. Otherwise, we are dealing with a 
				 * sparse run and we just memset the user buffer
				 * to 0 for the length of the run, which should
				 * be 16 (= compression unit size).
				 */
				r = ntfs_pread(vol->fd, b + total,
					       vol->cluster_size, 
					       rl[i].length,
					       rl[i].lcn <<
							vol->cluster_size_bits);
				if (r != rl[i].length) {
#define ESTR "Error reading attribute value"
					if (r == -1) {
						int eo = errno;
						perror(ESTR);
						errno = eo;
					} else if (r < rl[i].length) {
						fprintf(stderr, ESTR ": Ran "
							"out of input data.\n");
						errno = EIO;
					} else {
			                        fprintf(stderr, ESTR ": "
							   "unknown error\n");
						errno = EIO;
					}
#undef ESTR
					return 0;
				}
				total += r * vol->cluster_size;
			}
		}
		free(rl);
		return total;
	}
	errno = EINVAL;
	return 0;
}

int set_attribute_value(ntfs_volume *vol, ATTR_RECORD *a, 
			const __u8 *b, __s64 l)
{
	/* Sanity check. */
	if (!vol || !a || !b) {
		errno = EINVAL;
		return 0;
	}
	/* FIXME: We can't deal with changes in attribute value length at the
	   moment. */
	if ((l != get_attribute_value_length(a)) || !l) {
		errno = ENOTSUP;
		return 0;
	}
	/* Complex attribute? */
	if (a->flags) {
		puts("Enountered non-zero attribute flags. Cannot handle this "
		     "yet.");
		errno = ENOTSUP;
		return 0;
	}
	if (!a->non_resident) {		/* Attribute is resident. */
		/* Sanity check. */
		if (le32_to_cpu(a->value_length) +
		    le16_to_cpu(a->value_offset) > le32_to_cpu(a->length)) {
			errno = EINVAL;
			return 0;
		}
		memcpy((char*)a + le16_to_cpu(a->value_offset), b, l);
		errno = 0;
		return 1;
	} else {			/* Attribute is not resident. */
		run_list *rl;
		__s64 total;
		int w, i;
		
		/* If no data return 1. FIXME: This needs to change when we can
		   cope with changes in length of the attribute. */
		if (!(a->data_size)) {
			errno = 0;
			return 1;
		}
		/* Decompress the mapping pairs array into a run list. */
		rl = decompress_run_list(a);
		/* Now write all clusters to disk from b. */
		for (i = 0, total = 0; rl[i].length && (l > 0); i++) {
			w = ntfs_pwrite(vol->fd, b + total,
				min(l, rl[i].length << vol->cluster_size_bits),
				rl[i].lcn << vol->cluster_size_bits);
			if (w != min(l, rl[i].length <<
						vol->cluster_size_bits)) {
#define ESTR "Error writing attribute value"
				if (w == -1) {
					int eo = errno;
					perror(ESTR);
					errno = eo;
				} else if (w < min(l, 
						rl[i].length << 
						vol->cluster_size_bits)) {
					fprintf(stderr, ESTR ": Ran out of "
							"input data.\n");
					errno = EIO;
				} else {
		                        fprintf(stderr, ESTR ": unknown "
							"error\n");
					errno = EIO;
				}
#undef ESTR
				return 0;
			}
			total += w;
			l -= w;
			if ((l <= 0) && rl[i+1].length) {
				fprintf(stderr, 
				       "Warning: Amount of data to write "
				       "doesn't match attribute length in\n"
				       "run list! - The old data that was "
				       "present beyond the new data written\n"
				       "has been preserved.\n");
			} else if ((l > 0) && !rl[i+1].length) {
				fprintf(stderr, 
				       "Warning: Still have data to write but "
				       "the attribute length in the run list\n"
				       "was exceeded! - Data written has been "
				       "truncated!\n");
			}
		}
		free(rl);	
		return 1;
	}
	errno = EINVAL;
	return 0;
}

run_list *decompress_run_list(const ATTR_RECORD *attr)
{
	VCN vcn;		/* Current vcn. */
	LCN lcn; 		/* Current lcn. */
	__s64 deltaxcn; 	/* Change in [vl]cn. */
	run_list *rl = NULL;	/* The output run_list. */
	run_list *rl2;		/* Temporary run_list. */
	__u8* buf;		/* Current position in mapping pairs array. */
	int rlsize;		/* Size of run_list buffer. */
	__u16 rlpos;		/* Current run_list positions in units
				   of run_list_elements. */
	__u8 b;			/* Current byte offset in buf. */

	/* Make sure attr exists and is non-resident. */
	if (!attr || !attr->non_resident)
		return NULL;
	/* FIXME: Compression not supported yet. */
	if (attr->compression_unit) {
		puts("Encountered compressed non-resident attribute."
		     "Cannot handle this yet!");
		return NULL;
	}
	/* Start at vcn = lowest_vcn and lcn 0. */
	vcn = sle64_to_cpu(attr->lowest_vcn);
	lcn = 0;
	/* Get start of the mapping pairs array. */
	buf = (__u8*)attr + le16_to_cpu(attr->mapping_pairs_offset);
	/* Current position in run_list array. */
	rlpos = 0;
	/* Current run_list buffer size in bytes. */
	rlsize = 0;
	while (*buf) {
		/* Allocate (more) memory if needed. */
		if (rlpos * (int)sizeof(run_list_element) >= 
				rlsize - (int)sizeof(run_list_element)) {
			rl2 = malloc(rlsize + 16 * sizeof(run_list_element));
			if (!rl2) {
				if (rl)
					free(rl);
				return NULL;
			}
			if (rl) {
				memmove(rl2, rl, rlsize);
				free(rl);
			}
			rl = rl2;
			rlsize += 16 * sizeof(run_list_element);
		}
		/* Enter the current vcn into the current run_list_element. */
		(rl + rlpos)->vcn = vcn;
		/* 
		 * Get the change in vcn, i.e. the run length in clusters.
		 * Doing it this way ensures that we signextend negative values.
		 * A negative run length doesn't make any sense, but hey, I
		 * didn't make up the NTFS specs and Windows NT4 treats the run
		 * length as a signed value so that's how it is...
		 */
		b = *buf & 0xf;
		if (b) {
			for (deltaxcn = (__s8)buf[b--]; b; b--)
				deltaxcn = (deltaxcn << 8) + buf[b];
		} else { /* The length entry is compulsory. */
#ifdef DEBUG
			fprintf(stderr, "DEBUG Linux-NTFS: No length entry in "
					"mapping pairs array.\n"
					"decompress_run_list() failed.\n");
#endif
			deltaxcn = (__s64)-1;
		}
		/* Assume a negative length to indicate data corruption and
		   hence clean-up and return NULL. */
		if (deltaxcn < 0) {
			free(rl);
			return NULL;
		}
		/* Enter the current run length into the current 
		   run_list_element. */
		(rl + rlpos)->length = deltaxcn;
		/* Increment the current vcn by the current run length. */
		vcn += deltaxcn;
		/* There might be no lcn change at all, as is the case for
		 * sparse clusters on NTFS 3.0+, in which case we set the lcn
		 * to -1. */
		if (!(*buf & 0xf0))
			(rl + rlpos)->lcn = -1;
		else {
			/* Get the lcn change which really can be negative. */
			for (b = (*buf & 0xf) + ((*buf >> 4) & 0xf), deltaxcn =
					(__s8)buf[b--]; b > (*buf & 0xf); b--)
				deltaxcn = (deltaxcn << 8) + buf[b];
			/* Change the current lcn to it's new value. */
			lcn += deltaxcn;
#ifdef DEBUG
			/*
			 * On NTFS 1.2-, apparently can have lcn == -1 to
			 * indicate a hole. But we haven't verified ourselves
			 * whether it is really the lcn or the deltaxcn that is
			 * -1. So if either is found give us a message so we
			 * can investigate it further! (AIA)
			 */
			if (deltaxcn == -1LL)
				fprintf(stderr, "DEBUG Linux-NTFS: lcn delta "
						"== -1\n");
			else if (lcn == -1LL)
				fprintf(stderr, "DEBUG Linux-NTFS: lcn "
						"== -1\n");
			if (deltaxcn < -1LL)
				fprintf(stderr, "DEBUG Linux-NTFS: lcn delta "
						"< -1\n");
			else if (lcn < -1LL)
				fprintf(stderr, "DEBUG Linux-NTFS: lcn "
						"< -1\n");
#endif
			/* Enter the current lcn into the run_list_element. */
			(rl + rlpos)->lcn = lcn;
		}
		/* Get to the next run_list_element. */
		rlpos++;
		/* Increment the buffer position to the next mapping pair. */
		buf += (*buf & 0xf) + ((*buf >> 4) & 0xf) + 1;
	}
	/* If there were no mapping pairs in the run_list, then rl will stil
	   be NULL. */
	if (rl)	{
		/* Adjust buffer size to minimum before returning. */
		rl2 = malloc((rlpos + 1) * sizeof(run_list_element));
		if (!rl2) {
			if (rl)
				free(rl);
			return NULL;
		}
		memmove(rl2, rl, (rlpos + 1) * sizeof(run_list_element));
		free(rl);
		rl = rl2;
		/* Setup terminating run_list_element. */
		(rl + rlpos)->vcn = vcn;
		(rl + rlpos)->length = 0;
		/*
		 * This is not necessary, but do it for completeness sake.
		 * FIXME: Do we really want this -1? -1 means it is sparse,
		 * but I guess better that than an undefined random value...
		 */
		(rl + rlpos)->lcn = -1;
		/* The final vcn - 1 has to be equal to highest_vcn or
		   something has gone badly wrong. */
		if (vcn - 1 != sle64_to_cpu(attr->highest_vcn)) {
			free(rl);
			puts("Corrupt Mapping Pairs Array in non-resident "
			     "attribute!");
			return NULL;
		}
	}
#ifdef DEBUG
	if (rl) {
		int i;
		
		puts("\nMapping Pairs array successfully decompressed.");
		puts("Resulting run list (values in hexadecimal):");
		puts("VCN              LCN              Run length");
		for (i = 0; ; i++) {
			if ((rl + i)->lcn == -1)
				printf("%-16Lx %-16i %-16Lx (%s)\n",
					(rl + i)->vcn, -1, (rl + i)->length,
					(rl + i)->length ? "sparse run" :
							   "run list end");
			else
				printf("%-16Lx %-16Lx %-16Lx%s\n",
					(rl + i)->vcn, (rl + i)->lcn,
					(rl + i)->length, (rl + i)->length ?
					"" : " (run list end)");
			if (!(rl + i)->length)
				break;
		}
	}
#endif
	/* Finally finished! */
	return rl;
}

LCN vcn_to_lcn(const run_list *rl, const VCN vcn)
{
	int i = 0;
	
	if (!rl || vcn < 0) {
		errno = EINVAL;
		return 0;
	}
	if (!rl[0].length || vcn < rl[0].vcn) {
		errno = ENOENT;
		return 0;
	}
	do {
		if (vcn < rl[i+1].vcn) {
			if (rl[i].lcn == (LCN)-1)
				return (LCN)-1;
			errno = 0;
			return rl[i].lcn + (vcn - rl[i].vcn);
		}
	} while (rl[++i].length);
	errno = ENOENT;
	return 0;
}

