/*
 * $Id: ntfs_rec.h,v 1.7 2001/06/01 02:07:24 antona Exp $
 *
 * ntfs_rec.h - Exports for NTFS record handling. Part of the Linux-NTFS
 * 		project.
 *
 * Copyright (c) 2000,2001 Anton Altaparmakov.
 *
 * This program/include file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program/include file is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS 
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef NTFS_REC_H
#define NTFS_REC_H

#include "types.h"
#include "endians.h"

/**
 * __post_read_mst_fixup - fast deprotect multi sector transfer protected data
 * @b:		pointer to the data to deprotect
 * @size:	size in bytes of @b
 * 
 * Perform the necessary post read multi sector transfer fixup, not checking for
 * any errors. Defined inline for additional speed. 
 */
extern __inline__ void __post_read_mst_fixup(NTFS_RECORD *b, const __u32 size);

/**
 * post_read_mst_fixup - deprotect multi sector transfer protected data
 * @b:		pointer to the data to deprotect
 * @size:	size in bytes of @b
 * 
 * Perform the necessary post read multi sector transfer fixup and detect the
 * presence of incomplete multi sector transfers. - In that case, overwrite the
 * magic of the ntfs record header being processed with "BAAD" (in memory only!)
 * and abort processing.
 * Return 1 on success and 0 on error ("BAAD" magic will be present).
 * NOTE: We consider the absence / invalidity of an update sequence array to
 * mean that the structure is not protected at all and hence doesn't need to
 * be fixed up. Thus, we return success and not failure in this case.
 */
int post_read_mst_fixup(NTFS_RECORD *b, const __u32 size);

/**
 * pre_write_mst_fixup - apply multi sector transfer protection
 * @b:		pointer to the data to protect
 * @size:	size in bytes of @b
 * 
 * Perform the necessary pre write multi sector transfer fixup on the data
 * pointer to by @b of @size.
 * Return 1 if fixup applied (success) or 0 if no fixup was performed
 * (assumed not needed).
 * NOTE: We consider the absence / invalidity of an update sequence array to
 * mean that the structure is not subject to protection and hence doesn't need
 * to be fixed up. This means that you have to create a valid update sequence
 * array header in the ntfs record before calling this function, otherwise it
 * will fail and return 0 (the header needs to contain the position of the
 * update seqeuence array together with the number of elements in the array).
 * You also need to initialise the update sequence number before calling this
 * function otherwise a random word will be used (whatever was in the record
 * at that position at that time).
 */
int pre_write_mst_fixup(NTFS_RECORD *b, const __u32 size);

#endif /* defined NTFS_REC_H */

