/* $Id: wizard.c,v 1.4 2005/03/28 16:24:40 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2004-2005 Marcus von Appen. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* TODO:
 * - fix up title to respect GNOME HIG v2.0
 */

#include "../errors.h"
#include "hig-widgets.h"
#include "gtkdefs.h"

#include "util.h"
#include "wizard.h"

extern lingGtkMainWindow *main_app;

#define IS_LING_WIZARD(obj) (g_object_get_data (G_OBJECT (obj), "wizard_ref"))

typedef struct 
{
     GtkWidget *window;
     GtkWidget *notebook;
     GtkWidget *btn_next;
     GtkWidget *btn_prev;
     GList     *page_cbs;  /* creates the page contents */
     GList     *page_data; /* data to pass to the callbacks */
     GList     *callbacks;
     GList     *cb_data;
     gint     current;
} lingWizard;

static void destroy_page_content (lingWizard *wizard, gint page);
static void create_page_content (lingWizard *wizard, gint page);
static gboolean run_page_callback (lingWizard *wizard, gint page);
static void switch_page_next (lingWizard *wizard);
static void switch_page_prev (lingWizard *wizard);

static void 
destroy_page_content (lingWizard *wizard, gint page)
{
     GtkWidget *box_content;
     GtkBoxChild *last;

     /* destroy current page content */
     box_content = gtk_notebook_get_nth_page (GTK_NOTEBOOK (wizard->notebook),
                                              page);

     last = 
          (GtkBoxChild *) g_list_last (GTK_BOX (box_content)->children)->data;

     gtk_widget_destroy (last->widget);
     return;
}

static void
create_page_content (lingWizard *wizard, gint page)
{
     WizPageCallback callback;
     gpointer *data;
     GtkWidget *box_content;
     GtkWidget *widget;

     callback = g_list_nth_data (wizard->page_cbs, (guint) page - 1);
     data = g_list_nth_data (wizard->page_data, (guint) page - 1);
     
     box_content = gtk_notebook_get_nth_page (GTK_NOTEBOOK (wizard->notebook),
                                              page);
     widget = callback (data);
     
     gtk_box_pack_end (GTK_BOX (box_content), widget, TRUE, TRUE, 0);
     gtk_widget_show_all (widget);
     return;
}

static gboolean
run_page_callback (lingWizard *wizard, gint page)
{
     WizardCallback callback;
     gpointer *data;

     callback = g_list_nth_data (wizard->callbacks, (guint) page - 1);
     data = g_list_nth_data (wizard->cb_data, (guint) page - 1);
     
     if (callback)
          return callback (data);
     return TRUE;
}

static void
switch_page_next (lingWizard *wizard)
{
     debug ("Creating new page content...\n");
     if (wizard->current != 0)
     {
          if (!run_page_callback (wizard, wizard->current)) 
               return; /* return on failure */
          destroy_page_content (wizard, wizard->current);
     }

     wizard->current++;
     debug ("Switching to page %d\n", wizard->current);
     create_page_content (wizard, wizard->current);

     /* now set the current page */
     gtk_notebook_set_current_page (GTK_NOTEBOOK (wizard->notebook),
                                    wizard->current);

     if ((wizard->current > 0) && !wizard->btn_prev)
     {
          wizard->btn_prev = UTIL_CREATE_BUTTON_PREV ();
          g_signal_connect_swapped (G_OBJECT (wizard->btn_prev), "clicked",
                                    G_CALLBACK (switch_page_prev), wizard);
          gtk_box_pack_start (GTK_BOX (wizard->btn_next->parent),
                              wizard->btn_prev, FALSE, FALSE, 0);
          gtk_widget_show_all (wizard->btn_prev);

          /* dumb bug with pack_end && pack_start */
          gtk_widget_destroy (wizard->btn_next);
          wizard->btn_next = UTIL_CREATE_BUTTON_NEXT ();
          g_signal_connect_swapped (G_OBJECT (wizard->btn_next), "clicked",
                                    G_CALLBACK (switch_page_next), wizard);
          gtk_box_pack_end (GTK_BOX (wizard->btn_prev->parent),
                            wizard->btn_next, FALSE, FALSE, 0);
          gtk_widget_show_all (wizard->btn_next);
     }

     if (wizard->current ==
         (gtk_notebook_get_n_pages (GTK_NOTEBOOK (wizard->notebook)) - 1))
     {
          gtk_widget_destroy (wizard->btn_next);
          wizard->btn_next = util_create_button_with_image (_("_Finish"),
                                                            GTK_STOCK_APPLY,
                                                            TRUE);
          gtk_dialog_add_action_widget (GTK_DIALOG (wizard->window),
                                        wizard->btn_next, GTK_RESPONSE_OK);
          gtk_widget_show_all (wizard->btn_next);
     }
     return;
}

static void
switch_page_prev (lingWizard *wizard)
{
     GtkWidget *box;

     /* if (!run_page_callback (wizard, wizard->current))
      *      return;
      */

     destroy_page_content (wizard, wizard->current);

     wizard->current--;
     debug ("Switching to page %d\n", wizard->current);
     if (wizard->current != 0)
          create_page_content (wizard, wizard->current);

     gtk_notebook_set_current_page (GTK_NOTEBOOK (wizard->notebook),
                                    wizard->current);

     if (wizard->current == 0)
     {
          gtk_widget_destroy (wizard->btn_prev);
          wizard->btn_prev = NULL;
     }
     
     if (wizard->current < 
         (gtk_notebook_get_n_pages (GTK_NOTEBOOK (wizard->notebook)) - 1))
     {
          box = wizard->btn_next->parent;
          gtk_widget_destroy (wizard->btn_next);
          wizard->btn_next = UTIL_CREATE_BUTTON_NEXT();
          g_signal_connect_swapped (G_OBJECT (wizard->btn_next), "clicked",
                                    G_CALLBACK (switch_page_next), wizard);
          gtk_box_pack_end (GTK_BOX (box), wizard->btn_next, FALSE, FALSE, 0);
          gtk_widget_show_all (wizard->btn_next);
     }
     return;
}

GtkWidget*
wizard_create_new (const gchar *title, const gchar *init_text)
{
     lingWizard *wizard;
     GtkWidget *dlg_wizard;
     GtkWidget *box_div;
     GtkWidget *box_content;
     GtkWidget *lbl_title;
     GtkWidget *draw;
     GtkWidget *view_text;
     gchar *text = NULL;

     g_return_val_if_fail (title != NULL, NULL);
     g_return_val_if_fail (init_text != NULL, NULL);

     debug ("Creating new wizard...\n");

     wizard = g_new (lingWizard, 1);
     if (!wizard)
          return NULL;

     wizard->notebook = NULL;
     wizard->btn_next = NULL;
     wizard->btn_prev = NULL;
     wizard->current = 0;
     wizard->window = NULL;
     wizard->page_cbs = NULL;
     wizard->page_data = NULL;
     wizard->callbacks = NULL;
     wizard->cb_data = NULL;

     dlg_wizard = gtk_dialog_new ();
     gtk_dialog_set_has_separator (GTK_DIALOG (dlg_wizard), FALSE);
     gtk_window_set_destroy_with_parent (GTK_WINDOW (dlg_wizard), TRUE);
     gtk_window_set_title (GTK_WINDOW (dlg_wizard), title);
     gtk_window_set_modal (GTK_WINDOW (dlg_wizard), TRUE);
     gtk_window_set_transient_for (GTK_WINDOW (dlg_wizard),
                                   GTK_WINDOW (main_app->window));
     gtk_window_set_default_size (GTK_WINDOW (dlg_wizard), 500, 400);
     gtk_container_set_border_width (GTK_CONTAINER (dlg_wizard),
                                     WIDGET_BORDER);
     wizard->window = dlg_wizard;
     
     /* reference the wizard structure here */
     g_object_set_data (G_OBJECT (dlg_wizard), "wizard_ref", wizard);

     /* create inline main box */
     box_div = hig_hbox_new ();
     gtk_container_set_border_width (GTK_CONTAINER (box_div), 0);
     gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg_wizard)->vbox), box_div,
                         TRUE, TRUE, 0);

     /* drawing area for left indentation */
     draw = gtk_drawing_area_new ();
     gtk_widget_set_size_request (draw, 50, -1);
     gtk_box_pack_start (GTK_BOX (box_div), draw, FALSE, TRUE, 0);
     
     /* the main notebook for the pages */
     wizard->notebook = gtk_notebook_new ();
     gtk_notebook_set_show_tabs (GTK_NOTEBOOK (wizard->notebook), FALSE);
     gtk_notebook_set_show_border (GTK_NOTEBOOK (wizard->notebook), FALSE);
     gtk_box_pack_start (GTK_BOX (box_div), wizard->notebook, TRUE, TRUE, 0);

     box_content = hig_vbox_new ();
     gtk_notebook_append_page (GTK_NOTEBOOK (wizard->notebook), box_content,
                               NULL);

     text = g_strdup_printf (MARKUP_XXLARGE_BOLD, title);
     lbl_title = gtk_label_new (NULL);
     gtk_label_set_markup (GTK_LABEL (lbl_title), text);
     gtk_misc_set_alignment (GTK_MISC (lbl_title), 0, 0.5);
     gtk_box_pack_start (GTK_BOX (box_content), lbl_title, FALSE, FALSE, 0);
     g_free (text);

     view_text = gtk_text_view_new ();
     gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (view_text), GTK_WRAP_WORD);
     gtk_text_view_set_left_margin (GTK_TEXT_VIEW (view_text), 5); 
     gtk_text_view_set_right_margin (GTK_TEXT_VIEW (view_text), 25);
     gtk_text_view_set_pixels_above_lines (GTK_TEXT_VIEW (view_text), 10);
     gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW (view_text), FALSE);
     gtk_text_view_set_editable (GTK_TEXT_VIEW (view_text), FALSE);
     gtk_text_buffer_set_text
          (gtk_text_view_get_buffer (GTK_TEXT_VIEW (view_text)), init_text,
           -1);
     gtk_box_pack_start (GTK_BOX (box_content), view_text, TRUE, TRUE, 0);

     /* create the necessary buttons */
     gtk_dialog_add_button (GTK_DIALOG (dlg_wizard), GTK_STOCK_CANCEL,
                            GTK_RESPONSE_CANCEL);

     wizard->btn_next = util_create_button_with_image (_("_Finish"),
                                                       GTK_STOCK_APPLY, TRUE);
     gtk_dialog_add_action_widget (GTK_DIALOG (dlg_wizard), wizard->btn_next,
                                   GTK_RESPONSE_OK);
     return dlg_wizard;
}

void
wizard_append_page_with_callbacks (GtkWidget *wizard, const gchar *title, 
                                   WizPageCallback page_cb, gpointer page_data,
                                   WizardCallback callback, gpointer data)
{
     lingWizard *wiz;
     GtkWidget *box;
     GtkWidget *box_content;
     GtkWidget *lbl_title;
     gchar *text = NULL;

     debug ("Appending page to wizard...\n");

     g_return_if_fail (wizard != NULL);
     g_return_if_fail (title != NULL);
     g_return_if_fail (page_cb != NULL);

     if (!IS_LING_WIZARD (wizard))
     {
          error_warning (_("The object is not a wizard!"),
                         _("The passed object is not a valid wizard widget."),
                         ERR_NOT_AVAILABLE);
          return;
     }

     wiz = g_object_get_data (G_OBJECT (wizard), "wizard_ref");
     if (gtk_notebook_get_n_pages (GTK_NOTEBOOK (wiz->notebook)) > 1)
     {
          /* first element, start page */
          box = wiz->btn_next->parent;
          gtk_widget_destroy (wiz->btn_next);
          wiz->btn_next = UTIL_CREATE_BUTTON_NEXT ();
          g_signal_connect_swapped (G_OBJECT (wiz->btn_next), "clicked",
                                    G_CALLBACK (switch_page_next), wiz);
          gtk_box_pack_end (GTK_BOX (box), wiz->btn_next, FALSE, FALSE, 0);
     }

     box_content = hig_vbox_new ();
     gtk_notebook_append_page (GTK_NOTEBOOK (wiz->notebook), box_content,
                               NULL);

     /* append the callback and associated data to the internal list */
     wiz->page_cbs = g_list_append (wiz->page_cbs, page_cb);
     wiz->page_data = g_list_append (wiz->page_data, page_data);
     wiz->callbacks = g_list_append (wiz->callbacks, callback);
     wiz->cb_data = g_list_append (wiz->cb_data, data);

     text = g_strdup_printf (MARKUP_XXLARGE_BOLD, title);
     lbl_title = gtk_label_new (NULL);
     gtk_label_set_markup (GTK_LABEL (lbl_title), text);
     gtk_misc_set_alignment (GTK_MISC (lbl_title), 0, 0.5);
     gtk_box_pack_start (GTK_BOX (box_content), lbl_title, FALSE, FALSE, 0);
     g_free (text);

     return;
}

gint
wizard_run (GtkWidget *wizard)
{
     lingWizard *wiz;
     gint what;

     if (!IS_LING_WIZARD (wizard))
     {
          error_warning (_("The object is not a wizard!"),
                         _("The passed object is not a valid wizard widget.!"),
                         ERR_NOT_AVAILABLE);
          return GTK_RESPONSE_CANCEL;
     }
     wiz = g_object_get_data (G_OBJECT (wizard), "wizard_ref");
     gtk_widget_show_all (GTK_DIALOG (wizard)->vbox);
     gtk_widget_show_all (GTK_DIALOG (wizard)->action_area);
     what = gtk_dialog_run (GTK_DIALOG (wizard));
     gtk_widget_destroy (wizard);
     /* TODO: add correct free() for wizard structures */
     g_free (wiz);
     return what;
}
