/* $Id: lingmeaningtree.c,v 1.7 2005/03/28 19:39:47 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2004-2005 Marcus von Appen. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../errors.h"
#include "lingmeaningtree.h"
#include "gtkconfig.h"

extern lingGtkPrefs *settings;

/* basic interface methods for the tree */
static void ling_tree_store_class_init (LingTreeStoreClass *klass);
static void ling_tree_store_tree_model_init (GtkTreeModelIface *iface);
static void ling_tree_store_init (LingTreeStore *tree);
static void ling_tree_store_finalize (GObject *object);
static GtkTreeModelFlags ling_tree_store_get_flags (GtkTreeModel *model);
static gint ling_tree_store_get_n_columns (GtkTreeModel *model);
static GType ling_tree_store_get_column_type (GtkTreeModel *tree_model,
                                              gint index);
static gboolean ling_tree_store_get_iter (GtkTreeModel *tree_model,
                                          GtkTreeIter *iter,
                                          GtkTreePath *path);
static GtkTreePath* ling_tree_store_get_path (GtkTreeModel *tree_model,
                                              GtkTreeIter *iter);
static void ling_tree_store_get_value (GtkTreeModel *tree_model,
                                       GtkTreeIter *iter, gint column,
                                       GValue *value);
static gboolean ling_tree_store_iter_next (GtkTreeModel *tree_model,
                                           GtkTreeIter *iter);
static gboolean ling_tree_store_iter_children (GtkTreeModel *tree_model,
                                               GtkTreeIter *iter,
                                               GtkTreeIter *parent);
static gboolean ling_tree_store_iter_has_child (GtkTreeModel *tree_model,
                                                GtkTreeIter *iter);
static gint ling_tree_store_iter_n_children (GtkTreeModel *tree_model,
                                             GtkTreeIter *iter);
static gboolean ling_tree_store_iter_nth_child (GtkTreeModel *tree_model,
                                                GtkTreeIter *iter,
                                                GtkTreeIter *parent, gint n);
static gboolean ling_tree_store_iter_parent (GtkTreeModel *tree_model,
                                             GtkTreeIter *iter,
                                             GtkTreeIter *child);

static GObjectClass *parent_class = NULL;


static void 
ling_tree_store_class_init (LingTreeStoreClass *klass)
{
     GObjectClass *object_class;
     
     parent_class = (GObjectClass*) g_type_class_peek_parent (klass);
     object_class = (GObjectClass*) klass;
     object_class->finalize = ling_tree_store_finalize;
     return;
}

static void
ling_tree_store_tree_model_init (GtkTreeModelIface *iface)
{
  iface->get_flags = ling_tree_store_get_flags;
  iface->get_n_columns = ling_tree_store_get_n_columns;
  iface->get_column_type = ling_tree_store_get_column_type;
  iface->get_iter = ling_tree_store_get_iter;
  iface->get_path = ling_tree_store_get_path;
  iface->get_value = ling_tree_store_get_value;
  iface->iter_next = ling_tree_store_iter_next;
  iface->iter_children = ling_tree_store_iter_children;
  iface->iter_has_child = ling_tree_store_iter_has_child;
  iface->iter_n_children = ling_tree_store_iter_n_children;
  iface->iter_nth_child = ling_tree_store_iter_nth_child;
  iface->iter_parent = ling_tree_store_iter_parent;
    return;
}

static void
ling_tree_store_init (LingTreeStore *tree)
{
     tree->n_columns = LING_TREE_N_COLUMNS;

     tree->column_types[LING_TREE_ID] = G_TYPE_UINT;
     tree->column_types[LING_TREE_LANGUAGE] = G_TYPE_STRING;
     tree->column_types[LING_TREE_TRANS] = G_TYPE_STRING;
     tree->column_types[LING_TREE_POINTER] = G_TYPE_POINTER;
     tree->column_types[LING_TREE_EDITED] = G_TYPE_BOOLEAN;

     tree->num_rows = 0;
     tree->meaning = NULL;
     tree->root = ling_meaning_new (); /* 0th element, which won't be saved */

     do
          tree->stamp = g_random_int ();
     while (tree->stamp == 0);

     return;
}

static void
ling_tree_store_finalize (GObject *object)
{
     LingTreeStore *tree = LING_TREE_STORE (object);
     debug ("Finalizing tree..\n");
     ling_meaning_free (tree->root);

     (* parent_class->finalize) (object);
     return;
}

static GtkTreeModelFlags
ling_tree_store_get_flags (GtkTreeModel *tree_model)
{
     g_return_val_if_fail (LING_IS_TREE_STORE (tree_model), 0);
     return GTK_TREE_MODEL_ITERS_PERSIST;
}

static gint
ling_tree_store_get_n_columns (GtkTreeModel *tree_model)
{
  LingTreeStore *tree_store = (LingTreeStore *) tree_model;

  g_return_val_if_fail (LING_IS_TREE_STORE (tree_model), 0);

  return tree_store->n_columns;
}

static GType
ling_tree_store_get_column_type (GtkTreeModel *tree_model, gint index)
{
     g_return_val_if_fail (LING_IS_TREE_STORE (tree_model), G_TYPE_INVALID);
     g_return_val_if_fail  (index < LING_TREE_STORE (tree_model)->n_columns 
                            && index >= 0, G_TYPE_INVALID);
     
     return LING_TREE_STORE (tree_model)->column_types[index];
}

static gboolean
ling_tree_store_get_iter (GtkTreeModel *tree_model, GtkTreeIter *iter,
                          GtkTreePath *path)
{
     LingTreeStore *tree = LING_TREE_STORE (tree_model);
     gint *indices;
     gint depth = 0;
     luint cur_id = 0;
     gint pos = 0;
     lingMeaning *mn = NULL;

     g_return_val_if_fail (LING_IS_TREE_STORE (tree), FALSE);

     indices = gtk_tree_path_get_indices (path);
     depth = gtk_tree_path_get_depth (path);

     g_return_val_if_fail (depth > 0, FALSE);

     /* the depth cannot be greater than 2, thus we can speed things
      * up by calculating the total amount traverse */
     mn = (lingMeaning *) tree->root;
     cur_id = mn->id;
     while (pos <= indices[0])
     {
          while (mn && mn->id == cur_id)
               mn = mn->next;
          if (!mn)
               return FALSE;
          cur_id = mn->id;
          pos++;
     }

     pos = 0;
     if (depth == 2)
          while (pos <= indices[1])
          {
               mn = mn->next;
               pos++;
          }

     iter->stamp = tree->stamp;
     iter->user_data = mn;

     return TRUE;
}

static GtkTreePath*
ling_tree_store_get_path (GtkTreeModel *tree_model, GtkTreeIter *iter)
{
     GtkTreePath *path;
     lingMeaning *mn;
     lingMeaning *meanings;
     LingTreeStore *tree;
     luint cur_id = 0;
     gint pos = 0;

     g_return_val_if_fail (LING_IS_TREE_STORE (tree_model), NULL);
     g_return_val_if_fail (iter != NULL, NULL);
     g_return_val_if_fail (iter->user_data != NULL, NULL);
     g_return_val_if_fail (iter->stamp == LING_TREE_STORE (tree_model)->stamp,
                           NULL);

     tree = LING_TREE_STORE (tree_model);

     mn = (lingMeaning *) iter->user_data;
 
     /* TODO: if this is too slow, we should speed things up by using an
      * array with the additional path information */

     /* special case: iter == root */
     if (((lingMeaning *) iter->user_data)->prev == NULL &&
         ((lingMeaning *) iter->user_data)
         == LING_TREE_STORE (tree_model)->root)
          return gtk_tree_path_new ();

     /* calculate the path of the meaning */
     meanings = tree->root;
     cur_id = meanings->id;
     while (mn->id != meanings->id) /* get the position for the first depth */
     {
          /* increase position, if the meaning id changes */
          if (cur_id != meanings->id)
          {
               pos++;
               cur_id = meanings->id;
          }
          meanings = meanings->next;
     }

     path = gtk_tree_path_new ();
     gtk_tree_path_append_index (path, pos); /* append the first depth */

     /* we got the id, now we have to check for the position of the
      * second depth */
     pos = -1;
     while (mn != meanings)
     {
          pos++;
          meanings = meanings->next;
     }
     if (pos >= 0) /* this is true, if it is _not_ the first element */
          gtk_tree_path_append_index (path, pos);

     return path;
}

static void
ling_tree_store_get_value (GtkTreeModel *tree_model, GtkTreeIter *iter,
                           gint column, GValue *value)
{
     lingMeaning *mn;
     lingchar *language = NULL;

     g_return_if_fail (LING_IS_TREE_STORE (tree_model));
     g_return_if_fail (iter != NULL);
     g_return_if_fail (iter->stamp == LING_TREE_STORE (tree_model)->stamp);
     g_return_if_fail (column < LING_TREE_STORE (tree_model)->n_columns);
     
     g_value_init (value, LING_TREE_STORE (tree_model)->column_types[column]);

     mn = (lingMeaning *) iter->user_data;
     g_return_if_fail (mn != NULL);

     switch (column)
     {
     case LING_TREE_ID:
          g_value_set_uint (value, mn->id);
          break;
     case LING_TREE_LANGUAGE:
          /* we will return the translation instead of the real one,
             this should be kept in mind on future enhancements */
          if (settings->edit_lang)
               language =
                    ling_lang_get_lang_translation (settings->prefs->config,
                                                    mn->language,
                                                    settings->edit_lang);
          if (!language)
               g_value_set_string (value, mn->language);
          else
          {
               g_value_set_string (value, language);
               ling_free (language);
          }
          break;
     case LING_TREE_TRANS:
          g_value_set_string (value, mn->translation);
          break;
     case LING_TREE_POINTER:
          g_value_set_pointer (value, mn);
          break;
     default:
          break;
     }
     return;
}

static gboolean
ling_tree_store_iter_next (GtkTreeModel *tree_model, GtkTreeIter *iter)
{
     lingMeaning *mn = NULL;
     lingMeaning *next = NULL;

     g_return_val_if_fail (iter != NULL, FALSE);
     g_return_val_if_fail (iter->user_data != NULL, FALSE);
     g_return_val_if_fail (iter->stamp == LING_TREE_STORE (tree_model)->stamp,
                           FALSE);

     mn = (lingMeaning *) iter->user_data;

     if (!mn->prev) /* root node */
          next = mn->next;
     else if (mn->prev && mn->prev->id != mn->id) /* 1st level */
     {
          next = mn->next;
          while (next && next->id == mn->id)
               next = next->next;
     }
     else if (mn->next && mn->next->id == mn->id && mn->prev->id == mn->id)
           /* 2nd level */
          next = mn->next;

     if (!next)
          return FALSE;

     iter->user_data = next;
     return TRUE;
}

static gboolean
ling_tree_store_iter_children (GtkTreeModel *tree_model, GtkTreeIter *iter,
                               GtkTreeIter *parent)
{
     lingMeaning *mn;
     lingMeaning *next;

     g_return_val_if_fail (LING_IS_TREE_STORE (tree_model), FALSE);
     g_return_val_if_fail (parent == NULL || parent->user_data != NULL, FALSE);
     g_return_val_if_fail (parent == NULL ||
                           parent->stamp == LING_TREE_STORE (tree_model)->stamp,
                           FALSE);
     
     /* check the meaning id, if the previous one has another one and if the
      * next one has the same */
     if (parent)
     {
          mn = (lingMeaning *) parent->user_data;
          g_return_val_if_fail (mn->prev->id != mn->id, FALSE);
          if (!mn->next)
               return FALSE;
          g_return_val_if_fail (mn->next->id == mn->id, FALSE);
          next = mn->next;
     }
     else
          next = (lingMeaning *) LING_TREE_STORE (tree_model)->root;

     if (!next)
          return FALSE;

     iter->stamp = LING_TREE_STORE (tree_model)->stamp;
     iter->user_data = next;
     return TRUE;
}

static gboolean
ling_tree_store_iter_has_child (GtkTreeModel *tree_model, GtkTreeIter *iter)
{
     lingMeaning *mn;

     g_return_val_if_fail (LING_IS_TREE_STORE (tree_model), FALSE);
     g_return_val_if_fail (iter->stamp == LING_TREE_STORE (tree_model)->stamp,
                           FALSE);
     g_return_val_if_fail (iter->user_data != NULL, FALSE);

     mn = (lingMeaning *) iter->user_data;
     if (mn->next && mn->next->id == mn->id)
     {
          if (mn->prev && mn->prev->id == mn->id)
               return FALSE;
          return TRUE;
     }
     return FALSE;
}

static gint
ling_tree_store_iter_n_children (GtkTreeModel *tree_model, GtkTreeIter *iter)
{
     lingMeaning *mn;
     gint i = 0;

     g_return_val_if_fail (LING_IS_TREE_STORE (tree_model), 0);
     g_return_val_if_fail (iter == NULL || iter->user_data != NULL, 0);
     g_return_val_if_fail (iter == NULL || 
                           iter->stamp == LING_TREE_STORE (tree_model)->stamp,
                           0);
     
     if (!iter)
          return LING_TREE_STORE (tree_model)->num_rows;
     
     mn = (lingMeaning *) iter->user_data;
     while (mn)
     {
          i++;
          mn = mn->next;
     }

     return i;
}

static gboolean
ling_tree_store_iter_nth_child (GtkTreeModel *tree_model, GtkTreeIter *iter,
                                GtkTreeIter *parent, gint n)
{
     lingMeaning *mn;
     lingMeaning *par;
     gboolean root_flag = FALSE;

     g_return_val_if_fail (LING_IS_TREE_STORE (tree_model), FALSE);
     g_return_val_if_fail (parent == NULL || parent->user_data != NULL, FALSE);

     if (!parent)
     {
          par = LING_TREE_STORE (tree_model)->meaning;
          root_flag = TRUE;
     }
     else
          par = (lingMeaning *) parent->user_data;
     
     mn = par;
     if (!root_flag) /* it is not the root */
     {
          while (n-- > 0 && mn)
               mn = mn->next;
          if (mn && mn->id != par->id)
               return FALSE;
     }
     else /* root iter, just browse through the children of the first level */
     {
          while (n-- > 0 && mn)
          {
               while (mn->id == par->id && mn)
                    mn = mn->next;
               par = mn;
          }
          if (!mn)
               return FALSE;
     }
     
     iter->user_data = mn;
     iter->stamp = LING_TREE_STORE (tree_model)->stamp;
     return TRUE;
}

static gboolean
ling_tree_store_iter_parent (GtkTreeModel *tree_model, GtkTreeIter *iter,
                             GtkTreeIter *child)
{
     lingMeaning *parent;
     lingMeaning *mn;

     g_return_val_if_fail (iter != NULL, FALSE);
     g_return_val_if_fail (child != NULL, FALSE);
     g_return_val_if_fail (child->user_data != NULL, FALSE);
     g_return_val_if_fail (child->stamp == LING_TREE_STORE (tree_model)->stamp,
                           FALSE);

     mn = (lingMeaning *) child->user_data;
     parent = mn;

     /* root entry of the meaning list or other meaning id */
     if (parent->prev->id != mn->id)
          return FALSE;

     while (parent->prev && parent->prev->id == mn->id)
          parent = parent->prev;
     
     iter->user_data = parent;
     iter->stamp = LING_TREE_STORE (tree_model)->stamp;

     return TRUE;
}

/* public functions start here */

GType
ling_tree_store_get_type (void)
{
     static GType tree_type = 0;

     if (!tree_type)
     {
          static const GTypeInfo tree_store_info =
               {
                    sizeof (LingTreeStoreClass),
                    NULL, /* base_init */
                    NULL, /* base_finalize */
                    (GClassInitFunc) ling_tree_store_class_init,
                    NULL, /* class_finalize */
                    NULL, /* class_data */
                    sizeof (LingTreeStore),
                    0,    /* n_preallocs */
                    (GInstanceInitFunc) ling_tree_store_init
               };

          static const GInterfaceInfo tree_model_info =
               {
                    (GInterfaceInitFunc) ling_tree_store_tree_model_init,
                    NULL,
                    NULL
               };
          tree_type = g_type_register_static (G_TYPE_OBJECT, "LingTreeStore",
                                              &tree_store_info, 0);

          g_type_add_interface_static (tree_type, GTK_TYPE_TREE_MODEL,
                                       &tree_model_info);
     }
     return tree_type;
}

LingTreeStore*
ling_tree_store_new (void)
{
     LingTreeStore *retval = g_object_new (LING_TYPE_TREE_STORE, NULL);
     return retval;
}

LingTreeStore*
ling_tree_store_new_with_meanings (lingMeaning *meaning)
{
     lingMeaning *mn = meaning;
     LingTreeStore *retval = g_object_new (LING_TYPE_TREE_STORE, NULL);

     retval->meaning = meaning;

     /* preserve lesson of the meaning in root node */
     ((lingMeaning *) retval->root)->lesson = meaning->lesson;
     ((lingMeaning *) retval->root)->next = meaning;

     meaning->prev = ((lingMeaning *) retval->root);

     /* set row amount */
     while (mn)
     {
          retval->num_rows++;
          mn = mn->next;
     }

     return retval;
}

/* FIXME: if the user inserts while being on a toplevel node,
   insert after it, increase the id and renumber all other behind it.
   if the user inserts while being on a child node, just add it after the
   child node using the same id */
void
ling_tree_store_insert_after (LingTreeStore *tree_store, GtkTreeIter *iter,
                              GtkTreeIter *parent, GtkTreeIter *sibling)
{
     GtkTreePath *path;
     lingMeaning *mn;
     lingMeaning *par;
     lingMeaning *tree;
     gboolean toplevel = FALSE;

     g_return_if_fail (LING_IS_TREE_STORE (tree_store));
     g_return_if_fail (iter != NULL);

     if (parent)
     {
          g_return_if_fail (parent->stamp == tree_store->stamp);
          g_return_if_fail (parent->user_data != NULL);
     }

     if (sibling)
     {
          g_return_if_fail (sibling->stamp == tree_store->stamp);
          g_return_if_fail (sibling->user_data != NULL);
     }

     if (!parent && !sibling)
     {
          par = tree_store->meaning;
          if (!par)
               par = (lingMeaning *) tree_store->root;
     }
     else if (!sibling) /* toplevel node */
     {
          /* toplevel, jump to the last one of it, and add a new */
          toplevel = TRUE;
          par = (lingMeaning *) parent->user_data;
          if (par->prev->id != par->id)
               while (par->next && par->next->id == par->id)
                    par = par->next;
     }
     else
     {
          /* check, if the sibling is really a child of parent */
          g_return_if_fail (((lingMeaning *) sibling->user_data)->id ==
                            ((lingMeaning *) parent->user_data)->id);
          par = (lingMeaning *) sibling->user_data;
     }

     /* create new, empty meaning and add it to the tree */
     mn = ling_meaning_new ();
     mn->lesson = par->lesson;
     mn->id = par->id; /* set correct id */
     if (toplevel)
          mn->id++;

     if (!tree_store->meaning) /* no meanings */
     {
          mn->id = 1;
          tree = mn;
          /* make links */
          ((lingMeaning *) tree_store->root)->next = tree;
          tree->prev = (lingMeaning *) tree_store->root;
     }
     else
          tree = ling_meaning_insert_after (tree_store->meaning, par, mn);

     if (!tree)
     {
          ling_meaning_free (mn);
          return;
     }
     tree_store->meaning = tree;
     tree_store->num_rows++;

     iter->stamp = tree_store->stamp;
     iter->user_data = mn;

     /* we will renumber all meanings, if it is a toplevel entry */
     tree = mn->next;
     if (toplevel)
          while (tree)
          {
               tree->id++;
               tree = tree->next;
          }

     /* inform all others about the change */
     path = ling_tree_store_get_path (GTK_TREE_MODEL (tree_store), iter);
     gtk_tree_model_row_inserted (GTK_TREE_MODEL (tree_store), path, iter);
     
     gtk_tree_path_free (path);
     return;
}

gboolean
ling_tree_store_remove (LingTreeStore *tree_store, GtkTreeIter *iter)
{
     lingMeaning *mn;
     lingMeaning *next = NULL;
     lingMeaning *parent;
     lingMeaning *tree;
     guint id = 0;
     GtkTreeIter new_iter;
     GtkTreePath *path;

     g_return_val_if_fail (LING_IS_TREE_STORE (tree_store), FALSE);
     g_return_val_if_fail (iter != NULL, FALSE);
     g_return_val_if_fail (iter->user_data != NULL, FALSE);
     g_return_val_if_fail (iter->stamp == tree_store->stamp, FALSE);

     mn = (lingMeaning *) iter->user_data;

     /* create path for deleted signal and emit deleted signal */
     path = ling_tree_store_get_path (GTK_TREE_MODEL (tree_store), iter);
     gtk_tree_model_row_deleted (GTK_TREE_MODEL (tree_store), path);

     parent = mn->prev;
     id = mn->id;

     /* two cases: parent and child */
     if (parent->id != mn->id) /* parent, remove all children as well */
     {
          next = mn;
          while (next && next->id == id)
          {
               mn = next;
               next = next->next;
               tree_store->root =
                    ling_meaning_free_1 ((lingMeaning *) tree_store->root, mn);

          }
          /* we will renumber all meanings, if it is a toplevel entry */
          tree = next;
          while (tree)
          {
               tree->id--;
               tree = tree->next;
          }
     }
     else
          tree_store->root =
               ling_meaning_free_1 ((lingMeaning *) tree_store->root, mn);

     tree_store->meaning = ((lingMeaning *) tree_store->root)->next;

     /* now check, if it was a child of another meaning */
     if (parent != tree_store->root)
     {
          if (parent->id == id
              && ((parent->next && parent->next->id != id)
                  || (!parent->next && parent->prev
                      && parent->prev->id != id)))
          {
               /* no children with the same id */
               gtk_tree_path_up (path);
               new_iter.stamp = tree_store->stamp;
               new_iter.user_data = parent;
               gtk_tree_model_row_has_child_toggled
                    (GTK_TREE_MODEL (tree_store), path, &new_iter);
          }
     }
     gtk_tree_path_free (path);

     if (next)
     {
          iter->stamp = tree_store->stamp;
          iter->user_data = next;
          return TRUE;
     }
     else
          iter->user_data = NULL;
     return FALSE;
}

gboolean
ling_tree_store_search (GtkTreeModel  *tree_model, GtkTreeIter *iter,
                        gint column_id, gpointer data)
{
     guint id;
     gchar *search;
     gchar *val = NULL;
     gboolean retval = FALSE;
     lingMeaning *meaning;

     g_return_val_if_fail (LING_IS_TREE_STORE (tree_model), FALSE);
     g_return_val_if_fail (LING_TREE_STORE (tree_model)->meaning != NULL,
                           FALSE);
     g_return_val_if_fail ((lingMeaning *) iter->user_data != NULL, FALSE);
     g_return_val_if_fail (iter->stamp == LING_TREE_STORE (tree_model)->stamp,
                           FALSE);
     g_return_val_if_fail (column_id < LING_TREE_N_COLUMNS, FALSE);
     g_return_val_if_fail (data != NULL, FALSE);
    
     meaning = (lingMeaning *) iter->user_data;
     meaning = meaning->next;

     switch (column_id)
     {
     case LING_TREE_ID:
          id = GPOINTER_TO_INT (data);
          while (meaning && meaning->id != id)
               meaning = meaning->next;
          break;
     case LING_TREE_LANGUAGE:
          search = (gchar *) data;
          while (meaning)
          {
               if (settings->edit_lang)
                    val = ling_lang_get_lang_translation
                            (settings->prefs->config, meaning->language,
                             settings->edit_lang);
               else
                    val = meaning->language;

               if (g_strrstr (val, search) != NULL)
               {
                    if (settings->edit_lang)
                         ling_free (val);
                    break;
               }

               if (settings->edit_lang)
                    ling_free (val);
               meaning = meaning->next;
          }
          break;
     case LING_TREE_TRANS:
          search = (gchar *) data; 
          while (meaning && g_strrstr (meaning->translation, search) == NULL)
               meaning = meaning->next;
          break;
     default:
          return FALSE;
          break;
     }
     if (!meaning)
          return FALSE;
     
     retval = TRUE;
     iter->user_data = meaning;
     iter->stamp = LING_TREE_STORE (tree_model)->stamp;
     return retval;
}
