// lcwin32.cxx  Win32 stuff - part of lin-city
// Copyright (c) I J Peters 1995,1996.  Please read the file 'COPYRIGHT'.
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <assert.h>

#include "lin-city.h"
#include "common.h"
#include "lctypes.h"

#define USE_WINDOWS_FONT 1


int
AdjustX (int x)
{
  x <<= pix_double;
  x += borderx;
  return x;
}

int
AdjustY (int y)
{
  y <<= pix_double;
  y += bordery;
  return y;
}

int
UnAdjustX (int x)
{
  x -= borderx;
  x >>= pix_double;
  return x;
}

int
UnAdjustY (int y)
{
  y -= bordery;
  y >>= pix_double;
  return y;
}

void
HandleError (char *description, int degree)
{
  MessageBox (NULL, description, "ERROR", MB_OK);
  if (degree == FATAL)
    {
      exit (-1);
    }
}

void
setcustompalette (void)
{
  char s[100];
  unsigned int n, r, g, b;
  int i, flag[256];
  FILE *inf;

  for (i = 0; i < 256; i++)
    flag[i] = 0;
  if ((inf = fopen (colour_pal_file, "r")) == 0)
    HandleError ("Can't find the colour pallet file", FATAL);

  while (feof (inf) == 0)
    {
      fgets (s, 99, inf);
      if (sscanf (s, "%u %u %u %u", &n, &r, &g, &b) == 4)
	{
	  r = ((r * (1 - gamma_correct_red))
	       + (64 * sin ((float) r * M_PI / 128))
	       * gamma_correct_red);
	  g = ((g * (1 - gamma_correct_green))
	       + (64 * sin ((float) g * M_PI / 128))
	       * gamma_correct_green);
	  b = ((b * (1 - gamma_correct_blue))
	       + (64 * sin ((float) b * M_PI / 128))
	       * gamma_correct_blue);
	  AddPaletteEntry (n, r, g, b);
	  flag[n] = 1;
	}
    }
  fclose (inf);
  for (i = 0; i < 256; i++)
    {
      if (flag[i] == 0)
	{
	  printf ("Colour %d not loaded\n", i);
	  HandleError ("Can't continue", FATAL);
	}
    }
  UpdatePalette ();
}

COLORREF
GetPaletteColorref (int col)
{
  assert (col >= 0 || col <= 255);
  if (display.hasPalette)
    return PALETTEINDEX (col);
  else
    return display.colorrefPal[col];
}

HBRUSH
GetPaletteBrush (int col)
{
  assert (col >= 0 || col <= 255);
  if (display.brushPal[col] == 0)
    {
      display.brushPal[col] = CreateSolidBrush (GetPaletteColorref (col));
    }
  return display.brushPal[col];
}

#if defined (WIN32)
int CALLBACK
EnumFontFamProc (
		  ENUMLOGFONT FAR * lpelf,	// pointer to logical-font data 
		   NEWTEXTMETRIC FAR * lpntm,	// pointer to physical-font data 
		   int FontType,	// type of font 
		   LPARAM lParam	// address of application-defined data  
)
{
  if (0 != strcmp (lpelf->elfLogFont.lfFaceName, "Lincity"))
    //  if (0 != strcmp (lpelf->elfLogFont.lfFaceName, "Tester"))
    return 0;

  // GCS:  I'm not sure if it's OK to just copy the pointer here.
  //       This may cause problems, but let's give it a whirl.
  // *((LOGFONT**) lParam) = &(lpelf->elfLogFont);

  // GCS:  No wait, I changed my mind.  Copy the entire LOGFONT struct.
  *(LOGFONT *) lParam = lpelf->elfLogFont;
  return 1;
}
#endif


void
initfont ()
{
#if defined (USE_WINDOWS_FONT)
  int fonts_added = AddFontResource (windowsfontfile);
  if (fonts_added != 1)
    {
      HandleError ("Can't open the font file", FATAL);
    }
  SendMessage (HWND_BROADCAST, WM_FONTCHANGE, 0, 0);
  ProcessPendingEvents ();

  // GCS: This typedef is for the typecast...
  // typedef int (FAR WINAPI *voidfnptr)(void);
  // KSH: Changed the typedef to get around VC 6 strict type enformcement
#if defined (commentout)
  typedef int (FAR WINAPI * voidfnptr) (const struct tagLOGFONTA *, const struct tagTEXTMETRICA *, unsigned long, long);
  EnumFontFamilies (display.hdcMem, "Lincity", (voidfnptr) EnumFontFamProc, (LPARAM) & logfont);
#endif
  // GCS: Hmm.  There must be a way to get this to work on both compilers.  
  //            How about this?
  LOGFONT logfont;
  EnumFontFamilies (display.hdcMem, "Lincity", (FONTENUMPROC) EnumFontFamProc, (LPARAM) & logfont);

#if defined (commentout)
  LOGFONT logfont;
  logfont.lfHeight = 8;		// ??

  logfont.lfWidth = 8;		// ??

  logfont.lfEscapement = 0;	// ??

  logfont.lfOrientation = 0;	// ??

  logfont.lfWeight = 0;
  logfont.lfItalic = FALSE;
  logfont.lfUnderline = FALSE;
  logfont.lfStrikeOut = FALSE;
  logfont.lfCharSet = ANSI_CHARSET;
  logfont.lfOutPrecision = OUT_RASTER_PRECIS;
  logfont.lfClipPrecision = 0;
  logfont.lfQuality = DEFAULT_QUALITY;
  logfont.lfPitchAndFamily = FIXED_PITCH | FF_ROMAN;
  strcpy (logfont.lfFaceName, "alt-8x8");
#endif

  display.hFont = CreateFontIndirect (&logfont);
  if (!display.hFont)
    {
      HandleError ("Error executing CreateFontIndirect", FATAL);
    }
  SelectObject (display.hdcMem, display.hFont);
#endif

  // Load non-windows version of font
  int i;
  FILE *finf;
#if defined (WIN32)
  if ((finf = fopen (fontfile, "rb")) == 0)
    HandleError ("Can't open the font file", FATAL);
#else
  if ((finf = fopen (fontfile, "r")) == 0)
    HandleError ("Can't open the font file", FATAL);
#endif
  for (i = 0; i < 256 * 8; i++)
    myfont[i] = fgetc (finf);
  fclose (finf);
}

void
gl_setpalettecolor (long x, long r, long g, long b)
{
  if (x >= 0 && x <= 255)
    {
      AddPaletteEntry (x, r, g, b);
    }
}

void
Fgl_setfontcolors (int bg, int fg)
{
  text_fg = fg;
  text_bg = bg;
}

void
Fgl_setpixel (int x, int y, int col)
{
  int i;

  if (clipping_flag)
    if (x < xclip_x1 || x > xclip_x2 || y < xclip_y1 || y > xclip_y2)
      return;
  col &= 0xff;

  // Draw to pixmap buffer
  i = (y + bordery) * (640 + BORDERX) + x + borderx;
  *(pixmap + i) = (unsigned char) col;

  // Draw to DC
  if (display.useDIB)
    {
      BYTE *pixel = display.pBits;
      unsigned long row = display.pbminfo->bmiHeader.biHeight - (y + bordery);
      unsigned long rowlength = display.pbminfo->bmiHeader.biWidth;
      unsigned long mod = rowlength % sizeof (DWORD);
      if (mod)
	{
	  rowlength += (sizeof (DWORD) - mod);
	}			// Pad out to DWORD boundary

      pixel += row * rowlength + x + borderx;	// Ptr arith w/ 4 bytes per pixel

      *pixel = col;
    }
  else
    {
      if (pix_double)
	{
	  RECT rect;
	  HBRUSH hbr, hbrOld;
	  rect.left = AdjustX (x);
	  rect.top = AdjustY (y);
	  rect.right = AdjustX (x + 1);
	  rect.bottom = AdjustY (y + 1);
	  hbr = GetPaletteBrush (col);
	  hbrOld = (HBRUSH) SelectObject (display.hdcMem, hbr);
	  FillRect (display.hdcMem, &rect, hbr);
	  hbr = (HBRUSH) SelectObject (display.hdcMem, hbrOld);
	}
      else
	{
	  SetPixel (display.hdcMem, x + borderx, y + bordery, GetPaletteColorref (col));
	}
    }

  // It's too slow to write directly to screen or to queue update event
  // for every pixel.  Instead, the caller will have to do this.
}


int
Fgl_getpixel (int x, int y)
{
  return (*(pixmap + (y + bordery) * (640 + BORDERX) + x + borderx));
}


void
Fgl_hline (int x1, int y1, int x2, int col)
{
  int x, i;

  int xor = 0;
  if (col == -1)
    {
      xor = 1;
    }

  col &= 0xff;

  // Draw to pixmap buffer
  i = (y1 + bordery) * (640 + BORDERX);
  for (x = x1 + borderx; x <= x2 + borderx; x++)
    *(pixmap + i + x) = col;

  // Calculate extent of update rectangle
  RECT rect;
  rect.left = AdjustX (x1);
  rect.top = AdjustY (y1);
  rect.right = AdjustX (x2);	// LineTo() does not include final pixel of line

  rect.bottom = AdjustY (y1 + 1);



  // Draw to DC
  if (display.useDIB)
    {
      HPEN hpen, hpenOld;
      HBITMAP hOldBitmapMem;
      hpen = CreatePen (PS_SOLID, 0, GetPaletteColorref (col));
      hpenOld = (HPEN) SelectObject (display.hdcMem, hpen);
      hOldBitmapMem = (HBITMAP) SelectObject (display.hdcMem, display.hDIB);
      MoveToEx (display.hdcMem, x1 + borderx, y1 + bordery, NULL);
      int oldrop2;
      if (xor)
	oldrop2 = SetROP2 (display.hdcMem, R2_NOT);
      LineTo (display.hdcMem, x2 + borderx, y1 + bordery);
      if (xor)
	SetROP2 (display.hdcMem, oldrop2);
      display.hDIB = (HBITMAP) SelectObject (display.hdcMem, hOldBitmapMem);
      hpen = (HPEN) SelectObject (display.hdcMem, hpenOld);
      DeleteObject (hpen);
    }
  else
    {
      if (pix_double)
	{
	  HBRUSH hbr, hbrOld;
	  hbr = GetPaletteBrush (col);
	  hbrOld = (HBRUSH) SelectObject (display.hdcMem, hbr);
	  int oldrop2;
	  if (xor)
	    oldrop2 = SetROP2 (display.hdcMem, R2_NOT);
	  FillRect (display.hdcMem, &rect, hbr);
	  if (xor)
	    SetROP2 (display.hdcMem, oldrop2);
	  hbr = (HBRUSH) SelectObject (display.hdcMem, hbrOld);
	}
      else
	{
	  HPEN hpen, hpenOld;
	  hpen = CreatePen (PS_SOLID, 0, GetPaletteColorref (col));
	  hpenOld = (HPEN) SelectObject (display.hdcMem, hpen);
	  MoveToEx (display.hdcMem, x1 + borderx, y1 + bordery, NULL);
	  int oldrop2;
	  if (xor)
	    oldrop2 = SetROP2 (display.hdcMem, R2_NOT);
	  LineTo (display.hdcMem, x2 + borderx, y1 + bordery);
	  if (xor)
	    SetROP2 (display.hdcMem, oldrop2);
	  hpen = (HPEN) SelectObject (display.hdcMem, hpenOld);
	  DeleteObject (hpen);
	}
    }

  // Queue update event
  InvalidateRect (display.hWnd, &rect, FALSE);
}

void
Fgl_line (int x1, int y1, int dummy, int y2, int col)
// vertical lines only.
{
  int y;

  col &= 0xff;

  // Draw to pixmap buffer
  for (y = y1 + bordery; y <= y2 + bordery; y++)
    *(pixmap + x1 + borderx + (y * (640 + BORDERX))) = col;

  // Calculate extent of update rectangle
  RECT rect;
  rect.left = AdjustX (x1);
  rect.top = AdjustY (y1);
  rect.right = AdjustX (x1 + 1);
  rect.bottom = AdjustY (y2);	// LineTo() does not include final pixel of line

  // Draw to DC
  if (display.useDIB)
    {
      HPEN hpen, hpenOld;
      HBITMAP hOldBitmapMem;
      hpen = CreatePen (PS_SOLID, 0, GetPaletteColorref (col));
      hpenOld = (HPEN) SelectObject (display.hdcMem, hpen);
      hOldBitmapMem = (HBITMAP) SelectObject (display.hdcMem, display.hDIB);
      MoveToEx (display.hdcMem, x1 + borderx, y1 + bordery, NULL);
      LineTo (display.hdcMem, x1 + borderx, y2 + bordery);
      display.hDIB = (HBITMAP) SelectObject (display.hdcMem, hOldBitmapMem);
      hpen = (HPEN) SelectObject (display.hdcMem, hpenOld);
      DeleteObject (hpen);
    }
  else
    {
      if (pix_double)
	{
	  HBRUSH hbr, hbrOld;
	  hbr = GetPaletteBrush (col);
	  hbrOld = (HBRUSH) SelectObject (display.hdcMem, hbr);
	  FillRect (display.hdcMem, &rect, hbr);
	  hbr = (HBRUSH) SelectObject (display.hdcMem, hbrOld);
	}
      else
	{
	  HPEN hpen, hpenOld;
	  hpen = CreatePen (PS_SOLID, 0, GetPaletteColorref (col));
	  hpenOld = (HPEN) SelectObject (display.hdcMem, hpen);
	  MoveToEx (display.hdcMem, AdjustX (x1), AdjustY (y1), NULL);
	  LineTo (display.hdcMem, AdjustX (x1), AdjustY (y2));
	  hpen = (HPEN) SelectObject (display.hdcMem, hpenOld);
	  DeleteObject (hpen);
	}
    }

  // Queue update event
  InvalidateRect (display.hWnd, &rect, FALSE);
}


void
Fgl_write (int x, int y, char *s)
{
#if defined (USE_WINDOWS_FONT)
  // Select background and foreground colors
  SetTextColor (display.hdcMem, GetPaletteColorref (text_fg));
  SetBkColor (display.hdcMem, GetPaletteColorref (text_bg));

  // Draw text to backing store
  ExtTextOut (display.hdcMem, AdjustX (x), AdjustY (y), 0, NULL,
	      s, strlen (s), NULL);

  // Calculate size of text
  SIZE size;
  GetTextExtentPoint32 (display.hdcMem, s, strlen (s), &size);
#endif

  int i;
  for (i = 0; i < (int) (strlen (s)); i++)
    my_x_putchar (x + i * 8, y, s[i]);

  // Queue update event
#if defined (USE_WINDOWS_FONT)
  RECT rect;
  rect.left = AdjustX (x);
  rect.top = AdjustY (y);
  rect.right = AdjustX (x + size.cx);
  rect.bottom = AdjustY (y + size.cy);
  InvalidateRect (display.hWnd, &rect, FALSE);
#else
  RECT rect;
  rect.left = AdjustX (x);
  rect.top = AdjustY (y);
  rect.right = AdjustX (x + i * 8);
  rect.bottom = AdjustY (y + 8);
  InvalidateRect (display.hWnd, &rect, FALSE);
#endif
}


void
open_write (int x, int y, char *s)
{
  int i;
  for (i = 0; i < (int) (strlen (s)); i++)
    open_x_putchar (x + i * 8, y, s[i]);

  // Queue update event
  RECT rect;
  rect.left = AdjustX (x);
  rect.top = AdjustY (y);
  rect.right = AdjustX (x + i * 8);
  rect.bottom = AdjustY (y + open_font_height);
  InvalidateRect (display.hWnd, &rect, FALSE);
}

void
my_x_putchar (int xx, int yy, int c)
{
  int x, y, b;
#if defined (USE_WINDOWS_FONT)
  int i;
#endif
  for (y = 0; y < 8; y++)
    {
      b = myfont[c * 8 + y];
      for (x = 0; x < 8; x++)
	{
	  if ((b & 0x80) == 0)
	    {
#if defined (USE_WINDOWS_FONT)
	      // Draw to pixmap buffer
	      i = (yy + y + bordery) * (640 + BORDERX) + xx + x + borderx;
	      *(pixmap + i) = (unsigned char) text_bg;
#else
	      Fgl_setpixel (xx + x, yy + y, text_bg);
#endif
	    }
	  else
	    {
#if defined (USE_WINDOWS_FONT)
	      // Draw to pixmap buffer
	      i = (yy + y + bordery) * (640 + BORDERX) + xx + x + borderx;
	      *(pixmap + i) = (unsigned char) text_fg;
#else
	      Fgl_setpixel (xx + x, yy + y, text_fg);
#endif
	    }
	  b = b << 1;
	}
    }
}


void
open_x_putchar (int xx, int yy, int c)
{
  int x, y, b;
  for (y = 0; y < open_font_height; y++)
    {
      b = open_font[c * open_font_height + y];
      for (x = 0; x < 8; x++)
	{
	  if ((b & 0x80) == 0)
	    Fgl_setpixel (xx + x, yy + y, text_bg);
	  else
	    Fgl_setpixel (xx + x, yy + y, text_fg);
	  b = b << 1;
	}
    }
}


void
Fgl_fillbox (int x1, int y1, int w, int h, int col)
{
  int x, y;
  if (clipping_flag)
    {
      if (x1 < xclip_x1)
	x1 = xclip_x1;
      if (x1 + w > xclip_x2)
	w = xclip_x2 - x1;
      if (y1 < xclip_y1)
	y1 = xclip_y1;
      if (y1 + h > xclip_y2)
	h = xclip_y2 - y1;
    }
  col &= 0xff;

  // Draw to pixmap buffer
  for (y = y1 + bordery; y < y1 + h + bordery; y++)
    for (x = x1 + borderx; x < x1 + w + borderx; x++)
      *(pixmap + y * (640 + BORDERX) + x) = col;

  // Draw to DC
  RECT rect;
  if (display.useDIB)
    {
      HBITMAP hOldBitmapMem;
      HBRUSH hbr, hbrOld;
      rect.left = x1 + borderx;
      rect.top = y1 + bordery;
      rect.right = x1 + borderx + w;
      rect.bottom = y1 + bordery + h;
      hbr = GetPaletteBrush (col);
      hbrOld = (HBRUSH) SelectObject (display.hdcMem, hbr);
      hOldBitmapMem = (HBITMAP) SelectObject (display.hdcMem, display.hDIB);
      FillRect (display.hdcMem, &rect, hbr);
      display.hDIB = (HBITMAP) SelectObject (display.hdcMem, hOldBitmapMem);
      hbr = (HBRUSH) SelectObject (display.hdcMem, hbrOld);
    }
  else
    {
      HBRUSH hbr, hbrOld;
      rect.left = AdjustX (x1);
      rect.top = AdjustY (y1);
      rect.right = AdjustX (x1 + w);
      rect.bottom = AdjustY (y1 + h);
      hbr = GetPaletteBrush (col);
      hbrOld = (HBRUSH) SelectObject (display.hdcMem, hbr);
      FillRect (display.hdcMem, &rect, hbr);
      hbr = (HBRUSH) SelectObject (display.hdcMem, hbrOld);
    }

  // Queue update event
  InvalidateRect (display.hWnd, &rect, FALSE);
}

void
Fgl_putbox (int x1, int y1, int w, int h, void *buf)
{
  unsigned char *b;
  b = (unsigned char *) buf;
  int x, y;
  for (y = y1; y < y1 + h; y++)
    for (x = x1; x < x1 + w; x++)
      {
	Fgl_setpixel (x, y, *(b++));
      }

  // Queue update event
  RECT rect;
  rect.left = AdjustX (x1);
  rect.top = AdjustY (y1);
  rect.right = AdjustX (x1 + w);
  rect.bottom = AdjustY (y1 + h);
  InvalidateRect (display.hWnd, &rect, FALSE);
  // GCS: Don't update window here -- it causes farms to jump
  //      around when panning main screen
  //UpdateWindow (display.hWnd);
}

void
Fgl_getbox (int x1, int y1, int w, int h, void *buf)
{
  unsigned char *b;
  b = (unsigned char *) buf;
  int x, y;
  for (y = y1; y < y1 + h; y++)
    for (x = x1; x < x1 + w; x++)
      *(b++) = (unsigned char) Fgl_getpixel (x, y);
}

void
RefreshScreen ()
{
  InvalidateRect (display.hWnd, NULL, FALSE);
  ProcessNextEvent ();
}

void
RefreshArea (int x1, int y1, int x2, int y2)	// bounds of refresh area
 {
  RECT rect;
  rect.left = AdjustX (x1);
  rect.top = AdjustY (y1);
  rect.right = AdjustX (x2);
  rect.bottom = AdjustY (y2);
  InvalidateRect (display.hWnd, &rect, FALSE);
  UpdateWindow (display.hWnd);
}


void
SaveUnder (int x1, int y1, int w, int h, void *buf)
{
  HBITMAP hBitmapOld;
  HPALETTE hPalOld;

  // Copy pixmap buf to caller's buf
  unsigned char *b;
  b = (unsigned char *) buf;
  int x, y;
  for (y = y1; y < y1 + h; y++)
    for (x = x1; x < x1 + w; x++)
      *(b++) = (unsigned char) Fgl_getpixel (x, y);

  RECT rect;
  rect.left = AdjustX (x1);
  rect.top = AdjustY (y1);
  rect.right = AdjustX (x1 + w);
  rect.bottom = AdjustY (y1 + h);
  int saveunder_width = rect.right - rect.left;
  int saveunder_height = rect.bottom - rect.top;

  assert (display.hSaveUnderHdc == 0);
  assert (display.hSaveUnderBitmap == 0);

  // Create HDC
  display.hSaveUnderHdc = CreateCompatibleDC (display.hdcMem);
  if (display.hasPalette)
    {
      hPalOld = SelectPalette (display.hSaveUnderHdc, (HPALETTE) display.hPal, FALSE);
      RealizePalette (display.hSaveUnderHdc);
    }

  // Create and select bitmap
  display.hSaveUnderBitmap = CreateCompatibleBitmap (display.hdcMem, saveunder_width, saveunder_height);
  hBitmapOld = (HBITMAP) SelectObject (display.hSaveUnderHdc, display.hSaveUnderBitmap);

  // Copy from backing store bitmap to save under bitmap
  BitBlt (display.hSaveUnderHdc, 0, 0, saveunder_width, saveunder_height, display.hdcMem, rect.left, rect.top, SRCCOPY);
}


void
RestoreSaveUnder (int x1, int y1, int w, int h, void *buf)
{
  // Copy pixmap buf to caller's buf
  unsigned char *b;
  b = (unsigned char *) buf;
  int x, y;
  for (y = y1; y < y1 + h; y++)
    for (x = x1; x < x1 + w; x++)
      {
	// Draw to pixmap buffer
	int i = (y + bordery) * (640 + BORDERX) + x + borderx;
	*(pixmap + i) = (unsigned char) *(b++);
      }

  RECT rect;
  rect.left = AdjustX (x1);
  rect.top = AdjustY (y1);
  rect.right = AdjustX (x1 + w);
  rect.bottom = AdjustY (y1 + h);
  int saveunder_width = rect.right - rect.left;
  int saveunder_height = rect.bottom - rect.top;

  assert (display.hSaveUnderHdc != 0);
  assert (display.hSaveUnderBitmap != 0);

  // Copy from backing store bitmap to save under bitmap
  BitBlt (display.hdcMem, rect.left, rect.top, saveunder_width, saveunder_height, display.hSaveUnderHdc, 0, 0, SRCCOPY);

  // Destroy Bitmap
  DeleteObject (display.hSaveUnderBitmap);
  display.hSaveUnderBitmap = 0;

  // Destroy HDC
  DeleteDC (display.hSaveUnderHdc);
  display.hSaveUnderHdc = 0;

  // Refresh Window
  InvalidateRect (display.hWnd, &rect, FALSE);
  UpdateWindow (display.hWnd);
}


void
Fgl_enableclipping (void)
{
  clipping_flag = 1;
}

void
Fgl_setclippingwindow (int x1, int y1, int x2, int y2)
{
  xclip_x1 = x1;
  xclip_y1 = y1;
  xclip_x2 = x2;
  xclip_y2 = y2;
}

void
Fgl_disableclipping (void)
{
  clipping_flag = 0;
}

void
do_call_event (int wait)
{
  // Wait (if requested to do so)
  if (wait)
    lc_usleep (1000);

  // Process mouse events
  HandleMouse ();
}

void
call_event (void)
{
  do_call_event (0);
}

void
call_wait_event (void)
{
  do_call_event (1);
}

#ifdef USE_PIXMAPS
void
init_pixmaps ()
{
#if defined (WIN32)
  for (int i = 1; i < NUM_OF_TYPES; i++)
    {
      icon_pixmap[i] = 0;
    }
#endif
  init_icon_pixmap (CST_GREEN);
// powerlines
  init_icon_pixmap (CST_POWERL_H_L);
  init_icon_pixmap (CST_POWERL_V_L);
  init_icon_pixmap (CST_POWERL_LD_L);
  init_icon_pixmap (CST_POWERL_RD_L);
  init_icon_pixmap (CST_POWERL_LU_L);
  init_icon_pixmap (CST_POWERL_RU_L);
  init_icon_pixmap (CST_POWERL_LDU_L);
  init_icon_pixmap (CST_POWERL_LDR_L);
  init_icon_pixmap (CST_POWERL_LUR_L);
  init_icon_pixmap (CST_POWERL_UDR_L);
  init_icon_pixmap (CST_POWERL_LUDR_L);
  prog_box ("", 6);
  init_icon_pixmap (CST_POWERL_H_D);
  init_icon_pixmap (CST_POWERL_V_D);
  init_icon_pixmap (CST_POWERL_LD_D);
  init_icon_pixmap (CST_POWERL_RD_D);
  init_icon_pixmap (CST_POWERL_LU_D);
  init_icon_pixmap (CST_POWERL_RU_D);
  init_icon_pixmap (CST_POWERL_LDU_D);
  init_icon_pixmap (CST_POWERL_LDR_D);
  init_icon_pixmap (CST_POWERL_LUR_D);
  init_icon_pixmap (CST_POWERL_UDR_D);
  init_icon_pixmap (CST_POWERL_LUDR_D);
// HT windmills
  init_icon_pixmap (CST_WINDMILL_1_G);
  init_icon_pixmap (CST_WINDMILL_2_G);
  init_icon_pixmap (CST_WINDMILL_3_G);
  prog_box ("", 12);
  init_icon_pixmap (CST_WINDMILL_1_RG);
  init_icon_pixmap (CST_WINDMILL_2_RG);
  init_icon_pixmap (CST_WINDMILL_3_RG);
  init_icon_pixmap (CST_WINDMILL_1_R);
  init_icon_pixmap (CST_WINDMILL_2_R);
  init_icon_pixmap (CST_WINDMILL_3_R);
//LT windmills
  init_icon_pixmap (CST_WINDMILL_1_W);
  init_icon_pixmap (CST_WINDMILL_2_W);
  init_icon_pixmap (CST_WINDMILL_3_W);
// communes
  init_icon_pixmap (CST_COMMUNE_1);
  init_icon_pixmap (CST_COMMUNE_2);
  init_icon_pixmap (CST_COMMUNE_3);
  init_icon_pixmap (CST_COMMUNE_4);
  init_icon_pixmap (CST_COMMUNE_5);
  prog_box ("", 18);
  init_icon_pixmap (CST_COMMUNE_6);
  init_icon_pixmap (CST_COMMUNE_7);
  init_icon_pixmap (CST_COMMUNE_8);
  init_icon_pixmap (CST_COMMUNE_9);
  init_icon_pixmap (CST_COMMUNE_10);
  init_icon_pixmap (CST_COMMUNE_11);
  init_icon_pixmap (CST_COMMUNE_12);
  init_icon_pixmap (CST_COMMUNE_13);
  init_icon_pixmap (CST_COMMUNE_14);
// farms   (3 7 11 and 15 are the only ones needed)
  init_icon_pixmap (CST_FARM_O3);
  init_icon_pixmap (CST_FARM_O7);
  init_icon_pixmap (CST_FARM_O11);
  init_icon_pixmap (CST_FARM_O15);
// Lt. Industry
  prog_box ("", 24);
  init_icon_pixmap (CST_INDUSTRY_L_C);
  init_icon_pixmap (CST_INDUSTRY_L_Q1);
  init_icon_pixmap (CST_INDUSTRY_L_Q2);
  init_icon_pixmap (CST_INDUSTRY_L_Q3);
  init_icon_pixmap (CST_INDUSTRY_L_Q4);
  init_icon_pixmap (CST_INDUSTRY_L_L1);
  init_icon_pixmap (CST_INDUSTRY_L_L2);
  init_icon_pixmap (CST_INDUSTRY_L_L3);
  init_icon_pixmap (CST_INDUSTRY_L_L4);
  init_icon_pixmap (CST_INDUSTRY_L_M1);
  init_icon_pixmap (CST_INDUSTRY_L_M2);
  init_icon_pixmap (CST_INDUSTRY_L_M3);
  init_icon_pixmap (CST_INDUSTRY_L_M4);
  init_icon_pixmap (CST_INDUSTRY_L_H1);
  init_icon_pixmap (CST_INDUSTRY_L_H2);
  prog_box ("", 30);
  init_icon_pixmap (CST_INDUSTRY_L_H3);
  init_icon_pixmap (CST_INDUSTRY_L_H4);
// Hv. Industry
  init_icon_pixmap (CST_INDUSTRY_H_C);
  init_icon_pixmap (CST_INDUSTRY_H_L1);
  init_icon_pixmap (CST_INDUSTRY_H_L2);
  init_icon_pixmap (CST_INDUSTRY_H_L3);
  init_icon_pixmap (CST_INDUSTRY_H_L4);
  init_icon_pixmap (CST_INDUSTRY_H_L5);
  init_icon_pixmap (CST_INDUSTRY_H_L6);
  init_icon_pixmap (CST_INDUSTRY_H_L7);
  init_icon_pixmap (CST_INDUSTRY_H_L8);
  init_icon_pixmap (CST_INDUSTRY_H_M1);
  init_icon_pixmap (CST_INDUSTRY_H_M2);
  init_icon_pixmap (CST_INDUSTRY_H_M3);
  init_icon_pixmap (CST_INDUSTRY_H_M4);
  prog_box ("", 36);
  init_icon_pixmap (CST_INDUSTRY_H_M5);
  init_icon_pixmap (CST_INDUSTRY_H_M6);
  init_icon_pixmap (CST_INDUSTRY_H_M7);
  init_icon_pixmap (CST_INDUSTRY_H_M8);
  init_icon_pixmap (CST_INDUSTRY_H_H1);
  init_icon_pixmap (CST_INDUSTRY_H_H2);
  init_icon_pixmap (CST_INDUSTRY_H_H3);
  init_icon_pixmap (CST_INDUSTRY_H_H4);
  init_icon_pixmap (CST_INDUSTRY_H_H5);
  init_icon_pixmap (CST_INDUSTRY_H_H6);
  init_icon_pixmap (CST_INDUSTRY_H_H7);
  init_icon_pixmap (CST_INDUSTRY_H_H8);
// water
  prog_box ("", 42);
  init_icon_pixmap (CST_WATER);
  init_icon_pixmap (CST_WATER_D);
  init_icon_pixmap (CST_WATER_R);
  init_icon_pixmap (CST_WATER_U);
  init_icon_pixmap (CST_WATER_L);
  init_icon_pixmap (CST_WATER_LR);
  init_icon_pixmap (CST_WATER_UD);
  init_icon_pixmap (CST_WATER_LD);
  init_icon_pixmap (CST_WATER_RD);
  init_icon_pixmap (CST_WATER_LU);
  init_icon_pixmap (CST_WATER_UR);
  init_icon_pixmap (CST_WATER_LUD);
  init_icon_pixmap (CST_WATER_LRD);
  init_icon_pixmap (CST_WATER_LUR);
  init_icon_pixmap (CST_WATER_URD);
  init_icon_pixmap (CST_WATER_LURD);
// tracks
  init_icon_pixmap (CST_TRACK_LR);
  prog_box ("", 48);
  init_icon_pixmap (CST_TRACK_LU);
  init_icon_pixmap (CST_TRACK_LD);
  init_icon_pixmap (CST_TRACK_UD);
  init_icon_pixmap (CST_TRACK_UR);
  init_icon_pixmap (CST_TRACK_DR);
  init_icon_pixmap (CST_TRACK_LUR);
  init_icon_pixmap (CST_TRACK_LDR);
  init_icon_pixmap (CST_TRACK_LUD);
  init_icon_pixmap (CST_TRACK_UDR);
  init_icon_pixmap (CST_TRACK_LUDR);
// roads
  init_icon_pixmap (CST_ROAD_LR);
  init_icon_pixmap (CST_ROAD_LU);
  init_icon_pixmap (CST_ROAD_LD);
  init_icon_pixmap (CST_ROAD_UD);
  init_icon_pixmap (CST_ROAD_UR);
  init_icon_pixmap (CST_ROAD_DR);
  init_icon_pixmap (CST_ROAD_LUR);
  init_icon_pixmap (CST_ROAD_LDR);
  prog_box ("", 54);
  init_icon_pixmap (CST_ROAD_LUD);
  init_icon_pixmap (CST_ROAD_UDR);
  init_icon_pixmap (CST_ROAD_LUDR);
// rail
  init_icon_pixmap (CST_RAIL_LR);
  init_icon_pixmap (CST_RAIL_LU);
  init_icon_pixmap (CST_RAIL_LD);
  init_icon_pixmap (CST_RAIL_UD);
  init_icon_pixmap (CST_RAIL_UR);
  init_icon_pixmap (CST_RAIL_DR);
  init_icon_pixmap (CST_RAIL_LUR);
  init_icon_pixmap (CST_RAIL_LDR);
  init_icon_pixmap (CST_RAIL_LUD);
  init_icon_pixmap (CST_RAIL_UDR);
  init_icon_pixmap (CST_RAIL_LUDR);
// potteries
  prog_box ("", 60);
  init_icon_pixmap (CST_POTTERY_0);
  init_icon_pixmap (CST_POTTERY_1);
  init_icon_pixmap (CST_POTTERY_2);
  init_icon_pixmap (CST_POTTERY_3);
  init_icon_pixmap (CST_POTTERY_4);
  init_icon_pixmap (CST_POTTERY_5);
  init_icon_pixmap (CST_POTTERY_6);
  init_icon_pixmap (CST_POTTERY_7);
  init_icon_pixmap (CST_POTTERY_8);
  init_icon_pixmap (CST_POTTERY_9);
  init_icon_pixmap (CST_POTTERY_10);
// mills
  init_icon_pixmap (CST_MILL_0);
  init_icon_pixmap (CST_MILL_1);
  init_icon_pixmap (CST_MILL_2);
  init_icon_pixmap (CST_MILL_3);
  init_icon_pixmap (CST_MILL_4);
  prog_box ("", 66);
  init_icon_pixmap (CST_MILL_5);
  init_icon_pixmap (CST_MILL_6);
// blacksmiths
  init_icon_pixmap (CST_BLACKSMITH_0);
  init_icon_pixmap (CST_BLACKSMITH_1);
  init_icon_pixmap (CST_BLACKSMITH_2);
  init_icon_pixmap (CST_BLACKSMITH_3);
  init_icon_pixmap (CST_BLACKSMITH_4);
  init_icon_pixmap (CST_BLACKSMITH_5);
  init_icon_pixmap (CST_BLACKSMITH_6);
// residences
  init_icon_pixmap (CST_RESIDENCE_LL);
  init_icon_pixmap (CST_RESIDENCE_ML);
  init_icon_pixmap (CST_RESIDENCE_HL);
  init_icon_pixmap (CST_RESIDENCE_LH);
  init_icon_pixmap (CST_RESIDENCE_MH);
  init_icon_pixmap (CST_RESIDENCE_HH);
// coal power
  init_icon_pixmap (CST_POWERS_COAL_EMPTY);
  init_icon_pixmap (CST_POWERS_COAL_LOW);
  prog_box ("", 72);
  init_icon_pixmap (CST_POWERS_COAL_MED);
  init_icon_pixmap (CST_POWERS_COAL_FULL);
// substations
  init_icon_pixmap (CST_SUBSTATION_R);
  init_icon_pixmap (CST_SUBSTATION_G);
  init_icon_pixmap (CST_SUBSTATION_RG);
// markets
  init_icon_pixmap (CST_MARKET_EMPTY);
  init_icon_pixmap (CST_MARKET_LOW);
  init_icon_pixmap (CST_MARKET_MED);
  init_icon_pixmap (CST_MARKET_FULL);
// coal mines
  init_icon_pixmap (CST_COALMINE_EMPTY);
  init_icon_pixmap (CST_COALMINE_LOW);
  init_icon_pixmap (CST_COALMINE_MED);
  init_icon_pixmap (CST_COALMINE_FULL);
// ore mines
  prog_box ("", 78);
  init_icon_pixmap (CST_OREMINE_1);
  init_icon_pixmap (CST_OREMINE_2);
  init_icon_pixmap (CST_OREMINE_3);
  init_icon_pixmap (CST_OREMINE_4);
  init_icon_pixmap (CST_OREMINE_5);
  init_icon_pixmap (CST_OREMINE_6);
  init_icon_pixmap (CST_OREMINE_7);
  init_icon_pixmap (CST_OREMINE_8);
// tips
  init_icon_pixmap (CST_TIP_0);
  init_icon_pixmap (CST_TIP_1);
  init_icon_pixmap (CST_TIP_2);
  init_icon_pixmap (CST_TIP_3);
  init_icon_pixmap (CST_TIP_4);
  init_icon_pixmap (CST_TIP_5);
  init_icon_pixmap (CST_TIP_6);
  init_icon_pixmap (CST_TIP_7);
  init_icon_pixmap (CST_TIP_8);
// rockets
  prog_box ("", 84);
  init_icon_pixmap (CST_ROCKET_1);
  init_icon_pixmap (CST_ROCKET_2);
  init_icon_pixmap (CST_ROCKET_3);
  init_icon_pixmap (CST_ROCKET_4);
  init_icon_pixmap (CST_ROCKET_5);
  init_icon_pixmap (CST_ROCKET_6);
  init_icon_pixmap (CST_ROCKET_7);
  init_icon_pixmap (CST_ROCKET_FLOWN);
// fire stations
  init_icon_pixmap (CST_FIRESTATION_1);
  init_icon_pixmap (CST_FIRESTATION_2);
  init_icon_pixmap (CST_FIRESTATION_3);
  init_icon_pixmap (CST_FIRESTATION_4);
  init_icon_pixmap (CST_FIRESTATION_5);
  init_icon_pixmap (CST_FIRESTATION_6);
  init_icon_pixmap (CST_FIRESTATION_7);
  init_icon_pixmap (CST_FIRESTATION_8);
  init_icon_pixmap (CST_FIRESTATION_9);
  init_icon_pixmap (CST_FIRESTATION_10);
// cricket
  prog_box ("", 90);
  init_icon_pixmap (CST_CRICKET_1);
  init_icon_pixmap (CST_CRICKET_2);
  init_icon_pixmap (CST_CRICKET_3);
  init_icon_pixmap (CST_CRICKET_4);
  init_icon_pixmap (CST_CRICKET_5);
  init_icon_pixmap (CST_CRICKET_6);
  init_icon_pixmap (CST_CRICKET_7);
// fire
  init_icon_pixmap (CST_FIRE_1);
  init_icon_pixmap (CST_FIRE_2);
  init_icon_pixmap (CST_FIRE_3);
  init_icon_pixmap (CST_FIRE_4);
  init_icon_pixmap (CST_FIRE_5);
  init_icon_pixmap (CST_FIRE_DONE1);
  init_icon_pixmap (CST_FIRE_DONE2);
  init_icon_pixmap (CST_FIRE_DONE3);
  init_icon_pixmap (CST_FIRE_DONE4);

// others
  init_icon_pixmap (CST_PARKLAND_PLANE);
  init_icon_pixmap (CST_RECYCLE);
  init_icon_pixmap (CST_HEALTH);
  init_icon_pixmap (CST_BURNT);
  init_icon_pixmap (CST_MONUMENT_5);
  init_icon_pixmap (CST_SCHOOL);
  init_icon_pixmap (CST_SHANTY);
  prog_box ("", 96);
  init_icon_pixmap (CST_POWERS_SOLAR);
  init_icon_pixmap (CST_UNIVERSITY);
  init_icon_pixmap (CST_EX_PORT);
}

void
init_icon_pixmap (int type)
{
  unsigned char *g;
  int x, y;
  HBITMAP hOldBitmapMem;
  int w, h;

  w = main_types[type].size * 16;
  h = main_types[type].size * 16;
  w <<= pix_double;
  h <<= pix_double;
  icon_pixmap[type] = CreateCompatibleBitmap (display.hdcMem, w, h);
  hOldBitmapMem = (HBITMAP) SelectObject (display.hdcMem, icon_pixmap[type]);

  // Copy bits to bitmap
  g = (unsigned char *) main_types[type].graphic;
  for (y = 0; y < main_types[type].size * 16; y++)
    {
      for (x = 0; x < main_types[type].size * 16; x++)
	{
	  if (pix_double)
	    {
	      RECT rect;
	      HBRUSH hbr, hbrOld;
	      rect.left = x << 1;
	      rect.top = y << 1;
	      rect.right = (x + 1) << 1;
	      rect.bottom = (y + 1) << 1;
	      hbr = GetPaletteBrush (*g++);
	      hbrOld = (HBRUSH) SelectObject (display.hdcMem, hbr);
	      FillRect (display.hdcMem, &rect, hbr);
	      hbr = (HBRUSH) SelectObject (display.hdcMem, hbrOld);
	    }
	  else
	    {
	      SetPixel (display.hdcMem, x, y, GetPaletteColorref (*g++));
	    }
	}
    }
  icon_pixmap[type] = (HBITMAP) SelectObject (display.hdcMem, hOldBitmapMem);
}

void
update_pixmap (int x1, int y1, int sizex, int sizey, int dx, int dy
	       ,int real_size, char *g)
{
  char *i, *j;
  int x, y;
//printf("x1=%d y1=%d sizex=%d sizey=%d dx=%d dy=%d\n",x1,y1,sizex
  //,sizey,dx,dy);
  for (y = 0; y < sizey; y++)
    {
      i = (pixmap + ((dy + y + bordery) * (640 + BORDERX) + dx + borderx));
      j = (g + ((y1 + y) * real_size * 16 + x1));
//printf("%p %p -> %p %p size=%d\n",pixmap,g,i,j,real_size);
      for (x = 0; x < sizex; x++)
	*(i++) = *(j++);
    }
}

#endif
