/***********************************************************************************

	Copyright (C) 2007-2009 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include "lifeograph.hpp"


using namespace LIFEO;


inline void
Lifeograph::create_action(	Glib::RefPtr< Gtk::Action >			&action,
							Glib::RefPtr< Gtk::ActionGroup >	&actiongroup,
							const Glib::ustring					&name,
							const Gtk::StockID					&stockid,
							const Glib::ustring					&label,
							const Glib::ustring					&tooltip,
							const Gtk::AccelKey					&accelkey,
							const Glib::SignalProxy0< void >::SlotType
																&slot,
							Gtk::Widget							*widget_proxy )
{
	action  = Gtk::Action::create( name, stockid, label, tooltip );
	actiongroup->add( action, accelkey );
	action->set_accel_group( m_window->get_accel_group() );
	action->connect_accelerator();
	if( widget_proxy != NULL )
		action->connect_proxy( *widget_proxy );
	action->signal_activate().connect( slot );
}

void
Lifeograph::draw_welcomescreen( void )
{
	Gtk::Alignment *alignment = Gtk::manage( new Gtk::Alignment( 0.5, 0.5, 0.1, 0 ) );
	Gtk::Label *label_salutation = Gtk::manage(
		new Gtk::Label( _( STRING::SALUTATION ), 0.5, 0.5 ) );
	Gtk::Button *button_start = Gtk::manage( new Gtk::Button( _("_Continue"), true ) );
	Gtk::Image *icon_forward = Gtk::manage(
		new Gtk::Image( Gtk::Stock::GO_FORWARD, Gtk::ICON_SIZE_MENU ) );
	Gtk::VBox *vbox = Gtk::manage( new Gtk::VBox );

	alignment->set_padding( 120, 120, 150, 150 );
	m_window->add( *alignment );

	label_salutation->set_use_markup( true );

	button_start->set_image( *icon_forward );
	button_start->set_image_position( Gtk::POS_RIGHT );

	vbox->set_spacing( 50 );

	vbox->pack_start( *label_salutation );
	vbox->pack_start( *button_start );

	alignment->add( *vbox );

	button_start->signal_clicked().connect(
		sigc::mem_fun( *this, &Lifeograph::draw_loginscreen ) );

	if( bool( m_pixbuf_background ) )
		m_connection_event_expose = m_window->signal_expose_event().connect(
				sigc::mem_fun( *this, &Lifeograph::handle_event_expose ) );

	m_window->show_all();
}

void
Lifeograph::draw_loginscreen( void )
{
	m_internaloperation++;

	m_window->remove();

	Gtk::VBox			*vbox;
	Gtk::Alignment		*align1, *align2;
	Gtk::HBox			*hbox;
	Gtk::Label			*label_diary, *label_password;
	Gtk::Button			*button_about, *button_quit;
	Gtk::HSeparator		*hseparator;

	try {
		vbox = Gtk::manage( new Gtk::VBox );
		align1 = Gtk::manage( new Gtk::Alignment( 0.5, 0.5, 0.4, 0.1 ) );
		align2 = Gtk::manage( new Gtk::Alignment( 1.0, 0.5, 0, 1.0 ) );
		m_table_login = Gtk::manage( new Gtk::Table );
		hbox = Gtk::manage( new Gtk::HBox );
		label_diary = Gtk::manage( new Gtk::Label( _( "Diary:" ), 0, 0.5 ) );
		label_password = Gtk::manage( new Gtk::Label( _( "Password:" ), 0, 0.5 ) );
		m_label_startupmessage = Gtk::manage( new Gtk::Label( "", 0, 0.5 ) );
		m_button_opendiary = Gtk::manage( new Gtk::Button( Gtk::Stock::OK ) );
		m_entry_password = Gtk::manage( new Gtk::Entry );
		button_about = Gtk::manage( new Gtk::Button( Gtk::Stock::ABOUT ) );
		button_quit = Gtk::manage( new Gtk::Button( Gtk::Stock::QUIT ) );
		hseparator = Gtk::manage( new Gtk::HSeparator );
	}
	catch( std::exception & ex )
	{
		throw ex;
	}

	align1->set_padding( 110, 90, 30, 30 );

	m_table_login->set_spacings( 5 );

	m_button_opendiary->set_flags( Gtk::CAN_DEFAULT | Gtk::RECEIVES_DEFAULT );
	m_window->set_default( *m_button_opendiary );

	m_entry_password->set_visibility( false );
	m_entry_password->set_activates_default( true );

	m_label_startupmessage->set_line_wrap( true );
	m_label_startupmessage->set_alignment( Gtk::ALIGN_CENTER );

	button_about->set_relief( Gtk::RELIEF_NONE );
	button_about->set_focus_on_click( false );
	button_quit->set_relief( Gtk::RELIEF_NONE );
	button_quit->set_focus_on_click( false );

	// PACKING
	hbox->pack_start( *button_about, Gtk::PACK_SHRINK, 5 );
	hbox->pack_start( *button_quit, Gtk::PACK_SHRINK, 5 );

	m_table_login->attach( *label_diary, 0, 1, 0, 1,
						Gtk::SHRINK | Gtk::FILL, Gtk::SHRINK );
	m_table_login->attach( *label_password, 0, 1, 1, 2,
						Gtk::SHRINK | Gtk::FILL, Gtk::SHRINK );

	m_table_login->attach( *m_filebuttonrecent, 1, 2, 0, 1,
						Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );
	m_table_login->attach( *m_entry_password, 1, 2, 1, 2,
						Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );

	m_table_login->attach( *m_button_opendiary, 2, 3, 0, 2,
						Gtk::SHRINK | Gtk::FILL, Gtk::SHRINK | Gtk::FILL );

	m_table_login->attach( *hseparator, 0, 3, 2, 3,
						Gtk::SHRINK | Gtk::FILL, Gtk::SHRINK );

	m_table_login->attach( *m_label_startupmessage, 0, 3, 3, 4,
						Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );

	align1->add( *m_table_login );
	align2->add( *hbox );

	vbox->pack_start( *align1, Gtk::PACK_EXPAND_WIDGET, 5 );
	vbox->pack_start( *align2, Gtk::PACK_SHRINK, 5 );

	m_window->add( *vbox );

	// SIGNALS
	m_entry_password->signal_changed().connect(
			sigc::mem_fun( *this, &Lifeograph::handle_entry_password_changed ) );
	m_button_opendiary->signal_clicked().connect(
			sigc::mem_fun( *this, &Lifeograph::handle_button_opendb_clicked ) );
	button_quit->signal_clicked().connect(
			sigc::mem_fun( *this, &Lifeograph::handle_button_quit_clicked ) );
	button_about->signal_clicked().connect(
			sigc::mem_fun( *this, &Lifeograph::handle_button_about_clicked ) );

	if( bool( m_pixbuf_background ) && m_connection_event_expose.empty() )
		m_connection_event_expose = m_window->signal_expose_event().connect(
				sigc::mem_fun( *this, &Lifeograph::handle_event_expose ) );

	// SETTING DB FILE
	if( m_database.is_path_set() )
	{
		update_login_file();
	}
	else
	{
		m_entry_password->set_sensitive( false );
		m_button_opendiary->set_sensitive( false );
		m_label_startupmessage->set_label( _( "Welcome new user!" ) );
	}

	m_window->show_all();
	m_window->set_title( PROGRAM_NAME );

	m_internaloperation--;
}

void
Lifeograph::draw_editscreen( void )
{
	m_connection_event_expose.disconnect();
	// Filebutton has to be unparented to be used again:
	if( m_filebuttonrecent->get_parent() )
		m_table_login->remove( *m_filebuttonrecent );
	m_window->remove();

	Gtk::VBox				*vbox, *vbox_entry, *vbox_navigation;
	Gtk::Toolbar			*toolbar;
	Gtk::ToolItem			*toolitem_space;
	Gtk::Menu				*menu_logout;
	Menutoolbutton			*toolitem_tools;
	Gtk::SeparatorToolItem	*toolbarseparator1, *toolbarseparator2;
	Gtk::ImageMenuItem		*menuitem_password, *menuitem_about,
							*menuitem_logoutwosaving;
	Gtk::MenuItem			*menuitem_chaptereditor, *menuitem_backup;
	Gtk::Image				*image_logoutwosaving, *image_remove;
	Gtk::SeparatorMenuItem	*menuitem_separator1;
	Gtk::ScrolledWindow		*scrolledwindow1, *scrolledwindow2,
							*scrolledwindow3;

	try {
		vbox = Gtk::manage( new Gtk::VBox );
		vbox_entry = Gtk::manage( new Gtk::VBox );
		vbox_navigation = Gtk::manage( new Gtk::VBox );

		// TOOLBAR AND ITS ITEMS
		toolbar = Gtk::manage( new Gtk::Toolbar );
		m_toolbutton_previous = Gtk::manage( new Gtk::ToolButton );
		m_toolbutton_next = Gtk::manage( new Gtk::ToolButton );
		m_toolbutton_today = Gtk::manage( new Gtk::ToolButton );
		m_toolitem_filter = Gtk::manage( new Gtk::ToolItem );
		m_toolbutton_findprevious = Gtk::manage( new Gtk::ToolButton );
		m_toolbutton_findnext = Gtk::manage( new Gtk::ToolButton );
		m_toolbutton_logout = Gtk::manage(
			new Gtk::MenuToolButton( Gtk::Stock::DIALOG_AUTHENTICATION ) );
		menu_logout = Gtk::manage( new Gtk::Menu );
		toolitem_tools = Gtk::manage(
			new Menutoolbutton( Gtk::Stock::PREFERENCES, _( "Tools" ) ) );
		toolitem_space = Gtk::manage( new Gtk::ToolItem );
		toolbarseparator1 = Gtk::manage( new Gtk::SeparatorToolItem );
		toolbarseparator2 = Gtk::manage( new Gtk::SeparatorToolItem );

		// TOOLS MENU
		menuitem_chaptereditor = Gtk::manage(
			new Gtk::MenuItem( _( "_Edit Chapters..." ), true ) );
		menuitem_password = Gtk::manage(
			new Gtk::ImageMenuItem( _( "_Change Password..." ), true ) );
		menuitem_backup = Gtk::manage(
			new Gtk::MenuItem( _( "E_xport/Backup Current Diary..." ), true ) );
		image_logoutwosaving = Gtk::manage(
			new Gtk::Image( Gtk::Stock::STOP, Gtk::ICON_SIZE_MENU ) );
		menuitem_about = Gtk::manage( new Gtk::ImageMenuItem( Gtk::Stock::ABOUT ) );
		menuitem_separator1 = Gtk::manage( new Gtk::SeparatorMenuItem );

		// LOGOUT MENU
		menuitem_logoutwosaving = Gtk::manage(
			new Gtk::ImageMenuItem( *image_logoutwosaving,
									_( "_Log out without Saving" ),
									true ) );
		m_menuitem_logoutonidle = Gtk::manage(
			new Gtk::CheckMenuItem( _( "Log out Automatically on _Idle" ),
									true ) );

		m_textviewdiary = Gtk::manage( new TextviewDiary( m_textbufferdiary ) );

		scrolledwindow1 = Gtk::manage( new Gtk::ScrolledWindow );
		scrolledwindow2 = Gtk::manage( new Gtk::ScrolledWindow );
		scrolledwindow3 = Gtk::manage( new Gtk::ScrolledWindow );

		m_hpaned_main = Gtk::manage( new Gtk::HPaned );
		m_hbox_tagtools = Gtk::manage( new Gtk::HBox );
		m_hbox_replace = Gtk::manage( new Gtk::HBox );
		m_entry_filter = Gtk::manage( new EntryIdletext( _( "Filter" ) ) );
		m_entry_replace = Gtk::manage( new Gtk::Entry );
		m_button_replace = Gtk::manage( new Gtk::Button( _( "Replace" ) ) );
		m_button_replaceall = Gtk::manage( new Gtk::Button( _( "Replace All" ) ) );
		m_calendar = Gtk::manage( new Gtk::Calendar );
		m_treeview_entries = Gtk::manage( new Gtk::TreeView );
		m_combobox_liststyle = Gtk::manage( new Gtk::ComboBoxText );
		m_iconview_tags = Gtk::manage( new Gtk::IconView );
		m_tagwidget = Gtk::manage( new Tagwidget( Tagwidget::OT_CREATE ) );
		image_remove = Gtk::manage(
			new Gtk::Image( Gtk::Stock::REMOVE, Gtk::ICON_SIZE_BUTTON ) );
	}
	catch( std::exception &ex )
	{
		throw ex;
	}

	// MENU ITEMS
	toolitem_tools->get_menu()->attach( *menuitem_chaptereditor, 0, 1, 0, 1 );
	toolitem_tools->get_menu()->attach( *menuitem_password, 0, 1, 1, 2 );
	toolitem_tools->get_menu()->attach( *menuitem_backup, 0, 1, 2, 3 );
	toolitem_tools->get_menu()->attach( *menuitem_separator1, 0, 1, 3, 4 );
	toolitem_tools->get_menu()->attach( *menuitem_about, 0, 1, 4, 5 );
	toolitem_tools->get_menu()->show_all_children();

	// TOOLBAR
	m_entry_filter->set_icon_from_stock( Gtk::Stock::CLEAR, Gtk::ENTRY_ICON_SECONDARY );
	m_entry_filter->set_icon_sensitive( Gtk::ENTRY_ICON_SECONDARY, false );
	m_toolitem_filter->add( *m_entry_filter );

	toolitem_space->set_expand( true );

	m_toolbutton_logout->set_label( _( "Log out" ) );
	m_toolbutton_logout->set_is_important( true );
	m_toolbutton_logout->set_tooltip_text( m_database.get_path() );
	m_toolbutton_logout->set_menu( *menu_logout );
	menu_logout->attach( *menuitem_logoutwosaving, 0, 1, 0, 1 );
	menu_logout->attach( *m_menuitem_logoutonidle, 0, 1, 1, 2 );
	menu_logout->show_all_children();

	toolbar->append( *m_toolbutton_previous );
	toolbar->append( *m_toolbutton_next );
	toolbar->append( *m_toolbutton_today );
	toolbar->append( *toolbarseparator1 );
	toolbar->append( *toolitem_tools );
	toolbar->append( *toolbarseparator2 );
	toolbar->append( *m_toolitem_filter );
	toolbar->append( *m_toolbutton_findprevious );
	toolbar->append( *m_toolbutton_findnext );
	toolbar->append( *toolitem_space );
	toolbar->append( *m_toolbutton_logout );

	// TEXTVIEW
	if( m_database.has_theme() )
		m_textbufferdiary->set_theme( m_database.get_theme() );
	else
		m_textbufferdiary->reset_theme();

	scrolledwindow1->set_policy( Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC );
	scrolledwindow1->set_size_request( 300, -1 );
	scrolledwindow1->add( *m_textviewdiary );

	// CALENDAR
	m_calendar->set_tooltip_text( _( "Double click a day to add a new entry" ) );

	// PACKING
	vbox_navigation->pack_start( *scrolledwindow3 );
	vbox_navigation->pack_start( *m_combobox_liststyle, Gtk::PACK_SHRINK );
	vbox_navigation->pack_start( *m_calendar, Gtk::PACK_SHRINK );

	m_hbox_tagtools->pack_start( *m_tagwidget );

	vbox_entry->pack_start( *m_hbox_replace, Gtk::PACK_SHRINK );
	vbox_entry->pack_start( *scrolledwindow1 );
	vbox_entry->pack_start( *m_hbox_tagtools, Gtk::PACK_SHRINK );
	vbox_entry->pack_start( *scrolledwindow2, Gtk::PACK_SHRINK );
	m_hpaned_main->pack1( *vbox_entry );
	m_hpaned_main->pack2( *vbox_navigation );

	vbox->pack_start( *toolbar, Gtk::PACK_SHRINK );
	vbox->pack_start( *m_hpaned_main );

	scrolledwindow2->set_policy( Gtk::POLICY_NEVER, Gtk::POLICY_NEVER );
	scrolledwindow2->add( *m_iconview_tags );
	scrolledwindow2->set_size_request( 250, -1 );

	scrolledwindow3->set_policy( Gtk::POLICY_NEVER, Gtk::POLICY_AUTOMATIC );
	scrolledwindow3->add( *m_treeview_entries );

	m_menuitem_logoutonidle->set_active( m_option_autologout );

	int panePosition = m_gconfclient->get_int( CONFIG_PATH + "/pane_position" );
	if( panePosition > 0 )
		m_hpaned_main->set_position( panePosition );

	// ICONS
	Glib::RefPtr< Gtk::IconTheme > theme = Gtk::IconTheme::get_default();
	try {
		m_pixbuf_folder = theme->load_icon( "folder", 16 );
		m_pixbuf_add = theme->load_icon( "add", 22 );
		m_pixbuf_favored = Gdk::Pixbuf::create_from_file( ICONDIR "/favorite-16.png" );
		m_pixbuf_unfavored = Gdk::Pixbuf::create_from_file( ICONDIR "/entry-16.png" );
		m_pixbuf_tag = Gdk::Pixbuf::create_from_file( ICONDIR "/emblem-tag-22.png" );
	}
	catch( Glib::Error & ex )
	{
		//TODO: check whether program is not installed and look for icons in appropriate
		//		directories
		//	if ( !m_pixbuf_tag )
		//		m_pixbuf_tag = Gdk::Pixbuf::create( Gdk::COLORSPACE_RGB, false, 8, 1, 1 );
	}

	// ENTRIES LIST
	Gtk::CellRendererPixbuf *cellrendererPixbuf = Gtk::manage(
		new Gtk::CellRendererPixbuf );
	Gtk::CellRendererText *cellrendererDate = Gtk::manage(
		new Gtk::CellRendererText );
	Gtk::CellRendererText *cellrendererTitle = Gtk::manage(
		new Gtk::CellRendererText );
	Gtk::TreeViewColumn *columnDate = Gtk::manage(
		new Gtk::TreeViewColumn( _( "Date" ) ) );
	Gtk::TreeViewColumn *columnTitle = Gtk::manage(
		new Gtk::TreeViewColumn( _( "Title" ) ) );

	columnDate->pack_start( *cellrendererPixbuf );
	columnDate->pack_start( *cellrendererDate );
	columnDate->add_attribute( cellrendererDate->property_markup(),
								 m_colrec_entry.m_id );
	columnDate->add_attribute( cellrendererPixbuf->property_pixbuf(),
								 m_colrec_entry.m_icon );

	cellrendererTitle->property_ellipsize() = Pango::ELLIPSIZE_END;
	cellrendererTitle->property_scale() = .90;
	cellrendererTitle->set_fixed_height_from_font( 1 );
	columnTitle->pack_start( *cellrendererTitle );
	columnTitle->add_attribute( cellrendererTitle->property_text(),
								  m_colrec_entry.m_info );

	m_treeview_entries->set_rules_hint( true );
	m_treeview_entries->set_model( m_treestore_entries );
	m_treeview_entries->append_column( *columnDate );
	m_treeview_entries->append_column( *columnTitle );
	m_treeview_entries->set_search_column( -1 );     // disable built-in search
	m_treeview_entries->get_column( 0 )->set_sort_column( m_colrec_entry.m_id );
	m_treeview_entries->get_column( 1 )->set_sort_column( m_colrec_entry.m_info );
	// initially sort entries by date
	m_treestore_entries->set_sort_column( m_colrec_entry.m_id, Gtk::SORT_ASCENDING );

	// TAG VIEW
	m_iconview_tags->set_pixbuf_column( colrec_tag->m_icon );
	m_iconview_tags->set_markup_column( colrec_tag->m_name );
	m_iconview_tags->set_model( m_liststore_tagtable );
	m_iconview_tags->set_orientation( Gtk::ORIENTATION_HORIZONTAL );
	m_iconview_tags->set_selection_mode( Gtk::SELECTION_MULTIPLE );
	m_iconview_tags->set_events( Gdk::ENTER_NOTIFY_MASK );

	m_iconview_tags->set_item_width( 150 );

	m_tagwidget->set_model( m_liststore_tagcombo );
	m_tagwidget->set_text_column( colrec_tag->m_name );

	// REPLACE BAR
	m_entry_replace->set_tooltip_text( _( "Enter text to replace matches with" ) );
	m_button_replace->set_tooltip_text( _( "Replace selected (if any) or next match" ) );
	m_button_replaceall->set_tooltip_text(
			_( "Replace all matches throughout entire diary" ) );
	m_hbox_replace->set_spacing( 5 );
	m_hbox_replace->set_border_width( 5 );
	m_hbox_replace->pack_start( *m_entry_replace );
	m_hbox_replace->pack_start( *m_button_replace, Gtk::PACK_SHRINK );
	m_hbox_replace->pack_start( *m_button_replaceall, Gtk::PACK_SHRINK );

	m_window->add( *vbox );

	update_combobox_entryliststyle();

	// ACTIONS
	Glib::RefPtr< Gtk::ActionGroup > actiongroup =
		Gtk::ActionGroup::create( "lifeograph actions" );

	create_action(	m_action_undo, actiongroup,
					"Undo",
					Gtk::Stock::UNDO,
					_( "Undo" ),
					"",		// this tooltip will be dynamic
					Gtk::AccelKey( GDK_z, Gdk::CONTROL_MASK ),
					sigc::mem_fun( *this, &Lifeograph::handle_undo ) );

	create_action(	m_action_redo, actiongroup,
					"Redo",
					Gtk::Stock::REDO,
					_( "Redo" ),
					"",		// this tooltip will be dynamic
					Gtk::AccelKey( GDK_z, Gdk::CONTROL_MASK | Gdk::SHIFT_MASK ),
					sigc::mem_fun( *this, &Lifeograph::handle_redo ) );

	create_action(	m_action_previous_list, actiongroup,
					"ListPrevious",
					Gtk::Stock::GO_UP,
					_( "Previous" ),
					_( "Go to the previous entry in the list" ),
					Gtk::AccelKey( GDK_Page_Up, Gdk::CONTROL_MASK ),
					sigc::mem_fun( *this, &Lifeograph::handle_toolbutton_previous ),
					m_toolbutton_previous );

	create_action(	m_action_next_list, actiongroup,
					"ListNext",
					Gtk::Stock::GO_DOWN,
					_( "Next" ),
					_( "Go to the next entry in the list" ),
					Gtk::AccelKey( GDK_Page_Down, Gdk::CONTROL_MASK ),
					sigc::mem_fun( *this, &Lifeograph::handle_toolbutton_next ),
					m_toolbutton_next );

	create_action(	m_action_previous_browsing, actiongroup,
					"BrowseBack",
					Gtk::Stock::GO_BACK,
					"", "",		// no grachical interface yet!
//					_( "Go back in the browsing history" ),
					Gtk::AccelKey( GDK_Left, Gdk::MOD1_MASK ),
					sigc::mem_fun( *this, &Lifeograph::go_back ) );

	create_action(	m_action_previous_match, actiongroup,
					"MatchPrevious",
					Gtk::Stock::GO_BACK,
// TRANSLATORS: label and tooltip of the go-to-previous-match-of-filtered-text action
					_( "Previous Match" ),
					_( "Go to previous match" ),
					Gtk::AccelKey( GDK_F3, Gdk::SHIFT_MASK ),
					sigc::mem_fun( *this, &Lifeograph::handle_matchprevious ),
					m_toolbutton_findprevious );

	create_action(	m_action_next_match, actiongroup,
					"MatchNext",
					Gtk::Stock::GO_FORWARD,
// TRANSLATORS: label and tooltip of the go-to-next-match-of-filtered-text action
					_( "Next Match" ),
					_( "Go to next match" ),
					Gtk::AccelKey( GDK_F3, Gdk::ModifierType( 0 ) ),
					sigc::mem_fun( *this, &Lifeograph::handle_matchnext ),
					m_toolbutton_findnext );

	create_action(	m_action_jump2current, actiongroup,
					"Jump",
					Gtk::Stock::JUMP_TO,
					"", "",		// no grachical interface yet!
					Gtk::AccelKey( GDK_j, Gdk::CONTROL_MASK ),
					sigc::mem_fun( *this, &Lifeograph::present_currentrow ) );

	create_action(	m_action_focusfilter, actiongroup,
					"FocusFilter",
					Gtk::Stock::FIND,
					"", "",		// no grachical interface yet!
					Gtk::AccelKey( GDK_f, Gdk::CONTROL_MASK ),
					sigc::mem_fun( *this, &Lifeograph::focus_filter ) );

	create_action(	m_action_focustag, actiongroup,
					"FocusTag",
					Gtk::Stock::NO,
					"", "",		// no grachical interface yet!
					Gtk::AccelKey( GDK_t, Gdk::CONTROL_MASK ),
					sigc::mem_fun( *this, &Lifeograph::focus_tag ) );

	m_action_today = Gtk::Action::create_with_icon_name(
		"Today",
		"x-office-calendar",
		_( "Today" ), _( "Go to today (creates a new entry if there isn't any)" ) );
	actiongroup->add( m_action_today, Gtk::AccelKey( GDK_d, Gdk::CONTROL_MASK ) );
	m_action_today->set_accel_group( m_window->get_accel_group() );
	m_action_today->connect_accelerator();
	m_action_today->connect_proxy( *m_toolbutton_today );
	m_action_today->property_is_important() = true;

	m_toolbutton_logout->add_accelerator(
		"clicked",
		m_window->get_accel_group(),
		GDK_Escape,
		Gdk::CONTROL_MASK,
		Gtk::ACCEL_VISIBLE );

	// SIGNALS
	m_action_today->signal_activate().connect(
		sigc::mem_fun( *this, &Lifeograph::show_today ) );

	menuitem_chaptereditor->signal_activate().connect(
		sigc::mem_fun( *this, &Lifeograph::start_dialog_chaptereditor ) );
	menuitem_password->signal_activate().connect(
		sigc::mem_fun( *this, &Lifeograph::start_dialog_password ) );
	menuitem_backup->signal_activate().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_button_backup_clicked ) );
	menuitem_about->signal_activate().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_button_about_clicked ) );
	menuitem_logoutwosaving->signal_activate().connect(
		sigc::bind( sigc::mem_fun( *this, &Lifeograph::logout ), false ) );
	m_menuitem_logoutonidle->signal_toggled().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_autologout_toggled ) );

	m_toolbutton_logout->signal_clicked().connect(
		sigc::bind( sigc::mem_fun( *this, &Lifeograph::logout ), true ) );

	m_treeview_entries->get_selection()->signal_changed().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_treeview_entries_selectionchanged ) );
	m_treeview_entries->signal_button_press_event().connect_notify(
		sigc::mem_fun( *this, &Lifeograph::handle_treeview_entries_event_buttonpress ) );
	m_treestore_entries->signal_sort_column_changed().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_treeview_sortcolumnchanged ) );

	m_combobox_liststyle->signal_changed().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_entryliststyle_changed ) );

	m_iconview_tags->signal_enter_notify_event().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_hboxtagtools_enternotify ) );
	m_iconview_tags->signal_button_press_event().connect_notify(
		sigc::mem_fun( *this, &Lifeograph::handle_iconviewtags_buttonpress ) );
	m_iconview_tags->signal_key_release_event().connect_notify(
		sigc::mem_fun( *this, &Lifeograph::handle_iconviewtags_keyrelease ) );
	m_tagwidget->signal_activate().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_tagwidget_activated ) );

	m_calendar->signal_day_selected_double_click().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_calendar_doubleclicked ) );
	m_calendar->signal_month_changed().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_calendar_monthchanged ) );
	m_calendar->signal_day_selected().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_calendar_dayselected ) );

	m_entry_filter->signal_changed().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_entry_filter_changed ) );
	m_entry_filter->signal_activate().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_matchnext ) );
	m_entry_filter->signal_icon_release().connect(
		sigc::mem_fun( *this, &Lifeograph::handle_entry_filter_clear ) );
	m_button_replace->signal_clicked().connect(
		sigc::mem_fun( *this, &Lifeograph::replace_match ) );
	m_button_replaceall->signal_clicked().connect(
		sigc::mem_fun( *this, &Lifeograph::replace_allmatches ) );


	if( m_database.is_encrypted() && m_option_autologout )
		m_connection_event = m_window->signal_event().connect(
				sigc::mem_fun( *this, &Lifeograph::handle_event ) );

	m_connection_backup = Glib::signal_timeout().connect_seconds(
			sigc::mem_fun( *this, &Lifeograph::handle_backup ),
			BACKUP_INTERVAL );

	m_window->show_all();
	m_hbox_replace->hide();
	m_toolbutton_findprevious->hide();
	m_toolbutton_findnext->hide();
}

