/* -*- mode: c++ -*-

  This file is part of the Life library

  Author(s): Christophe Prud'homme <christophe.prudhomme@ujf-grenoble.fr>
       Date: 2005-01-20

  Copyright (C) 2005,2006 EPFL
  Copyright (C) 2006,2007 Universit Joseph Fourier (Grenoble I)

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3.0 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
/**
   \file integrators.hpp
   \author Christophe Prud'homme <christophe.prudhomme@ujf-grenoble.fr>
   \date 2005-01-20
 */
#ifndef __Integrators_H
#define __Integrators_H 1

#include <boost/timer.hpp>
#include <life/lifevf/block.hpp>

//#include <boost/numeric/bindings/traits/traits.hpp>
//#include <boost/numeric/bindings/traits/sparse_traits.hpp>
//#include <boost/numeric/bindings/traits/ublas_sparse.hpp>

#include <life/lifemesh/filters.hpp>
#include <life/lifepoly/quadmapped.hpp>
#include <life/lifevf/formcontextbase.hpp>
#if defined( HAVE_GOOGLE_PROFILER_H )
#include <google/profiler.h>
#endif

namespace Life
{
namespace vf
{
/// \cond detail
enum IntegratorType
    {
        INT_0D = -1,                 /**< 0D integrator */
        INT_1D = 1,                 /**< 1D integrator */
        INT_2D = 2,                 /**< 2D integrator */
        INT_3D = 3,                 /**< 3D integrator */
        INT_ELEMENTS = 4,           /**< integration over elements */
        INT_ELEMENT_FACES = 5,      /**< integration over face elements */
        INT_INTERNAL_FACES = 6      /**< integration over internal faces */

    };
/**
 * \class Integrator
 * \brief base class for integrators
 *
 * @author Christophe Prud'homme
 * @see IntegratorOn
 */
template<typename Elements, typename Im, typename Expr>
class Integrator
{
public:

    /** @name Constants
     */
    //@{

    static const size_type context = Expr::context;

    static const uint16_type imorder = 0;
    static const bool imIsPoly = true;

    template<typename Func>
    struct HasTestFunction
    {
        static const bool result = Expr::template HasTestFunction<Func>::result;
    };

    template<typename Func>
    struct HasTrialFunction
    {
        static const bool result = Expr::template HasTrialFunction<Func>::result;
    };

    static const size_type iDim = boost::tuples::template element<0, Elements>::type::value;
    //@}

    /** @name Typedefs
     */
    //@{

    typedef Integrator<Elements, Im, Expr> self_type;

    typedef typename boost::tuples::template element<1, Elements>::type element_iterator;

    typedef Expr expression_type;
    typedef typename expression_type::value_type value_type;
    typedef ublas::vector<int> vector_dof_type;

    struct eval
    {
        //
        // some typedefs
        //
        typedef typename boost::remove_reference<typename element_iterator::reference>::type const_t;
        typedef typename boost::remove_const<const_t>::type the_face_element_type;
        typedef typename the_face_element_type::super2::template Element<the_face_element_type>::type the_element_type;
        typedef the_element_type element_type;
        typedef typename the_element_type::gm_type gm_type;
        typedef boost::shared_ptr<gm_type> gm_ptrtype;
        typedef typename gm_type::template Context<expression_type::context, the_element_type> gmc_type;
        typedef boost::shared_ptr<gmc_type> gmc_ptrtype;
        typedef typename gm_type::precompute_ptrtype gmcpc_ptrtype;
        //typedef typename eval_expr_type::value_type value_type;
        //typedef typename strongest_numeric_type<typename Im::value_type, typename expression_type::value_type>::type value_type;
        typedef typename expression_type::value_type value_type;

        //
        // Precompute some data in the reference element for
        // geometric mapping and reference finite element
        //
        typedef fusion::map<fusion::pair<detail::gmc<0>, gmc_ptrtype> > map_gmc_type;
        typedef typename expression_type::template tensor<map_gmc_type> eval_expr_type;
        typedef typename eval_expr_type::shape shape;
        //typedef typename shape_type::storage<value_type> storage_type;
        /*
          typedef mpl::if_<mpl::bool_<shape_type::is_scalar>,
          mpl::identity<value_type>,
          mpl::identity<ublas::vector<value_type,storage_type> > >::type::type ret_type;*/
        typedef ublas::matrix<value_type> ret_type;
    };

    typedef typename mpl::if_<mpl::bool_<eval::the_element_type::is_simplex>,
                              mpl::identity<typename Im::template apply<eval::the_element_type::nDim, value_type, Simplex>::type >,
                              mpl::identity<typename Im::template apply<eval::the_element_type::nDim, value_type, SimplexProduct>::type >
                              >::type::type im_type;
    typedef typename im_type::face_quadrature_type im_face_type;

    //@}

    /** @name Constructors, destructor
     */
    //@{

    Integrator( Elements const& elts, Im const& __im, expression_type const& __expr )
        :
        _M_eltbegin( elts.template get<1>() ),
        _M_eltend( elts.template get<2>() ),
        _M_im(),
        _M_expr( __expr )
    {
        Debug( 5065 ) << "Integrator constructor from expression\n";
    }

    Integrator( Integrator const& __vfi )
        :
        _M_eltbegin( __vfi._M_eltbegin ),
        _M_eltend( __vfi._M_eltend ),
        _M_im( __vfi._M_im ),
        _M_expr( __vfi._M_expr )
    {
        Debug( 5065 ) << "Integrator copy constructor\n";
    }

    virtual ~Integrator() {}

    //@}

    /** @name Operator overloads
     */
    //@{


    //@}

    /** @name Accessors
     */
    //@{

    /**
     * get the integration method
     *
     *
     * @return the integration method
     */
    im_type const& im() const { return _M_im; }

    /**
     * get the integration method on face f
     *
     *
     * @return the integration method on face f
     */
    im_face_type  im( uint16_type f ) const { return _M_im.face( f ); }

    /**
     * get the variational expression
     *
     *
     * @return the variational expression
     */
    expression_type const& expression() const { return _M_expr; }


    /**
     * iterator that points at the beginning of the container that
     * holds the data that will apply the integration upon
     */
    element_iterator beginElement() const { return _M_eltbegin; }

    /**
     * iterator that points at the end of the container that
     * holds the data that will apply the integration upon
     */
    element_iterator endElement() const { return _M_eltend; }

    /**
     * tell whether the expression is symmetric or not
     *
     *
     * @return true if symmetric, false otherwise
     */
    bool isSymmetric() const { return _M_expr.isSymmetric(); }

    //@}

    /** @name  Mutators
     */
    //@{

    //@}

    /** @name  Methods
     */
    //@{

    template<typename Elem1, typename Elem2, typename FormType>
    void assemble( boost::shared_ptr<Elem1> const& __u,
                   boost::shared_ptr<Elem2> const& __v,
                   FormType& __form ) const;

    template<typename Elem1, typename FormType>
    void assemble( boost::shared_ptr<Elem1> const& __v,
                   FormType& __form ) const;

    //typename expression_type::template tensor<Geo_t>::value_type
    typename eval::ret_type
    evaluate() const
    {
        return evaluate( mpl::int_<iDim>() );
    }

    typename eval::ret_type
    evaluateAndSum() const
    {
        typename eval::ret_type loc =  evaluate( mpl::int_<iDim>() );
        typename eval::ret_type glo( loc );
#if defined( HAVE_MPI )
        if ( M_comm.size() > 1 )
            {
                MPI_Allreduce( loc.data().begin(),
                               glo.data().begin(),
                               loc.size1()*loc.size2(),
                               MPI_DOUBLE,
                               MPI_SUM,
                               M_comm );
            }
#endif // HAVE_MPI
        return glo;
    }

    //@}

private:
    template<typename FormType>
    void assemble( FormType& __form, mpl::int_<MESH_ELEMENTS> ) const;

    template<typename FormType>
    void assemble( FormType& __form, mpl::int_<MESH_FACES> ) const;



    typename eval::ret_type evaluate( mpl::int_<MESH_ELEMENTS> ) const;
    typename eval::ret_type evaluate( mpl::int_<MESH_FACES> ) const;

private:

    mpi::communicator M_comm;
    element_iterator _M_eltbegin;
    element_iterator _M_eltend;
    mutable im_type _M_im;
    expression_type const&  _M_expr;

    //     mutable boost::prof::basic_profiler<boost::prof::basic_profile_manager<std::string, double, boost::high_resolution_timer, boost::prof::empty_logging_policy, boost::prof::default_stats_policy<std::string, double> > > _M_profile_local_assembly;

    //     mutable boost::prof::basic_profiler<boost::prof::basic_profile_manager<std::string, double, boost::high_resolution_timer, boost::prof::empty_logging_policy, boost::prof::default_stats_policy<std::string, double> > > _M_profile_global_assembly;
};

template<typename Elements, typename Im, typename Expr>
template<typename Elem1, typename Elem2, typename FormType>
void
Integrator<Elements, Im, Expr>::assemble( boost::shared_ptr<Elem1> const& /*__u*/,
                                          boost::shared_ptr<Elem2> const& /*__v*/,
                                          FormType& __form ) const
{
#if 0
    details::GlobalMatrixAssembler<iDim, self_type, Elem1, Elem2, FormType>( *this,
                                                                             __u,
                                                                             __v,
                                                                             __form );
#endif
    assemble( __form, mpl::int_<iDim>() );

}
template<typename Elements, typename Im, typename Expr>
template<typename Elem1, typename FormType>
void
Integrator<Elements, Im, Expr>::assemble( boost::shared_ptr<Elem1> const& /*__v*/,
                                          FormType& __form ) const
{
#if 0
    details::GlobalVectorAssembler<iDim, self_type, Elem1, FormType>( *this,
                                                                      __v,
                                                                      __form );
#endif

    assemble( __form, mpl::int_<iDim>() );
}
template<typename Elements, typename Im, typename Expr>
template<typename FormType>
void
Integrator<Elements, Im, Expr>::assemble( FormType& __form, mpl::int_<MESH_ELEMENTS> ) const
{
    Debug( 5065 ) << "integrating over "
                  << std::distance( this->beginElement(), this->endElement() )  << " elements\n";
    boost::timer __timer;

    //
    // some typedefs
    //
    typedef typename FormType::gm_type gm_type;
    typedef typename gm_type::template Context<expression_type::context, typename eval::element_type> gmc_type;
    typedef boost::shared_ptr<gmc_type> gmc_ptrtype;

    //
    // Precompute some data in the reference element for
    // geometric mapping and reference finite element
    //
    typename gm_type::precompute_ptrtype __geopc( new typename gm_type::precompute_type( __form.gm(),
                                                                                         this->im().points() ) );

    __form.precomputeBasisAtPoints( this->im().points() );

    element_iterator it = this->beginElement();
    element_iterator en = this->endElement();

    // check that we have elements to iterate over
    if ( it == en )
        return;

    gmc_ptrtype __c( new gmc_type( __form.gm(), *it, __geopc ) );

    typedef fusion::map<fusion::pair<detail::gmc<0>, gmc_ptrtype> > map_gmc_type;
    typedef typename FormType::template Context<map_gmc_type, expression_type, im_type> form_context_type;
    //typedef detail::FormContextBase<map_gmc_type,im_type> fcb_type;
    typedef form_context_type fcb_type;

    typedef boost::shared_ptr<fcb_type> fcb_ptrtype;
    typedef typename form_context_type::value_type value_type;

    map_gmc_type mapgmc( fusion::make_pair<detail::gmc<0> >( __c ) );

    fcb_ptrtype formc( new form_context_type( __form,
                                              mapgmc,
                                              this->expression(),
                                              this->im() ) );

    //int nelt = std::distance( this->beginElement(), this->endElement() );
    boost::timer ti0,ti1, ti2, ti3;
    double t0 = 0, t1 = 0,t2 = 0,t3 = 0;
    //
    // start the real intensive job:
    // -# iterate over all elements to integrate over
    // -# construct the associated geometric mapping with the reference element
    // -# loop over quadrature loop and assemble the local matrix associated with the bilinear form
    // -# assemble the local contribution in the global representation of the bilinear form
    //
    for ( ; it != en; ++it )

        //for( int i = 0; i < nelt; ++i )
        {
#if 1
            ti0.restart();
            __c->update( *it );
            t0+=ti0.elapsed();
#if 0
            Debug( 5065 ) << "Element: " << it->id() << "\n"
                          << " o - points : " << it->G() << "\n"
                          << " o - quadrature :\n"
                          << "     ref : " << this->im().points() << "\n"
                          << "     real : " << __c->xReal() << "\n";
#endif
            ti1.restart();
            map_gmc_type mapgmc( fusion::make_pair<detail::gmc<0> >( __c ) );
            formc->update( mapgmc );
            //Debug( 5065 )  << "update gmc : " << ti1.elapsed() << "\n";
            t1+=ti1.elapsed();

            ti2.restart();
            formc->integrate();
            //Debug( 5065 )  << "integrate : " << ti2.elapsed() << "\n";
            t2+=ti2.elapsed();

            ti3.restart();
            formc->assemble();
            //Debug( 5065 )  << "assemble : " << ti3.elapsed() << "\n";
            t3+=ti3.elapsed();
#else
            __c->update( *it );
            map_gmc_type mapgmc( fusion::make_pair<detail::gmc<0> >( __c ) );
            formc.update( mapgmc );
            formc.integrate();
#endif
        } // end loop on elements

    Debug( 5065 ) << "[elements] Overall geometric mapping update time : " << (t0+t1+t2) << " per element:" << (t0+t1+t2)/std::distance( this->beginElement(), this->endElement() ) << "\n";
    Debug( 5065 ) << "[elements] Overall geometric mapping update time : " << t0 << "\n";
    Debug( 5065 ) << "[elements] Overall form update time : " << t1 << "\n";
    Debug( 5065 ) << "[elements] Overall local assembly time : " << t2 << "\n";
    Debug( 5065 ) << "[elements] Overall global assembly time : " << t3 << "\n";
    Debug( 5065 ) << "integrating over elements done in " << __timer.elapsed() << "s\n";
}
template<typename Elements, typename Im, typename Expr>
template<typename FormType>
void
Integrator<Elements, Im, Expr>::assemble( FormType& __form, mpl::int_<MESH_FACES> ) const
{
    Debug( 5065 ) << "integrating over "
                  << std::distance( this->beginElement(), this->endElement() )  << " faces\n";
    boost::timer __timer;

    //
    // some typedefs
    //
    typedef typename FormType::gm_type gm_type;
    typedef boost::shared_ptr<gm_type> gm_ptrtype;
    typedef typename gm_type::template Context<expression_type::context|vm::JACOBIAN|vm::KB|vm::NORMAL|vm::POINT, typename eval::element_type> gmc_type;
    typedef boost::shared_ptr<gmc_type> gmc_ptrtype;
    //
    // Precompute some data in the reference element for
    // geometric mapping and reference finite element
    //
    typedef typename gm_type::precompute_type pc_type;
    typedef typename gm_type::precompute_ptrtype pc_ptrtype;
    //typedef typename mpl::if_<mpl::equal_to<mpl::int_<FormType::nDim>, mpl::int_<2> >, mpl::identity<typename eval::element_type::edge_permutation_type>, mpl::identity<typename eval::element_type::face_permutation_type> >::type::type permutation_type;

    QuadMapped<im_type> qm;
    typedef typename QuadMapped<im_type>::permutation_type permutation_type;
    typename QuadMapped<im_type>::permutation_points_type ppts( qm( im() ) );


    std::vector<std::map<permutation_type, pc_ptrtype> > __geopc( im().nFaces() );
    typedef typename im_type::face_quadrature_type face_im_type;

    //__typeof__(im(__face_id_in_elt_0 ) ) im_face ( im(__face_id_in_elt_0 ) );
    std::vector<face_im_type> face_ims( im().nFaces() );

    for ( uint16_type __f = 0; __f < im().nFaces(); ++__f )
        {
            face_ims[__f] = this->im( __f );

            for( permutation_type __p( permutation_type::IDENTITY );
                 __p < permutation_type( permutation_type::N_PERMUTATIONS ); ++__p )
            {
                //LIFE_ASSERT( ppts[__f].find(__p)->second.size2() != 0 ).warn( "invalid quadrature type" );
                __geopc[__f][__p] = pc_ptrtype(  new pc_type( __form.gm(), ppts[__f].find(__p)->second ) );
                __form.precomputeBasisAtPoints( __f, __p, ppts[__f].find(__p)->second );
            }
        }

    element_iterator it = beginElement();
    element_iterator en = endElement();

    uint16_type __face_id_in_elt_0 = it->pos_first();

    // check that we have elements to iterate over
    if ( it == en )
        return;

    // get the geometric mapping associated with element 0
    //Debug( 5065 ) << "element " << it->element(0)  << "face " << __face_id_in_elt_0 << " permutation " << it->element(0).permutation( __face_id_in_elt_0 ) << "\n";
    gm_ptrtype __gm = it->element(0).gm();
    Debug( 5065 ) << "[integrator] evaluate(faces), gm is cached: " << __gm->isCached() << "\n";
    gmc_ptrtype __c0( new gmc_type( __gm, it->element( 0 ), __geopc, __face_id_in_elt_0 ) );



    //
    // the case where the face is connected only to one element
    //
    typedef fusion::map<fusion::pair<detail::gmc<0>, gmc_ptrtype> > map_gmc_type;
    typedef typename FormType::template Context<map_gmc_type, expression_type, face_im_type> form_context_type;

    map_gmc_type mapgmc( fusion::make_pair<detail::gmc<0> >( __c0 ) );
    form_context_type form( __form, mapgmc, expression(), face_ims[__face_id_in_elt_0], this->im() );

    //
    // the case where the face is connected only to two elements
    //
    // get the geometric mapping associated with element 1
    gmc_ptrtype __c1;

    typedef fusion::map<fusion::pair<detail::gmc<0>, gmc_ptrtype>, fusion::pair<detail::gmc<1>, gmc_ptrtype> > map2_gmc_type;
    typedef typename FormType::template Context<map2_gmc_type, expression_type, face_im_type> form2_context_type;
    typedef boost::shared_ptr<form2_context_type> form2_context_ptrtype;
    form2_context_ptrtype form2;
    // true if connected to another element, false otherwise
    if ( it->isConnectedTo1() )
        {
            uint16_type __face_id_in_elt_1 = it->pos_second();

            __c1 = gmc_ptrtype( new gmc_type( __gm, it->element( 1 ), __geopc, __face_id_in_elt_1 ) );
            map2_gmc_type mapgmc2( fusion::make_pair<detail::gmc<0> >( __c0 ),
                                   fusion::make_pair<detail::gmc<1> >( __c1 ) );

            form2 = form2_context_ptrtype( new form2_context_type( __form, mapgmc2, expression(), face_ims[__face_id_in_elt_0], this->im(), mpl::int_<2>() ) );
        }

    boost::timer ti0,ti1, ti2, ti3;
    double t0 = 0, t1 = 0,t2 = 0,t3 = 0;
    Debug( 5065 ) << "[Integrator::faces/forms] starting...\n";
    //
    // start the real intensive job:
    // -# iterate over all elements to integrate over
    // -# construct the associated geometric mapping with the reference element
    // -# loop over quadrature loop and assemble the local matrix associated with the bilinear form
    // -# assemble the local contribution in the global representation of the bilinear form
    //
    for ( ; it != en; ++it )
        {
            if ( it->isConnectedTo1())
                {
                    LIFE_ASSERT( it->isOnBoundary() == false  )
                        ( it->id() ).error( "face on boundary but connected on both sides");
                    ti0.restart();
                    // get the id of the face in each adjacent element
                    uint16_type __face_id_in_elt_0 = it->pos_first();
                    uint16_type __face_id_in_elt_1 = it->pos_second();

                    __c0->update( it->element(0), __face_id_in_elt_0 );
                    __c1->update( it->element(1), __face_id_in_elt_1 );
                    t0 += ti0.elapsed();

                    ti1.restart();
                    map2_gmc_type mapgmc2 = map2_gmc_type( fusion::make_pair<detail::gmc<0> >( __c0 ),
                                                           fusion::make_pair<detail::gmc<1> >( __c1 ) );
                    form2->update( mapgmc2, face_ims[__face_id_in_elt_0], mpl::int_<2>() );
                    t1 += ti1.elapsed();


                    ti2.restart();
                    form2->integrate( );
                    t2 += ti2.elapsed();

                    ti3.restart();
                    form2->assemble();
                    t3 += ti3.elapsed();
                }
            else
                {
#if 1
                    ti0.restart();
                    uint16_type __face_id_in_elt_0 = it->pos_first();
                    __c0->update( it->element(0),__face_id_in_elt_0 );
                    t0 += ti0.elapsed();

                    LIFE_ASSERT( __face_id_in_elt_0 == __c0->faceId() )
                        ( __face_id_in_elt_0 )
                        ( __c0->faceId() ).warn ( "invalid face id" );

                    ti1.restart();
                    map_gmc_type mapgmc( fusion::make_pair<detail::gmc<0> >( __c0 ) );
                    form.update( mapgmc, face_ims[__face_id_in_elt_0] );
                    t1 += ti1.elapsed();

                    ti2.restart();
                    form.integrate( );
                    t2 += ti2.elapsed();

                    ti3.restart();
                    form.assemble( it->element(0).id() );
                    t3 += ti3.elapsed();
#else
                    map_gmc_type mapgmc( fusion::make_pair<detail::gmc<0> >( __c0 ) );
                    form.update( mapgmc, face_ims[__face_id_in_elt_0] );
                    form.integrate( );
#endif
                } // end loop on elements

        }

    Debug( 5065 ) << "[faces] Overall integration time : " << (t0+t1+t2+t3) << " per element:" << (t0+t1+t2+t3)/std::distance( this->beginElement(), this->endElement() ) << "for " << std::distance( this->beginElement(), this->endElement() ) << "elements\n";
    Debug( 5065 ) << "[faces] Overall geometric mapping update time : " << t0 << "\n";
    Debug( 5065 ) << "[faces] Overall form update time : " << t1 << "\n";
    Debug( 5065 ) << "[faces] Overall local assembly time : " << t2 << "\n";
    Debug( 5065 ) << "[faces] Overall global assembly time : " << t3 << "\n";

    Debug( 5065 ) << "integrating over faces done in " << __timer.elapsed() << "s\n";
}


template<typename Elements, typename Im, typename Expr>
typename Integrator<Elements, Im, Expr>::eval::ret_type
Integrator<Elements, Im, Expr>::evaluate( mpl::int_<MESH_ELEMENTS> ) const
{
    Debug( 5065 ) << "integrating over "
                  << std::distance( this->beginElement(), this->endElement() )  << " elements\n";
    boost::timer __timer;

    //
    // some typedefs
    //
    typedef typename boost::remove_reference<typename element_iterator::reference>::type const_t;
    typedef typename boost::remove_const<const_t>::type the_element_type;
    typedef the_element_type element_type;
    typedef typename the_element_type::gm_type gm_type;
    typedef boost::shared_ptr<gm_type> gm_ptrtype;
    typedef typename gm_type::template Context<expression_type::context, the_element_type> gmc_type;
    typedef boost::shared_ptr<gmc_type> gmc_ptrtype;

    //typedef typename eval_expr_type::value_type value_type;
    //typedef typename Im::value_type value_type;

    element_iterator it = this->beginElement();
    element_iterator en = this->endElement();

    //
    // Precompute some data in the reference element for
    // geometric mapping and reference finite element
    //
    gm_ptrtype gm = it->gm();
    Debug(5065) << "[integrator] evaluate(elements), gm is cached: " << gm->isCached() << "\n";
    typename gm_type::precompute_ptrtype __geopc( new typename gm_type::precompute_type( gm,
                                                                                         this->im().points() ) );


    it = this->beginElement();
    // wait for all the guys
#ifdef HAVE_MPI
        if ( M_comm.size() > 1 )
            {
                M_comm.barrier();
            }
#endif

    // make sure that we have elements to iterate over (return 0
    // otherwise)
    if ( it == en )
        return typename eval::ret_type( eval::shape::M, eval::shape::N );;

    gmc_ptrtype __c( new gmc_type( gm, *it, __geopc ) );
    typedef fusion::map<fusion::pair<detail::gmc<0>, gmc_ptrtype> > map_gmc_type;
    map_gmc_type mapgmc( fusion::make_pair<detail::gmc<0> >( __c ) );

    typedef typename expression_type::template tensor<map_gmc_type> eval_expr_type;
    eval_expr_type expr( expression(), mapgmc );
    typedef typename eval_expr_type::shape shape;

    typename eval::ret_type res(eval::shape::M, eval::shape::N );
    res.clear();


    //value_type res1 = 0;
    for ( ; it != en; ++it )
        {
            boost::timer ti;
            __c->update( *it );
            map_gmc_type mapgmc( fusion::make_pair<detail::gmc<0> >( __c ) );
            expr.update( mapgmc );
            const gmc_type& gmc = *__c;

            _M_im.update( gmc );


            for( uint16_type c1 = 0; c1 < eval::shape::M; ++c1 )
                for( uint16_type c2 = 0; c2 < eval::shape::N; ++c2 )
                    {
                        res(c1,c2) += _M_im( expr, c1, c2 );
                    }
        }
    //std::cout << "res=" << res << "\n";
    //std::cout << "res1=" << res1 << "\n";
    Debug( 5065 ) << "integrating over elements done in " << __timer.elapsed() << "s\n";

    return res;
}
template<typename Elements, typename Im, typename Expr>
typename Integrator<Elements, Im, Expr>::eval::ret_type
Integrator<Elements, Im, Expr>::evaluate( mpl::int_<MESH_FACES> ) const
{
    Debug( 5065 ) << "integrating over "
                  << std::distance( this->beginElement(), this->endElement() )  << "faces\n";
    boost::timer __timer;

    //
    // some typedefs
    //
    typedef typename boost::remove_reference<typename element_iterator::reference>::type const_t;
    typedef typename boost::remove_const<const_t>::type the_face_element_type;
    typedef typename the_face_element_type::super2::entity_type the_element_type;
    typedef typename the_element_type::gm_type gm_type;
    typedef boost::shared_ptr<gm_type> gm_ptrtype;
    typedef typename gm_type::template Context<expression_type::context|vm::JACOBIAN|vm::KB|vm::NORMAL|vm::POINT, the_element_type> gmc_type;
    typedef boost::shared_ptr<gmc_type> gmc_ptrtype;
    //typedef typename eval_expr_type::value_type value_type;
    //typedef typename Im::value_type value_type;

    //BOOST_MPL_ASSERT_MSG( the_element_type::nDim > 1, INVALID_DIM, (mpl::int_<the_element_type::nDim>, mpl::int_<the_face_element_type::nDim>, mpl::identity<the_face_element_type>, mpl::identity<the_element_type> ) );;

    QuadMapped<im_type> qm;
    typename QuadMapped<im_type>::permutation_points_type ppts( qm( im() ) );
    typedef typename QuadMapped<im_type>::permutation_type permutation_type;

    //
    // Precompute some data in the reference element for
    // geometric mapping and reference finite element
    //
    typedef typename gm_type::precompute_type pc_type;
    typedef typename gm_type::precompute_ptrtype pc_ptrtype;
    std::vector<std::map<permutation_type, pc_ptrtype> > __geopc( im().nFaces() );
    typedef typename im_type::face_quadrature_type face_im_type;

    std::vector<im_face_type> __integrators;

    element_iterator it = beginElement();
    element_iterator en = endElement();

    gm_ptrtype gm = it->element(0).gm();
    Debug(5065) << "[integrator] evaluate(faces), gm is cached: " << gm->isCached() << "\n";
    for ( uint16_type __f = 0; __f < im().nFaces(); ++__f )
        {
            __integrators.push_back( im(__f) );
            for( permutation_type __p( permutation_type::IDENTITY );
                 __p < permutation_type( permutation_type::N_PERMUTATIONS ); ++__p )
            {
                //LIFE_ASSERT( ppts[__f][__p]->size2() != 0 ).warn( "invalid quadrature type" );
                __geopc[__f][__p] = pc_ptrtype(  new pc_type( gm, ppts[__f].find(__p)->second ) );
            }
        }


    // make sure that we have elements to iterate over (return 0
    // otherwise)
    if ( it == en )
      return typename eval::ret_type(eval::shape::M, eval::shape::N );

    uint16_type __face_id_in_elt_0 = it->pos_first();

    // get the geometric mapping associated with element 0
    gmc_ptrtype __c0( new gmc_type( gm, it->element( 0 ), __geopc, __face_id_in_elt_0 ) );

    typedef fusion::map<fusion::pair<detail::gmc<0>, gmc_ptrtype> > map_gmc_type;

    typedef typename expression_type::template tensor<map_gmc_type> eval_expr_type;
    typedef boost::shared_ptr<eval_expr_type> eval_expr_ptrtype;
    map_gmc_type mapgmc( fusion::make_pair<detail::gmc<0> >( __c0 ) );
    eval_expr_ptrtype expr( new eval_expr_type( expression(), mapgmc ) );
    expr->init( im() );

    typedef fusion::map<fusion::pair<detail::gmc<0>, gmc_ptrtype>, fusion::pair<detail::gmc<1>, gmc_ptrtype> > map2_gmc_type;
    typedef typename expression_type::template tensor<map2_gmc_type> eval2_expr_type;
    typedef boost::shared_ptr<eval2_expr_type> eval2_expr_ptrtype;
    eval2_expr_ptrtype expr2;

    // true if connected to another element, false otherwise
    bool isConnectedTo1 = it->isConnectedTo1();

    // get the geometric mapping associated with element 1
    gmc_ptrtype __c1;

   //value_type res = 0;
    //value_type res1 = 0;
    if ( isConnectedTo1 )
        {
            uint16_type __face_id_in_elt_1 = it->pos_second();

            __c1 = gmc_ptrtype( new gmc_type( gm, it->element( 1 ), __geopc, __face_id_in_elt_1 ) );

            map2_gmc_type mapgmc( fusion::make_pair<detail::gmc<0> >( __c0 ),
                                  fusion::make_pair<detail::gmc<1> >( __c1 ) );

            expr2 = eval2_expr_ptrtype( new eval2_expr_type( expression(), mapgmc ) );
            expr2->init( im() );
        }
    // make sure that we have elements to iterate over (return 0
    // otherwise)
    if ( it == en )
        return typename eval::ret_type(eval::shape::M, eval::shape::N );

    typename eval::ret_type res(eval::shape::M, eval::shape::N );
    res.clear();
    typename eval::ret_type res0(eval::shape::M, eval::shape::N );
    res0.clear();
    typename eval::ret_type res1(eval::shape::M, eval::shape::N );
    res1.clear();

    //
    // start the real intensive job:
    // -# iterate over all elements to integrate over
    // -# construct the associated geometric mapping with the reference element
    // -# loop over quadrature loop and assemble the local matrix associated with the bilinear form
    // -# assemble the local contribution in the global representation of the bilinear form
    //
    for ( ; it != en; ++it )
        {

            if ( it->isConnectedTo1() )
                {
                    LIFE_ASSERT( it->isOnBoundary() == false   )
                        ( it->id() ).error( "face on boundary but connected on both sides");
                    uint16_type __face_id_in_elt_0 = it->pos_first();
                    uint16_type __face_id_in_elt_1 = it->pos_second();

                    __c0->update( it->element(0), __face_id_in_elt_0 );
                    __c1->update( it->element(1), __face_id_in_elt_1 );

#if 0
                    std::cout << "face " << it->id() << "\n"
                              << " id in elt = " << __face_id_in_elt_1 << "\n"
                              << "  elt 0 : " << it->element(0).id() << "\n"
                              << "  elt 0 G: " << it->element(0).G() << "\n"
                              << "  node elt 0 0 :" << it->element(0).point( it->element(0).fToP( __face_id_in_elt_0, 0 ) ).node() << "\n"
                              << "  node elt 0 1 :" << it->element(0).point( it->element(0).fToP( __face_id_in_elt_0, 1 ) ).node() << "\n"
                              << "  ref nodes 0 :" << __c0->xRefs() << "\n"
                              << "  real nodes 0: " << __c0->xReal() << "\n";
                    std::cout << "face " << it->id() << "\n"
                              << " id in elt = " << __face_id_in_elt_1 << "\n"
                              << " elt 1 : " << it->element(1).id() << "\n"
                              << "  elt 1 G: " << it->element(1).G() << "\n"
                              << "  node elt 1 0 :" << it->element(1).point( it->element(1).fToP( __face_id_in_elt_1, 1 ) ).node() << "\n"
                              << "  node elt 1 1 :" << it->element(1).point( it->element(1).fToP( __face_id_in_elt_1, 0 ) ).node() << "\n"
                              << " ref nodes 1 :" << __c1->xRefs() << "\n"
                              << " real nodes 1:" << __c1->xReal() << "\n";
#endif

                    __typeof__(im(__face_id_in_elt_0 ) ) im_face ( im(__face_id_in_elt_0 ) );
                    //std::cout << "pts = " << im_face.points() << "\n";
                    map2_gmc_type mapgmc( fusion::make_pair<detail::gmc<0> >( __c0 ),
                                          fusion::make_pair<detail::gmc<1> >( __c1 ) );

                    expr2->update( mapgmc );
                    const gmc_type& gmc = *__c0;

                    __integrators[__face_id_in_elt_0].update( gmc );
                    for( uint16_type c1 = 0; c1 < eval::shape::M; ++c1 )
                        for( uint16_type c2 = 0; c2 < eval::shape::N; ++c2 )
                            {
                                res(c1,c2) += __integrators[__face_id_in_elt_0]( *expr2, c1, c2 );
                            }
                }
            else
                {
                    uint16_type __face_id_in_elt_0 = it->pos_first();
                    __c0->update( it->element(0), __face_id_in_elt_0 );
                    map_gmc_type mapgmc( fusion::make_pair<detail::gmc<0> >( __c0 ) );
                    expr->update( mapgmc, __face_id_in_elt_0 );
                    //expr->update( mapgmc );
                    const gmc_type& gmc = *__c0;

                    __integrators[__face_id_in_elt_0].update( gmc );

                    for( uint16_type c1 = 0; c1 < eval::shape::M; ++c1 )
                        for( uint16_type c2 = 0; c2 < eval::shape::N; ++c2 )
                            {
                                res(c1,c2) += __integrators[__face_id_in_elt_0]( *expr, c1, c2 );
                            }
                } // !isConnectedTo1
        } // for loop on face

    //std::cout << "res=" << res << "\n";
    //std::cout << "res1=" << res1 << "\n";
    Debug( 5065 ) << "integrating over faces done in " << __timer.elapsed() << "s\n";
    return res;
}
/// \endcond

/**
 * integrate an expression \c expr over a set of convexes \c elts
 * using the integration rule \c im .
 */
template<typename IntElts, typename Im, typename ExprT>
Expr<Integrator<IntElts, Im, ExprT> >
integrate( IntElts const& elts,
           Im const& im,
           ExprT const& expr )
{
    typedef Integrator<IntElts, Im, ExprT> expr_t;
    return Expr<expr_t>( expr_t( elts, im, expr ) );
}


/**
 * \class ExpressionOrder
 *
 * Class that compute the expression polynomial order of \p ExprT
 *
 * \tparam ExprT expression whose approximate order must be computed
 *
 * Note that if the expression is polynomial then the order is exact, however if
 * analytic functions such as exp, cos, sin ... then the order is only an
 * approximation.
 */
template<typename ExprT>
struct ExpressionOrder
{
    static const bool is_polynomial = ExprT::imIsPoly;
    static const int  value = boost::mpl::if_< boost::mpl::bool_< ExprT::imIsPoly > ,
                                               typename boost::mpl::if_< boost::mpl::greater< boost::mpl::int_<ExprT::imorder>,
                                                                                              boost::mpl::int_<19> > ,
                                                                         boost::mpl::int_<19>,
                                                                         boost::mpl::int_<ExprT::imorder> >::type,
                                               boost::mpl::int_<10> >::type::value;

};

//Macro which get the good integration order
# define VF_VALUE_OF_IM(O)                                              \
    boost::mpl::if_< boost::mpl::bool_< O::imIsPoly > ,                 \
                     typename boost::mpl::if_< boost::mpl::greater< boost::mpl::int_<O::imorder>, boost::mpl::int_<19> > , \
                                               boost::mpl::int_<19>,    \
                                               boost::mpl::int_<O::imorder> >::type >::type , \
                                                                               boost::mpl::int_<10> >::type::value \
/**/

/**
 * integrate an expression \c expr over a set of convexes \c elts
 * using an automatic integration rule .
 */
template<typename IntElts, typename ExprT>
Expr<Integrator<IntElts, _Q< ExpressionOrder<ExprT>::value >, ExprT> >
integrate( IntElts const& elts,
           ExprT const& expr )
{
    typedef Integrator<IntElts, _Q< ExpressionOrder<ExprT>::value >, ExprT> expr_t;
    Debug(5065) << "[integrate] order to integrate = " << ExpressionOrder<ExprT>::value << "\n";
    return Expr<expr_t>( expr_t( elts, _Q< ExpressionOrder<ExprT>::value >(), expr ) );
}


} // vf
} // life
#include <life/lifevf/integratoron.hpp>


#endif /* __Integrator_H */
