//
// The contents of this file are subject to the Mozilla Public License
// Version 1.0 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License
// at http://www.mozilla.org/MPL/
// 
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
// the License for the specific language governing rights and
// limitations under the License.
//
// This software was developed as part of the legOS project.
//
// Contributor: Pat Welch (legOS@mousebrains.com)

#ifndef _Sensor_H_
#define _Sensor_H_

// This is the base class for all sensors
// The methods include:
// Sensor(sensor address)
// unsigned int get() returns the current sensor reading
// void mode(bool active) if active is true, then the sensor is made active, 
//                        else it is passive
// void passive() makes sensor passive
// void active() makes sensor active
// void off() makes sensor passive
// void on() makes sensor active
// void strobe() turn sensor on then off
// void strobe(const int delay) turn sensor on, wait delay(ms), then turn off
// unsigned int sample(unsigned int size, int wait), samples the sensor size
//                     times, with wait ms delays between samples

#include <dsensor.h>
#include <conio.h>

class Sensor {
public:
  enum Port {
    S1 = 0,	// RCX sensor port 1
    S2,		// RCX sensor port 2
    S3,		// RCX sensor port 3
    Battery	// RCX battery sensor
  };

  Sensor(const Port port, bool makeActive = false)
    : sensor((port == S1) ? SENSOR_1 : 
             (port == S2) ? SENSOR_2 :
             (port == S3) ? SENSOR_3 :
	     BATTERY) 
  {
    if (makeActive) active();    // Don't use mode here, since this saves memory
    else            passive();
  }
  ~Sensor() {off();}

  unsigned int get() const {return sensor;}
  void mode(bool makeActive) const {
    if (makeActive) active();
    else            passive();
  }
  void passive() const {ds_passive(&sensor);}
  void active() const {ds_active(&sensor);}
  void on () const {active();}
  void off() const {passive();}
  void strobe() const {off(); on();}
  void strobe(const int ms) const {off(); delay(ms); on();}

  unsigned int sample(unsigned int size = 10, int wait = 2) const {
    if (size == 0) size = 1;
    unsigned int sum(get());
    for (unsigned int i = 1; i < size; ++i) {
      sum += get();
      delay(wait);
    }
    return sum / size;
  }

protected:
  volatile unsigned int& sensor;
};

#endif // _Sensor_H_
