/*
 * $Id: kl_task.c,v 1.2 2005/02/23 01:09:12 tjm Exp $
 *
 * This file is part of libklib.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, NEC, and others
 *
 * Copyright (C) 1999 - 2005 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <klib.h>

#define DUMP_ASM_MAGIC_NUMBER   0xdeaddeadULL  /* magic number */

kaddr_t deftask = (kaddr_t)NULL;

/* 
 * kl_set_deftask()
 */
k_error_t
kl_set_deftask(kaddr_t task)
{
	void *tsp = 0;

	kl_reset_error();
	if (task) {
		tsp = kl_alloc_block(TASK_STRUCT_SZ, K_TEMP);
		if (tsp) {
			kl_get_task_struct(task, 2, tsp);
			if (!KL_ERROR) {
				deftask = task;
			}
			kl_free_block(tsp);
		}
	} else {
		deftask = 0;
	}
	return(KL_ERROR);
}

/* 
 * kl_parent_pid()
 */
int
kl_parent_pid(void *tsp)
{
	int ppid;
	kaddr_t parent;
	void *tpp;

	if (kl_is_member("task_struct", "parent")) {
		parent = kl_kaddr(tsp, "task_struct", "parent");
	} else {
		parent = kl_kaddr(tsp, "task_struct", "p_pptr");
	}
	if (parent) {
		tpp = kl_alloc_block(TASK_STRUCT_SZ, K_TEMP);
		if (!tpp) {
			return(-1);
		}
		GET_BLOCK(parent, TASK_STRUCT_SZ, tpp);
		if (KL_ERROR) {
			return(-1);
		}
		ppid = (uint64_t)KL_UINT(tpp, "task_struct", "pid");
		kl_free_block(tpp);
		return(ppid);
	}
	return(-1);
}

/*
 * kl_pid_to_task()
 */
kaddr_t
kl_pid_to_task(kaddr_t pid)
{
	kaddr_t addr, first_task;
	void *tsp;
        int offset;

	if (!(first_task = kl_first_task())) {
		if (!(tsp = kl_alloc_block(TASK_STRUCT_SZ, K_TEMP))) {
			return((kaddr_t)NULL);
		}
		if (LINUX_2_6_X(KL_LINUX_RELEASE)) {
			offset = kl_member_offset("task_struct","tasks");
		}
		addr = first_task;
		do {
			/* take care, that mode == 2 otherwise recursion */
			if (kl_get_task_struct(addr, 2, tsp)) {
				break;
			}
			if (KL_UINT(tsp, "task_struct", "pid") == pid) {
				kl_free_block(tsp);
				return(addr);
			}
			if(KL_LINUX_RELEASE < LINUX_2_6_0){
				addr = kl_kaddr(tsp, "task_struct", "next_task");
			} else {
				addr = kl_kaddr(((char*)tsp) + offset,
					"list_head", "next") - offset;
			}
		} while (addr != first_task);
		kl_free_block(tsp);
	}
	KL_ERROR = KLE_PID_NOT_FOUND;
	return((kaddr_t)NULL);
}

/* 
 * kl_get_task_struct()
 */
k_error_t
kl_get_task_struct(kaddr_t value, int mode, void *tsp)
{
	kaddr_t addr;

	kl_reset_error();
	if (!tsp) {
		KL_ERROR = KLE_NULL_BUFF;
	} else {
		if (mode == 1) {
			addr = kl_pid_to_task(value);
		} else if (mode == 2) {
			addr = value;
		} else {
			KL_ERROR = KLE_BAD_TASK_STRUCT;
		}
	}
	if (!KL_ERROR) {
		GET_BLOCK(addr, TASK_STRUCT_SZ, tsp);
	}
	return(KL_ERROR);
}

/*
 * kl_first_task()
 */
kaddr_t
kl_first_task(void)
{
	kaddr_t addr = 0;
	syment_t *symp;

	if ((symp = kl_lkup_symname("init_task_union"))) {
		addr = symp->s_addr;
	} else if ((symp = kl_lkup_symname("init_task"))) {
		addr = symp->s_addr;
	} else if ((symp = kl_lkup_symname("init_tasks"))) {
		symp = kl_lkup_symname("init_tasks");
		addr = KL_VREAD_PTR(symp->s_addr);
	} 
	return(addr);
}

/* 
 * kl_next_task()
 */
kaddr_t
kl_next_task(void *tsp)
{       
	kaddr_t next;
	
	if (kl_is_member("task_struct", "next_task")) {
		next = kl_kaddr(tsp, "task_struct", "next_task");
	} else {
		next = kl_kaddr(K_PTR(tsp, "task_struct", "tasks"),
			"list_head", "next");
		next -= kl_member_offset("task_struct", "tasks");
	}
	return (next);
}

/*
 * kl_prev_task()
 */
kaddr_t
kl_prev_task(void *tsp)
{
	kaddr_t prev;

	if (kl_is_member("task_struct", "prev_task")) {
		prev = kl_kaddr(tsp, "task_struct", "prev_task");
	} else {
		prev = kl_kaddr(K_PTR(tsp, "task_struct", "tasks"),
			"list_head", "prev");
		prev -= kl_member_offset("task_struct", "tasks");
	}
	return (prev);
}

/*                              
 * kl_task_size()               
 */                     
int                     
kl_task_size(kaddr_t task)
{               
	int task_size = 0;
	void *tsp = kl_alloc_block(TASK_STRUCT_SZ, K_TEMP);
	
	kl_get_task_struct(task, 2, tsp);
	if (!KL_ERROR) {
		task_size = KL_UINT(K_PTR(tsp, "task_struct", "thread"),
			"thread_struct", "task_size");
	}
	kl_free_block(tsp);
	return(task_size);
}
