/*
 * $Id: kl_page_s390.c,v 1.1 2004/12/21 23:26:20 tjm Exp $
 *
 * This file is part of libklib.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, NEC, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <klib.h>

/*
 * declarations of static functions
 */
static kaddr_t _kl_pgd_offset_s390(
	kaddr_t         /* kernel virtual address of page directory */,
	kaddr_t         /* kernel virtual address */);

static kaddr_t _kl_pte_offset_s390(
	kaddr_t         /* kernel virtual address of page table */,
	kaddr_t         /* kernel virtual address */);

/*
 * function definitions
 */

inline int kl_pte_present_s390(kaddr_t x){
	if(KL_LINUX_RELEASE >= LINUX_2_6_0){
		return !((x) & KL_PAGE_INVALID_S390) ||
			((x) & KL_PAGE_INVALID_MASK_S390) == KL_PAGE_INVALID_NONE_S390;
	} else {
		return ((x) & KL_PAGE_PRESENT_S390);
	}
}

/* page table traversal functions */
kaddr_t _kl_pgd_offset_s390(kaddr_t pgd_base, kaddr_t vaddr)
{
	kaddr_t pgd_off, pmd_base;

	pgd_off = ((vaddr >> KL_PGDIR_SHIFT_S390) & (KL_PTRS_PER_PGD_S390 - 1))
		* KL_NBPW;
	pmd_base = KL_READ_PTR(pgd_base + pgd_off - KL_PAGE_OFFSET_S390);

	return pmd_base;
}

kaddr_t _kl_pte_offset_s390(kaddr_t pte_base, kaddr_t vaddr)
{
	kaddr_t pte_off, pte_val;

	pte_off = ((vaddr >> KL_PAGE_SHIFT_S390) & (KL_PTRS_PER_PTE_S390 - 1))
		* KL_NBPW;
	pte_val = KL_READ_PTR(pte_base + pte_off);

	return pte_val;
}

/* lookup virtual address in page tables */
kaddr_t kl_mmap_virtop_s390(kaddr_t vaddr, void *mmp)
{
	kaddr_t pgd_base;
	kaddr_t pte_base, pte_val, paddr;

	/* get the pgd entry */
	pgd_base = kl_kaddr(mmp, "mm_struct", "pgd");
	pte_base = _kl_pgd_offset_s390(pgd_base,vaddr);
	if(KL_PMD_INVALID_S390(pte_base) ||
	   kl_pmd_bad_s390(pte_base) ||
	   kl_pmd_none_s390(pte_base) ||
	   KL_ERROR){
		KL_ERROR = KLE_INVALID_MAPPING;
		return 0;
	}

	/* get the pte */
	pte_base = pte_base & KL_PT_BASE_MASK_S390;
	pte_val = _kl_pte_offset_s390(pte_base,vaddr);
	if(KL_PTE_INVALID_S390(pte_val) ||
	   kl_pte_none_s390(pte_val) ||
	   KL_ERROR){
		KL_ERROR = KLE_INVALID_MAPPING;
		return 0;
	}
	if(!kl_pte_present_s390(pte_val)){
		/* printf("swapped out: %x\n",pte_val); */
		KL_ERROR = KLE_PAGE_NOT_PRESENT;
		return (0);
	}
	paddr = (pte_val & KL_PAGE_BASE_MASK_S390) | 
		(vaddr & (~(KL_PAGE_MASK_S390)));
	return(paddr);
}
