/*
 * $Id: module.c,v 1.1 2004/12/21 23:26:17 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002, 2004 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>

/* #define _DBG_COMPONENT KL_DBGCOMP_MODULE */

/*
 * module_banner()
 */
void
module_banner(FILE *ofp, int flags)
{
	if(flags & C_FULL){
		if(!(flags & C_PLAIN)){
			print_banner(ofp, "EXPORTED MODULE SYMBOLS: %}75",
				     flags);
		}
	} else {
		print_banner(ofp, "%>      ADDR%      SIZE USED NAME"
			     "                          REFS %}75", flags);
	}
}

char noname[] = "NONAME";

/* 
 * print_kallsyms()
 */
void
print_kallsyms(void *module, FILE *ofp)
{
	int i, nsyms, symsize;
	uint64_t nameoff;
	unsigned char stype;
	char symname[KL_SYMBOL_NAME_LEN + 1];
	kaddr_t symtab_addr, strtab_addr, name_addr, symaddr;
	void *symp;
	void *symtab;

	symsize = kl_struct_len("elf64_sym");
	symtab_addr = kl_kaddr(module, "module", "symtab");
	if (!symtab_addr) {
		return;
	}
	strtab_addr = kl_kaddr(module, "module", "strtab");
	nsyms = KL_UINT(module, "module", "num_symtab");
	symtab = kl_alloc_block((nsyms * symsize), K_TEMP);
	GET_BLOCK(symtab_addr, (nsyms * symsize), symtab);
	if (KL_ERROR) {
		/* XXX Error message?
		 */
		return;
	}
	for (i = 0; i < nsyms; i++) {
		symp = (void*)((unsigned long)symtab + (i * symsize));
		symaddr = kl_kaddr(symp, "elf64_sym", "st_value");
		stype = *(char*)CHAR(symp, "elf64_sym", "st_info");
		nameoff = KL_UINT(symp, "elf64_sym", "st_name");
		if (nameoff) {
			name_addr = strtab_addr + nameoff;
			GET_BLOCK(name_addr, KL_SYMBOL_NAME_LEN, symname);
		} else {
			strcpy(symname, noname);
		}	
		print_value(ofp, NULL, symaddr, 10, ADDR_FLG);
		fprintf(ofp, " %c ", stype);
		fprintf(ofp," %-36s\n", (char*)symname);
	}
}

/*
 * print_module()
 */
void
print_module(kaddr_t addr, void *ptr, int flags, FILE *ofp)
{
	int32_t dummy;
	void *dump_page = NULL;
	const char *modname = NULL;

	dump_page = kl_alloc_block(KL_PAGE_SIZE, K_TEMP);
	if (!dump_page) 
		return; /* XXX Error */

	if(KL_LINUX_RELEASE >= LINUX_2_6_0){
		memcpy(dump_page, ptr + kl_member_offset("module","name"),kl_member_size("module","name"));
	} else {
		GET_BLOCK(kl_kaddr(ptr, "module", "name"),
		  KL_PAGE_SIZE, dump_page);
		if((KL_ERROR == KLE_INVALID_MAPPING) ||
	   	(KL_ERROR == KLE_PAGE_NOT_PRESENT)){
/* 			kl_trace2(0, "Ignoring PAGE_NOT_PRESENT " */
/* 				  "and INVALID_MAPPING errors.\n"); */
			kl_reset_error();
		}
	}

	if(*(char*) dump_page == '\000') {
		modname = KL_KERNEL_MODULE;
	} else {
		modname = (char *) dump_page;
	}

	if(flags & C_FULL){
		if(!(flags & C_PLAIN)){
			fprintf(ofp, "        Module: %s\n", modname);
		}
		print_modsyms(ptr, modname, flags, ofp);
	} else {
		if(KL_LINUX_RELEASE >= LINUX_2_6_0){
			int i,offset,ref_count = 0;

			if(!kl_is_member("module","ref")){
        		        /* no unload support -> no ref count */
				ref_count = 0;
			} else {
				/* calculate reference count */
				offset = kl_member_offset("module","ref");
				for(i = 0; i < KL_NUM_CPUS; i++){
					if(kl_struct_len("local_t") == 2){
                                                ref_count += KL_GET_INT16(ptr +
offset);
                                        }
					else if(kl_struct_len("local_t") == 4){
						ref_count += KL_GET_INT32(ptr + offset);
					} else if(kl_struct_len("local_t") == 8){
						ref_count += KL_GET_INT64(ptr + offset);
					}
					offset += kl_struct_len("module_ref");
				}
			}

			print_value(ofp, "", ADDR64(addr), 8, ADDR_FLG);
			print_value(ofp, " ", UINT64(KL_UINT(ptr, "module", "core_size")), 9, UNSIGNED_FLG);
			print_value(ofp, " ", ref_count, 4, 0);
			fprintf(ofp, " %-30s", modname);
			print_modrefs(ptr, modname, ofp);

		} else {
			/* Linux 2.2 and Linux 2.4 */
			print_value(ofp, "", ADDR64(addr), 8, ADDR_FLG);
			print_value(ofp, " ", UINT64(KL_UINT(ptr, "module", "size")), 9, UNSIGNED_FLG);
			dummy = KL_GET_INT32(K_ADDR(ptr, "module", "uc"));
			print_value(ofp, " ", dummy, 4, 0);
			fprintf(ofp, " %-30s", modname);
			print_modrefs(ptr, modname, ofp);
		}
	}
	if (flags & C_KALLSYMS) {
		print_kallsyms(ptr, ofp);
	}
	if(!(flags & C_PLAIN)){
		fprintf(ofp, "\n");
	}

	kl_free_block(dump_page);
}


/*
 * print_modrefs_2_6()
 */
static void
print_modrefs_2_6(void *dump_module, const char *modname,  FILE *ofp)
{
	kaddr_t next_addr = 0;
	kaddr_t first_addr;
	void *mod_buf = NULL;
	int indent = 0;
	int counter = 0;

	if(!modname)
		return;

	if(!kl_is_member("module","modules_which_use_me")){
		/* no unload support */
		return;
	}

	first_addr = next_addr = 
		kl_kaddr(dump_module + kl_member_offset("module", 
				"modules_which_use_me"), "list_head", "next");
	fprintf(ofp, "[");

	/* We assume the following datatype here:   */
	/* struct module_use                        */
	/* {                                        */
       	/* 	struct list_head list;              */
       	/*	struct module *module_which_uses;   */
	/* };                                       */
	/* Unfortunately this type is not contained */
	/* in Kerntypes.                            */

	do{
		size_t mod_size = 0;
		uint64_t mod_addr;
		char* modname;
		char  buf[3*8];

		GET_BLOCK(next_addr, 3*KL_NBPW, buf);
		mod_addr=KL_GET_PTR(&buf[2*KL_NBPW]);
		next_addr=KL_GET_PTR(buf);

		if(next_addr == first_addr)
			break;

		kl_get_structure(mod_addr, "module", &mod_size, &mod_buf);
		modname = ((char*)mod_buf) + kl_member_offset("module","name");
		if(indent){
			if(KL_PTRSZ==64){
				fprintf(ofp, "\n %64s%s", "\0",
					modname);

			} else {
				fprintf(ofp, "\n %56s%s", "\0",
					modname);
			}
		} else {
			fprintf(ofp, "%s", modname);
			indent = 1;
		}
	} while(CHECK_ITER_THRESHOLD(counter++));
	fprintf(ofp, "]");
	kl_free_block(mod_buf);
	return;
}

/*
 * print_modrefs_2_4()
 */
static void
print_modrefs_2_4(void *dump_module, const char *modname,  FILE *ofp)
{
	void *dump_ref = NULL;
	size_t size_ref = 0, size_module;
	kaddr_t next_ref = 0;
	int counter = 0;
	void *dump_page = NULL;
	long indent = 0;

	if(!modname)
		return;

	dump_page = kl_alloc_block(KL_PAGE_SIZE * 2, K_TEMP);
	size_module = MODULE_SZ;
	next_ref = kl_kaddr(dump_module, "module", "refs");
	fprintf(ofp, "[");
	while(next_ref && CHECK_ITER_THRESHOLD(counter++)){
		memset(dump_page, 0, KL_PAGE_SIZE);
		if(kl_get_structure(next_ref, "module_ref",
					&size_ref, &dump_ref)){
			kl_free_block(dump_page);
			kl_free_block(dump_ref);
			return;
		}
		if(kl_get_structure(kl_kaddr(dump_ref, "module_ref",
					     "ref"), "module",
				    &size_module, &dump_module)){
			kl_free_block(dump_page);
			kl_free_block(dump_ref);
			return;
		}
		GET_BLOCK(kl_kaddr(dump_module, "module", "name"),
			  KL_PAGE_SIZE, dump_page);
		if((KL_ERROR == KLE_INVALID_MAPPING) ||
		   (KL_ERROR == KLE_PAGE_NOT_PRESENT)){
			/* ignore the error; possibly we crossed a
			 * page boundary and tried to access next
			 * page after page containing our zero
			 * terminated string 
			 */
			kl_reset_error();
		}
		modname = dump_page;
		if(indent){
			if(KL_PTRSZ==64){
				fprintf(ofp, "\n %64s%s", "\0",
					modname);

			} else {
				fprintf(ofp, "\n %56s%s", "\0",
					modname);
			}
		} else {
			fprintf(ofp, "%s", modname);
		}
		indent=1;
		next_ref = kl_kaddr(dump_ref, "module_ref", "next_ref");
	}
	ITER_THRESHOLD_MSG(ofp, counter-1);
	fprintf(ofp, "]");

	kl_free_block(dump_page);
	kl_free_block(dump_ref);

	return;
}

/*
 * print_modrefs()
 */

void
print_modrefs(void *dump_module, const char *modname,  FILE *ofp)
{
	if(KL_LINUX_RELEASE >= LINUX_2_6_0)
		print_modrefs_2_6(dump_module, modname, ofp);
	else
		print_modrefs_2_4(dump_module, modname, ofp);
}

/*
 * print_modsyms()
 */
void
print_modsyms(void *dump_module, const char *modname, int flags, FILE *ofp)
{
	uint64_t value;
	int i;
	int nsyms = 0;
	size_t size_modsym = 0;
	kaddr_t name, syms;
	void *dump_symname = NULL, *dump_modsym = NULL;
	char *module_name = NULL;

	if(!((flags & C_PLAIN) && (strcmp(modname, KL_KERNEL_MODULE)==0))){
		module_name = kl_alloc_block(strlen(modname)+3, K_TEMP);
		sprintf(module_name, "[%s]", modname);
	}
	dump_symname = kl_alloc_block(KL_SYMBOL_NAME_LEN, K_TEMP);
	if(KL_ERROR){
		return;
	}

	syms = kl_kaddr(dump_module, "module", "syms");
	if(KL_LINUX_RELEASE >= LINUX_2_6_0)
		/* XXX maybe implement also gpl_syms in the future */
		nsyms = KL_UINT(dump_module, "module", "num_syms");
	else
		nsyms = KL_UINT(dump_module, "module", "nsyms");

	if(!(flags & C_PLAIN)){
		fprintf(ofp, "        Number of exported symbols: %lu\n\n", 
			(unsigned long) nsyms);
	}

	for(i=0; (i < nsyms) && CHECK_ITER_THRESHOLD(i); i++){
		if(!i && !(flags & C_PLAIN)){
		        print_banner(ofp, "%>      ADDR% NAME [MODULE]%}75",
				     BANNER|SMINOR);
		}
		memset(dump_symname, 0, KL_SYMBOL_NAME_LEN);
		if(KL_LINUX_RELEASE >= LINUX_2_6_0){
			if(kl_get_structure(syms, "kernel_symbol",
                                            &size_modsym, &dump_modsym)){
                                kl_free_block(dump_symname);
                                kl_free_block(dump_modsym);
                                return;
                        }
                        value=KL_UINT(dump_modsym, "kernel_symbol", "value");
                        name=kl_kaddr(dump_modsym, "kernel_symbol", "name");
                        GET_BLOCK(name, KL_SYMBOL_NAME_LEN, dump_symname);
		} else {
			/* Linux 2.2 and 2.4 */
			if(kl_get_structure(syms, "module_symbol",
					    &size_modsym, &dump_modsym)){
				kl_free_block(dump_symname);
				kl_free_block(dump_modsym);
				return;
			}
			value=KL_UINT(dump_modsym, "module_symbol", "value");
			name=kl_kaddr(dump_modsym, "module_symbol", "name");
			GET_BLOCK(name, KL_SYMBOL_NAME_LEN, dump_symname);
		}
		if(flags & C_PLAIN){
			if(module_name){
				fprintf(ofp, "%"FMT64"x %s\t%s\n", value,
				(char*) dump_symname, module_name);
			} else {
				fprintf(ofp, "%"FMT64"x %s\n", value,
				(char*) dump_symname);
			}
		} else {
			print_value(ofp, NULL, value, 10, ADDR_FLG);
			fprintf(ofp," %-36s%s\n", (char*) dump_symname,
				module_name);
		}
		syms += size_modsym;
	}
	ITER_THRESHOLD_MSG(ofp, i);
	kl_free_block(dump_symname);
	kl_free_block(dump_modsym);
	kl_free_block(module_name);
	return;
}
