/*
 * $Id: dis_alpha.c,v 1.1 2004/12/21 23:26:17 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */
#include <lcrash.h>

#define SLOT_ALPHA(pc) (((pc & 0xf) < 6) ? 0 : (((pc & 0xf) < 0xc) ? 1 : 2))

/*
 * function declarations
 */
static kaddr_t start_instr_alpha(kaddr_t, int);
static kaddr_t do_dis_alpha(kaddr_t, int, FILE*);
static kaddr_t print_instr_stream_alpha(kaddr_t, int, int, int, FILE*);
static void dump_instr_alpha(kaddr_t, uint64_t, int, FILE*);
static int  print_instr_alpha(kaddr_t, FILE*, int);

/*
 * function definitions
 */

/*
 * start_instr_alpha() - static function to ensure proper padding of instruction
 */
static kaddr_t
start_instr_alpha(kaddr_t pc, int before)
{
	while (before--) {
		if (SLOT_ALPHA(pc)) {
			pc -= 6;
		} else {
			pc -= 4;
		}
	}
	return(pc);
}
/*
 * do_dis_alpha() - static function to disassemble i instructions 
 */
static kaddr_t
do_dis_alpha(kaddr_t value, int lines, FILE *ofp)
{
	bfd_vma pc;
	int i;

	set_dis_ofp(ofp);

        pc = value;
	for (i = 0; i < lines; i++) {
		dis_printintaddr(pc, &DISINFO, 1);
		pc += print_insn_alpha(pc, &DISINFO);
		DISINFO.fprintf_func(ofp, "\n");
	}
	return((kaddr_t)pc);
}

/*              
 * print_instr_stream_alpha() - static function to disassemble an
 *                              instructions stream
 */
static kaddr_t
print_instr_stream_alpha(kaddr_t v, int bcount, int acount, int flags,
			 FILE *ofp)
{
	int count;
	kaddr_t pc = v;

	/* Make sure that output goes to the right place
	 */ 
	set_dis_ofp(ofp);

	/* Make sure that pc is aligned properly
	 */
	if (SLOT_ALPHA(pc) == 1) {
		pc = ((pc >> 4) << 4) | 0x6;
	} else if (SLOT_ALPHA(pc) == 2) {
		pc = ((pc >> 4) << 4) | 0xc;
	}
	count = acount + 1;
	if (bcount) {
		pc = start_instr_alpha(pc, bcount);
		count += bcount;	
	}
	pc = do_dis_alpha(pc, count, ofp);
	return(pc);
}

/* 
 * print_instr_alpha() - static function
 */
static int                     
print_instr_alpha(kaddr_t pc, FILE *ofp, int flag)
{               
	return(do_dis_alpha(pc, 1, ofp) - pc);
}

/*
 * dump_instr_alpha() - static function, arch specific instruction dump routine
 */
static void
dump_instr_alpha(kaddr_t addr, uint64_t count, int flags, FILE *ofp)
{
	unsigned char template;
	uint64_t t0, t1, slot0, slot1, slot2;
	kaddr_t bundlep;
	bfd_byte bundle[16];
	
	bundlep = (addr & 0xfffffffffffffff0);
	GET_BLOCK(bundlep, sizeof(bundle), bundle);
	t0 = bfd_getl64(bundle);
	t1 = bfd_getl64(bundle + 8);
	template = (t0 >> 1) & 0xf;
	slot0 = (t0 >>  5) & 0x1ffffffffffLL;
	slot1 = ((t0 >> 46) & 0x3ffff) | ((t1 & 0x7fffff) << 18);
	slot2 = (t1 >> 23) & 0x1ffffffffffLL;

	fprintf(ofp, "\nbundlep  : 0x%llx\n", bundlep);
	fprintf(ofp, "template : %02x\n", template);
	fprintf(ofp, "   slot0 : %011llx\n", slot0);
	fprintf(ofp, "   slot1 : %011llx\n", slot1);
	fprintf(ofp, "   slot2 : %011llx\n", slot2);
}

/*
 * dis_init_alpha() - init arch specific stuff for disassembling
 */
int
dis_init_alpha(FILE *ofp, int dumparch)
{
	PRINT_INSTR        = print_instr_alpha;
	PRINT_INSTR_STREAM = print_instr_stream_alpha;
	DUMP_INSTR         = dump_instr_alpha;
	USE_OPCODE         = 1;

	DISINFO.fprintf_func   = dis_fprintf;
	DISINFO.stream         = ofp;
	DISINFO.flavour        = bfd_target_elf_flavour;
	DISINFO.arch           = bfd_arch_alpha;
	/* don't know what mach to specify here */
/* 	DISINFO.mach           = bfd_mach_alpha_ev????; */
	DISINFO.endian         = BFD_ENDIAN_LITTLE;
	DISINFO.read_memory_func 	   = getidmem;
	DISINFO.print_address_func 	   = dis_printaddr;
	DISINFO.symbol_at_address_func = dis_getsym;
	/* set display_endianess according to alpha endianess */
	DISINFO.display_endian = BFD_ENDIAN_LITTLE;

	return(0);
}
