/*
 * $Id: cmd_page.c,v 1.1 2004/12/21 23:26:18 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>

extern kaddr_t get_pg_struct_addr(uint64_t, void *, int, unsigned long *);
/*
 * page_cmd() -- Run the 'page' command.
 */
int
page_cmd(command_t *cmd)
{
	int mode, first_time = TRUE, page_cnt = 0;
	unsigned long start_pfn, spanned_pages, pfn;
	void  *pp, *pgdat = NULL;
	kaddr_t p, pgdat_list, mmap;
	uint64_t value;
	unsigned long pgdat_size, i;

	pp = kl_alloc_block(PAGE_SZ, K_TEMP);
	if (KL_ERROR) {
		return(1);
	}
	pgdat_size = kl_struct_len("pg_data_t");
	pgdat_list = KL_PGDAT_LIST;
	pgdat  = kl_alloc_block(pgdat_size, K_TEMP);
	if (KL_ERROR) {
		return(1);
	}

	if (cmd->nargs == 0) {
		if (LINUX_2_6_X(KL_LINUX_RELEASE) && !MEM_MAP) {
			/*
			 * Analyzing dump generated from NUMA machine.
			 */
			while (pgdat_list) {
				GET_BLOCK(pgdat_list, pgdat_size, pgdat);
				if (KL_ERROR) {
					kl_free_block(pp);
					kl_free_block(pgdat);
					return KL_ERROR;
				}
				start_pfn = KL_UINT(pgdat, "pg_data_t",
						"node_start_pfn");
				spanned_pages = KL_UINT(pgdat, "pg_data_t",
						"node_spanned_pages");
				for (pfn = start_pfn;
					pfn < start_pfn + spanned_pages; pfn++){
					mmap = kl_kaddr(pgdat, "pg_data_t",
						"node_mem_map");
					p = mmap + (pfn * PAGE_SZ);
					GET_BLOCK(p, PAGE_SZ, pp);
					if (KL_ERROR) {
						continue;
					}
					if (first_time ||
						(cmd->flags & (C_FULL|C_NEXT))){
						page_banner(cmd->ofp, BANNER|SMAJOR);
						first_time = FALSE;
					}
					print_page_struct(p, pp, cmd->flags,
						 pfn, cmd->ofp);
					page_cnt++;
				}
				pgdat_list =
					kl_kaddr(pgdat,"pg_data_t","pgdat_next")
;
 			} 	
		} else { 
			for (i = 0; i < NUM_PHYSPAGES; i++) {            
				p = MEM_MAP + (i * PAGE_SZ);
				GET_BLOCK(p, PAGE_SZ, pp);
				if (KL_ERROR) {
					continue;
				}
				if (first_time || (cmd->flags & (C_FULL|C_NEXT))
) {
					page_banner(cmd->ofp, BANNER|SMAJOR);	
					first_time = FALSE;
				}
				print_page_struct(p, pp, cmd->flags, i, 
					cmd->ofp);
				page_cnt++;
			}
		}       
	} else {
		for (i = 0; i < cmd->nargs; i++) {
			kl_get_value(cmd->args[i], 
				&mode, NUM_PHYSPAGES, &value);
			if (KL_ERROR) {
				continue;
			}
			if (!(p = get_pg_struct_addr(value, pgdat, mode, &pfn)))
 {
				if (mode == 1) {
					fprintf(KL_ERRORFP, "Invalid page "
						"number : %"FMT64"d\n", value);
				} else if (mode == 2) {
					fprintf(KL_ERRORFP, "Invalid page "
						"struct address: "
						"0x%"FMTPTR"x\n", p);
				} else {
					fprintf(KL_ERRORFP,"Invalid page struct"
						" value : %s\n", cmd->args[i]);
				}
				continue;
			}
			GET_BLOCK(p, PAGE_SZ, pp);
			if (KL_ERROR) {
				continue;
			}
			if (first_time || (cmd->flags & (C_FULL|C_NEXT))) {
				page_banner(cmd->ofp, BANNER|SMAJOR);
				first_time = FALSE;
			}
			print_page_struct(p, pp, cmd->flags, pfn, cmd->ofp);
			page_cnt++;
		}
	}
	kl_free_block(pp);
	kl_free_block(pgdat);
	if (page_cnt) {
		page_banner(cmd->ofp, SMAJOR);
	}
	PLURAL("active page struct", page_cnt, cmd->ofp);
	return(0);
}

#define _PAGE_USAGE "[-f] [-n] [-w outfile] [page_list]"

/*
 * page_usage() -- Print the usage string for the 'page' command.
 */
void
page_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _PAGE_USAGE);
}

/*
 * page_help() -- Print the help information for the 'page' command.
 */
void
page_help(command_t *cmd)
{
	CMD_HELP(cmd, _PAGE_USAGE,
        "Display relevant information from the page struct for each entry "
	"in page_list. Entries in page_list can take the form of a page "
	"number (following a '#') or a virtual address of a page struct "
	"in memory. If no entries are specified, an entry for every "
	"page of physical memory will be displayed.");
}

/*
 * page_parse() -- Parse the command line arguments for 'page'.
 */
int
page_parse(command_t *cmd)
{
	if (set_cmd_flags(cmd, (C_FULL|C_NEXT|C_WRITE), 0)) {
		return(1);
	}
	return(0);
}

/*
 * page_complete() -- Complete arguments of 'page' command.
 */
char *
page_complete(command_t *cmd)
{
	char *ret;

	/* complete standard options (for example, -w option) arguments
	 */
	if ((ret = complete_standard_options(cmd)) != NOT_COMPLETED) {
		return(ret);
	}
	fprintf(cmd->ofp, "\n");
	page_usage(cmd);
	return(DRAW_NEW_ENTIRE_LINE);
}
