/*
 * $Id: cmd_dump.c,v 1.2 2005/02/25 22:04:04 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2005 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>

/*
 * dump_cmd() -- Run the 'dump' command.
 */
int
dump_cmd(command_t *cmd)
{
        int mode;
        uint64_t size, value;
        kaddr_t start_addr, paddr;
        syment_t *sp;

        sp = kl_lkup_symname(cmd->args[0]);
        if (!KL_ERROR) {
                start_addr = sp->s_addr;
        } else {
                kl_get_value(cmd->args[0], &mode, 0, &value);
                if (KL_ERROR) {
			fprintf(KL_ERRORFP, "Invalid value: %s\n", 
				cmd->args[0]); 
                        return(1);
                }
		start_addr = (kaddr_t)value;

                /* If entered as a PFN, convert to physical address
                 */
                if (mode == 1) {
			start_addr = start_addr << KL_PAGE_SHIFT;
			cmd->flags |= C_PADDR;
		} else {
			/* Check to see if the entered address is a
			 * physical address.
			 */
			kl_virtop(start_addr, (void*)NULL, &paddr);
			if (start_addr == paddr) {
				cmd->flags |= C_PADDR;
			}
		}
        }

        if (cmd->nargs > 1) {
                GET_VALUE(cmd->args[1], &size);
                if (KL_ERROR) {
			fprintf(KL_ERRORFP, "Invalid size: %s\n", 
				cmd->args[1]);
                        return(1);
                }
        } else {
                size = 1;
        }

        if ((start_addr == 0) &&
                ((cmd->args[0][0] != '0') && (cmd->args[0][0] != '#'))) {
                        fprintf(cmd->efp, "%s not found in symbol table.\n", 
				cmd->args[0]);
        }

        dump_memory(start_addr, size, cmd->flags, cmd->ofp);
        if (KL_ERROR) {
                return(1);
        }
        return(0);
}

#define _DUMP_USAGE \
	"[-c] [-d] [-i] [-o] [-p] [-s] [-x] [-B] [-D] [-H] [-W] [-w outfile]\n" \
	"              addr [count]"

/*
 * dump_usage() -- Print the usage string for the 'dump' command.
 */
void
dump_usage(command_t *cmd)
{
        CMD_USAGE(cmd, _DUMP_USAGE);
}

/*
 * dump_help() -- Print the help information for the 'dump' command.
 */
void
dump_help(command_t *cmd)
{
        CMD_HELP(cmd, _DUMP_USAGE,
		"Display count values starting at kernel virtual (or "
		"physical) address addr in one of the following formats: "
		"decimal (-d), octal (-o), or hexadecimal (-x) (addr is "
		"considered to be a physical address if it is not in "
		"kernel address space). addr is also considered to be "
                 "a physical address if -p option is specified. "
		"If addr is preceded by a pound "
		"sign ('#'), it is treated as a page number (PFN) and is "
		"converted to a physical address. Values can be "
		"displayed as bytes (-B), half words (-H), words (-W), "
		"and double words (-D). Architecture word size is "
		"the default display size.\n\n"
		"If the -c command line flag is included, then individual "
		"values will be displayed concatenated together, with no "
		"space between them. This option is useful when "
		"constructing patterns for the search command. Note that "
		"this option only has effect when values are displayed in "
		"hexadecimal format.\n\n"
		"If the -i command line flag is included, then program "
		"instruction values will be displayed instead of numeric "
		"values.\n\n"
		"If the -s command line flag is included, then count "
		"number of NULL terminated strings, starting from addr, "
		"will be displayed.\n\n"
		"The default format is hexadecimal, and the default "
		"count is 1.");
}

/*
 * dump_parse() -- Parse the command line arguments for the 'dump' command.
 */
int
dump_parse(command_t *cmd)
{
	option_t *op;

	if (set_cmd_flags(cmd, (C_TRUE|C_WRITE), "cdiosxBDHPW")) {
		return(1);
	}
	op = cmd->options;
	while (op) {
		switch(op->op_char) {
			case 'c':
				cmd->flags |= C_CONCAT;
				break;

			case 'd':
				cmd->flags |= C_DECIMAL;
				break;

			case 'i':
				cmd->flags |= C_INSTR;
				break;

			case 'o':
				cmd->flags |= C_OCTAL;
				break;

			case 's':
				cmd->flags |= C_STRING;
				break;

			case 'p':
				cmd->flags |= C_PADDR;
				break;

			case 'x':
				cmd->flags |= C_HEX;
				break;

			case 'B':
				cmd->flags |= C_BYTE;
				break;

			case 'D':
				cmd->flags |= C_DWORD;
				break;

			case 'H':
				cmd->flags |= C_HWORD;
				break;

			case 'W':
				cmd->flags |= C_WORD;
				break;
		}
		op = op->op_next;
	}
	return(0);
}

/*
 * dump_complete() -- Complete arguments of 'dump' command.
 */
char *
dump_complete(command_t *cmd)
{
	char *ret;

	/* complete standard options (for example, -w option) arguments
	 */
	if ((ret = complete_standard_options(cmd)) != NOT_COMPLETED) {
		return(ret);
	}
	fprintf(cmd->ofp, "\n");
	dump_usage(cmd);
	return(DRAW_NEW_ENTIRE_LINE);
}
