/*
 * $Id: cmd_base.c,v 1.1 2004/12/21 23:26:18 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>

/* 
 * base_print() -- print a number in hex, decimal, octal, and binary
 */
int
base_print(char *string, FILE *ofp)
{
	uint64_t num;
	char *loc;

	errno=0;
	num = (uint64_t)kl_strtoull(string, &loc, 0);
	if (errno || loc) {
		return(1);
	}
	fprintf(ofp, "    hex: 0x%"FMT64"x\n", num);
	fprintf(ofp, "decimal: %"FMT64"u\n", num);
	fprintf(ofp, "  octal: 0%"FMT64"o\n", num);
	fprintf(ofp, " binary: 0b");
	kl_binary_print(num, ofp);
	fprintf(ofp, "\n");
	return(0);
}

/*
 * base_cmd() -- Run the 'base' command.
 */
int
base_cmd(command_t *cmd)
{
	int i;

	fprintf(cmd->ofp, "\n");
	for (i = 0; i < cmd->nargs; i++) {
		fprintf(cmd->ofp, "------------------------------------------"
			"--------\n");
		if (base_print(cmd->args[i], cmd->ofp)) {
			fprintf(cmd->efp, "Error with input value (%s)\n",
				cmd->args[i]);
		}
	}
	if (cmd->nargs) {
		fprintf(cmd->ofp, "------------------------------------------" 
			"--------\n");
	}
	return(0);
}

#define _BASE_USAGE "[-w outfile] numeric_values[s]"

/*
 * base_usage() -- Print the usage string for the 'base' command.
 */
void
base_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _BASE_USAGE);
}

/*
 * base_help() -- Print the help information for the 'base' command.
 */
void
base_help(command_t *cmd)
{
	CMD_HELP(cmd, _BASE_USAGE,
		"Display a number in binary, octal, decimal, and hexadecimal. "
		"A number in a radix other then decimal should be preceded by a "
		"prefix that indicates its radix as follows:\n\n"
		"              0x      hexadecimal\n"
		"              0       octal\n"
		"              0b      binary");
}

/*
 * base_parse() -- Parse the command line arguments.
 */
int
base_parse(command_t *cmd)
{
	if (set_cmd_flags(cmd, (C_TRUE|C_WRITE), 0)) {
		return(1);
	}
	return(0);
}

/*
 * base_complete() -- Complete arguments of 'base' command.
 */
char *
base_complete(command_t *cmd)
{
	char *ret;

	/* complete standard options (for example, -w option) arguments
	 */
	if ((ret = complete_standard_options(cmd)) != NOT_COMPLETED) {
		return(ret);
	}
	fprintf(cmd->ofp, "\n");
	base_usage(cmd);
	return(DRAW_NEW_ENTIRE_LINE);
}
