/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */


#ifndef PUBLIC_VERIFY_X509
 #define PUBLIC_VERIFY_X509

#include <openssl/x509.h>

#include "verify_x509_datatypes.h"

/************************************************************************
 * Main library functions 
 ************************************************************************/

/* 
 * The life cycle is the following:
 * 1. verify_X509_init (&verify_datastruct);
 * 2. verify_X509_setParameter (&verify_datastruct, VERIFY_X509_CERTIFICATE_PEM, pemstring);
 * .. verify_X509_setParameter (&verify_datastruct, VERIFY_X509_PRIVATEKEY_PEM, pemprivatekeystring);
 * .. verify_X509_setParameter (&verify_datastruct, VERIFY_X509_CA_PATH, capath);
 * 3. verify_X509_verify (&verify_datastruct);
 * 4. verify_X509_term (&verify_datastruct);
 */
 
 
/* These are the interacting functions for a stand-alone verification call out.
   To prepare for multi-threaded environments, a pointer to the
   internal_verify_x509_data_t structure is initialized, and needs to be passed
   on throughout the life cycle of the functions until its destruction */


/**
 * initializes verify_x509_data and loads OpenSSL_add_all_algorithms and
 * ERR_load_crypto_strings
 * Returns 0 on success, 1 on error */
int verify_X509_init(internal_verify_x509_data_t **);

/**
 * Parses key/value pair verify_x509_option/... and stores in the internal data
 * verify_x509_data.
 * Returns VER_R_* error code or VER_R_X509_PARAMS_OK on success
 */
verify_x509_error_t verify_X509_setParameter(
	internal_verify_x509_data_t ** verify_x509_data,
	verify_x509_option_t verify_x509_option,
	...);

/**
 * This is the verification function that will first process the provided
 * setParameters into usable datatypes for the actual verification. Then
 * the verification of the certificate chain (includes GT2 and GT4/RFC
 * proxies) will start. First the chain will be checked and then (if there
 * is a private key) the private key must match the certificate chain.
 * 
 * Returns 0 on success or a error code from ERR_peek_error()
 */
long unsigned verify_X509_verify(internal_verify_x509_data_t **);

/**
 * Terminates the verification library by freeing the internal memory
 * Returns 0 on success, 1 on failure.
 */
int verify_X509_term(internal_verify_x509_data_t **);

/************************************************************************
 * Utility functions from verify_x509_utils.c
 ************************************************************************/

/**
 * Check if certificate can be used as a CA to sign standard X509 certs
 * Return 1 if true; 0 if not.
 */
int verify_x509IsCA(X509 *cert);

/**
 * Converts ASN1 time string (in a const char *) to time_t. See also
 * verify_asn1TimeToTimeT()
 */
time_t verify_str_asn1TimeToTimeT(const char *asn1time);

/**
 * Converts ASN1 time string (in a ASN1_TIME) to time_t
 * using ASN1_STRING_data() ASN1_STRING_get0_data() to convert
 * ASN1_GENERALIZEDTIME to const char *, then calling
 * verify_str_asn1TimeToTimeT()
 */
time_t verify_asn1TimeToTimeT(ASN1_TIME *asn1time);

/**
 * Returns type of proxy certificate.
 * Valid values are:
 *	  NONE
 *	  CA
 *	  EEC
 *	  GT2_PROXY
 *	  RFC_PROXY
 *	  GT2_LIMITED_PROXY
 *	  RFC_LIMITED_PROXY
 *	  GT3_PROXY
 *	  GT3_LIMITED_PROXY
 */
proxy_type_t verify_type_of_proxy(X509 *);


#endif /* PUBLIC_VERIFY_X509 */
