(* Copyright (C) 2000, Luca Padovani <luca.padovani@cs.unibo.it>.
 *
 * This file is part of lablgtkmathview, the Ocaml binding
 * for the GtkMathView widget.
 * 
 * lablgtkmathview is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * lablgtkmathview is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with lablgtkmathview; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * For details, send a mail to the author.
 *)

open Gaux
open Gtk
open Gtk_mathview
open GtkBase
open GtkMathView
open GObj

exception ErrorLoadingFile of string;;
exception ErrorWritingFile of string;;
exception ErrorLoadingDOM;;
exception NoSelection;;

class math_view_signals obj = object
  inherit GContainer.container_signals obj
  method clicked =
   let module S = GtkSignal in
    let new_clicked =
     let new_marshaller f x y =
      MathView.Signals.clicked.S.marshaller
       (fun e -> f (new Gdome.element e)) x y
     in
      { S.name = "clicked"; S.marshaller = new_marshaller }
    in
     GtkSignal.connect ~sgn:new_clicked obj ~after
  method selection_changed =
   let module S = GtkSignal in
    let new_selection_changed =
     let new_marshaller f x y =
      MathView.Signals.selection_changed.S.marshaller
       (function None -> f None | Some e -> f (Some (new Gdome.element e))) x y
     in
      { S.name = "selection_changed"; S.marshaller = new_marshaller }
    in
     GtkSignal.connect ~sgn:new_selection_changed obj ~after
  method element_changed =
   let module S = GtkSignal in
    let new_element_changed =
     let new_marshaller f x y =
      MathView.Signals.element_changed.S.marshaller
       (function None -> f None | Some e -> f (Some (new Gdome.element e))) x y
     in
      { S.name = "element_changed"; S.marshaller = new_marshaller }
    in
     GtkSignal.connect ~sgn:new_element_changed obj ~after
end

class math_view obj = object
 inherit GContainer.container (obj : Gtk_mathview.math_view obj)
 method connect = new math_view_signals obj
 method load ~filename =
  if not (MathView.load obj ~filename) then raise (ErrorLoadingFile filename)
 method load_tree ~dom =
  if not (MathView.load_tree obj ~dom:((dom : Gdome.document)#as_Document)) then
   raise ErrorLoadingDOM
 method unload = MathView.unload obj
 method get_selection =
  match MathView.get_selection obj with
     None -> None
   | Some element -> Some (new Gdome.element element)
 method set_selection element =
  let element =
   match element with
      None -> None
    | Some element -> Some ((element : Gdome.element)#as_Element)
  in
   MathView.set_selection obj element
 method get_width = MathView.get_width obj
 method get_height = MathView.get_height obj
 method get_top = MathView.get_top obj
 method set_top = MathView.set_top obj
 method set_adjustments =
  fun adj1 adj2 ->
   MathView.set_adjustments obj (GData.as_adjustment adj1)
   (GData.as_adjustment adj2)
 method get_hadjustment = new GData.adjustment (MathView.get_hadjustment obj)
 method get_vadjustment = new GData.adjustment (MathView.get_vadjustment obj)
 method get_buffer = MathView.get_buffer obj
 method get_frame = new GBin.frame (MathView.get_frame obj)
 method set_font_size = MathView.set_font_size obj
 method get_font_size = MathView.get_font_size obj
 method set_anti_aliasing = MathView.set_anti_aliasing obj
 method get_anti_aliasing = MathView.get_anti_aliasing obj
 method set_kerning = MathView.set_kerning obj
 method get_kerning = MathView.get_kerning obj
 method set_transparency = MathView.set_transparency obj
 method get_transparency = MathView.get_transparency obj
 method set_log_verbosity = MathView.set_log_verbosity obj
 method get_log_verbosity = MathView.get_log_verbosity obj
 method export_to_postscript
 	?(width = 595) ?(height = 822) ?(x_margin = 72) ?(y_margin = 72)
	?(disable_colors = false) ~filename () =
  let result = MathView.export_to_postscript obj
	~width ~height ~x_margin ~y_margin ~disable_colors ~filename
  in
  if not result then raise (ErrorWritingFile filename)
 method get_font_manager_type = MathView.get_font_manager_type obj
 method set_font_manager_type ~fm_type = MathView.set_font_manager_type obj ~fm_type
 method get_element =
  match MathView.get_element obj with
     None -> None
   | Some element -> Some (new Gdome.element element)
 method action_get_selected = MathView.action_get_selected obj
 method action_set_selected = MathView.action_set_selected obj
 method get_action =
  match MathView.get_action obj with
     None -> None
   | Some ac -> Some (new Gdome.element ac)
 method action_toggle = MathView.action_toggle obj
end

let math_view ?adjustmenth ?adjustmentv ?font_size ?font_manager ?border_width
 ?width ?height ?packing ?show () =
 let w =
   MathView.create
    ?adjustmenth:(may_map ~f:GData.as_adjustment adjustmenth)
    ?adjustmentv:(may_map ~f:GData.as_adjustment adjustmentv)
    ()
 in
  Container.set w ?border_width ?width ?height;
 let mathview = pack_return (new math_view w) ~packing ~show in
 begin
    match font_size with
    | Some size -> mathview#set_font_size size
    | None      -> ()
  end;
  begin
    match font_manager with
    | Some manager -> mathview#set_font_manager_type ~fm_type:manager
    | None         -> ()
  end;
  mathview
;;
