/***************************************************************************
 *
 *  $Id: kzenmediafiles.cpp,v 1.15 2005/06/29 00:05:27 muszilla Exp $
 *
 *  Copyright (C) 2005 by Andreas Mussgiller
 *  muszilla@users.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the
 *  Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <iostream>

#include <qdir.h>
#include <qregexp.h>
#include <qfileinfo.h>

#include <kmimetype.h>
#include <klocale.h>
#include <kfilemetainfo.h>
#include <kdebug.h>

#include <fileref.h>
#include <mpegfile.h>
#include <tag.h>

#include "kzenconfig.h"
#include "kzenmediafiles.h"
#include "kzentrack.h"

bool KZenMediaFiles::isMediaFile(const KURL & url)
{
  return isMP3(url);
}

bool KZenMediaFiles::isPlaylistFile(const KURL & url)
{
  KMimeType::Ptr result = KMimeType::findByURL(url, 0);
  return (result->name() == "audio/mpegurl" ||
	  result->name() == "audio/x-mpegurl");
}

bool KZenMediaFiles::isMP3(const KURL & url)
{
  KMimeType::Ptr result = KMimeType::findByURL(url, 0);
  return result->name() == "audio/x-mp3";
}

bool KZenMediaFiles::isWMA(const KURL & url)
{
  KMimeType::Ptr result = KMimeType::findByURL(url, 0);
  return result->name() == "audio/x-ms-wma";
}

bool KZenMediaFiles::isWAV(const KURL & url)
{
  KMimeType::Ptr result = KMimeType::findByURL(url, 0);
  return result->name() == "audio/x-wav";
}

bool KZenMediaFiles::isDirectory(const KURL & url)
{
  QFileInfo info(url.path());
  if (info.exists() && info.isDir()) return true;
  return false;
}

KZenTrack * KZenMediaFiles::createTrackMP3(const KURL & url)
{
  QFile file(url.path());
  if (!file.exists()) return 0;

  TagLib::FileRef f = TagLib::FileRef(new TagLib::MPEG::File(QFile::encodeName(url.path()).data()));
  if(f.isNull() || !f.tag()) return 0;
  
  KZenTrack * newTrack = new KZenTrack();
  newTrack->setCodec(KZenTrack::CodecMP3);

  newTrack->setFileSize(file.size());

  TagLib::Tag * tag = f.tag();
  QString temp;

  temp = TStringToQString(tag->artist());
  if (temp.length()>0) {
    newTrack->setArtist(temp);
  } else {
    newTrack->setArtist(i18n("Unknown Artist"));
  }
    
  temp = TStringToQString(tag->album());
  if (temp.length()>0) {
    newTrack->setAlbum(temp);
  } else {
    newTrack->setAlbum(i18n("Unknown Album"));
  }
  
  temp = TStringToQString(tag->title());
  if (temp.length()>0) {
    newTrack->setTitle(temp);
  } else {
    newTrack->setTitle("");
  }
    
  temp = TStringToQString(tag->genre());
  if (temp.length()>0) {
    newTrack->setGenre(temp);
  } else {
    newTrack->setGenre("Other");
  }
  
  newTrack->setTrackNumber(tag->track());
  newTrack->setYear(tag->year());
  
  TagLib::AudioProperties * properties = f.audioProperties();
  
  newTrack->setLength(properties->length());
  
  QString filename = url.path().utf8();
  filename = filename.right(filename.length()-filename.findRev(QDir::separator(),-1)-1);
  newTrack->setFileName(filename);
  
  if (newTrack->getTitle().length()==0) {
    while ((filename.right(4)).lower()==".mp3")
      filename = filename.left(filename.length()-4);
    
    newTrack->setTitle(filename);
  }
  
  return newTrack;
}

bool KZenMediaFiles::checkLibraryPath(const KURL & url)
{
  QString path = url.path();
  QStringList list = QStringList::split("/",path);
  
  QDir dir;
  path = "";
  for (QStringList::Iterator it=list.begin();it!=list.end();++it) {
    path += "/";
    path += *it;
    
    QFileInfo info(path);
    if (!info.exists()) {
      if (!dir.mkdir(path)) {
	return false;
      }
    }
    
    dir.cd(path);
  }
  
  return true;
}

KURL KZenMediaFiles::saveFilename(KZenTrack * track, const KURL & surl, bool & valid, bool create)
{
  KURL url = surl;
  
  QString tracknumber;
  tracknumber.sprintf("%02d",track->getTrackNumber());

  QString artist = track->getArtist();
  artist.replace("/","_");
  QString album = track->getAlbum();
  album.replace("/","_");
  QString title = track->getTitle();
  title.replace("/","_");

  QString path = KZenConfig::get()->savePath();
  path.replace("%artist",artist,false);
  path.replace("%album",album,false);
  path.replace("%title",title,false);
  path.replace("%track",tracknumber,false);
  path.replace(" ","_");
  path.replace("'","_");
  path.replace("__","_");
  url.addPath(path.lower());

  if (create && !checkLibraryPath(url)) {
    valid = false;
    return url;
  }
  
  QFileInfo info; 
  info.setFile(url.path());
  if (create && !info.isWritable()) {
    valid = false;
    return url;
  }

  QString codecname = track->getCodecName();
  codecname = codecname.lower();
  QString name = KZenConfig::get()->saveName();
  name.replace("%artist",artist,false);
  name.replace("%album",album,false);
  name.replace("%title",title,false);
  name.replace("%track",tracknumber,false);
  name.replace("/","_");
  name.replace(" ","_");
  name.replace("'","_");
  name.replace("__","_");
  url.addPath(name.lower());
  
  int count = 0;
  name.sprintf("%s.%s",url.path().utf8().data(),codecname.utf8().data());
  while (create && QFile::exists(name)) {
    count++;
    name.sprintf("%s_%d.%s",url.path().utf8().data(),count,codecname.utf8().data());
  }

  url.setPath(name);
  
  valid = true;
  
  return url;
}

void KZenMediaFiles::getMediaFiles(const KURL & url, QPtrList<KZenMediaFiles::Track> & t, bool exclude)
{ 
  QStringList::Iterator it = KZenConfig::get()->excludeFromSync().find(url.path());
  if (exclude || it!=KZenConfig::get()->excludeFromSync().end()) {
    exclude = true;
  } else {
    exclude = false;
  }
  
  QDir d(url.path());
  d.setFilter(QDir::Files | QDir::Dirs | QDir::NoSymLinks);
  
  const QFileInfoList *list = d.entryInfoList();
  QFileInfoListIterator itl(*list);
  QFileInfo *fi;
  
  KURL u;
  while ((fi=itl.current())!=0) {
    u = url;
    u.addPath(fi->fileName());

    KZenMediaFiles::Track * track;
    if (!fi->fileName().startsWith(".")) {

      if (fi->isFile() && KZenMediaFiles::isMP3(u)) {
	track = new KZenMediaFiles::Track();

	track->fExclude = exclude;
	track->fURL = u;
	t.append(track);
      }
	
      if (fi->isDir()) getMediaFiles(u, t, exclude);
    }
    ++itl;
  }
}

void KZenMediaFiles::getM3UList(const KURL & url, QStringList & t)
{ 
  QDir d(url.path());
  d.setFilter(QDir::Files | QDir::Dirs | QDir::NoSymLinks);
  
  const QFileInfoList *list = d.entryInfoList();
  QFileInfoListIterator it( *list );
  QFileInfo *fi;
  
  KURL u;
  while ((fi=it.current())!=0) {
    u = url;
    u.addPath(fi->fileName());
    
    if (!fi->fileName().startsWith(".")) {
      if (fi->isFile() && KZenMediaFiles::isPlaylistFile(u)) {
	t << u.path();	
      }
      
      if (fi->isDir()) getM3UList(u, t);
    }
    ++it;
  }
}

KZenMediaFiles::Track::Track()
{
  fTagRead = false; 
  fExclude = false; 
}

KZenMediaFiles::Track::~Track() 
{

}

void KZenMediaFiles::Track::readTag()
{
  if (fTagRead) return;
  fTagRead = true;
  
  TagLib::FileRef f = TagLib::FileRef(new TagLib::MPEG::File(QFile::encodeName(fURL.path()).data()));
  if(f.isNull() || !f.tag()) {
    fExclude = true;
    return;
  }
  
  TagLib::Tag * tag = f.tag();
  QString temp;
  
  temp = TStringToQString(tag->title());
  if (temp.length()>0) {
    fTitle = temp;
  } else {
    fTitle = i18n("Unknown Title");
  }
  
  temp = TStringToQString(tag->artist());
  if (temp.length()>0) {
    fArtist = temp;
  } else {
    fArtist = i18n("Unknown Artist");
  }
  
  temp = TStringToQString(tag->album());
  if (temp.length()>0) {
    fAlbum = temp;
  } else {
    fAlbum = i18n("Unknown Album");
  }

  fTagRead = true;
}
