// =============================================================================
//
//      --- kvi_dcc_voice_thread.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2003 Robin Verduijn <robin@debian.org>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviDccVoiceThread"

#include "kvi_settings.h"

#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include <unistd.h>
// Check for the OSS Api. We do not want to fail here
#ifndef COMPILE_WITH_NO_SOUND
	#ifdef HAVE_LINUX_SOUNDCARD_H
		#include <linux/soundcard.h>
	#else
		#ifdef HAVE_SYS_SOUNDCARD_H
			#include <sys/soundcard.h>
		#else
			#ifdef HAVE_SOUNDCARD_H
				#include <soundcard.h>
			#else
				// Cannot compile
				#define COMPILE_WITH_NO_SOUND
				#warning "Cannot find the soundcard.h header; you will NOT be able to use DCC Voice"
			#endif
		#endif
	#endif
#endif

#include "kvi_application.h"
#include "kvi_dcc_event.h"
#include "kvi_dcc_voice.h"
#include "kvi_dcc_voice_thread.h"
#include "kvi_debug.h"
#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_mutex.h"
#include "kvi_netutils.h"

// DO NOT CHANGE THIS
#define KVI_AUDIO_DEVICE    "/dev/dsp"
#define KVI_FRAG_SIZE       0x00080008
#define KVI_FORMAT          AFMT_S16_LE
#define KVI_NUM_CHANNELS    1
#define KVI_SPEED           8000

KviDccVoiceThread::KviDccVoiceThread(KviDccVoiceData *data)
	: KviDccThread()
{
	m_pData = data;
}

KviDccVoiceThread::~KviDccVoiceThread()
{
	// Nothing here
}

/**
 * Main DCC Voice loop.
 */
void KviDccVoiceThread::eventLoop()
{
#ifndef COMPILE_WITH_NO_SOUND
	KviDccEvent *ev = new KviDccEvent(KVI_DCC_EVENT_CONNECTIONESTABLISHED, "");
	KviApplication::postEvent(m_pData->parent, ev);

	ev = new KviDccEvent(KVI_DCC_EVENT_ENABLETALK, "");
	KviApplication::postEvent(m_pData->parent, ev);

	for( ; !m_bStopRequested; ) {
		if( m_pData->state != KVI_DCC_VOICE_STATE_TALK ) {
			// Sampling nothing... just take it easy
			if( m_pData->prevState == KVI_DCC_VOICE_STATE_TALK ) {
				// Just finished sampling
				appendEOF();
				__range_valid(m_pData->soundFd != -1);
				close(m_pData->soundFd);
				m_pData->soundFd = -1;
				m_pData->prevState = KVI_DCC_VOICE_STATE_IDLE;
			}
			if( m_pData->sendHead ) {
				// First send data
				if( !sendSample() ) {
					// Check for incoming data
					if( !receiveData() )
						usleep(300);
				}
			} else {
				// First check for incoming data
				if( !receiveData() ) {
					if( m_bStopRequested ) return;
				}
			}
			// If we have complete buffers, start playback
			if( m_pData->completeBuffers > 0 )
				playbackLoop();
			else
				usleep(100);
		} else {
			// Sampling... check if just started
			if( m_pData->prevState != KVI_DCC_VOICE_STATE_TALK ) {
				// Just started to talk
				__range_valid(m_pData->prevState == KVI_DCC_VOICE_STATE_IDLE);
				int iError;
				if( m_pData->fullDuplex ) {
					if( m_pData->soundFd < 0 )
						iError = openSoundCard(true);
					else
						iError = KVI_ERROR_Success;
				} else iError = openSoundCard(true);
				if( iError != KVI_ERROR_Success ) {
					if( m_pData->soundFd != -1 )
						close(m_pData->soundFd);
					m_pData->soundFd = -1;
					m_pData->tmpBuffer.sprintf(
						_i18n_("WARNING: cannot open sound card for reading (%s)"),
						kvi_getErrorString(iError)
					);
					KviDccEvent *ev = new KviDccEvent(KVI_DCC_EVENT_MSG, m_pData->tmpBuffer.ptr());
					KviApplication::postEvent(m_pData->parent, ev);

					if( m_bStopRequested ) return;
					usleep(5000); // Give time to release the button, process the message and so on...
					continue;     // And retry
				} else m_pData->prevState = KVI_DCC_VOICE_STATE_TALK;
				m_pData->readAdpcm.index   = 0;
				m_pData->readAdpcm.valprev = 0;
			}
			if( m_bStopRequested ) return;
			readSampleFromSoundcard();
			if( !sendSample() ) {
				if( m_bStopRequested ) return;
				// Check for incoming data
				receiveData();
			}
		}
	}
#endif
}

void KviDccVoiceThread::appendEOF()
{
#ifndef COMPILE_WITH_NO_SOUND
	SampleStruct *s = new SampleStruct();
	for( int i = 0; i < 6; i++ )
		s->buffer[i] = 0;

	// Append to the list
	s->next = 0;
	if( m_pData->sendHead ) {
		// Other data in queue
		m_pData->sendTail->next = s;
		m_pData->sendTail       = s;
	} else {
		// No data in queue
		m_pData->sendHead = s;
		m_pData->sendTail = s;
	}
#endif
}

bool KviDccVoiceThread::sendSample()
{
#ifndef COMPILE_WITH_NO_SOUND
	struct timeval t;
	fd_set fset;
	t.tv_sec  = 0;
	t.tv_usec = 1; // We could wait more?
	FD_ZERO(&fset);
	FD_SET(m_socket, &fset);

	if( m_bStopRequested ) return false;

	if( select(m_socket+1, 0, &fset, 0, &t)>0 ) {
		// Get the first available buffer and try to send it
		SampleStruct *s = m_pData->sendHead;
		// m_pData->lastSentSize can be either 0 here...
		int toSend = KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS - m_pData->lastSentSize;
		// ... so toSend is KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS;
		int wrote = write(m_socket, (void *) ((s->buffer) + m_pData->lastSentSize), toSend);

		if( wrote < 0 ) {
			abort(_i18n_("Error while writing to the socket: connection lost"), m_pData->parent);
			return false;
		} else {
			if( wrote == toSend ) {
				m_pData->lastSentSize = 0;
				m_pData->sendHead     = s->next;
				if( !m_pData->sendHead )
					m_pData->sendTail = 0;
				delete s;
			} else m_pData->lastSentSize += wrote; // And finish next time
			return true;
		}
	} else return false;
#endif
	return true; // Never here
}

bool KviDccVoiceThread::receiveData()
{
#ifndef COMPILE_WITH_NO_SOUND
	if( !selectForRead(m_socket) )
		return false;

	// There is data to read
	if( (!m_pData->lastReadSize) && (!m_pData->tempSample) )
		m_pData->tempSample = new SampleStruct(); // New buffer begin
	// If m_lastReadSize is 0...
	int toRead = KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS - m_pData->lastReadSize;
	// ... toread is KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS;
	if( m_bStopRequested ) return false;
	int bRead = read(m_socket, (void *) ((m_pData->tempSample->buffer) + m_pData->lastReadSize), toRead);
	if( bRead <= 0 ) {
		// Oops?
		if( bRead == 0 ) {
			abort(_i18n_("Remote end has closed the connection"), m_pData->parent);
			return false;
		} else { // Error?
			if( (errno == EINTR) || (errno == EAGAIN) )
				return false;
			// Yes... error :(
			int iError;
			switch( errno ) {
				case ECONNREFUSED: iError = KVI_ERROR_ConnectionRefused;     break;
				case ENOTSOCK:     iError = KVI_ERROR_KernelNetworkingPanic; break;
				case ETIMEDOUT:    iError = KVI_ERROR_ConnectionTimedOut;    break;
				case ENETUNREACH:  iError = KVI_ERROR_NetworkUnreachable;    break;
				case EPIPE:        iError = KVI_ERROR_BrokenPipe;            break;
				// Unhandled error; pass errno to the strerror function
				default:           iError = -errno;                          break;
			}
			// Post the error event to the parent and exit
			m_pData->tmpBuffer.sprintf(_i18n_("READ ERROR: %s"), kvi_getErrorString(iError));
			abort(m_pData->tmpBuffer.ptr(), m_pData->parent);
			return false;
		}
	} else {
		if( bRead == toRead ) {
			m_pData->lastReadSize = 0; // Next time start a new buffer
			// Append the buf
			m_pData->tempSample->next = 0;
			if( m_pData->recvHead ) { // Other data in queue
				m_pData->recvTail->next = m_pData->tempSample;
				m_pData->recvTail       = m_pData->tempSample;
			} else { // No data in queue
				m_pData->recvHead = m_pData->tempSample;
				m_pData->recvTail = m_pData->tempSample;
			}
			// Check if it was an EOF
			if( m_pData->tempSample->buffer[0] == 0 ) {
				if( m_pData->tempSample->buffer[1] == 0 ) {
					if( m_pData->tempSample->buffer[2] == 0 ) {
						if( m_pData->tempSample->buffer[3] == 0 ) {
							if( m_pData->tempSample->buffer[4] == 0 ) {
								if( m_pData->tempSample->buffer[5] == 0 )
									m_pData->completeBuffers++;
							}
						}
					}
				}
			}
			m_pData->recvQueueCount++;

			m_pData->tmpBuffer.sprintf(_i18n_("Buffered: %d milliseconds"), m_pData->recvQueueCount * 128);
			KviDccEvent *ev = new KviDccEvent(KVI_DCC_EVENT_BUFFEREDTIME, m_pData->tmpBuffer.ptr());
			KviApplication::postEvent(m_pData->parent, ev);

			m_pData->tempSample = 0;
		} else m_pData->lastReadSize += bRead; // Continue here
	}
#endif
	return true;
}

void KviDccVoiceThread::playbackLoop()
{
#ifndef COMPILE_WITH_NO_SOUND
	m_pData->playMutex->lock();
	if( (m_pData->state == KVI_DCC_VOICE_STATE_TALK) && !m_pData->fullDuplex ) {
		m_pData->playMutex->unlock();
		return;
	}
	m_pData->bPlaying = true;
	m_pData->playMutex->unlock();

	KviDccEvent *ev = new KviDccEvent(KVI_DCC_EVENT_DISABLETALK, "");
	if( !m_pData->fullDuplex )
		KviApplication::postEvent(m_pData->parent, ev);

	m_pData->writeAdpcm.index   = 0;
	m_pData->writeAdpcm.valprev = 0;

	int iError;
	if( m_pData->fullDuplex ) {
		if( m_pData->soundFd < 0 )
			iError = openSoundCard(true);
		else
			iError = KVI_ERROR_Success;
	} else iError = openSoundCard(false);
	if( iError != KVI_ERROR_Success ) {
		if( m_pData->soundFd != -1 )
			close(m_pData->soundFd);
		m_pData->soundFd = -1;
		m_pData->tmpBuffer.sprintf(_i18n_("WARNING: cannot open sound card for writing (%s)"), kvi_getErrorString(iError));
		KviDccEvent *ev = new KviDccEvent(KVI_DCC_EVENT_MSG, m_pData->tmpBuffer.ptr());
		KviApplication::postEvent(m_pData->parent, ev);

		if( m_bStopRequested ) return;
		usleep(1000); // Give time to release the button, process the message and so on...
	} else {
		bool notEof = true;
		while( m_pData->recvHead && notEof ) {
			SampleStruct *s = m_pData->recvHead;
			// Check if it is EOF for us.
			// Gnarly hack but it works (6 bytes 0 is a complete rumour);
			if( s->buffer[0] == 0 ) {
				if( s->buffer[1] == 0 ) {
					if( s->buffer[2] == 0 ) {
						if( s->buffer[3] == 0 ) {
							if( s->buffer[4] == 0 ) {
								if( s->buffer[5] == 0 ) {
									notEof = false;
									m_pData->completeBuffers--;
								}
							}
						}
					}
				}
			}
			if( notEof ) { // Play the sample
				if( m_pData->use090CompatibleCompression ) {
					m_pData->writeAdpcm.index   = 0;
					m_pData->writeAdpcm.valprev = 0;
				}
				ADPCM_uncompress(
					s->buffer, (short *) (m_pData->soundWriteBuffer),
					KVI_WRITE_BUF_SIZE_IN_SHORTS, &(m_pData->writeAdpcm)
				);
				if( m_bStopRequested ) return;
				write(m_pData->soundFd, (void *) (m_pData->soundWriteBuffer), KVI_WRITE_BUF_SIZE_IN_CHARS);
				// Do something while playing
				if( m_pData->sendHead ) {
					if( !sendSample() ) {
						if( !receiveData() ) {
							usleep(50);
						}
					}
				} else {
					receiveData();
				}
			}
			m_pData->recvHead = s->next;
			if( !(m_pData->recvHead) )
				m_pData->recvTail = 0;
			m_pData->recvQueueCount--;
			delete s;
		}
		// Close the soundcard
		if( m_bStopRequested ) return;
		usleep(100);
		close(m_pData->soundFd);
		m_pData->soundFd = -1;
	}

	m_pData->tmpBuffer.sprintf(_i18n_("Buffered: %d milliseconds"), m_pData->recvQueueCount * 128);
	ev = new KviDccEvent(KVI_DCC_EVENT_BUFFEREDTIME, m_pData->tmpBuffer.ptr());
	KviApplication::postEvent(m_pData->parent, ev);

	ev = new KviDccEvent(KVI_DCC_EVENT_ENABLETALK, "");
	KviApplication::postEvent(m_pData->parent, ev);

	m_pData->playMutex->lock();
	m_pData->bPlaying = false;
	m_pData->playMutex->unlock();
#endif
}

int KviDccVoiceThread::openSoundCard(bool read)
{
	return KviDccVoiceThread::openSoundCard(m_pData, read);
}

int KviDccVoiceThread::openSoundCard(KviDccVoiceData *data, bool read)
{
#ifndef COMPILE_WITH_NO_SOUND
	if( data->soundFd != -1 )
		return KVI_ERROR_SoundCardAlreadyOpen;
	if( data->fullDuplex ) {
		if( data->soundDevice.hasData() )
			data->soundFd = open(data->soundDevice.ptr(), O_RDWR);
		if( data->soundFd < 0 )
			data->soundFd = open(KVI_AUDIO_DEVICE, O_RDWR);
	} else {
		if( data->soundDevice.hasData() )
			data->soundFd = open(data->soundDevice.ptr(), read ? O_RDONLY : O_WRONLY);
		if( data->soundFd < 0 )
			data->soundFd = open(KVI_AUDIO_DEVICE, read ? O_RDONLY : O_WRONLY);
	}
	if( data->soundFd < 0 )
		return KVI_ERROR_CanNotOpenDevDsp;
	static int frag = KVI_FRAG_SIZE;
	if( ioctl(data->soundFd, SNDCTL_DSP_SETFRAGMENT, &frag) < 0 )
		return KVI_ERROR_CanNotInitializeSoundCard;
	static int fmt = KVI_FORMAT;
	if( ioctl(data->soundFd, SNDCTL_DSP_SETFMT, &fmt) < 0 )
		return KVI_ERROR_CanNotInitializeSoundCard;
	static int chans = KVI_NUM_CHANNELS;
	if( ioctl(data->soundFd, SNDCTL_DSP_CHANNELS, &chans) < 0 )
		return KVI_ERROR_CanNotInitializeSoundCard;
	static int speed = KVI_SPEED;
	if( ioctl(data->soundFd, SNDCTL_DSP_SPEED, &speed) < 0 )
		return KVI_ERROR_CanNotInitializeSoundCard;
	return KVI_ERROR_Success;
#else
	return KVI_ERROR_CompiledWithNoSound;
#endif
}

void KviDccVoiceThread::readSampleFromSoundcard()
{
#ifndef COMPILE_WITH_NO_SOUND
	// Read a sample (2048 bytes).
	// Should be blocking in this thread
	if( m_bStopRequested ) return;
	read(m_pData->soundFd, (void *) m_pData->soundReadBuffer, KVI_READ_BUF_SIZE_IN_CHARS);

	// Compress
	if( m_pData->use090CompatibleCompression ) {
		m_pData->readAdpcm.index   = 0;
		m_pData->readAdpcm.valprev = 0;
	}

	SampleStruct *s = new SampleStruct();
	// Compress (1024 shorts to 512 byte samples)
	ADPCM_compress((short *) (m_pData->soundReadBuffer), s->buffer, KVI_READ_BUF_SIZE_IN_SHORTS, &(m_pData->readAdpcm));
	// Append to the list
	s->next = 0;

	if( m_pData->sendHead ) {
		// Other data in queue
		m_pData->sendTail->next = s;
		m_pData->sendTail       = s;
	} else {
		// No data in queue
		m_pData->sendHead = s;
		m_pData->sendTail = s;
	}
#endif
}

////////////////////////////////////////////////////////////////////////////////

KviDccVoiceAcceptThread::KviDccVoiceAcceptThread(KviDccVoiceData *data)
	: KviDccVoiceThread(data)
{
	// Nothing here
}

KviDccVoiceAcceptThread::~KviDccVoiceAcceptThread()
{
	// Nothing here
}

/**
 * Accept a DCC VOICE request from another user on IRC.
 * Specific routine for accepting DCC Voice requests
 * from users on IRC.
 * A user sent us a CTCP DCC with a port and an address.
 * Now we try to contact that host and initiate a DCC Voice.
 */
void KviDccVoiceAcceptThread::run()
{
	// Prepare the target data
	struct in_addr     inAddress;
	struct sockaddr_in hostSockAddr;

	inAddress.s_addr        = m_pData->uAddress;
	hostSockAddr.sin_family = AF_INET;
	hostSockAddr.sin_port   = htons(m_pData->uPort);
	hostSockAddr.sin_addr   = inAddress;

	// Let's go...
	m_socket = socket(PF_INET, SOCK_STREAM, 0);

	if( m_socket < 0 ) {
		abort(_i18n_("Unable to create a stream socket. DCC voice failed"), m_pData->parent);
		return;
	}

	if( fcntl(m_socket, F_SETFL, O_NONBLOCK) < 0 ) {
		abort(_i18n_("Unable to create a non-blocking stream socket. DCC voice failed"), m_pData->parent);
		return;
	}

	if( m_bStopRequested ) return;

	// Ok... now try to connect
	int iError = connect((struct sockaddr *) (&hostSockAddr), sizeof(hostSockAddr));

	if( iError != KVI_ERROR_Success ) {
		m_pData->tmpBuffer.sprintf(_i18n_("CONNECT ERROR: %s"), kvi_getErrorString(iError));
		abort(m_pData->tmpBuffer.ptr(), m_pData->parent);
		return;
	}

	// Now wait for connection...
	KviStr tmp(KviStr::Format, _i18n_("Connecting to %s on port %u"), m_pData->szAddress.ptr(), m_pData->uPort);
	KviDccEvent *ev = new KviDccEvent(KVI_DCC_EVENT_MSG, tmp.ptr());
	KviApplication::postEvent(m_pData->parent, ev);

	iError = waitForOutgoingConnection();
	if( iError != KVI_ERROR_Success ) {
		m_pData->tmpBuffer.sprintf(_i18n_("CONNECT ERROR: %s"), kvi_getErrorString(iError));
		abort(m_pData->tmpBuffer.ptr(), m_pData->parent);
		return;
	}

	ev = new KviDccEvent(KVI_DCC_EVENT_MSG, _i18n_("Connection established"));
	KviApplication::postEvent(m_pData->parent, ev);

	eventLoop();
}

////////////////////////////////////////////////////////////////////////////////

KviDccVoiceRequestThread::KviDccVoiceRequestThread(KviDccVoiceData *data)
	: KviDccVoiceThread(data)
{
	// Nothing here
}

KviDccVoiceRequestThread::~KviDccVoiceRequestThread()
{
	// Nothing here
}

/**
 * Request a DCC Voice from an IRC user
 */
void KviDccVoiceRequestThread::run()
{

	KviDccEvent *ev;
	int newsock = -1;

	struct sockaddr_in sockAddress;
	sockAddress.sin_family      = AF_INET;
	sockAddress.sin_port        = htons(m_pData->uPortToListenOn);
	sockAddress.sin_addr.s_addr = INADDR_ANY;

	// Let's go...
	m_socket = socket(PF_INET, SOCK_STREAM, 0);

	if( m_socket < 0 ) {
		abort(_i18n_("Unable to create a listening socket. DCC voice failed"), m_pData->parent);
		return;
	}

	if( m_bStopRequested ) return;

	if( (bind(m_socket, (struct sockaddr *) &sockAddress, sizeof(sockAddress)) < 0) || (listen(m_socket, 100) < 0) ) {
		abort(_i18n_("Unable to set up a listening socket. DCC voice failed"), m_pData->parent);
		return;
	}

	socklen_t iSize = sizeof(sockAddress);
	getsockname(m_socket, (struct sockaddr *) &sockAddress, &iSize);
	m_pData->uPort = ntohs(sockAddress.sin_port);

	m_pData->tmpBuffer.sprintf(_i18n_("Listening on port %u"), m_pData->uPort);
	ev = new KviDccEvent(KVI_DCC_EVENT_MSG, m_pData->tmpBuffer.ptr());
	KviApplication::postEvent(m_pData->parent, ev);

	m_pData->tmpBuffer.sprintf(
		"PRIVMSG %s :%cDCC VOICE adpcm %u %u 8000%c",
		m_pData->nick.ptr(), 0x01, ntohl(m_pData->uAddress), m_pData->uPort, 0x01
	);
	ev = new KviDccEvent(KVI_DCC_EVENT_LISTENING, m_pData->tmpBuffer.ptr());
	KviApplication::postEvent(m_pData->parent, ev);

	newsock = waitForIncomingConnection(&(m_pData->uAddress), &(m_pData->uPort), &(m_pData->szAddress));

	close(m_socket); // Close the old socket...
	m_socket = newsock;

	m_pData->tmpBuffer.sprintf(_i18n_("Connected to %s on port %u"), m_pData->szAddress.ptr(), m_pData->uPort);
	ev = new KviDccEvent(KVI_DCC_EVENT_MSG, m_pData->tmpBuffer.ptr());
	KviApplication::postEvent(m_pData->parent, ev);

	eventLoop();
}
