//=============================================================================
//
//   File : optw_theme.cpp
//   Creation date : Sun May 25 2003 18:00:00 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2003 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#include "optw_theme.h"
#include "kvi_xmladdonloader.h"

#include <qlayout.h>
#include <qpushbutton.h>
#include <qlistbox.h>
#include <qlineedit.h>
#include <qtextedit.h>
#include <qregexp.h>
#include <qmessagebox.h>
#include <qdir.h>
#include <qstringlist.h>
#include <qdatetime.h>
#include <qfiledialog.h>
#include <qpopupmenu.h>
#include <qcombobox.h>

#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_config.h"
#include "kvi_fileutils.h"
#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_iconmanager.h"
#include "kvi_internalcmd.h"
#include "kvi_styled_controls.h"

#include <stdlib.h> // rand & srand


KviSaveThemeDialog::KviSaveThemeDialog(QWidget * par)
: QDialog(par)
{
	setCaption(__tr2qs_ctx("Save Current Theme","options"));

	QGridLayout * g = new QGridLayout(this,8,2,4,4);

	QLabel * l;

	l = new QLabel(__tr2qs_ctx("Theme name:","options"),this);
	g->addWidget(l,0,0);

	m_pThemeNameEdit = new QLineEdit(this);
	g->addMultiCellWidget(m_pThemeNameEdit,0,0,1,2);
	connect(m_pThemeNameEdit,SIGNAL(textChanged(const QString &)),this,SLOT(themeNameChanged(const QString &)));


	l = new QLabel(__tr2qs_ctx("Version:","options"),this);
	g->addWidget(l,1,0);

	m_pVersionEdit = new QLineEdit(this);
	g->addMultiCellWidget(m_pVersionEdit,1,1,1,2);


	l = new QLabel(__tr2qs_ctx("Author:","options"),this);
	g->addWidget(l,2,0);

	m_pAuthorEdit = new QLineEdit(this);
	g->addMultiCellWidget(m_pAuthorEdit,2,2,1,2);

	l = new QLabel(__tr2qs_ctx("Description:","options"),this);
	g->addWidget(l,3,0);

	m_pDescriptionEdit = new QTextEdit(this);
	g->addMultiCellWidget(m_pDescriptionEdit,3,4,1,2);
	
	m_pIncludeMsgcolors = new KviStyledCheckBox( __tr2qs_ctx("Include message colors information","options"), this );
	m_pIncludeMsgcolors->setChecked(TRUE);
	g->addMultiCellWidget(m_pIncludeMsgcolors,5,5,0,1);
	
	l = new QLabel(__tr2qs_ctx("Package format:","options"),this);
	g->addWidget(l,6,0);

	m_pFormatCombo = new QComboBox(this);
	m_pFormatCombo->insertItem(__tr2qs_ctx("XML single-file format","options"),0);
	m_pFormatCombo->insertItem(__tr2qs_ctx("New multi-file format","options"),1);
	m_pFormatCombo->insertItem(__tr2qs_ctx("Old format","options"),2);
	g->addWidget(m_pFormatCombo,6,1);

	QHBox * h = new QHBox(this);
	h->setSpacing(7);
	g->addWidget(h,7,2);
	m_pOkButton = new QPushButton(__tr2qs_ctx("OK","options"),h);
	m_pOkButton->setEnabled(false);
	connect(m_pOkButton,SIGNAL(clicked()),this,SLOT(saveTheme()));
	m_pOkButton->setIconSet(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ACCEPT)));

	QPushButton * pb = new QPushButton(__tr2qs_ctx("Cancel","options"),h);
	connect(pb,SIGNAL(clicked()),this,SLOT(reject()));
	pb->setIconSet(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_DISCARD)));

}

KviSaveThemeDialog::~KviSaveThemeDialog()
{

}

void KviSaveThemeDialog::themeNameChanged(const QString &txt)
{
	m_pOkButton->setEnabled(!txt.isEmpty());
}

void KviSaveThemeDialog::saveThemeInOldFormat(KviThemeInfo& sto)
{
	if(!KviFileUtils::directoryExists(sto.szAbsoluteDirectory))
	{
		g_pApp->getLocalKvircDirectory(sto.szAbsoluteDirectory,KviApp::Themes,sto.szSubdirectory,true);
		if(!KviFileUtils::makeDir(sto.szAbsoluteDirectory))
		{
			QMessageBox::critical(this,__tr2qs_ctx("Save Theme - KVIrc","options"),__tr2qs_ctx("Unable to create theme directory.","options"),
				QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
			return;
		}
	}
	g_pApp->saveTheme(sto);
	// write down the images

	QString szMsg = __tr2qs_ctx("Theme saved successfully to ","options");
	szMsg += sto.szAbsoluteDirectory;

	QMessageBox::information(this,__tr2qs_ctx("Save Theme - KVIrc","options"),szMsg,QMessageBox::Ok,
		QMessageBox::NoButton,QMessageBox::NoButton);
}

void KviSaveThemeDialog::saveThemeInNewFormat(KviThemeInfo& sto)
{
	QString szBaseDir;
	QFileDialog * d = new QFileDialog(this);
	d->setCaption(__tr2qs_ctx("Choose save directory","options"));
	d->setMode(QFileDialog::DirectoryOnly);
	if(d->exec() == QDialog::Accepted)
	{
		szBaseDir = d->selectedFile();
		KviFileUtils::adjustFilePath(szBaseDir);
		delete d;
		if(szBaseDir.isEmpty())
		{
			QMessageBox::critical(this,__tr2qs_ctx("Save Theme - KVIrc","options"),__tr2qs_ctx("You must select save directory","options"),
				QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
			return;
		}
	} else {
		delete d;
		QMessageBox::critical(this,__tr2qs_ctx("Save Theme - KVIrc","options"),__tr2qs_ctx("You must select save directory","options"),
			QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
		return;	
	}
	KviQString::ensureLastCharIs(szBaseDir,KVI_PATH_SEPARATOR_CHAR);
	sto.szAbsoluteDirectory=szBaseDir+sto.szSubdirectory;
	KviQString::ensureLastCharIs(sto.szAbsoluteDirectory,KVI_PATH_SEPARATOR_CHAR);
	QString szThemeDir=sto.szAbsoluteDirectory;
	sto.szAbsoluteDirectory+="files";
	
	if(!KviFileUtils::makeDir(sto.szAbsoluteDirectory))
	{
		QMessageBox::critical(this,__tr2qs_ctx("Save Theme - KVIrc","options"),__tr2qs_ctx("Unable to create theme files directory.","options"),
			QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
		return;
	}
	
	g_pApp->saveTheme(sto);

	if(!KviFileUtils::makeDir(szThemeDir+"utils"))
	{
		QMessageBox::critical(this,__tr2qs_ctx("Save Theme - KVIrc","options"),__tr2qs_ctx("Unable to create theme utils directory.","options"),
			QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
		return;
	}
	
	QString szInstallerScript;
	g_pApp->getGlobalKvircDirectory(szInstallerScript,KviApp::DefScript,"installer.kvs");
	if(!KviFileUtils::fileExists(szInstallerScript))
	{
		QMessageBox::critical(this,__tr2qs_ctx("Save Theme - KVIrc","options"),__tr2qs_ctx("Cannot find installer.kvs.","options"),
			QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
		return;
	}
	
	if(!KviFileUtils::copyFile(szInstallerScript,szThemeDir+"utils/installer.kvs"))
	{
		QMessageBox::critical(this,__tr2qs_ctx("Save Theme - KVIrc","options"),__tr2qs_ctx("Cannot write installer.kvs.","options"),
			QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
		return;
	}
	
	g_pApp->getGlobalKvircDirectory(szInstallerScript,KviApp::DefScript,"theme-install.kvs");
	QString szMainScript=szThemeDir+"install.kvs";
	if(!KviFileUtils::copyFile(szInstallerScript,szMainScript))
	{
		QMessageBox::critical(this,__tr2qs_ctx("Save Theme - KVIrc","options"),__tr2qs_ctx("Cannot write install.kvs.","options"),
			QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
		return;
	}
	QFile installFile(szMainScript);
	if(!installFile.open(IO_ReadOnly))
	{
		QMessageBox::critical(this,__tr2qs_ctx("Save Theme - KVIrc","options"),__tr2qs_ctx("Cannot open install.kvs for reading.","options"),
			QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
		return;
	}
	
	QCString szFileContents;
	szFileContents=installFile.readAll();
	installFile.close();
	
	sto.szName.replace("\"","\\\"");

	szFileContents.replace("@THEMENAME@",sto.szName.utf8().data());
//		szFileContents.replace("@THEMENAME@",sto.szName.utf8().data());
	sto.szDescription.replace("\"","\\\"");
	if(sto.szDescription.isEmpty()) sto.szDescription="empty";
	szFileContents.replace("@THEMEDESCRIPTION@",sto.szDescription.utf8().data());
	sto.szVersion.replace("\"","\\\"");
	szFileContents.replace("@THEMEVERSION@",sto.szVersion.utf8().data());
	szFileContents.replace("@SUBDIR@",sto.szSubdirectory.utf8().data());
	
	QString szAliasName="";
	QString m_szId;
	
	struct timeval tv;
	kvi_gettimeofday(&tv,0);
	
	KviQString::sprintf(m_szId,"theme_%d%d%d",tv.tv_usec,rand() % 1000,rand() % 1000);
	szAliasName.prepend(m_szId);
	szFileContents.replace("@ALIASNAME@",szAliasName.utf8().data());
	if(!installFile.open(IO_WriteOnly))
	{
		QMessageBox::critical(this,__tr2qs_ctx("Save Theme - KVIrc","options"),__tr2qs_ctx("Cannot open install.kvs for writing.","options"),
			QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
		return;
	}
	installFile.writeBlock(szFileContents.data(),szFileContents.length());
	installFile.close();
	
	QString szMsg = __tr2qs_ctx("Theme saved successfully to ","options");
	szMsg += szThemeDir;

	QMessageBox::information(this,__tr2qs_ctx("Save Theme - KVIrc","options"),szMsg,QMessageBox::Ok,
		QMessageBox::NoButton,QMessageBox::NoButton);
}

void KviSaveThemeDialog::saveThemeInXmlFormat(KviThemeInfo& sto)
{
	QString szFilename;
	QFileDialog * d = new QFileDialog(this);
	d->setCaption(__tr2qs_ctx("Choose filaname","options"));
	d->setMode(QFileDialog::AnyFile);
	d->setFilter( "KVIrc XML files (*.kvxml)" );
	if(d->exec() == QDialog::Accepted)
	{
		szFilename = d->selectedFile();
		KviFileUtils::adjustFilePath(szFilename);
		delete d;
		if(szFilename.isEmpty())
		{
			QMessageBox::critical(this,__tr2qs_ctx("Save Theme - KVIrc","options"),__tr2qs_ctx("You must select filename","options"),
				QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
			return;
		}
	} else {
		delete d;
		QMessageBox::critical(this,__tr2qs_ctx("Save Theme - KVIrc","options"),__tr2qs_ctx("You must select filename","options"),
			QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
		return;	
	}
	if(QFileInfo(szFilename).extension()!="kvxml") szFilename.append(".kvxml");
	g_pApp->saveXmlTheme(sto,szFilename);
}

void KviSaveThemeDialog::saveTheme()
{
	KviThemeInfo sto;
	sto.szName = m_pThemeNameEdit->text();
	if(sto.szName.isEmpty())
	{
		QMessageBox::critical(this,__tr2qs_ctx("Save Theme - KVIrc","options"),__tr2qs_ctx("You must choose a theme name!","options"),QMessageBox::Ok,
			QMessageBox::NoButton,QMessageBox::NoButton);
		return;
	}

	sto.szAuthor = m_pAuthorEdit->text();
	sto.szDescription = m_pDescriptionEdit->text();
	sto.szDate = QDateTime::currentDateTime().toString();
	sto.szVersion = m_pVersionEdit->text();
	sto.szKvircVersion = KVI_VERSION;
	sto.bIncludeMsgColors=m_pIncludeMsgcolors->isChecked();
	
	if(sto.szVersion.isEmpty())sto.szVersion = "1.0.0";

	sto.szSubdirectory = sto.szName + QString("-") + sto.szVersion;
	sto.szSubdirectory.replace(QRegExp("[ \\\\/:][ \\\\/:]*"),"_");

	switch(m_pFormatCombo->currentItem())
	{
		case 0:
			saveThemeInXmlFormat(sto);
			break;
		case 1:
			saveThemeInNewFormat(sto);
			break;
		case 2:
			saveThemeInOldFormat(sto);
			break;
	}

	accept();
}




KviThemeListBoxItem::KviThemeListBoxItem(QListBox * box,KviThemeInfo * inf)
: QListBoxText(box)
{
	QString szTxt = inf->szName;
	szTxt += " ";
	szTxt += inf->szVersion;
	setText(szTxt);
	m_pThemeInfo = inf;
}

KviThemeListBoxItem::~KviThemeListBoxItem()
{
	delete m_pThemeInfo;
}



KviThemeOptionsWidget::KviThemeOptionsWidget(QWidget * parent)
: KviOptionsWidget(parent,"theme_options_widget")
{
	m_pSaveThemeDialog = 0;

	createLayout(11,2);

	QString szDir;
	g_pApp->getLocalKvircDirectory(szDir,KviApp::Themes);
	addLabel(0,0,1,0,__tr2qs_ctx("To apply a theme select it and click \"Ok\" or \"Apply\".<br>" \
	"Your theme dir is:<br><b>","options") + szDir);
	m_pListBox = new QListBox(this);
	layout()->addMultiCellWidget(m_pListBox,1,10,0,0);
	connect(m_pListBox,SIGNAL(highlighted(QListBoxItem *)),this,SLOT(loadThemeInfo(QListBoxItem *)));

	m_pNameLabel = new QLabel(this);
	m_pNameLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	layout()->addWidget(m_pNameLabel,1,1);

	m_pVersionLabel = new QLabel(this);
	m_pVersionLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	layout()->addWidget(m_pVersionLabel,2,1);

	m_pAuthorLabel = new QLabel(this);
	m_pAuthorLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	layout()->addWidget(m_pAuthorLabel,3,1);

	m_pDateLabel = new QLabel(this);
	m_pDateLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	layout()->addWidget(m_pDateLabel,4,1);

	m_pDescriptionLabel = new QLabel(this);
	m_pDescriptionLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	m_pDescriptionLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
	layout()->addWidget(m_pDescriptionLabel,5,1);

	QPushButton * pb = new QPushButton(__tr2qs_ctx("Refresh Theme List","options"),this);
	layout()->addWidget(pb,6,1);
	connect(pb,SIGNAL(clicked()),this,SLOT(fillThemeBox()));
	
	pb = new QPushButton(__tr2qs_ctx("Install XML theme pack","options"),this);
	layout()->addWidget(pb,7,1);
	connect(pb,SIGNAL(clicked()),this,SLOT(installFromXml()));

	pb = new QPushButton(__tr2qs_ctx("Save Current Theme...","options"),this);
	layout()->addWidget(pb,8,1);
	connect(pb,SIGNAL(clicked()),this,SLOT(saveCurrentTheme()));
	
	pb = new QPushButton(__tr2qs_ctx("Delete Selected Theme","options"),this);
	layout()->addWidget(pb,9,1);
	connect(pb,SIGNAL(clicked()),this,SLOT(deleteTheme()));

	pb = new QPushButton(__tr2qs_ctx("Get More Themes...","options"),this);
	layout()->addWidget(pb,10,1);
	connect(pb,SIGNAL(clicked()),this,SLOT(getMoreThemes()));


	layout()->setRowStretch(5,1);
	layout()->setColStretch(1,1);

	fillThemeBox();
}

KviThemeOptionsWidget::~KviThemeOptionsWidget()
{
	if(m_pSaveThemeDialog)
	{
		delete m_pSaveThemeDialog;
		m_pSaveThemeDialog = 0;
	}
}

void KviThemeOptionsWidget::deleteTheme()
{
	KviThemeListBoxItem* pItem=(KviThemeListBoxItem* )(m_pListBox->selectedItem());
	if(pItem)
	{
		KviThemeInfo * pInfo=pItem->themeInfo();
		QDir d(pInfo->szAbsoluteDirectory);
		KviQString::ensureLastCharIs(pInfo->szAbsoluteDirectory,KVI_PATH_SEPARATOR_CHAR);
		QStringList sl = d.entryList(QDir::Files);
		for(QStringList::Iterator it=sl.begin();it != sl.end();it++)
		{
			KviFileUtils::removeFile(pInfo->szAbsoluteDirectory+*it);
		}
		KviFileUtils::removeDir(pInfo->szAbsoluteDirectory);
		fillThemeBox();
	}
}

void KviThemeOptionsWidget::installFromXml()
{
	QString szFilename;
	QFileDialog * d = new QFileDialog(this);
	d->setCaption(__tr2qs_ctx("Choose filaname","options"));
	d->setMode(QFileDialog::ExistingFile);
	d->setFilter( "KVIrc XML files (*.kvxml)" );
	if(d->exec() == QDialog::Accepted)
	{
		szFilename = d->selectedFile();
		KviFileUtils::adjustFilePath(szFilename);
		delete d;
		if(szFilename.isEmpty())
		{
			QMessageBox::critical(this,__tr2qs_ctx("Open Theme - KVIrc","options"),__tr2qs_ctx("You must select file","options"),
				QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
			return;
		}
	} else {
		delete d;
		return;
	}
	KviXmlAddonLoader * pLoader = new KviXmlAddonLoader(szFilename,this);
	pLoader->exec();
	fillThemeBox();
}

void KviThemeOptionsWidget::getMoreThemes()
{
	if(!g_pFrame)return;
	g_pFrame->executeInternalCommand(KVI_INTERNALCOMMAND_OPENURL_KVIRC_THEMES);
}

void KviThemeOptionsWidget::loadThemeInfo(QListBoxItem * i)
{
	KviThemeListBoxItem * it = (KviThemeListBoxItem *)i;

	if(!it)
	{
		m_pNameLabel->setText("");
		m_pAuthorLabel->setText("");
		m_pVersionLabel->setText("");
		m_pDateLabel->setText("");
		m_pDescriptionLabel->setText("");
		return;
	}

	if(it->text().isEmpty())return; //hm, not set yet ?

	if(it->themeInfo()->szAbsoluteDirectory.isEmpty())
	{
		m_pNameLabel->setText(__tr2qs_ctx("Current Theme","options"));
		m_pAuthorLabel->setText("");
		m_pVersionLabel->setText("");
		m_pDateLabel->setText("");
		m_pDescriptionLabel->setText(__tr2qs_ctx("<p>The active theme settings</p>","options"));
		return;
	}

	m_pNameLabel->setText(QString(__tr2qs_ctx("<p>Theme: %1</p>","options")).arg(it->themeInfo()->szName));
	m_pVersionLabel->setText(QString(__tr2qs_ctx("<p>Version: %1</p>","options")).arg(it->themeInfo()->szVersion));
	m_pDateLabel->setText(QString(__tr2qs_ctx("<p>Date: %1</p>","options")).arg(it->themeInfo()->szDate));
	m_pAuthorLabel->setText(QString(__tr2qs_ctx("<p>Author: %1</p>","options")).arg(it->themeInfo()->szAuthor));
	m_pDescriptionLabel->setText(it->themeInfo()->szDescription);
}

void KviThemeOptionsWidget::saveCurrentTheme()
{
	if(m_pSaveThemeDialog)return;
	m_pSaveThemeDialog = new KviSaveThemeDialog(this);
	m_pSaveThemeDialog->exec();
	if(m_pSaveThemeDialog)
	{
		delete m_pSaveThemeDialog;
		m_pSaveThemeDialog = 0;
		fillThemeBox();
	}
}

void KviThemeOptionsWidget::fillThemeBox(const QString &szDir)
{
	QDir d(szDir);

	QStringList sl = d.entryList(QDir::Dirs);

	for(QStringList::Iterator it = sl.begin();it != sl.end();++it)
	{
		if(*it == ".")continue;
		if(*it == "..")continue;

		QString szTest = szDir;
		szTest += KVI_PATH_SEPARATOR_CHAR;
		szTest += *it;

		KviThemeInfo * inf = new KviThemeInfo;
		if(g_pApp->loadThemeInfo(szTest,*inf))
		{
			KviThemeListBoxItem * item = 0;
			item = new KviThemeListBoxItem(m_pListBox,inf);
		} else {
			delete inf;
		}
	}
}

void KviThemeOptionsWidget::fillThemeBox()
{
	m_pListBox->clear();

	KviThemeInfo * ti = new KviThemeInfo;
	ti->szName = __tr2qs_ctx("Current Theme","options");

	KviThemeListBoxItem * itewrq = new KviThemeListBoxItem(m_pListBox,ti);

	QString szDir;
	g_pApp->getGlobalKvircDirectory(szDir,KviApp::Themes);
	fillThemeBox(szDir);
	g_pApp->getLocalKvircDirectory(szDir,KviApp::Themes);
	fillThemeBox(szDir);

	m_pListBox->setCurrentItem(itewrq);
}

void KviThemeOptionsWidget::commit()
{
	KviOptionsWidget::commit();

	KviThemeListBoxItem * it = (KviThemeListBoxItem *)m_pListBox->item(m_pListBox->currentItem());
	if(!it)return;

	QString szPath = it->themeInfo()->szAbsoluteDirectory;
	if(szPath.isEmpty())return;

	g_pApp->loadTheme(szPath);

	// make sure that we do not re-commit again
	it = 0;
	QString szCur = __tr2qs_ctx("Current Theme","options");
	for(unsigned int i = 0;i< m_pListBox->count();i++)
	{
		KviThemeListBoxItem * daItem = (KviThemeListBoxItem *)m_pListBox->item(i);
		if(daItem)
		{
			if(daItem->themeInfo()->szName == szCur)
			{
				it = daItem;
				break;
			}
		}
	}
	
	if(!it)fillThemeBox();
	else m_pListBox->setCurrentItem(it);
}

#include "m_optw_theme.moc"
