//
//   Charpicker plugin for the KVIrc IRC client
//   Copyright (C) 1999-2000 Stefanek Szymon (kvirc@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __KVIRC_PLUGIN__

#include "kvirc_plugin.h"

#include "libkvicharpicker.h"
#include "kvi_input.h"
#include "kvi_locale.h"
#include "kvi_options.h"

#include <qtooltip.h>

static QList<KviCharPicker> * m_pCharPickerList = 0;
static bool m_bInCleanup = false;

KviCharPickerButton::KviCharPickerButton(QWidget * parent,char display)
: QPushButton(parent)
{
	KviStr str;
	str.append(display);
	setText(str.ptr());
	connect(this,SIGNAL(clicked()),this,SLOT(hasBeenClicked()));
}

KviCharPickerButton::~KviCharPickerButton()
{
}

void KviCharPickerButton::hasBeenClicked()
{
	emit charPicked(text());
}

KviCharPicker::KviCharPicker(QSplitter * parent,KviWindow * wnd)
: QScrollView(parent,"char_picker_widget")
{
	m_pWnd = wnd;
	setFont(g_pOptions->m_fntInput);
	m_pCharPickerList->append(this);
	setFrameStyle(QFrame::Sunken | QFrame::Panel);
	enableClipper(true);

	m_pCloseButton = new QPushButton(__tr("X"),this);
	QToolTip::add(m_pCloseButton,__tr("Close"));
	setCornerWidget(m_pCloseButton);
	connect(m_pCloseButton,SIGNAL(clicked()),this,SLOT(closeClicked()));
	m_pCloseButton->show();

	m_pUpdateButton = new QPushButton(__tr("Update font"),viewport());
	m_pUpdateButton->resize(m_pUpdateButton->sizeHint());
	m_pUpdateButton->show();
	addChild(m_pUpdateButton,0,0);
	connect(m_pUpdateButton,SIGNAL(clicked()),this,SLOT(updateFont()));

	int baseY = m_pUpdateButton->height();

	m_pBtn[0] = new KviCharPickerButton(viewport(),'X');
	m_pBtn[0]->setEnabled(false);
	QSize min = m_pBtn[0]->sizeHint();
	m_pBtn[0]->resize(min);
	m_pBtn[0]->show();
	addChild(m_pBtn[0],0,baseY);
	int maxY = baseY;
	int maxX = 0;
	for(int i=1;i<256;i++)
	{
//		printf("Adding charpicker %d at %d,%d",i,(i % 16) * min.width(),(i / 16) * min.height());
		m_pBtn[i] = new KviCharPickerButton(viewport(),i);
		m_pBtn[i]->resize(min);
		int ax = (i % 16) * min.width();
		int ay = (i / 16) * min.height() + baseY;
		if(maxX < (ax + min.width()))maxX = ax + min.width();
		if(maxY < (ay + min.height()))maxY = ay + min.height();
//		debug("Adding child at %d,%d",ax,ay);
		addChild(m_pBtn[i],ax,ay);
		m_pBtn[i]->show();
		KviStr tmp(KviStr::Format,"Ascii %d",i);
		QToolTip::add(m_pBtn[i],tmp.ptr());
		connect(m_pBtn[i],SIGNAL(charPicked(const QString &)),this,SLOT(charPicked(const QString &)));
	}
	setVScrollBarMode(QScrollView::Auto);
	setHScrollBarMode(QScrollView::Auto);
	setResizePolicy(QScrollView::Manual);
	resizeContents(maxX,maxY);
}

void KviCharPicker::updateFont()
{
	m_pUpdateButton->setFont(g_pOptions->m_fntInput);
	m_pUpdateButton->resize(m_pUpdateButton->sizeHint());
	int baseY = m_pUpdateButton->height();

	m_pCloseButton->setFont(g_pOptions->m_fntInput);

	m_pBtn[0]->setFont(g_pOptions->m_fntInput);
	QSize min = m_pBtn[0]->sizeHint();
	m_pBtn[0]->resize(min);
	moveChild(m_pBtn[0],0,baseY);
	int maxY = baseY;
	int maxX = 0;
	for(int i=1;i<256;i++)
	{
		m_pBtn[i]->setFont(g_pOptions->m_fntInput);
		m_pBtn[i]->resize(min);
		int ax = (i % 16) * min.width();
		int ay = (i / 16) * min.height() + baseY;
		if(maxX < (ax + min.width()))maxX = ax + min.width();
		if(maxY < (ay + min.height()))maxY = ay + min.height();
		moveChild(m_pBtn[i],ax,ay);
	}
	resizeContents(maxX,maxY);
}

void KviCharPicker::closeClicked()
{
	delete this;
}

KviCharPicker::~KviCharPicker()
{
	if(!m_bInCleanup)
	{
		m_pCharPickerList->removeRef(this); //<--autoDelete disabled
	}
}

void KviCharPicker::charPicked(const QString &str)
{
	KviStr s = str;
	if(m_pWnd->m_pInput && *(s.ptr()))m_pWnd->m_pInput->insertChar(*(s.ptr()));
//	debug("inserted char %d",*(s.ptr()));
}

static KviCharPicker * lookup_charpicker(KviWindow *wnd)
{
	for(KviCharPicker *p = m_pCharPickerList->first();p;p = m_pCharPickerList->next())
	{
		if(p->m_pWnd == wnd)return p;
	}
	return 0;
}


bool charpicker_plugin_command_charpicker(KviPluginCommandStruct *cmd)
{
//#warning "Integer parameter out of range here"
	if(kvirc_plugin_param(cmd,1))
	{
		// Turn it off
		KviCharPicker * p = lookup_charpicker(cmd->window);
		if(p)delete p;
		else {
			cmd->error = KVI_ERROR_InvalidOperation;
			cmd->errorstr = __tr("No charpicker to remove in this window");
			return false;
		}
		return true;
	}

	if(kvirc_plugin_mdi_window_can_add_widget(cmd->window))
	{
		if(!lookup_charpicker(cmd->window))
		{
			KviCharPicker * p = new KviCharPicker(kvirc_plugin_mdi_window_splitter(cmd->window),cmd->window);
			kvirc_plugin_mdi_window_add_widget(p,cmd->window);
			p->show();
		} else {
			cmd->error = KVI_ERROR_InvalidOperation;
			cmd->errorstr = __tr("A charpicker is already in this window");
			return false;
		}
	} else {
		cmd->error = KVI_ERROR_InvalidOperation;
		cmd->errorstr = __tr("Can't add charpicker to this window");
		return false;
	}
	return true;
}

bool charpicker_plugin_init(KviPluginCommandStruct *cmd)
{
	m_pCharPickerList = new QList<KviCharPicker>;
	m_pCharPickerList->setAutoDelete(false);
	kvirc_plugin_register_command(cmd->handle,"CHARPICKER",charpicker_plugin_command_charpicker);
	return true; // success
}

void charpicker_plugin_cleanup()
{
	// Kill all the char picker widgets
	m_bInCleanup = true;
	m_pCharPickerList->setAutoDelete(true);
	delete m_pCharPickerList;
	m_pCharPickerList = 0;
	kvirc_plugin_unregister_meta_object("KviCharPickerButton");
	kvirc_plugin_unregister_meta_object("KviCharPicker");
}

KviPlugin kvirc_plugin =
{
	"charpicker" ,
	"An ASCII character picker" ,
	"0.1" ,
	"Szymon Stefanek <stefanek@tin.it>" ,
	"Allows to add an ascii character\n" \
	"picker to almost any MDI window in KVIrc\n" \
	"exports the /CHARPICKER command",
	charpicker_plugin_init ,
	charpicker_plugin_cleanup ,
	0 ,
	0
};

#include "m_libkvicharpicker.moc"
