//
//   File : kvi_selectors.cpp (/usr/build/NEW_kvirc/kvirc/kvilib/kvi_selectors.cpp)
//   Last major modification : Sat Jan 16 1999 14:57:02 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//#define _KVI_DEBUG_CHECK_RANGE_

#define _KVI_SELECTORS_CPP_

#include "kvi_debug.h"

#include "kvi_selectors.h"
#include "kvi_defines.h"
#include "kvi_malloc.h"

#include "kvi_buttons.h"

#include "kvi_locale.h"
#include "kvi_msgbox.h"

#include <qvalidator.h>
#include <qlayout.h>
#include <qfiledialog.h>
#include <qfileinfo.h>

#include <qobjcoll.h>
#include <ctype.h>

KviLineEdit::KviLineEdit(QWidget * parent,const char *name)
:QLineEdit(parent,name)
{
	setFocusPolicy(QWidget::ClickFocus);
	m_szAltKeyCode = "";
}

KviLineEdit::~KviLineEdit()
{
}

QString KviLineEdit::text() const
{
	KviStr text(QLineEdit::text());
	char * p = (char *)kvi_malloc(text.len()+1);
        qstrcpy(p, text.ptr());
	text = "";
	while(*p){
		if(*p == '~'){
			p++;
			switch(*p){
				case '~': text.append(*p);
				break;
				case 'b': text.append((char)KVI_TEXT_BOLD);
				break;
				case 'c': text.append((char)KVI_TEXT_RESET);
				break;
				case 'r': text.append((char)KVI_TEXT_REVERSE);
				break;
				case 'u': text.append((char)KVI_TEXT_UNDERLINE);
				break;
				default:
					text.append((char)KVI_TEXT_COLOR);
					text.append(*p);
				break;
			}
		} else text.append(*p);
		p++;
	}
	return QString(text.ptr());
}

void KviLineEdit::keyPressEvent(QKeyEvent *e)
{
	if(e->state() & ControlButton){
		switch(e->key()){
			case Qt::Key_B:
				QLineEdit::insert(QString("~b"));
				break;
			case Qt::Key_K:
				QLineEdit::insert(QString("~"));
				break;
			case Qt::Key_O:
				QLineEdit::insert(QString("~c"));
				break;
			case Qt::Key_R:
				QLineEdit::insert(QString("~r"));
				break;
			case Qt::Key_U:
				QLineEdit::insert(QString("~u"));
				break;
			default:
//				e->ignore();
				QLineEdit::keyPressEvent(e);
				break;
		}
		return;
	}

	if(e->state() & AltButton){
		// Qt::Key_Meta seems to substitute Key_Alt on some keyboards
		if((e->key() == Qt::Key_Alt) || (e->key() == Qt::Key_Meta)){
			m_szAltKeyCode = "";
			return;
		} else if((e->ascii() >= '0') && (e->ascii() <= '9')){
			m_szAltKeyCode += e->ascii();
			return;
		}
	}
	QLineEdit::keyPressEvent(e);
}

void KviLineEdit::keyReleaseEvent(QKeyEvent *e)
{
	if((e->key() == Qt::Key_Alt) || (e->key() == Qt::Key_Meta)){
		if(m_szAltKeyCode.hasData()){
			bool bOk;
			char ch = m_szAltKeyCode.toChar(&bOk);
			//(const char *)ch;
			if(bOk && ch > ' '){
				insert(KviStr(ch).ptr());
				e->accept();
			}
		}
		m_szAltKeyCode = "";
	}
	e->ignore();
}

void KviLineEdit::setText(const char * text)
{
	KviStr t;
	while(*text){
		switch(*text){
			case '~':
				t.append("~~");
			break;
			case KVI_TEXT_BOLD:
				t.append("~b");
			break;
			case KVI_TEXT_RESET:
				t.append("~c");
			break;
			case KVI_TEXT_REVERSE:
				t.append("~r");
			break;
			case KVI_TEXT_UNDERLINE:
				t.append("~u");
			break;
			case KVI_TEXT_COLOR:
				t.append('~');
			break;
			default:
				t.append(*text);
			break;
		}
		text++;
	}
	QLineEdit::setText(QString(t.ptr()));
}

void KviLineEdit::insert(const char *newText)
{
	KviStr t;
	while(*newText){
		switch(*newText){
			case '~':
				t.append("~~");
			break;
			case KVI_TEXT_BOLD:
				t.append("~b");
			break;
			case KVI_TEXT_RESET:
				t.append("~c");
			break;
			case KVI_TEXT_REVERSE:
				t.append("~r");
			break;
			case KVI_TEXT_UNDERLINE:
				t.append("~u");
			break;
			case KVI_TEXT_COLOR:
				t.append("~");
			break;
			default:
				t.append(*newText);
			break;
		}
		newText++;
	}
	QLineEdit::insert(QString(t.ptr()));
}

KviIpEditor::KviIpEditor(QWidget * parent,const QString &ipAddr,const char *name)
:QHBox(parent,name)
{

	QFontMetrics fm(font());
	int minWidth = fm.width("000") + 4;

	for(int i=0;i<4;i++)
	{
		m_pEdit[i]  = new QLineEdit(this);
		m_pEdit[i]->installEventFilter(this);
		m_pEdit[i]->setFrame(false);
		m_pEdit[i]->setAlignment(Qt::AlignCenter);
		m_pEdit[i]->setMinimumWidth(minWidth);
		m_pEdit[i]->setMaxLength(3);

		if(i < 3)
		{
			m_pLabel[i] = new QLabel(".",this);
			m_pLabel[i]->setBackgroundMode(QWidget::PaletteBase);
		}
	}

	setFrameStyle(QFrame::Sunken|QFrame::WinPanel);
	if(ipAddr.isNull())
	{
		setAddress("0.0.0.0");
	} else setAddress(ipAddr);
}


KviIpEditor::~KviIpEditor()
{
	
}

void KviIpEditor::setEnabled(bool bEnabled)
{
	QHBox::setEnabled(bEnabled);
	for(int i=0;i<4;i++)
	{
		m_pEdit[i]->setEnabled(bEnabled);
		if(i < 3){
			m_pLabel[i]->setEnabled(bEnabled);
			m_pLabel[i]->setBackgroundMode(isEnabled() ? QWidget::PaletteBase : QWidget::PaletteBackground);
		}
	}
}

/*
	Sets the address displayed by the widget
*/

bool KviIpEditor::setAddress(const QString &ipAddr)
{
	if(ipAddr.isEmpty())return false;

	QCString ip = ipAddr.ascii(); // ip addresses are digits & latin letters abcdef (IpV6)
	ip = ip.stripWhiteSpace();
	const char * c = ip.data();

	if(!c)return false; // huh ?....(should never happen at this point)

	for(int i=0;i<4;i++)
	{
		const char *anchor = c;
		while(isdigit(*c))c++;
		if(c == anchor)return false; // Invalid empty field
		QCString str(anchor,(c - anchor) + 1);
		bool bOk;
		int num = str.toInt(&bOk);
		if(!bOk)return false; // should never happen , but just to be sure
		if((num < 0) || (num > 255))return false; // Invalid field
		m_pEdit[i]->setText(str.data());
		if(i < 3){
			if(*c == '.')c++;
			else return false; // missing separator
		}
	}

	return (*c == '\0'); // trailing garbage check (we could avoid this)
}

/*
	Returns the edited IP address in the string form.
	The string returned is always a valid IP address.
	In the IpV4 mode , blank fields are treated as zeroes
	and leading zeroes are removed.
*/

QString KviIpEditor::address()
{
	QString ret;


	for(int i=0;i<4;i++)
	{
		QString tmp = m_pEdit[i]->text();
		bool bOk;
		int num = tmp.toInt(&bOk);
		if(!bOk)num = 0;
		tmp.setNum(num);
		ret.append(tmp);
		if(i < 3)ret.append(".");
	}
	return ret;
}

bool KviIpEditor::eventFilter(QObject * o,QEvent *e)
{
	if(o->inherits("QLineEdit"))
	{
		if(e->type() == QEvent::KeyPress)
		{
			QString s;
			// find the editor
			int edIdx = -1;
			for(int i=0;i<4;i++)
			{
				if(m_pEdit[i] == o)
				{
					edIdx = i;
					break;
				}
			}
			if(edIdx == -1)return QHBox::eventFilter(o,e); // user added QLineEdit child ?
			int edMax = 3;
			int cursorPos = ((QLineEdit *)o)->cursorPosition();
			switch(((QKeyEvent *)e)->key())
			{
				case Qt::Key_Right:
					s = ((QLineEdit *)o)->text();
					if(((unsigned int)cursorPos) == s.length())
					{
						if(edIdx < edMax)
						{
							m_pEdit[++edIdx]->setCursorPosition(0);
							m_pEdit[edIdx]->setFocus();
							return true;
						}
					}
				break;
				case Qt::Key_Left:
				case Qt::Key_Backspace:
					if(cursorPos == 0)
					{
						if(edIdx > 0)
						{
							s = m_pEdit[--edIdx]->text();
							m_pEdit[edIdx]->setCursorPosition(s.length());
							m_pEdit[edIdx]->setFocus();
							return true;							
						}
					}
					return QHBox::eventFilter(o,e);
				break;
				case Qt::Key_End:
				case Qt::Key_Home:
				case Qt::Key_Delete:
				case Qt::Key_Tab:
					return QHBox::eventFilter(o,e);
				break;
				default:
					// a normal key (this part substitutes a QValidator)
					char c = tolower(((QKeyEvent *)e)->ascii());
					if((c >= '0') && (c <= '9'))
					{
						if(m_pEdit[edIdx]->hasMarkedText())m_pEdit[edIdx]->cut();
						cursorPos = m_pEdit[edIdx]->cursorPosition();
						s = m_pEdit[edIdx]->text();
						s.insert(cursorPos,c);
						bool bOk = false;
						int num = s.toInt(&bOk);
						if(!bOk)return true; //should never happen , but just to be sure
						if((num < 0) || (num > 255))return true; //invalid field
						m_pEdit[edIdx]->setText(s);
						if(num > 25)
						{
							// The focus goes to the next editor
							if(edIdx < edMax)
							{
								m_pEdit[++edIdx]->selectAll();
								m_pEdit[edIdx]->setCursorPosition(0);
								m_pEdit[edIdx]->setFocus();
								return true;
							}
						}
						m_pEdit[edIdx]->cursorRight(false);
					}
					return true;
				break;
			}
		}
	}
	return QHBox::eventFilter(o,e);
}

KviNewStringSelector::KviNewStringSelector(QWidget *parent,const char *description,KviStr *pString,bool bEnabled,int fieldWidth,const char *name)
: QWidget(parent,name)
{
	__range_valid(pString);
	m_pString = pString;

	m_pLabel = new QLabel(_CHAR_2_QSTRING(description),this);
	m_pEdit  = new QLineEdit(this);
	m_pEdit->setText(pString->ptr());
	m_iFieldWidth = (fieldWidth > 0) ? fieldWidth : 300;
	setMinimumWidth(m_iFieldWidth + m_pLabel->sizeHint().width() + 1);
	setMinimumHeight(m_pEdit->sizeHint().height());
	setEnabled(bEnabled);
}

KviNewStringSelector::~KviNewStringSelector()
{
}

void KviNewStringSelector::commitChanges()
{
	*m_pString = m_pEdit->text();
}

void KviNewStringSelector::commitAll(QWidget *w)
{
	QObjectList * l = w->queryList("KviNewStringSelector");
	if(!l)return;
	QObjectListIt it(*l);
	while(it.current())
	{
		((KviNewStringSelector *)it.current())->commitChanges();
		++it;
	}
	delete l;
}
//
//bool KviNewStringSelector::selectionChanged()
//{
//	KviStr tmp = m_pEdit->text();
//	return (!kvi_strEqualCS(m_pString->ptr(),tmp.ptr()));
//}

void KviNewStringSelector::resizeEvent(QResizeEvent *)
{
	m_pEdit->setGeometry(width() - m_iFieldWidth,0,m_iFieldWidth,height());
	m_pLabel->setGeometry(0,0,width() - (m_iFieldWidth + 1),height());
}

QSize KviNewStringSelector::sizeHint()
{
	return QSize(m_pLabel->sizeHint().width() + m_iFieldWidth + 1,m_pEdit->sizeHint().height());
}

void KviNewStringSelector::setEnabled(bool bEnabled)
{
	m_pLabel->setEnabled(bEnabled);
	m_pEdit->setEnabled(bEnabled);
	QWidget::setEnabled(bEnabled);
}



KviNewCharSelector::KviNewCharSelector(QWidget *parent,const char *description,char *pChar,bool bEnabled,int fieldWidth,const char *name)
: QWidget(parent,name)
{
	__range_valid(pChar);
	m_pChar = pChar;

	m_pLabel = new QLabel(_CHAR_2_QSTRING(description),this);
	m_pEdit  = new QLineEdit(this);
	KviStr tmp;
	tmp.append(*pChar);
	m_pEdit->setText(tmp.ptr());
	m_iFieldWidth = (fieldWidth > 0) ? fieldWidth : 300;
	setMinimumWidth(m_iFieldWidth + m_pLabel->sizeHint().width() + 1);
	setMinimumHeight(m_pEdit->sizeHint().height());
	setEnabled(bEnabled);
}

KviNewCharSelector::~KviNewCharSelector()
{
}

void KviNewCharSelector::commitChanges()
{
	KviStr tmp = m_pEdit->text();
	if(tmp.hasData())*m_pChar = *(tmp.ptr());
	else *m_pChar = '/';
}

//bool KviNewCharSelector::selectionChanged()
//{
//	KviStr tmp = m_pEdit->text();
//	return ((*m_pChar) != (*(tmp.ptr())));
//}

void KviNewCharSelector::resizeEvent(QResizeEvent *)
{
	m_pEdit->setGeometry(width() - m_iFieldWidth,0,m_iFieldWidth,height());
	m_pLabel->setGeometry(0,0,width() - (m_iFieldWidth + 1),height());
}

QSize KviNewCharSelector::sizeHint()
{
	return QSize(m_pLabel->sizeHint().width() + m_iFieldWidth + 1,m_pEdit->sizeHint().height());
}

void KviNewCharSelector::setEnabled(bool bEnabled)
{
	m_pLabel->setEnabled(bEnabled);
	m_pEdit->setEnabled(bEnabled);
	QWidget::setEnabled(bEnabled);
}




KviNewIntegerSelector::KviNewIntegerSelector(QWidget *parent,const char *description,void * pNumber,IntSelectorType type,int lowerBound,int upperBound,
	int defValue,bool bEnabled,int fieldWidth,const char *name,int lineStep,const char *suffix)
: QWidget(parent,name)
{
	__range_valid(pNumber);
	m_pNumber = pNumber;
	m_type = type;
	m_pLabel = new QLabel(_CHAR_2_QSTRING(description),this);
//	m_pEdit  = new QLineEdit(this);
	m_pSpinBox = new QSpinBox(this);
	m_iLowBound = lowerBound;
	m_iHighBound = upperBound;
	m_szDefault.setNum(defValue);
//	QIntValidator *v = new QIntValidator(this);
//	v->setRange(lowerBound,upperBound);
	m_pSpinBox->setMinValue(lowerBound);
	m_pSpinBox->setMaxValue(upperBound);
	if(lineStep > 0)m_pSpinBox->setLineStep(lineStep);
	if(*suffix)m_pSpinBox->setSuffix(suffix);
//	m_pEdit->setValidator(v);
//	m_pSpinBox->setValidator(v);

//	KviStr szOriginal = "0";
	switch(type)
	{
/*		case Int :
			szOriginal.setNum(*((int *)(pNumber)));
		break; */
		case Int :
			m_pSpinBox->setValue(*((int *)(pNumber)));
		break;
		case UInt :
			m_pSpinBox->setValue(*((unsigned int *)(pNumber)));
		break;
		case UShort:
			m_pSpinBox->setValue(*((unsigned short int *)(pNumber)));
		break;
		default:
			m_pSpinBox->setValue(*((short int *)(pNumber)));
		break;
	}
//	m_pEdit->setText(szOriginal.ptr());
	m_iFieldWidth = (fieldWidth > 0) ? fieldWidth : 100;
	setMinimumWidth(m_iFieldWidth + m_pLabel->sizeHint().width() + 1);
	setMinimumHeight(m_pSpinBox->sizeHint().height());
	setEnabled(bEnabled);
}

KviNewIntegerSelector::~KviNewIntegerSelector()
{
}

void KviNewIntegerSelector::setEnabled(bool bEnabled)
{
	m_pLabel->setEnabled(bEnabled);
	m_pSpinBox->setEnabled(bEnabled);
	QWidget::setEnabled(bEnabled);
}

void KviNewIntegerSelector::commitChanges()
{
	KviStr szT = m_pSpinBox->cleanText();
	bool bOk = false;
	//Adjust the value
	int value = szT.toInt(&bOk);
	if(!bOk)szT = m_szDefault.ptr();
	if(value < m_iLowBound)szT.setNum(m_iLowBound);
	if(value > m_iHighBound)szT.setNum(m_iHighBound);
	switch(m_type){
		case Int:
			(*((int *)m_pNumber)) = szT.toInt(&bOk);
			if(!bOk)(*((int *)m_pNumber)) = m_szDefault.toInt(&bOk);
		break;
		case UInt:
			(*((unsigned int *)m_pNumber)) = szT.toUInt(&bOk);
			if(!bOk)(*((unsigned int *)m_pNumber)) = m_szDefault.toUInt(&bOk);
		break;
		case UShort:
			(*((unsigned short *)m_pNumber)) = szT.toUShort(&bOk);
			if(!bOk)(*((unsigned short *)m_pNumber)) = m_szDefault.toUShort(&bOk);
		break;
		default:
			(*((short *)m_pNumber)) = szT.toShort(&bOk);
			if(!bOk)(*((short *)m_pNumber)) = m_szDefault.toShort(&bOk);
		break;
	}
}

void KviNewIntegerSelector::commitAll(QWidget *w)
{
	QObjectList * l = w->queryList("KviNewIntegerSelector");
	if(!l)return;
	QObjectListIt it(*l);
	while(it.current())
	{
		((KviNewIntegerSelector *)it.current())->commitChanges();
		++it;
	}
	delete l;
}

//bool KviNewIntegerSelector::selectionChanged()
//{
//	KviStr tmp = m_pEdit->text();
//	return (!kvi_strEqualCS(m_szOriginal.ptr(),tmp.ptr()));
//}

void KviNewIntegerSelector::resizeEvent(QResizeEvent *)
{
	m_pSpinBox->setGeometry(width() - m_iFieldWidth,0,m_iFieldWidth,height());
	m_pLabel->setGeometry(0,0,width() - (m_iFieldWidth + 1),height());
}

QSize KviNewIntegerSelector::sizeHint()
{
	return QSize(m_pLabel->sizeHint().width() + m_iFieldWidth + 1,m_pSpinBox->sizeHint().height());
}




KviNewBoolSelector::KviNewBoolSelector(QWidget *parent,const char *description,bool *pBool,bool bEnabled,const char *name)
: QCheckBox(_CHAR_2_QSTRING(description),parent,name)
{
	m_pBool = pBool;
	setChecked(*pBool);
	setEnabled(bEnabled);
}

KviNewBoolSelector::~KviNewBoolSelector()
{
}

void KviNewBoolSelector::commitChanges()
{
	*m_pBool = isChecked();
}

void KviNewBoolSelector::commitAll(QWidget *w)
{
	QObjectList * l = w->queryList("KviNewBoolSelector");
	if(!l)return;
	QObjectListIt it(*l);
	while(it.current())
	{
		((KviNewBoolSelector *)it.current())->commitChanges();
		++it;
	}
	delete l;
}

//
//bool KviNewBoolSelector::selectionChanged()
//{
//	return (isChecked() != (*m_pBool));
//}
//


KviNewFileSelector::KviNewFileSelector(QWidget * parent,KviStr *valuePtr,bool bMustBeReadable,bool bEnabled)
:QWidget(parent)
{
	QGridLayout * g= new QGridLayout(this,1,2,0,2);
	m_valuePtr = valuePtr;
	m_pValueLabel = new QLabel(valuePtr->ptr(),this);
	m_pValueLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	m_bMustBeReadable = bMustBeReadable;
	g->addWidget(m_pValueLabel,0,0);
	m_pButton = new QPushButton(_CHAR_2_QSTRING(__tr("&Browse...")),this);
	g->addWidget(m_pButton,0,1);
	connect(m_pButton,SIGNAL(clicked()),this,SLOT(changeFile()));
	setMinimumSize(sizeHint());
	setEnabled(bEnabled);
}

KviNewFileSelector::~KviNewFileSelector()
{
}
//
//QSize KviFileSelector::sizeHint()
//{
//	QSize sz=m_pValueLabel->sizeHint();
//	sz.setWidth(sz.width() + m_pButton->sizeHint().width() + 4);
//	return sz;
//}

void KviNewFileSelector::commitChanges()
{
	*m_valuePtr = m_pValueLabel->text();
}
//
//bool KviNewFileSelector::selectionChanged()
//{
//	KviStr tmp = m_pValueLabel->text();
//	return (!kvi_strEqualCS(m_valuePtr->ptr(),tmp.ptr()));
//}

void KviNewFileSelector::changeFile()
{
	QString s = QFileDialog::getOpenFileName(m_pValueLabel->text(),QString::null,this);
	if(!s.isNull()){
		if(m_bMustBeReadable)
		{
			QFileInfo fi(s);
			if(! (fi.exists() && fi.isFile() && fi.isReadable()) )
			{
				KviStr tmp(KviStr::Format,__tr("The file %s is not readable."),s.latin1());
				kvi_warningBox(_CHAR_2_QSTRING(__tr("KVIrc: File Selector")),_CHAR_2_QSTRING(tmp.ptr()));
				return;
			}
		}
		m_pValueLabel->setText(s);
	}
}

void KviNewFileSelector::setEnabled(bool bEnabled)
{
	m_pValueLabel->setEnabled(bEnabled);
	m_pButton->setEnabled(bEnabled);
	QWidget::setEnabled(bEnabled);
}


KviNewIpSelector::KviNewIpSelector(QWidget * parent,const char * description,KviStr *ipPtr,bool bEnabled)
: QHBox(parent)
{
	m_ipPtr = ipPtr;
	m_pDescription = new QLabel(description,this);
	m_pIpEditor = new KviIpEditor(this,ipPtr->ptr());
	setEnabled(bEnabled);
}

KviNewIpSelector::~KviNewIpSelector()
{
}

void KviNewIpSelector::setEnabled(bool bEnabled)
{
	QHBox::setEnabled(bEnabled);
	m_pIpEditor->setEnabled(bEnabled);
	m_pDescription->setEnabled(bEnabled);
}

void KviNewIpSelector::commitChanges()
{
	*m_ipPtr = m_pIpEditor->address();
}


//
// KviNewColorSelector
//

KviNewColorSelector::KviNewColorSelector(QWidget *parent,const char * label,QColor *valuePtr,
		bool bEnabled,int defInputWidth)
:QWidget(parent)
{
	m_valuePtr = valuePtr;
	m_pLabel   = new QLabel(_CHAR_2_QSTRING(label),this);
	m_pClrButton = new KviColorButton(this,*valuePtr);
	m_iDefInputWidth = defInputWidth;
	setMinimumSize(sizeHint());
	setEnabled(bEnabled);
}
KviNewColorSelector::~KviNewColorSelector(){}

void KviNewColorSelector::commitChanges()
{
	*m_valuePtr = m_pClrButton->getSelectedColor();
}
void KviNewColorSelector::setColor(const QColor &clr)
{
	m_pClrButton->setColor(clr);
}
void KviNewColorSelector::setEnabled(bool bEnabled)
{
	m_pClrButton->setEnabled(bEnabled);
	m_pLabel->setEnabled(bEnabled);
}

const QColor & KviNewColorSelector::getColor()
{
	return m_pClrButton->getSelectedColor();
}
QSize KviNewColorSelector::sizeHint()
{
	QSize sz=m_pLabel->sizeHint();
	sz.setWidth(sz.width()+m_iDefInputWidth);
	sz.setHeight(m_pLabel->sizeHint().height());
	if(sz.height() < 24)sz.setHeight(24);
	return sz;
}
void KviNewColorSelector::resizeEvent(QResizeEvent *)
{
	m_pLabel->setGeometry(0,0,width()-(m_iDefInputWidth +1),height());
	m_pClrButton->setGeometry(width()-m_iDefInputWidth,0,m_iDefInputWidth,height());
}



//
// KviFontSelector
//

KviNewFontSelector::KviNewFontSelector(QWidget *parent,const char * label,QFont *valuePtr,
		bool bEnabled,int defInputWidth)
:QWidget(parent)
{
	m_valuePtr = valuePtr;
	m_pLabel   = new QLabel(_CHAR_2_QSTRING(label),this);
	QFont tmp(*valuePtr);
	m_pFntButton = new KviFontButton(this,tmp);
	m_iDefInputWidth = defInputWidth;
	setMinimumSize(sizeHint());
	setEnabled(bEnabled);
}
KviNewFontSelector::~KviNewFontSelector()
{}
void KviNewFontSelector::commitChanges()
{
	*m_valuePtr = m_pFntButton->getSelectedFont();
}
QFont KviNewFontSelector::getFont()
{
	return m_pFntButton->getSelectedFont();
}
QSize KviNewFontSelector::sizeHint()
{
	QSize sz=m_pLabel->sizeHint();
	sz.setWidth(sz.width()+m_iDefInputWidth);
	sz.setHeight(m_pLabel->sizeHint().height());
	if(sz.height() < 24)sz.setHeight(24);
	return sz;
}
void KviNewFontSelector::resizeEvent(QResizeEvent *)
{
	m_pLabel->setGeometry(0,0,width()-(m_iDefInputWidth +1),height());
	m_pFntButton->setGeometry(width()-m_iDefInputWidth,0,m_iDefInputWidth,height());
}

//
// KviNewImageSelector
//

KviNewImageSelector::KviNewImageSelector(QWidget *parent,const char * label,KviStr *valuePtr,
		QPixmap * cachedPixmap,bool bEnabled)
:QWidget(parent)
{
	m_valuePtr = valuePtr;
	m_pCheckBox   = new QCheckBox(_CHAR_2_QSTRING(label),this);
	m_pCheckBox->setChecked(!valuePtr->isEmpty());
	m_pImgButton = new KviImageButton(this,valuePtr->ptr(),cachedPixmap);
	connect(m_pCheckBox,SIGNAL(toggled(bool)),m_pImgButton,SLOT(setEnabled(bool)));
	m_pImgButton->setEnabled(!valuePtr->isEmpty());
	setMinimumSize(sizeHint());
	setEnabled(bEnabled);
}
KviNewImageSelector::~KviNewImageSelector()
{}
void KviNewImageSelector::commitChanges()
{
	if(m_pCheckBox->isChecked()){
		const char *ptr = m_pImgButton->getSelectedImagePath();
		*m_valuePtr = ptr ? ptr : "";
	} else *m_valuePtr = ""; //no image
}
const QPixmap * KviNewImageSelector::getImage()
{
	if(m_pCheckBox->isChecked()){
		return m_pImgButton->pixmap();
	} else return 0;
}
QSize KviNewImageSelector::sizeHint()
{
	QSize sz=m_pCheckBox->sizeHint();
	sz.setHeight(sz.height() + 100);
	return sz;
}
void KviNewImageSelector::resizeEvent(QResizeEvent *)
{
	int hght = m_pCheckBox->sizeHint().height();
	m_pCheckBox->setGeometry(0,0,width(),hght);
	m_pImgButton->setGeometry(0,hght+1,width(),height() - (hght+1));
}

#include "m_kvi_selectors.moc"
