// KreateCD - CD recording software for the K desktop environment
//
// 2000 by Joseph Wenninger <jowenn@bigfoot.com>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

#include "TrackListView.h"
#include <klocale.h>
#include <qtimer.h>
#include <stdio.h>

#include "TrackListView.moc"

TrackListView::TrackListView(QWidget *parent, 
			     const char *name,
			     TrackListManager *tracklistmanager, 
			     bool ClassicStyle):KListView(parent,name) {
  listEntriesCnt=0;
  selectionChanged=false;
  TLM=tracklistmanager;
  connect(TLM,SIGNAL(needUpdate(int)),this,SLOT(needUpdate(int)));
  connect(TLM,SIGNAL(needCurrentChanged(int)),this,SLOT(changeCurrent(int)));

  connect(this,SIGNAL(moved(QListViewItem*, QListViewItem*,QListViewItem*)),
	  this,SLOT(Moved(QListViewItem*, QListViewItem*,QListViewItem*)));
  connect(this,SIGNAL(doubleClicked(QListViewItem *)),this,SLOT(editTrack()));
  connect(this,SIGNAL(dropped(QDropEvent*,QListViewItem*)),
	  this,SLOT(Dropped(QDropEvent *,QListViewItem *)));

  connect(this,SIGNAL(selectionChanged()),this,SLOT(selectionChangedInternal()));
  connect(this,SIGNAL(clicked(QListViewItem*)),this,SLOT(executedInternal()));

  connect(TLM,SIGNAL(trackAdded()),this,SLOT(appendTrack()));
  connect(TLM,SIGNAL(clearList()),this,SLOT(clearTracks()));
  connect(TLM,SIGNAL(trackDeleted()),this,SLOT(trackDeleted()));

  isClassic=ClassicStyle;
  //For D'N'D stuff
  this->setItemsMovable(true);
  this->setDragEnabled(true);
  this->setAcceptDrops(true);
  this->setDropVisualizer(true);
  this->setSorting(0);

  //Initialization
  this->addColumn(i18n("Track"),-1);
  this->addColumn(i18n("Description"),-1);
  this->addColumn(i18n("Size"),-1);
  this->addColumn(i18n("Mode"),-1);
  this->setAllColumnsShowFocus(true);
}

void TrackListView::selectionChangedInternal()
{
	selectionChanged=true;
}
void TrackListView::executedInternal()
{
	if (selectionChanged)
 	{
		selectionChanged=false;
		emit selectionChangedDelayed();
	}
}

void TrackListView::clearTracks() {
  clear();
  qDebug(">>>>>>>>>> List cleared");
  listEntriesCnt=0;
}

int TrackListView::getCurrentId() {
  QListViewItem *cur=currentItem();
  if (cur==0) return(-1);
  QString curtext=cur->text(0);
  if (curtext.isNull()){ /*qDebug("Can't move NULL item");*/ return (-1);}
  //  qDebug("Selected item:" + curtext);
  bool ok;
  int TrackID=curtext.toInt(&ok,10);
  if (!ok) {/*qDebug("Couldn't extract track number"); */ return (-1);}
  else return (TrackID-1);
}

void TrackListView::editTrack() {
  TLM->editTrack(getCurrentId());
}

void TrackListView::moveUp() {
  TLM->moveByOne(getCurrentId(),true);
};

void TrackListView::moveDown() {
  TLM->moveByOne(getCurrentId(),false);
};

QListViewItem *TrackListView::findTrack(int id) {
  QListViewItem *tmpItem;
  for (tmpItem=firstChild(); (tmpItem) && (tmpItem->text(0).toInt()!=id); tmpItem=tmpItem->nextSibling());
  return tmpItem;
}

void TrackListView::needUpdate(int trackId) {
  QListViewItem *item=findTrack(trackId+1);
  if (item) {
    item->setText(1,TLM->getTrackDescription(trackId));
    item->setText(2,TLM->getTrackDurationHMSB(trackId));
    item->setText(3,TLM->getTrackTypeText(trackId));
  } else {
    qDebug("NOT FOUND");
  }
}

void TrackListView::changeCurrent(int trackId) {
  QListViewItem *item=findTrack(trackId+1);
  if (!item) return;
  clearSelection();
  setSelected(item,true);
  ensureItemVisible(item);
}

bool TrackListView::acceptDrag(QDropEvent *e) const {
  return QUriDrag::canDecode(e) || KListView::acceptDrag(e);
	
}


void TrackListView::Moved(QListViewItem *item, QListViewItem *afterFirst,
			  QListViewItem *afterNow) {
  int firstId=-1;
  int nowId=-1;

  (afterFirst) ? firstId=afterFirst->text(0).toInt()-1 : -1;
  (afterNow) ? nowId=afterNow->text(0).toInt()-1 : -1;
  if (afterFirst==afterNow) return;
  qDebug(""+QString("%1").arg(firstId)+"=>"+QString("%1").arg(nowId));
  if ( ((firstId-nowId)==-2) || ((firstId-nowId)==1)) {
    itemToChange=item->text(0).toInt()-1;
    changeDirection=(nowId-firstId)<0;
    QTimer::singleShot(0,this,SLOT(changeNow()));
  } else {
    itemToChange=item->text(0).toInt()-1;
    itemNewPos=nowId;
    QTimer::singleShot(0,this,SLOT(changeNowFar()));
  }
}

void TrackListView::changeNow() {
  TLM->moveByOne(itemToChange,changeDirection);
}

void TrackListView::changeNowFar() {
  TLM->moveItemAfter(itemToChange,itemNewPos);
}

void TrackListView::Dropped(QDropEvent *e,QListViewItem *after) {
  if (!QUriDrag::canDecode(e)) return;
  TLM->insert(e,after ? after->text(0).toInt():0);
};


void TrackListView::appendTrack() {
  char col[80];
  QListViewItem *lItem = new QListViewItem(this);
  sprintf(col,"%3d",++listEntriesCnt);
  lItem->setText(0,col);
}

void TrackListView::trackDeleted() {
  if (listEntriesCnt==0) return;
  delete findTrack(listEntriesCnt--);
}

void TrackListView::deleteTrack() {
  TLM->deleteTrack(getCurrentId());
}

void TrackListView::updateTrack() {
  TLM->updateTrack(getCurrentId());
}
