/***************************************************************************
                          kpgtreeitem.h  -  description
                             -------------------
    begin                : Ne led 11 2004
    copyright            : (C) 2004 by Lumir Vanek
    email                : lvanek@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KPGTREEITEM_H
#define KPGTREEITEM_H

// Note that for new code which you wish to be strictly Unicode-clean, you can define the macro QT_NO_ASCII_CAST when compiling your code.
#define QT_NO_ASCII_CAST

#include <qlistview.h>

// include files for libpgxx 
#include <pqxx/transaction> 
#include <pqxx/result> 

#include "../kpgsqlexception.h"

class KPGConnection;
class QTable;


/** 
  * Abstract base class for all tree items. 
  * Mother of KPGObject and KPGItemsFolder
  * 
  * @author Lumir Vanek
  */
class KPGTreeItem : public QListViewItem
{
public: 
	KPGTreeItem(QListView *);
  	KPGTreeItem(KPGTreeItem *, const QString);
	KPGTreeItem(QListView *, KPGTreeItem *);
  	KPGTreeItem(KPGTreeItem *, KPGTreeItem *, const QString);
	
	~KPGTreeItem();

  
  enum eNodeType { 
    nodeUnselected = 0, nodeServer, nodeDatabase, nodeSchema, nodeTablespace, 
    nodeTable, nodeView, nodeFunction, nodeSequence, nodeDomainConstr, nodeTableColumn, nodeTypeAttribute,
    nodePrimaryKeyConstr, nodeForeignKeyConstr, nodeUniqueConstr, nodeCheckConstr,
	nodePrimaryKey, nodeIndex, nodeRule, nodeTrigger, nodeOperator, nodeOperatorClass, 
	nodeAggregate, nodeUser, nodeGroup, nodeType, nodeLanguage, nodeDomain, nodeConversion,
    nodeCast, nodeFolderTablespaces, nodeFolderTables, nodeFolderViews, 
    nodeFolderSequences, nodeFolderFunctions, nodeFolderTableColumns, 
    nodeFolderTableIndexes, nodeFolderTableConstraints, nodeFolderDomainConstraints,
	nodeFolderRules, nodeFolderTriggers,
	nodeFolderOperators, nodeFolderOperatorClasses, nodeFolderAggregates, nodeFolderUsers, 
	nodeFolderGroups, nodeFolderTypes, nodeFolderLanguages, nodeFolderDomains,
    nodeFolderConversions, nodeFolderCasts
	};

  	virtual eNodeType type() const = 0; // return tree node type
	
	// Return false from KPGTreeItem's, and true for KPGObject's
	virtual bool hasOid() const { return false; }
  	
  	
  	// Return connection to database. All items childs to KPGDatabase tells your parents for connection
	virtual KPGConnection * connection() const;
	
	// Refresh item, ald also childs objects
	virtual void refresh() throw(const KPGSqlException &) = 0;
  	
  	// Refresh only this item, not childs
	virtual void refreshItem() throw(const KPGSqlException &) { return refresh(); }
  	
  	// This same behavior as refreshItem(), but for some objects, may be optimalized to run faster
  	virtual void smartRefresh() throw(const KPGSqlException &)  { return refresh(); }
	
	virtual KPGTreeItem * searchObject(const QString &, pqxx::oid); // search object with given path and OID
	const QString path();
	
  	static const QString makeConnectionString(const QString&, const QString&, const QString&, const QString&, const QString&);
  	
  	static const QString makeConnectionString(const QString&, const QString&, const QString&, const QString&);

	// Return child item with given name
	KPGTreeItem * getChildByName(const QString &);

    // Return child item with given oid
	KPGTreeItem * getChildByOid(pqxx::oid);
	
  	// Return nearest parent item with given type
  	KPGTreeItem * getParentByType(const eNodeType);
	
public:
	static QPixmap *m_pIconServer;
	static QPixmap *m_pIconTablespaceSys;
	static QPixmap *m_pIconTablespaceUsr;
	static QPixmap *m_pIconDatabase;
	static QPixmap *m_pIconDatabaseSys;
	static QPixmap *m_pIconDatabaseUsr;
	static QPixmap *m_pIconSchema;
  	static QPixmap *m_pIconSchemaUnl;
	static QPixmap *m_pIconSchemaSys;
	static QPixmap *m_pIconSchemaUsr;
	static QPixmap *m_pIconTable;
	static QPixmap *m_pIconToastTable;
	static QPixmap *m_pIconView;
	static QPixmap *m_pIconSequence;
	static QPixmap *m_pIconOperator;
	static QPixmap *m_pIconOperatorClass;
	static QPixmap *m_pIconAggregate;
	static QPixmap *m_pIconFunction;
	static QPixmap *m_pIconFunctionC;
	static QPixmap *m_pIconFunctionSql;
	static QPixmap *m_pIconFunctionPl;
	static QPixmap *m_pIconUser;
	static QPixmap *m_pIconUserSuper;
	static QPixmap *m_pIconUserCanCreateDb;
	static QPixmap *m_pIconGroup;
	static QPixmap *m_pIconType;
	static QPixmap *m_pIconTypeBase;
	static QPixmap *m_pIconTypeComposite;
	static QPixmap *m_pIconTypeStandalone;
	static QPixmap *m_pIconLanguage;
	static QPixmap *m_pIconDomain;
	static QPixmap *m_pIconConversion;
	static QPixmap *m_pIconCast;
		
	static QPixmap *m_pIconColumnOther;
	static QPixmap *m_pIconColumnBool; 
	static QPixmap *m_pIconColumnGeo;   
	static QPixmap *m_pIconColumnNet; 
	static QPixmap *m_pIconColumnOid;  
	static QPixmap *m_pIconColumnXid;
	static QPixmap *m_pIconColumnTid;
	static QPixmap *m_pIconColumnCid;
	static QPixmap *m_pIconColumnDate; 
	static QPixmap *m_pIconColumnChar;  
	static QPixmap *m_pIconColumnNum;
	static QPixmap *m_pIconColumnTime;
	static QPixmap *m_pIconColumnBit;
	static QPixmap *m_pIconColumnBin;
	static QPixmap *m_pIconColumnXml;
	static QPixmap *m_pIconColumnUuid;
		
	static QPixmap *m_pIconPrimaryKeyConstr;
	static QPixmap *m_pIconForeignKeyConstr;
	static QPixmap *m_pIconUniqueConstr;
	static QPixmap *m_pIconCheckConstr;
	static QPixmap *m_pIconPrimaryKey;
	static QPixmap *m_pIconIndex;
	static QPixmap *m_pIconRule;
	static QPixmap *m_pIconTrigger;
	
	static QPixmap *m_pIconFolderBlue;
	static QPixmap *m_pIconFolderRed;
	static QPixmap *m_pIconFolderCyan;
	static QPixmap *m_pIconFolderGreen;
	static QPixmap *m_pIconFolderYellow;
	static QPixmap *m_pIconFolderViolet;
	static QPixmap *m_pIconFolderGrey;
	
	// Null pixmap for SQL keywords in code completion box
	static QPixmap *m_pIconNull;
	
	static QPixmap *m_pIconLock;
};

/**
  * Storage class for OID and their object name, and some another info
  *
  * @author Lumir Vanek
  */
class KPGOidName
{
public:
    // Default constructor
	KPGOidName() {;}
    ~KPGOidName() {;}
    
    // Standard constructor
	KPGOidName(pqxx::oid _oid, const QString & strName, KPGTreeItem::eNodeType nodeType, const QPixmap *pIcon, const QString & strDescription)
	{
		Q_ASSERT(nodeType != KPGTreeItem::nodeFunction);
		
		m_oid = _oid;
		m_strName = strName;
		m_nodeType = nodeType;
		m_pIcon = pIcon;
		m_strDescription = strDescription;
		m_oidUnderlyingType = 0;
	}
	
	// Standard constructor - for functions
	KPGOidName(pqxx::oid _oid, const QString & strName, KPGTreeItem::eNodeType nodeType, const QPixmap *pIcon, const QString & strDescription, const QString &strArgumentTypes)
	{
	    Q_ASSERT(nodeType == KPGTreeItem::nodeFunction);
	    
		m_oid = _oid;
		m_strName = strName;
		m_nodeType = nodeType;
		m_pIcon = pIcon;
		m_strDescription = strDescription;
		m_oidUnderlyingType = 0;
		m_strArgumentTypes = strArgumentTypes;
	}
	
	// Copy constructor
	KPGOidName(const KPGOidName &src)
	{
		m_oid = src.oid();
		m_strName = src.name();
		m_nodeType = src.type();
		m_pIcon = src.icon();
		m_strDescription = src.description();
		m_oidUnderlyingType = src.oidUnderlyingType();
		m_strArgumentTypes = src.argumentTypes();
	}
	
	void operator = (const KPGOidName &src)
	{
		m_oid = src.oid();
		m_strName = src.name();
		m_nodeType = src.type();
		m_pIcon = src.icon();
		m_strDescription = src.description();
		m_oidUnderlyingType = src.oidUnderlyingType();
		m_strArgumentTypes = src.argumentTypes();
	}
	
	bool operator == (const KPGOidName &oidName)
    {
       if(m_nodeType != oidName.type())
        return false;
       
       return ( QString::localeAwareCompare( m_strName, oidName.name() ) == 0 );
    }
 
    bool operator < (const KPGOidName &oidName)
    {
        return ( QString::localeAwareCompare( m_strName, oidName.name() ) < 0 );
    }
    
    bool operator <= (const KPGOidName &oidName)
    {
        return ( QString::localeAwareCompare( m_strName, oidName.name() ) <= 0 );
    }
    
    bool operator > (const KPGOidName &oidName)
    {
        return ( QString::localeAwareCompare( m_strName, oidName.name() ) > 0 );
    }
    
    bool operator >= (const KPGOidName &oidName)
    {
        return ( QString::localeAwareCompare( m_strName, oidName.name() ) >= 0 );
    }
    
	
	// Return PostgreSQL row identifier
	const pqxx::oid oid() const { return m_oid; }
		
	// Return name
	const QString & name() const { return m_strName; }
	
	// Return node type
	KPGTreeItem::eNodeType type() const { return m_nodeType; }
	
	// Return node pixmap
	const QPixmap *icon() const { return m_pIcon; }
		
	// Return description
	const QString & description() const { return m_strDescription; }
		
	// Return PostgreSQL row identifier of underlying type
	const pqxx::oid oidUnderlyingType() const { return m_oidUnderlyingType; }
	
	// Set underlying type OID
	void setOidUnderlyingType(pqxx::oid oidUnderlyingType) { m_oidUnderlyingType = oidUnderlyingType; }
	
	// Return function argument types
	const QString & argumentTypes() const { return m_strArgumentTypes; }
	
protected:

    // Object name
	QString m_strName;

	// PostgreSQL row identifier
	pqxx::oid m_oid;
		
	// Node type
	KPGTreeItem::eNodeType m_nodeType;
	
	// Node pixmap
	const QPixmap *m_pIcon;
	
	// Object description
	QString m_strDescription;
		
	// For tables, type of row structure. For functions returning rowset, it's type. Otherwise 0.
	pqxx::oid m_oidUnderlyingType;
	
	// Arguments list for functions
	QString m_strArgumentTypes;
};

/**
  * Extends KPGOidName for list of aliases
  *
  * @author Lumir Vanek
  */
class KPGOidNameAliases : public KPGOidName
{
public:
    // Default constructor
	KPGOidNameAliases() {;}
    ~KPGOidNameAliases() {;}
    
    // Return aliases
	const QStringList & aliases() const { return m_strAliases; }
	
	// Clear aliases
	void clearAliases() { m_strAliases.clear(); }
	
	// Append alias
	void appendAlias(const QString &alias) { m_strAliases.append(alias); }
	
	// Return true, if given alias already exists in list
	bool containAlias(const QString &alias) const 
	{
		return(m_strAliases.findIndex(alias) >= 0);
	}
	
	// Copy constructor
	KPGOidNameAliases(const KPGOidNameAliases &src)
	: KPGOidName(src)
	{
		m_strAliases = src.aliases();
	}
	
	// Copy constructor for base class
	KPGOidNameAliases(const KPGOidName &src)
	: KPGOidName(src)
	{
	}
	
	void operator = (const KPGOidNameAliases &src)
	{
		KPGOidName::operator = (src);
		m_strAliases = src.aliases();
	}
	
protected:
	// Object aliases, e.g.: SELECT a.* FROM aaa a -- "a" is alias
	QStringList m_strAliases;
    
};

/**
  * List of KPGOidName with sorting ability.
  */
class KPGOidNameList : public QValueList<KPGOidName>
{
public:
    KPGOidNameList() {;}
    
    // Sort the list
    void sort();
    
    // Get item from list by given name
    KPGOidName * getItemByName(const QString &, bool = true);
    
    // Get item from list by given name
    const KPGOidName * getConstItemByName(const QString &) const;
};

/**
  * List of KPGOidNameAliases with sorting ability.
  */
class KPGOidNameAliasesList : public QValueList<KPGOidNameAliases>
{
public:
    KPGOidNameAliasesList() {;}
    
    // Sort the list
    void sort();
    
    // Get item from list by given name
    KPGOidNameAliases * getItemByName(const QString &, bool = true);
    
    // Get item from list by given name
    const KPGOidNameAliases * getConstItemByName(const QString &) const;
    
    // Get item from list by given alias
    const KPGOidNameAliases * getConstItemByAlias(const QString &) const;
    
    // Return true, if given alias already exists in list
	bool containAlias(const QString &alias) const;
};

#endif
