/*******************************************************************************
**                       configurationdlg.cpp
**                             part of
**                konserve -- A small backup application
**                         -------------------
**  copyright: (C) 2002 - 2004 by Florian Simnacher
**  email    : simnacher AT gmx DOT de
*******************************************************************************/

/*******************************************************************************
**  This program is free software; you can redistribute it and/or             **
**  modify it under the terms of the GNU General Public License               **
**  as published by the Free Software Foundation; either version 2            **
**  of the License, or (at your option) any later version.                    **
**                                                                            **
**  This program is distributed in the hope that it will be useful,           **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of            **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             **
**  GNU General Public License for more details.                              **
**                                                                            **
**  You should have received a copy of the GNU General Public License         **
**  along with this program; if not, write to the Free Software               **
**  Foundation, Inc., 59 Temple Place - Suite 330, Boston,                    **
**  MA  02111-1307, USA.                                                      **
*******************************************************************************/

// Standard Includes
#include <iostream>

// Qt Includes
#include <qradiobutton.h>

// KDE Includes
#include <keditlistbox.h>
#include <kurlrequester.h>
#include <knuminput.h>
#include <kcombobox.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <klineedit.h>

// My Includes
#include "konservedebug.h"
#include "configurationdlg.h"

////////////////////////////////////////////////////////////////////////////////
///////               class ConfigurationDlg                             ///////
////////////////////////////////////////////////////////////////////////////////

ConfigurationDlg::ConfigurationDlg()
    : mbProfileDictModified( false ),
      msIdentifierBackup( QString::null ),
      mProfileDict()
{
    TRACE();
    mProfileDict.setAutoDelete( true );

    setupConfigKeditListBoxConnections();
    setupActionConnections();
    setupDataModifyConnections();

    configKEditListBox->listBox()->setSelectionMode( QListBox::Extended );

    sourceKURLRequester->setMode(
        KFile::File | KFile::Directory |
        KFile::ExistingOnly | KFile::LocalOnly
        );
    archiveKURLRequester->setMode(
        KFile::File | KFile::Directory | KFile::LocalOnly
        );

    saveKPushButton->setDisabled( true );
    restoreKPushButton->setDisabled( true );
}

ConfigurationDlg::~ConfigurationDlg()
{
    TRACE();
    // NOOP
}

void ConfigurationDlg::inserted( BackupProfile *b )
{
    TRACE();
    configKEditListBox->listBox()->insertItem( b->getIdentifier() );
    slotUpdateUI( new QListBoxText( b->getIdentifier() ) );

    mProfileDict.insert(
        b->getIdentifier(),
        new MyBackupProfileInfo( BackupProfile::newBackupProfileReadOnly( b ) )
        );
}

void ConfigurationDlg::changed( BackupProfile *b )
{
    TRACE();
    slotUpdateUI( new QListBoxText(b->getIdentifier() ) );

    mProfileDict.replace(
        b->getIdentifier(),
        new MyBackupProfileInfo( BackupProfile::newBackupProfileReadOnly( b ) )
        );
}

void ConfigurationDlg::removed( QString s )
{
    TRACE();
    mToDeleteItem = s;
    this->slotRemoveProfile();
    slotUpdateUI( new QListBoxText() );
    mProfileDict.remove( s );
}

void ConfigurationDlg::slotUpdateUI( QListBoxItem* l )
{
    TRACE();
    QString identifier;

    blockInputWidgetSignals();
    if ( 0L == l )
    {
        slotReset();
        return;
    }
    else
    {
        identifier = l->text();
    }
    if ( QString::null == identifier )
    {
        slotReset();
        return;
    }
    if ( 0L == mProfileDict.find( identifier ) )
    {
        slotReset();
        return;
    }
    sourceKURLRequester->lineEdit()->setText(
        mProfileDict.find( identifier )->getSourceUrl().url()
        );
    archiveKURLRequester->lineEdit()->setText(
        mProfileDict.find( identifier )->getArchiveUrl().url()
        );
    intervalKIntNumInput->setValue(
        mProfileDict.find( identifier )->getTimeBetweenBackups().value()
        );
    measureKComboBox->setCurrentItem(
        mProfileDict.find(
            identifier
            )->getTimeBetweenBackups().measurement( 0 )
        );
    activRadioButton->setChecked(
        mProfileDict.find( identifier )->isActivated()
        );
    Q_ASSERT( mProfileDict.find( identifier ) );
    setApplyOn( mProfileDict.find( identifier )->modified() );
    setIdentifierBackup();
    unblockInputWidgetSignals();
}

void ConfigurationDlg::slotUpdateDict( QListBoxItem* /* not used */ )
{
    TRACE();
    blockInputWidgetSignals();

    if ( 0L == mProfileDict.find( getIdentifierBackup() ) )
    {
        unblockInputWidgetSignals();
        return;
    }
    if ( mbProfileDictModified )
    {
        mProfileDict.find( getIdentifierBackup() )->modify(
            BackupProfile::newBackupProfileReadOnly(
                sourceKURLRequester->url(),
                archiveKURLRequester->url(),
                getIdentifierBackup(),
                intervalKIntNumInput->value(),
                measureKComboBox->currentText(),
                activRadioButton->isChecked(),
                QDateTime::currentDateTime()
                )
            );
        mbProfileDictModified = false;
    }
    unblockInputWidgetSignals();
}

void ConfigurationDlg::slotDataModified()
{
    TRACE();
    mbProfileDictModified = true;
    setApplyOn( true );
}

void ConfigurationDlg::slotDataModified( const QString& )
{
    TRACE();
    slotDataModified();
}

void ConfigurationDlg::slotDataModified( int )
{
    TRACE();
    slotDataModified();
}

void ConfigurationDlg::slotDataModified( bool )
{
    TRACE();
    slotDataModified();
}

void ConfigurationDlg::slotSaveProfile()
{
    TRACE();
    slotSaveProfile( configKEditListBox->currentText() );
}

void ConfigurationDlg::slotSaveProfile( const QString& identifier )
{
    TRACE();
    if ( QString::null == identifier )
    {
        KMessageBox::error( 0L, i18n( "Please select a backup profile" ));
        return;
    }
    if ( mbProfileDictModified )
        slotUpdateDict( 0L );
    MyBackupProfileInfo* mbpi = mProfileDict.find( identifier );
    mbpi->applyChanges();

    ProfileMngrInterface::insert(
        BackupProfile::newBackupProfile(
            mbpi->getSourceUrl(),
            mbpi->getArchiveUrl(),
            identifier,
            mbpi->getTimeBetweenBackups().value(),
            mbpi->getTimeBetweenBackups().measurement(),
            activRadioButton->isChecked(),
            QDateTime::currentDateTime()
            )
        );
    setApplyOn( false );
}

void ConfigurationDlg::slotRemoveProfile()
{
    TRACE();
    if ( ProfileMngrInterface::exists( mToDeleteItem ) )
        ProfileMngrInterface::remove( mToDeleteItem );
    slotUpdateUI( new QListBoxText() );
    mProfileDict.remove( mToDeleteItem );
}

void ConfigurationDlg::slotSaveToDeleteItem()
{
    TRACE();
    QListBox *lbox = configKEditListBox->listBox();
    mToDeleteItem  = lbox->text( lbox->currentItem() );
}

void ConfigurationDlg::slotReset()
{
    TRACE();
    blockInputWidgetSignals();
    configKEditListBox->listBox()->clearSelection();
    sourceKURLRequester->lineEdit()->setText( QString::null );
    archiveKURLRequester->lineEdit()->setText( QString::null );
    intervalKIntNumInput->setValue( 1 );

    measureKComboBox->setCurrentItem( TimeInterval::TIME_DAY );

    saveKPushButton->setDisabled( true );
    restoreKPushButton->setDisabled( true );
    activRadioButton->setChecked( false );
    unblockInputWidgetSignals();
}

void ConfigurationDlg::slotRestore()
{
    TRACE();
    ProfileMngrInterface::restore(
        configKEditListBox->listBox()->currentText()
        );
}

void ConfigurationDlg::slotClose()
{
    TRACE();
    slotUpdateDict( 0L );

    if ( unsavedChanges() )
    {
        switch( KMessageBox::warningYesNoCancel(
                    this,
                    i18n( "There are unsaved changes.\n"
                          "Do you want to save them now?" ),
                    i18n( "Unsaved Changes" ) ) )
        {
        case KMessageBox::Yes:
            saveChanges();
            break;
        case KMessageBox::No:
            forgetChanges();
            break;
        case KMessageBox::Cancel:
            return;
            break;
        default:
            Q_ASSERT( false );
        }
    }
    this->close();
    slotReset();
}

void ConfigurationDlg::slotAddIdentifierToDict()
{
    TRACE();
    if ( !mProfileDict.find( configKEditListBox->lineEdit()->text() ) )
    {
        mProfileDict.insert(
            // Note: Taken from the lineEdit
            configKEditListBox->lineEdit()->text(),
            new MyBackupProfileInfo(
                BackupProfile::newBackupProfileReadOnly(
                    sourceKURLRequester->url(),
                    archiveKURLRequester->url(),
                    configKEditListBox->currentText(),
                    intervalKIntNumInput->value(),
                    measureKComboBox->currentText(),
                    activRadioButton->isChecked(),
                    QDateTime::currentDateTime()
                    )
                )
            );
        mProfileDict.find( configKEditListBox->lineEdit()->text() )->modified();
    }
    else
        KMessageBox::sorry( this, i18n( "Backup profile already exists" ) );
}

void ConfigurationDlg::blockInputWidgetSignals()
{
    TRACE();
    sourceKURLRequester->blockSignals( true );
    archiveKURLRequester->blockSignals( true );
    intervalKIntNumInput->blockSignals( true );
    measureKComboBox->blockSignals( true );
    activRadioButton->blockSignals( true );
}

void ConfigurationDlg::unblockInputWidgetSignals()
{
    TRACE();
    sourceKURLRequester->blockSignals( false );
    archiveKURLRequester->blockSignals( false );
    intervalKIntNumInput->blockSignals( false );
    measureKComboBox->blockSignals( false );
    activRadioButton->blockSignals( false );
}

void ConfigurationDlg::setupDataModifyConnections()
{
    TRACE();
    connect(
        sourceKURLRequester, SIGNAL( textChanged( const QString& ) ),
        this,                SLOT( slotDataModified( const QString& ) )
        );
    connect(
        sourceKURLRequester, SIGNAL( urlSelected( const QString& ) ),
        this,                SLOT( slotDataModified( const QString& ) )
        );
    connect(
        archiveKURLRequester, SIGNAL( textChanged( const QString& ) ),
        this,                 SLOT( slotDataModified( const QString& ) )
        );
    connect(
        archiveKURLRequester, SIGNAL( urlSelected( const QString& ) ),
        this,                 SLOT( slotDataModified( const QString& ) )
        );
    connect(
        intervalKIntNumInput, SIGNAL( valueChanged( int ) ),
        this,                 SLOT( slotDataModified( int ) )
        );
    connect(
        measureKComboBox, SIGNAL( activated( const QString& ) ),
        this,             SLOT( slotDataModified( const QString& ) )
        );
    connect(
        activRadioButton, SIGNAL( toggled( bool ) ),
        this,             SLOT( slotDataModified( bool ) )
        );
}

void ConfigurationDlg::setupActionConnections()
{
    TRACE();
    connect(
        quitKPushButton, SIGNAL( clicked() ),
        this,            SLOT( slotClose() )
        );
    connect(
        restoreKPushButton, SIGNAL( clicked() ),
        this,               SLOT( slotRestore() )
        );
    connect(
        saveKPushButton, SIGNAL( clicked() ),
        SLOT( slotSaveProfile() )
        );
}

void ConfigurationDlg::setupConfigKeditListBoxConnections()
{
    TRACE();
    connect(
        configKEditListBox->removeButton(), SIGNAL( clicked() ),
        this,                               SLOT( slotRemoveProfile() )
        );
    connect(
        configKEditListBox->removeButton(), SIGNAL( pressed() ),
        this,                               SLOT( slotSaveToDeleteItem() )
        );
    connect(
        configKEditListBox->addButton(), SIGNAL( released() ),
        this,                            SLOT( slotAddIdentifierToDict() )
        );
    connect(
        configKEditListBox->listBox(), SIGNAL( pressed( QListBoxItem* )),
        this,                          SLOT( slotUpdateDict( QListBoxItem* ) )
        );
    connect(
        configKEditListBox->listBox(), SIGNAL( clicked( QListBoxItem* )),
        this,                          SLOT( slotUpdateUI( QListBoxItem* ) )
        );
}

void ConfigurationDlg::setIdentifierBackup()
{
    TRACE();
    msIdentifierBackup = configKEditListBox->currentText();
}

QString& ConfigurationDlg::getIdentifierBackup()
{
    TRACE();
    return msIdentifierBackup;
}

void ConfigurationDlg::setApplyOn( bool on )
{
    TRACE();
    if ( QString::null == configKEditListBox->currentText() )
    {
        saveKPushButton->setDisabled( true );
        restoreKPushButton->setDisabled( true );
        return;
    }
    if ( 0L == mProfileDict.find( configKEditListBox->currentText() ) )
    {
        saveKPushButton->setDisabled( true );
        restoreKPushButton->setDisabled( true );
    }
    if ( on )
    {
        saveKPushButton->setEnabled( true );
        restoreKPushButton->setDisabled( true );
    }
    else
    {
        saveKPushButton->setDisabled( true );
        restoreKPushButton->setEnabled( true );
    }
}

bool ConfigurationDlg::unsavedChanges()
{
    TRACE();
    QDictIterator<MyBackupProfileInfo> it( mProfileDict );
    for ( /* NOOP */; it.current(); ++it )
        if ( it.current()->modified() || mbProfileDictModified )
            return true;
    return false;
}

void ConfigurationDlg::saveChanges()
{
    TRACE();
    QDictIterator<MyBackupProfileInfo> itit( mProfileDict );
    for ( /* NOOP */; itit.current(); ++itit )
        if ( itit.current()->modified() )
        {
            slotSaveProfile( itit.current()->getIdentifier() );
            itit.current()->applyChanges();
        }
}

void ConfigurationDlg::forgetChanges()
{
    TRACE();
    QDictIterator<MyBackupProfileInfo> itit( mProfileDict );
    for ( /* NOOP */; itit.current(); ++itit )
        if ( itit.current()->modified() )
            itit.current()->discardChanges();
}

/////////////////////////////////////////////////////////////////////////////
//////////               class MyBackupProfileInfo                  /////////
/////////////////////////////////////////////////////////////////////////////

ConfigurationDlg::MyBackupProfileInfo::MyBackupProfileInfo(
    BackupProfile* profile
    )
    : mbModified( false ),
      mpCurrentData( profile ), // We take ownership of profile
      mpOriginalData( 0L )
{
    TRACE();
    // NOOP
}

ConfigurationDlg::MyBackupProfileInfo::~MyBackupProfileInfo()
{
    TRACE();
    if ( mpOriginalData )
        delete mpOriginalData;
    delete mpCurrentData;
}

bool ConfigurationDlg::MyBackupProfileInfo::modified()
{
    TRACE();
    return mbModified;
}

void ConfigurationDlg::MyBackupProfileInfo::modify( BackupProfile* profile )
{
    TRACE();
    if ( ! mpOriginalData )
        mpOriginalData = mpCurrentData;
    else
        delete mpCurrentData;
    mpCurrentData  = profile;
    mbModified     = true;

}

void ConfigurationDlg::MyBackupProfileInfo::discardChanges()
{
    TRACE();
    delete mpCurrentData;
    mpCurrentData  = mpOriginalData;
    mpOriginalData = 0L;
    mbModified     = false;
}

void ConfigurationDlg::MyBackupProfileInfo::applyChanges()
{
    TRACE();
    delete mpOriginalData;
    mpOriginalData = 0L;
    mbModified     = false;
}

bool ConfigurationDlg::MyBackupProfileInfo::isActivated()
{
    TRACE();
    return mpCurrentData->isActivated();
}

KURL ConfigurationDlg::MyBackupProfileInfo::getSourceUrl()
{
    TRACE();
    return mpCurrentData->getSourceUrl();
}

KURL ConfigurationDlg::MyBackupProfileInfo::getArchiveUrl()
{
    TRACE();
    return mpCurrentData->getArchiveUrl();
}

TimeInterval ConfigurationDlg::MyBackupProfileInfo::getTimeBetweenBackups()
{
    TRACE();
    return mpCurrentData->getTimeBetweenBackups();
}

QString ConfigurationDlg::MyBackupProfileInfo::getIdentifier()
{
    TRACE();
    return mpCurrentData->getIdentifier();
}

ConfigurationDlg::MyBackupProfileInfo::MyBackupProfileInfo(
    const MyBackupProfileInfo& /* copy */
    )
{
    TRACE();
    Q_ASSERT( false ); // Don't use
}

ConfigurationDlg::MyBackupProfileInfo&
ConfigurationDlg::MyBackupProfileInfo::operator=(
    const MyBackupProfileInfo& /* copy */
    )
{
    TRACE();
    Q_ASSERT( false ); // Don't use
    return (*this);
}

#include "configurationdlg.moc"
