/***************************************************************************
 *   Copyright (C) 2007-2008 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qheader.h>
#include <qlabel.h>
#include <qtooltip.h>
#include <qvbox.h>

#include <kdatetbl.h>
#include <kfiledialog.h>
#include <kiconloader.h>
#include <klistview.h>
#include <klocale.h>

#include "komparatorurllist.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

KomparatorUrlList::KomparatorUrlList( QWidget *parent, const char *name ) :
	KPushButton( parent, name ),
	m_popup_frame( NULL ), m_list_view( NULL )
{
	init();
}

KomparatorUrlList::KomparatorUrlList( const QString &text, QWidget *parent, const char *name ) :
	KPushButton( text, parent, name ),
	m_popup_frame( NULL ), m_list_view( NULL )
{
	init();
}

KomparatorUrlList::KomparatorUrlList( const QIconSet &icon, const QString &text, QWidget *parent, const char *name ) :
	KPushButton( icon, text, parent, name ),
	m_popup_frame( NULL ), m_list_view( NULL )
{
	init();
}

KomparatorUrlList::KomparatorUrlList( const KGuiItem &item, QWidget *parent, const char *name ) :
	KPushButton( item, parent, name ),
	m_popup_frame( NULL ), m_list_view( NULL )
{
	init();
}

KomparatorUrlList::~KomparatorUrlList()
{
	delete m_list_view;
	m_list_view = NULL;
	delete m_main_box;
	m_main_box = NULL;
	delete m_popup_frame;
	m_popup_frame = NULL;
}

void KomparatorUrlList::init()
{
	setText( i18n( "Virtual subdirectories..." ) );

	setToggleButton( true );

	m_popup_frame = new KPopupFrame( this );
	m_popup_frame->installEventFilter( this );

	m_main_box = new QHBox( m_popup_frame );

	m_list_view = new KListView( m_main_box, "m_list_view" );
	m_list_view->setSelectionMode( QListView::Single );
	m_list_view->addColumn( "" );
	m_list_view->header()->hide();

	QVBox *buttons_box = new QVBox( m_main_box );
	m_add_button = new KPushButton( SmallIcon( "add" ), "", buttons_box );
	m_add_button->setFixedSize( m_add_button->iconSet()->iconSize( QIconSet::Small ) );
	connect( m_add_button, SIGNAL( clicked() ), this, SLOT( slotAdd() ) );
	new QLabel( "", buttons_box );

	m_remove_button = new KPushButton( SmallIcon( "remove" ), "", buttons_box );
	m_remove_button->setFixedSize( m_remove_button->iconSet()->iconSize( QIconSet::Small ) );
	connect( m_remove_button, SIGNAL( clicked() ), this, SLOT( slotRemove() ) );
	new QLabel( "", buttons_box );

	m_clear_button = new KPushButton( SmallIcon( "clear_left" ), "", buttons_box );
	m_clear_button->setFixedSize( m_clear_button->iconSet()->iconSize( QIconSet::Small ) );
	connect( m_clear_button, SIGNAL( clicked() ), this, SLOT( slotClear() ) );
	
	buttons_box->setMargin( 2 );

	m_main_box->adjustSize();
	m_popup_frame->setMainWidget( m_main_box );

	const QString add_tt = i18n( "Add an url to the list" );
	QToolTip::add( m_add_button, add_tt );
	const QString remove_tt = i18n( "Remove the selected url from the list" );
	QToolTip::add( m_remove_button, remove_tt );
	const QString clear_tt = i18n( "Clears all urls from the list" );
	QToolTip::add( m_clear_button, clear_tt );

	connect( this, SIGNAL( toggled( bool ) ), this, SLOT( slotToggled( bool ) ) );
}

void KomparatorUrlList::slotToggled( bool )
{
	if ( m_popup_frame->isHidden() )
	{
		m_main_box->setMinimumWidth( width() );
		m_main_box->setMinimumHeight( height()*5 );
		m_popup_frame->setMainWidget( m_main_box );
		m_popup_frame->popup( mapToGlobal( QPoint( 0, height() ) ) );
	}
}

void KomparatorUrlList::slotAdd()
{
	KFileDialog *dlg = new KFileDialog( QString::null, "*", (QWidget*) this, "file dialog", true );
	dlg->setMode( KFile::Directory );
	if ( dlg->exec() != QDialog::Accepted )
	{
		return;
	}
	
	new KListViewItem( m_list_view, dlg->selectedURL().url() );
}

void KomparatorUrlList::slotRemove()
{
	delete m_list_view->selectedItem();
}

void KomparatorUrlList::slotClear()
{
	clear();
}

void KomparatorUrlList::clear()
{
	m_list_view->clear();
}

KURL::List KomparatorUrlList::getUrlList()
{
	KURL::List list;

	QListViewItemIterator it( m_list_view );
	while ( it.current() )
	{
		list.append( KURL( it.current()->text( 0 ) ) );
		++it;
	}

	return list;
}

void KomparatorUrlList::addUrl( const KURL &url )
{
	new KListViewItem( m_list_view, url.url() );
}

int KomparatorUrlList::getUrlNumber()
{
	return m_list_view->childCount();
}

bool KomparatorUrlList::eventFilter( QObject*, QEvent *e )
{
	if ( e->type() == QEvent::Hide )
	{
		if ( !m_popup_frame->isHidden() ) // This hide event should actually really hide the frame.
		{
			m_popup_frame->hide();
		}
		else if ( isOn() ) // frame is hidden now; "on" state shows us, that these events were not yet emitted.
		{
			setOn( false );
		}
		
		return true;
	}
	else if ( e->type() == QEvent::KeyRelease || e->type() == QEvent::KeyPress )
	{
		if( ((QKeyEvent *)e)->key() == Qt::Key_Escape ) // Esc would close main window, if it wasn't caught here !!!
		{
			if ( !m_popup_frame->isHidden() ) m_popup_frame->hide();
			return true;
		}
	}

  return false;
}

void KomparatorUrlList::setReadOnly( bool ro )
{
	m_add_button->setEnabled( !ro );
	m_remove_button->setEnabled( !ro );
	m_clear_button->setEnabled( !ro );
}
