/***************************************************************************
 *   Copyright (C) 2005-2010 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


/* This file uses main parts of "fdupes.c" from fdupes-1.40.
	Copyright: */

/* FDUPES Copyright (c) 1999 Adrian Lopez

	Permission is hereby granted, free of charge, to any person
	obtaining a copy of this software and associated documentation files
	(the "Software"), to deal in the Software without restriction,
	including without limitation the rights to use, copy, modify, merge,
	publish, distribute, sublicense, and/or sell copies of the Software,
	and to permit persons to whom the Software is furnished to do so,
	subject to the following conditions:

	The above copyright notice and this permission notice shall be
	included in all copies or substantial portions of the Software.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
	OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
	MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
	IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
	CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
	TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. */

#include <QFile>
#include <QCustomEvent>

#include <KDebug>
#include <KLocale>

#include "kmdcodec.h"

#include "kfileitemext.h"
#include "komparator.h"
#include "komparatorwidget.h"

#include "komparatorcomparejob.h"

#include <version.h>

KomparatorCompareJob::KomparatorCompareJob( QWidget *_parent )
	: KomparatorJob( _parent )
{
	m_files = NULL;
	m_ignore_empty = false;
	m_binary_comparison = false;
	m_canceled = false;
	m_initialized = false;

	connect( this, SIGNAL( emitDuplicate( KFileItemExt * ) ), (KomparatorWidget *)m_parent, SLOT( slotInteractionDuplicate( KFileItemExt * ) ) );
	connect( this, SIGNAL( emitCompMissing( KFileItemExt * ) ), (KomparatorWidget *)m_parent, SLOT( slotInteractionCompMissing( KFileItemExt * ) ) );
	connect( this, SIGNAL( emitCompNewer( KFileItemExt *, KFileItemExt * ) ), (KomparatorWidget *)m_parent, SLOT( slotInteractionCompNewer( KFileItemExt *, KFileItemExt * ) ) );
	connect( this, SIGNAL( emitCompNewerEqual( KFileItemExt *, KFileItemExt * ) ), (KomparatorWidget *)m_parent, SLOT( slotInteractionCompNewerEqual( KFileItemExt *, KFileItemExt * ) ) );
	connect( this, SIGNAL( emitCompNewerSameTime( KFileItemExt *, KFileItemExt * ) ), (KomparatorWidget *)m_parent, SLOT( slotInteractionCompNewerSameTime( KFileItemExt *, KFileItemExt * ) ) );
	connect( this, SIGNAL( emitCompFinished() ), (KomparatorWidget *)m_parent, SLOT( slotInteractionCompFinished() ) );
}

KomparatorCompareJob::~KomparatorCompareJob()
{
}

bool KomparatorCompareJob::initialize( KFileItemExt *_files, bool _ignore_empty, bool _size, bool _calculate_checksum, bool _binary_comparison,
														uint _binary_comparision_threshold, uint _number_of_files, const bool _enable_duplicates_search, bool _enable_missing_search, bool _enable_newer_search, bool _case_sensitive )
{
	m_mutex.lock();

	m_files = _files;
	m_ignore_empty = _ignore_empty;
	m_size = _size;
	m_calculate_checksum = _calculate_checksum;
	m_binary_comparison = _binary_comparison;
	m_binary_comparision_threshold = _binary_comparision_threshold;
	m_number_of_files = _number_of_files;

	m_enable_duplicates_search = _enable_duplicates_search;
	m_enable_missing_search = _enable_missing_search;
	m_enable_newer_search = _enable_newer_search;

	m_case_sensitive = _case_sensitive;

	m_initialized = true;
	m_canceled = false;

	m_mutex.unlock();

	return true;
}

void KomparatorCompareJob::run()
{
	if ( !m_initialized ) return;

	KFileItemExt *curfile = NULL;
	KFileItemExt *match = NULL;
	FileTree *checktree_dupes = NULL;
	FileTree *checktree_comp = NULL;

	QString status;
	uint current_file;

	if ( m_size && ( m_enable_duplicates_search || m_enable_missing_search || m_enable_newer_search ) )
	{ // comparing duplicates by size is the minimum we can do.
		status = i18n( "Searching for duplicates..." );
		current_file = 0;

		match = NULL;
		curfile = m_files;

		while ( curfile )
		{
			if ( isCanceled() ) break;

			emitProgress( status, (int)((current_file*100)/m_number_of_files) );

			if ( !checktree_dupes )
				registerFile( &checktree_dupes, curfile );
			else
				match = checkMatchDupes( &checktree_dupes, checktree_dupes, curfile );

			if ( match != NULL )
			{
				if ( !curfile->isReadable() )
				{
					curfile = curfile->next;
					continue;
				}

				if ( !match->isReadable() )
				{
					curfile = curfile->next;
					continue;
				}

				if ( m_ignore_empty )  if ( match->size() == 0 )
				{
					curfile = curfile->next;
					continue;
				}

				if ( m_binary_comparison )
				{
					if ( confirmMatchDupes( match, curfile ) )
					{
						match->hasdupes_size = 1;
						match->isdupe_size = 1;
						curfile->isdupe_size = 1;
						if ( match->duplicates_size )
						{
							match->duplicates_size->isdupe_size = 1;
							match->duplicates_size->dup_size_parent = curfile;
						}
						curfile->duplicates_size = match->duplicates_size;
						match->duplicates_size = curfile;
						curfile->dup_size_parent = match;
					}
				}
				else
				{
					match->hasdupes_size = 1;
					match->isdupe_size = 1;
					curfile->isdupe_size = 1;
					if ( match->duplicates_size )
					{
						match->duplicates_size->isdupe_size = 1;
						match->duplicates_size->dup_size_parent = curfile;
					}
					curfile->duplicates_size = match->duplicates_size;
					match->duplicates_size = curfile;
					curfile->dup_size_parent = match;
				}
			}

			current_file++;
			curfile = curfile->next;
		}

		emitProgress( status, -1 );

		if ( m_enable_duplicates_search )
		{
			status = i18n( "Appending duplicates to list view..." );
			current_file = 0;

			curfile = m_files;

			while ( curfile )
			{
				if ( isCanceled() )  break;

				emitProgress( status, (int)((current_file*100)/m_number_of_files) );

				if ( curfile->hasdupes_size )
				{
					emitDuplicate( curfile );
				}

				current_file++;
				curfile = curfile->next;
			}
		}

		emitProgress( status, -1 );
	}

	if ( ( m_enable_duplicates_search && !m_size ) || m_enable_missing_search || m_enable_newer_search )
	{
		status = i18n( "Searching for missing / newer files..." );
		current_file = 0;

		match = NULL;
		curfile = m_files;

		while ( curfile )
		{
			if ( isCanceled() )  break;

			emitProgress( status, (int)((current_file*100)/m_number_of_files) );

			if ( !checktree_comp )
				registerFile( &checktree_comp, curfile );
			else
				match = checkMatchComp( &checktree_comp, checktree_comp, curfile );

			if ( match != NULL )
			{
				match->hasdupes_path = 1;
				match->isdupe_path = 1;
				curfile->isdupe_path = 1;
				curfile->duplicates_path = match->duplicates_path;
				match->duplicates_path = curfile;
				curfile->dup_path_parent = match;
			}

			current_file++;
			curfile = curfile->next;
		}

		emitProgress( status, -1 );

		KFileItemExt *tmpfile;

		status = i18n( "Appending missing / newer files to list views..." );
		current_file = 0;

		curfile = m_files;

		while ( curfile )
		{
			if ( isCanceled() ) break;

			emitProgress( status, (int)((current_file*100)/m_number_of_files) );

			QList< QPair<KFileItemExt*, KFileItemExt*> * > list;

			if ( curfile->hasdupes_path )
			{
				QPair<KFileItemExt*, KFileItemExt*> *pair = new QPair<KFileItemExt*, KFileItemExt*>;
				*pair = qMakePair( curfile, (KFileItemExt*)NULL );
				list.push_back( pair );

				QPair<KFileItemExt*, KFileItemExt*> *tmppair;
				bool inserted;

				tmpfile = curfile->duplicates_path;
				while ( tmpfile )
				{
					if ( isCanceled() ) break;

					inserted = false;

					QListIterator< QPair<KFileItemExt*, KFileItemExt*> * > list_iterator( list );
					for ( tmppair = list.first(); list_iterator.hasNext(); tmppair = list_iterator.next() ) // if two paths are identical, we have a pair.
					{                                                              // otherwise it's <item, NULL>.
						if ( tmppair->second == NULL )
						{
							if ( comparePath( tmpfile, tmppair->first ) == 0 ) // comparePath returns qstrcmp result; 0 if equal
							{
								tmppair->second = tmpfile;
								inserted = true;
							}
						}
					}

					if ( !inserted )
					{
						QPair<KFileItemExt*, KFileItemExt*> *pair2 = new QPair<KFileItemExt*, KFileItemExt*>;
						*pair2 = qMakePair( curfile, (KFileItemExt*)NULL );
						list.push_back( pair2 );
					}

					tmpfile = tmpfile->duplicates_path;
				}

				QListIterator< QPair<KFileItemExt*, KFileItemExt*> * > list_iterator2( list );
				for ( tmppair = list.first(); list_iterator2.hasNext(); tmppair = list_iterator2.next() )
				{
					if ( tmppair->second == NULL ) // a file that has no equivalent on the other side.
					{
						if ( m_enable_missing_search )
						{
							emit emitCompMissing( tmppair->first );
						}
					}
					else                          // found two identical paths. must check for newer file.
					{
						KFileItemExt *neweritem = newerItem( tmppair->first, tmppair->second );
						if ( neweritem != NULL )    // we have files with the same path+name, but different mod time...
						{
							QPair<KFileItemExt*, KFileItemExt*> *resultpair = new QPair<KFileItemExt*, KFileItemExt*>;

							if ( neweritem == tmppair->first ) *resultpair = qMakePair( tmppair->first, tmppair->second );
							else *resultpair = qMakePair( tmppair->second, tmppair->first );

							// check if files are duplicates.
							bool are_duplicates = false;

							if ( resultpair->first->size() == 0 && resultpair->second->size() == 0 ) are_duplicates = true;

							tmpfile = resultpair->first;
							while( tmpfile && !are_duplicates )
							{
								if ( tmpfile == resultpair->second ) are_duplicates = true;
								tmpfile = tmpfile->duplicates_size;
							}

							tmpfile = resultpair->second;
							while( tmpfile && !are_duplicates )
							{
								if ( tmpfile == resultpair->first ) are_duplicates = true;
								tmpfile = tmpfile->duplicates_size;
							}

							if ( m_enable_newer_search )
							{ // if !size we always find newer, not newer_equal
								if ( are_duplicates )
								{
									emit emitCompNewerEqual( resultpair->first, resultpair->second );
								}
								else
								{
									emit emitCompNewer( resultpair->first, resultpair->second );
								}
								delete resultpair;
								resultpair = NULL;
							}
							else
							{
								delete resultpair;
							}
						}
						else // ... find files that are different, but have the same timestamp
						{
							if ( isCanceled() ) break;

							// check if files are duplicates.
							bool are_duplicates = false;

							if ( tmppair->first->size() == 0 && tmppair->second->size() == 0 ) are_duplicates = true;

							tmpfile = tmppair->first;
							while( tmpfile && !are_duplicates )
							{
								if ( tmpfile == tmppair->second ) are_duplicates = true;
								tmpfile = tmpfile->duplicates_size;
							}

							tmpfile = tmppair->second;
							while( tmpfile && !are_duplicates )
							{
								if ( tmpfile == tmppair->first ) are_duplicates = true;
								tmpfile = tmpfile->duplicates_size;
							}

							if ( !are_duplicates )  // if !size we shouldn't report all the files as different.
							{                       // the rare case of equal but same time isn't considered as user disabled search by size.
								if ( m_size )
								{
									if ( m_enable_newer_search )
									{
										QPair<KFileItemExt*, KFileItemExt*> *resultpair = new QPair<KFileItemExt*, KFileItemExt*>;
										*resultpair = qMakePair( tmppair->first, tmppair->second );

										emit emitCompNewerSameTime( resultpair->first, resultpair->second );
										delete resultpair;
									}
								}
								else // we don't search for size duplicates, and the file doesn't belong to missing or newer list.
								{    // so we abuse the duplicates list view to display the duplicates by path. (no duplicates by size will be displayed!).
									if ( m_enable_duplicates_search )
									{
										tmppair->first->duplicates_size = tmppair->second;
										tmppair->second->dup_size_parent = tmppair->first;
										tmppair->first->hasdupes_size = 1;
										tmppair->first->isdupe_size = 1;
										tmppair->second->isdupe_size = 1;

										emit emitDuplicate( tmppair->first );
									}
								}
							}
						}
					}
				}
			}
			else
			{
				if ( !curfile->isdupe_path )
				{
					if ( m_enable_missing_search )
					{
						emit emitCompMissing( curfile );
					}
				}
			}

			current_file++;
			curfile = curfile->next;
		}

		emitProgress( status, -1 );
	}

	purgeTree( checktree_dupes );
	purgeTree( checktree_comp );

	if ( !isCanceled() )
	{
		emit emitCompFinished();
	}

	m_initialized = false;
}

int KomparatorCompareJob::registerFile(FileTree **branch, KFileItemExt *file)
{
	*branch = new FileTree();

	(*branch)->file = file;
	(*branch)->left = NULL;
	(*branch)->right = NULL;

	return 1;
}

KFileItemExt *KomparatorCompareJob::checkMatchDupes( FileTree **root, FileTree *checktree, KFileItemExt *file )
{
	if ( isCanceled() ) return NULL;

	int cmpresult;
	unsigned long fsize;

	// If inodes are equal one of the files is a hard link, which
	// is usually not accidental. We don't want to flag them as
	// duplicates, unless the user specifies otherwise.

//   if (!ISFLAG(flags, F_CONSIDERHARDLINKS) && getinode(file->d_name) ==
//    checktree->file->inode) return NULL;
// FIXME: Hard link handling

	fsize = file->size();

	if ( fsize < checktree->file->size() ) cmpresult = -1;
	else if ( fsize > checktree->file->size() ) cmpresult = 1;
	else if ( m_calculate_checksum )
	{
		QByteArray file_md5 = file->MD5( (KomparatorWidget*)m_parent );
		if ( isCanceled() ) return NULL;
		QByteArray checktree_file_md5 = checktree->file->MD5( (KomparatorWidget*)m_parent );
		cmpresult = qstrcmp( file_md5, checktree_file_md5 );
	}
	else cmpresult = 0;

	if ( cmpresult < 0 )
	{
		if ( checktree->left != NULL )
		{
			return checkMatchDupes( root, checktree->left, file );
		}
		else
		{
			registerFile( &(checktree->left), file );
			return NULL;
		}
	}
	else if ( cmpresult > 0 )
	{
		if ( checktree->right != NULL )
		{
			return checkMatchDupes( root, checktree->right, file );
		}
		else
		{
			registerFile( &(checktree->right), file );
			return NULL;
		}
	}
	else return checktree->file;
}

KFileItemExt *KomparatorCompareJob::checkMatchComp( FileTree **root, FileTree *checktree, KFileItemExt *file )
{
	if ( isCanceled() ) return NULL;

	int cmpresult = comparePath( file, checktree->file );

	if ( cmpresult < 0 )
	{
		if ( checktree->left != NULL )
		{
			return checkMatchComp( root, checktree->left, file );
		}
		else
		{
			registerFile( &(checktree->left), file );
			return NULL;
		}
	}
	else if ( cmpresult > 0 )
	{
		if ( checktree->right != NULL )
		{
			return checkMatchComp( root, checktree->right, file );
		}
		else
		{
			registerFile( &(checktree->right), file );
			return NULL;
		}
	}
	else return checktree->file;
}

bool KomparatorCompareJob::confirmMatchDupes( KFileItemExt *item1, KFileItemExt *item2 )
{
	if ( isCanceled() ) return false;

	m_mutex.lock();

	if ( item1->size() > m_binary_comparision_threshold*1048576 ||
			item2->size() > m_binary_comparision_threshold*1048576 )
	{
		m_mutex.unlock();
		return true; // Don't perform binary comparision for files bigger than *** MB = *** *1048576 bytes.
	}

	m_mutex.unlock();

	bool ret = true;

	QFile file1( item1->getFile( (KomparatorWidget*)m_parent ) );

	if ( isCanceled() )
	{
		item1->deleteFile( (KomparatorWidget*)m_parent );
		return false;
	}

	QFile file2( item2->getFile( (KomparatorWidget*)m_parent ) );

	if ( file1.fileName().isEmpty() || file2.fileName().isEmpty() )
	{
		item1->deleteFile( (KomparatorWidget*)m_parent );
		item2->deleteFile( (KomparatorWidget*)m_parent );
		return true;  // better return true on failure to not lose possible duplicates.
	}

	if ( !file1.open( QIODevice::ReadOnly ) )
	{
		emitMessage( i18n( "Error opening file %1." ).arg( file1.fileName() ), i18n( "Error while comparing files" ) );
		item1->deleteFile( (KomparatorWidget*)m_parent );
		item2->deleteFile( (KomparatorWidget*)m_parent );
		return true;
	}

	if ( !file2.open( QIODevice::ReadOnly ) )
	{
		emitMessage( i18n( "Error opening file %1." ).arg( file2.fileName() ), i18n( "Error while comparing files" ) );
		item1->deleteFile( (KomparatorWidget*)m_parent );
		item2->deleteFile( (KomparatorWidget*)m_parent );
		return true;
	}

	if ( isCanceled() )
	{
		file1.close();
		file2.close();

		item1->deleteFile( (KomparatorWidget*)m_parent );
		item1->deleteFile( (KomparatorWidget*)m_parent );
		return false;
	}

	uint buffer_size = 65536;
	int read_size1, read_size2;

	char buffer1[buffer_size];
	char buffer2[buffer_size];

	while ( ( read_size1 = file1.read( buffer1, buffer_size ) ) > 0 )
	{
		if ( ( read_size2 = file2.read( buffer2, buffer_size ) ) <= 0 )
		{
			ret = false;
			break;
		}

		if ( isCanceled() )
		{
			file1.close();
			file2.close();

			item1->deleteFile( (KomparatorWidget*)m_parent );
			item1->deleteFile( (KomparatorWidget*)m_parent );
			return false;
		}

		if ( qstrncmp( buffer1, buffer2, read_size1 ) != 0 || read_size1 != read_size2 )
		{
			ret = false;
			break;
		}
	}

	if ( ret && file2.read( buffer2, buffer_size ) > 0 ) ret = false;

	file1.close();
	file2.close();

	item1->deleteFile( (KomparatorWidget*)m_parent );
	item2->deleteFile( (KomparatorWidget*)m_parent );

	return ret;
}

int KomparatorCompareJob::comparePath( KFileItemExt *item1, KFileItemExt *item2 )
{
	if ( isCanceled() ) return 0;


	QString item1_str = KUrl::relativePath( item1->parent_path.path( KUrl::RemoveTrailingSlash ),
		item1->url().path() );
	QString item2_str = KUrl::relativePath( item2->parent_path.path( KUrl::RemoveTrailingSlash ),
		item2->url().path() );

	if ( item1_str.length() < item2_str.length() ) return -1;
	else if ( item1_str.length() > item2_str.length() ) return 1;
	else return QString::compare( m_case_sensitive ? item1_str : item1_str.toLower(), m_case_sensitive ? item2_str : item2_str.toLower() );

	return 1;
}

KFileItemExt *KomparatorCompareJob::newerItem( KFileItemExt *item1, KFileItemExt *item2 )
{
	if ( isCanceled() ) return NULL;

	if ( item1->time( KFileItem::ModificationTime ).toTime_t() > item2->time( KFileItem::ModificationTime ).toTime_t() )
	{
		return item1; // bigger time means newer
	}
	else if ( item1->time( KFileItem::ModificationTime ).toTime_t() < item2->time( KFileItem::ModificationTime ).toTime_t() )
	{
		return item2;
	}
	else
	{
		return NULL; // both have equal timestamps.
	}
}

void KomparatorCompareJob::purgeTree( FileTree *tree )
{
	if ( tree == NULL ) return;

	if ( tree->left != NULL ) purgeTree( tree->left );

	if ( tree->right != NULL ) purgeTree( tree->right );

	delete tree;
	tree = NULL;
}
