/***************************************************************************
 *   Copyright (C) 2007 by Todor Gyumyushev   *
 *   yodor@developer.bg   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "querydev.h"

#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <kprocio.h>
#include <qdir.h>
#include <qfile.h>


QueryDev::QueryDev(QObject *parent): QObject(parent), fdsock(0),nlsock(0), firstRunComplete(false)
{
	fdsock = socket(AF_INET, SOCK_DGRAM, 0);
	if(fdsock == -1)
	{
		throw QString("Can not create inet socket");
	}
	//used later in updates
	if((nlsock = socket(PF_NETLINK, SOCK_DGRAM, NETLINK_ROUTE)) < 0)
	{
		throw QString("Can not create netlink socket");
	}
	msgSeq=0;
	dev_list.setAutoDelete(true);
	

}
QueryDev::~QueryDev()
{
	//if (fdsock) delete fdsock;
	//if (nlsock) delete nlsock;
	fdsock=0;
	nlsock=0;
}




void QueryDev::queryDevices() 
{
	QMutexLocker lock(&mutex);

	struct nlmsghdr *nlMsg;
	struct ifinfomsg *ifInfo;
	struct rtattr *rtAttr;
	
	int len, rtLen, msgSeq = 0;


	memset(msgBuf, 0, NL_BUFSIZE);

	nlMsg = (struct nlmsghdr *)msgBuf;
	
	nlMsg->nlmsg_len = NLMSG_LENGTH(sizeof(struct ifinfomsg));
	nlMsg->nlmsg_type = RTM_GETLINK;
	nlMsg->nlmsg_flags = NLM_F_REQUEST | NLM_F_DUMP;
	nlMsg->nlmsg_seq = msgSeq++;
	nlMsg->nlmsg_pid = getpid();
	
	if(write(nlsock, nlMsg, nlMsg->nlmsg_len) < 0){
		
		throw QString("Can not write to netlink socket");
	}

	if((len = readSock(nlsock, msgBuf, msgSeq, getpid())) < 0){
		
		throw QString("Read netlink socket error");
	}
	
	

	for(;NLMSG_OK(nlMsg,len);nlMsg = NLMSG_NEXT(nlMsg,len))
	{
		ifInfo = (struct ifinfomsg *)NLMSG_DATA(nlMsg);
	
		rtAttr = (struct rtattr *)IFLA_RTA(ifInfo);
		rtLen = IFLA_PAYLOAD(nlMsg);
		for(;RTA_OK(rtAttr,rtLen);rtAttr = RTA_NEXT(rtAttr,rtLen))
		{
			if (rtAttr->rta_type == IFLA_IFNAME)
			{
				QString if_name((const char *)RTA_DATA(rtAttr));
				if (!dev_list[if_name])
				{
					KNetDockIf *n_dev = new KNetDockIf;
					n_dev->initDevice(if_name);
					dev_list.insert(if_name,n_dev);
					n_dev->devpresence=CHECKING;
					emit devAdded(if_name);
				}
				else {
					KNetDockIf *p_dev = dev_list[if_name];
					p_dev->devpresence=CHECKING;
				}
			}
		}	
	}

 	QDictIterator<KNetDockIf> it( dev_list );
    for( ; it.current(); ++it )
	{
		KNetDockIf *curr = it.current();
		QString dev_name = it.currentKey();

		if (curr->devpresence==CHECKING){
			curr->devpresence=PRESENT;
			update(dev_name);
			emit devUpdated(dev_name);
			continue;
		}
		//removeDevice(it.currentKey());
		emit devRemoved(dev_name);
		dev_list.remove(dev_name);
	}

	if (!firstRunComplete){
		firstRunComplete=true;
		emit firstRunCompleted();
	}
}


KNetDockIf *QueryDev::update(const QString& dev_name) 
{
	

	KNetDockIf *curr = dev_list[dev_name];
	if (!curr)
	{
		//qWarning("not found in list");
		return 0;
	}

	curr->iconstatus=0;

	bool m_link_down=false;
	

	QString essid="";

	curr->wstat.link_qual=0;
	curr->wstat.noise_level=0;
	curr->wstat.signal_level=0;	
	
	
	struct ifreq devifreq;
	struct ifconf ifc;

	char buffer[8192];
	ifc.ifc_len = sizeof(buffer);
	ifc.ifc_buf = buffer;

	
	strcpy(devifreq.ifr_name, dev_name);

		
	if (ioctl(fdsock,SIOCGIFFLAGS,&devifreq) < 0)
	{
		m_link_down=true;
		//qWarning("unable to query");
		//updateTip(if_name + "\nUnable to qurey flags");
		curr->iconstatus=4;
		return curr;
	}

	if ((devifreq.ifr_flags & IFF_LOOPBACK)!=0)
	{
		curr->devtype=LOOP;
	}

	if ((devifreq.ifr_flags & IFF_POINTOPOINT)!=0)
	{
		curr->devtype=PPP;
	}			
	
	struct iwreq wireq;

	strncpy(wireq.ifr_name, dev_name, IFNAMSIZ);

	if(ioctl(fdsock, SIOCGIWNAME, &wireq) == -1)
	{	}
	else
	{
		curr->devtype=WIFI;
	}

	if ((devifreq.ifr_flags & IFF_UP)!=0)
	{
		if ((devifreq.ifr_flags & IFF_RUNNING)!=0 )
		{
			//qWarning("up running");
			m_link_down=false;
			curr->devstate = UP;
		}
		else
		{
			//qWarning("link down");
			m_link_down=true;
			curr->devstate = NOLINK;
		}
	}
	else
	{
		//qWarning("not up");
		m_link_down=true;
		curr->devstate = DOWN;
		
	}
	if (ioctl(fdsock, SIOCGIFMTU, &devifreq)<0)
	{
		curr->mtu="";
	}
	else
	{
		curr->mtu=QString::number(devifreq.ifr_ifru.ifru_mtu);
	}
		
	if (ioctl(fdsock, SIOCGIFADDR, &devifreq)<0) 
	{
		
		curr->addrstat.ip="Off";
	}
	else
	{
		sockaddr_in sin = ((sockaddr_in&)devifreq.ifr_addr);
		curr->addrstat.ip=QString(inet_ntoa(sin.sin_addr));
		
	}	

	if (ioctl(fdsock, SIOCGIFNETMASK, &devifreq)<0) 
	{
		curr->addrstat.mask="Off";
	}
	else
	{
		sockaddr_in mask = ((sockaddr_in&)devifreq.ifr_netmask);
		curr->addrstat.mask=QString(inet_ntoa(mask.sin_addr));
	}
			
	if(ioctl(fdsock, SIOCGIFDSTADDR, &devifreq)<0)
	{
		curr->addrstat.ptp="";
	}
	else
	{
		sockaddr_in sin = ((sockaddr_in&)devifreq.ifr_addr);
		curr->addrstat.ptp=QString(inet_ntoa(sin.sin_addr));
	}

	if(ioctl(fdsock, SIOCGIFHWADDR,  &devifreq)<0)
	{
		
		curr->addrstat.mac="Off";
	}
	else
	{
		unsigned char *hw = (unsigned char*)devifreq.ifr_hwaddr.sa_data;

		QString mac = QString("%1:%2:%3:%4:%5:%6").arg(hw[0],2,16)
				.arg(hw[1],2,16)
				.arg(hw[2],2,16)
				.arg(hw[3],2,16)
				.arg(hw[4],2,16)
				.arg(hw[5],2,16);
		mac.replace(' ', '0');
		curr->addrstat.mac=mac;
	}
			//emit updateInfo(allifs[v].addrstat);



	
	if (curr->devtype==WIFI){
		
		curr->wstat.ap="Unassociated";
		curr->wstat.essid=QString("Off/Any");
		
		struct iw_statistics wifistats;
	
		wireq.u.data.pointer = (caddr_t) &wifistats;
		wireq.u.data.length = sizeof(struct iw_statistics);
		wireq.u.data.flags = 1;
			
		
		if(ioctl(fdsock, SIOCGIWSTATS, &wireq) < 0)
		{
			
		}
		else
		{

			if(ioctl(fdsock, SIOCGIWAP, &wireq) <0)
			{}
			else
			{
				//memcpy(&(info.bitrate), &(wrq.u.bitrate), sizeof(iwparam));
				sockaddr a = wireq.u.ap_addr;
				unsigned char* hw = (unsigned char*)a.sa_data;

 				QString mac = QString("%1:%2:%3:%4:%5:%6").arg(hw[0],2,16)
				.arg(hw[1],2,16)
				.arg(hw[2],2,16)
				.arg(hw[3],2,16)
				.arg(hw[4],2,16)
				.arg(hw[5],2,16);
				mac.replace(' ', '0');
				if (QString::compare(mac,"00:00:00:00:00:00")==0)mac="Unassociated";
				curr->wstat.ap=mac;
			
			}

			if(ioctl(fdsock, SIOCGIWTXPOW, &wireq) <0)
			{}
			else
			{
				//memcpy(&(info.bitrate), &(wrq.u.bitrate), sizeof(iwparam));
				if (wireq.u.txpower.disabled)
					curr->wstat.txpow = -1;
				else 
					curr->wstat.txpow = wireq.u.txpower.value;
			}

			if(ioctl(fdsock, SIOCGIWRATE, &wireq) <0)
			{}
			else
			{
				//memcpy(&(info.bitrate), &(wrq.u.bitrate), sizeof(iwparam));
				curr->wstat.bitrate = wireq.u.bitrate.value;
			}
	
			//continue with range struct
			struct iw_range *range;

			memset(rngbuf, 0, sizeof(rngbuf));
			
			wireq.u.data.pointer = (caddr_t) rngbuf;
			wireq.u.data.length = sizeof(rngbuf);
			wireq.u.data.flags = 0;

			if(ioctl(fdsock, SIOCGIWRANGE, &wireq) < 0)
			{
				
			}
			else
			{
				/* Copy stuff at the right place, ignore extra */
				range = (struct iw_range *) rngbuf;						
				if(ioctl(fdsock,  SIOCGIWESSID, &wireq) < 0) {}
				else {
					if (wireq.u.essid.length>0){
						const char * nm = (const char *)wireq.u.essid.pointer;
						curr->wstat.essid=QString(nm);
					}
				}

				int rqn = range->max_qual.noise;
				int rql = range->max_qual.level;
				int rmq = range->max_qual.qual;
			
				// in case the quality levels are zero, do not allow division by zero, and
				// instead set the max. possible quality range to 255 (max of "unsigned char")
				if (!rqn) rqn = 255;
				if (!rql) rql = 255;
				if (!rmq) rmq = 255;
				
				bool isAbs; // false if values are relative to a peak value,
							// otherwise values are dBm
				isAbs = (float)wifistats.qual.level > (int)range->max_qual.level;
				if(isAbs)
				{
					curr->wstat.noise_level  = wifistats.qual.noise - 0x100;
					curr->wstat.signal_level = wifistats.qual.level - 0x100;
					curr->wstat.dbm=true;
				} else {
					curr->wstat.noise_level  =  (wifistats.qual.noise / rqn) * 100;
					curr->wstat.signal_level =  (wifistats.qual.level / rql) * 100;
					curr->wstat.dbm=false;
				}
			
				curr->wstat.link_qual = (((double)wifistats.qual.qual) / rmq) * 100;
			
			}
		}
		
		
	}//is wifi

	if (m_link_down)
	{
		return curr;
	}


	struct nlmsghdr *nlMsg;
	memset(msgBuf, 0, NL_BUFSIZE);
	nlMsg = (struct nlmsghdr *)msgBuf;
	
	
	/* For getting interface addresses */
	nlMsg->nlmsg_len = NLMSG_LENGTH(sizeof(struct ifinfomsg));
	nlMsg->nlmsg_type = RTM_GETLINK;
	nlMsg->nlmsg_flags = NLM_F_REQUEST | NLM_F_DUMP;
	nlMsg->nlmsg_seq = msgSeq++;
	nlMsg->nlmsg_pid = getpid();

	if(write(nlsock, nlMsg, nlMsg->nlmsg_len) < 0)
	{
		//disableDevice();
		return 0;
	}

	int len;
	if((len = readSock(nlsock, msgBuf, msgSeq, getpid())) < 0)
	{
		//disableDevice();
		return 0;
	}
	
	struct ifinfomsg *ifInfo;
	struct rtattr *rtAttr;
	struct rtnl_link_stats *stats;
	

	int rtLen;
	
	Q_ULLONG byte_sum = 0;

	bool recv_up=false;
	bool send_up=false;
	bool fnd=false;
		

	

	for(;NLMSG_OK(nlMsg,len);nlMsg = NLMSG_NEXT(nlMsg,len))
	{
		ifInfo = (struct ifinfomsg *)NLMSG_DATA(nlMsg);		
		rtAttr = (struct rtattr *)IFLA_RTA(ifInfo);		
		rtLen = IFLA_PAYLOAD(nlMsg);
	
		for(;RTA_OK(rtAttr,rtLen);rtAttr = RTA_NEXT(rtAttr,rtLen))
		{
			switch (rtAttr->rta_type)
			{

				case IFLA_IFNAME:
					{
						const char *nm = (const char *)RTA_DATA(rtAttr);
						if (nm == dev_name)
						{
							fnd=true;
						}
					}
					break;

				case IFLA_STATS:
					if (fnd)
					{
						
						stats = (struct rtnl_link_stats*)RTA_DATA(rtAttr);
						//long tx = stats->tx_packets;
						//qDebug(QString("statrx:%1").arg(tx));


						memcpy(&curr->stats, stats,sizeof(struct rtnl_link_stats));

						byte_sum = (Q_ULLONG)(stats->rx_bytes + stats->tx_bytes);

						

						if (curr->oldstat.bytes_in>stats->rx_bytes)
							curr->oldstat.bytes_in=0;
						if (curr->oldstat.bytes_out>stats->tx_bytes)
							curr->oldstat.bytes_out=0;
						
						Q_ULLONG indelta=stats->rx_bytes-curr->oldstat.bytes_in;
						Q_ULLONG outdelta=stats->tx_bytes-curr->oldstat.bytes_out;

						if (indelta > 0)
						{
							recv_up=true;
							curr->iconstatus=1;
						}
						if (outdelta > 0)
						{
							send_up=true;
							curr->iconstatus=2;
						}
						if (recv_up && send_up) curr->iconstatus=3;
						
						curr->total.bytes_in+=indelta;
						curr->total.bytes_out+=outdelta;
						curr->total.bytes=curr->total.bytes_in+curr->total.bytes_out;

						curr->oldstat.bytes_in=stats->rx_bytes;
						curr->oldstat.bytes_out=stats->tx_bytes;
					
						curr->total.speed=(indelta+outdelta);
						curr->total.speed_in=indelta;
						curr->total.speed_out=outdelta;

					}
					break;
			} //switch 
			
		} // rtAttr
		if (fnd)
		{
			break;
		}
	} //nlMsg
	
	return curr;
}


int QueryDev::readSock(int sockFd, char *bufPtr, int seqNum, int pId)
{
        struct nlmsghdr *nlHdr;
        int readLen = 0, flag = 0, msgLen = 0;

        do{
                /* Recieve response from kernel */
                if((readLen = recv(sockFd, bufPtr, NL_BUFSIZE - msgLen, 0)) < 0){
                        perror("SOCK READ: ");
                        return -1;
                }
                nlHdr = (struct nlmsghdr *)bufPtr;

                /* Check if header is valid */
                if((NLMSG_OK(nlHdr, readLen) == 0) || (nlHdr->nlmsg_type == NLMSG_ERROR)){
                        perror("Error in recieved packet");
                        return -1;
                }

                /* Check if its the last message */
                if(nlHdr->nlmsg_type == NLMSG_DONE)     {
                        flag = 1;
                        break;
                }
                else{
                        /* Move the buffer pointer appropriately */
                        bufPtr += readLen;
                        msgLen += readLen;
                }
                if((nlHdr->nlmsg_flags & NLM_F_MULTI) == 0){
                        flag = 1;
                        break;
                }
        } while((nlHdr->nlmsg_seq != seqNum) || (nlHdr->nlmsg_pid != pId) || (flag == 0));
        return msgLen;
}


DeviceList QueryDev::getDeviceList()
{
	//QMutexLocker lock(&mutex);
	return dev_list;

}

KNetDockIf* QueryDev::device(const QString & dev_name)
{
	KNetDockIf *device = dev_list[dev_name];
	return	device;
}




//some kode borrowed from kwifimanager
void QueryDev::wifiScan(const QString& dev_name, WifiNetworkList& list)
{
	QMutexLocker lock(&mutex);

	list.clear();

 	KProcIO iwlist;
   
    QString iwlist_bin = KStandardDirs::findExe("iwlist");
    if(iwlist_bin.isEmpty())
      iwlist_bin = KStandardDirs::findExe("iwlist", "/usr/local/sbin:/usr/sbin:/sbin");
    if(iwlist_bin.isEmpty())
      iwlist_bin = "iwlist"; // try our best ;/

    iwlist << iwlist_bin << dev_name << "scanning";

//    connect ( iwlist, SIGNAL ( readReady ( KProcIO * ) ), this, SLOT ( parseScanData ( KProcIO * ) ) );

   if ( !iwlist.start ( KProcess::Block ) )
        KMessageBox::sorry ( 0,  "Unable to perform the scan. Please make sure the executable \"iwlist\" is in your $PATH." , "Scanning not possible"  );

    // this should never happen. But there was a report about Block not being as blocking as it should, so let's be safe about it
    while (iwlist.isRunning()) sleep ( 1 );

	QString data;
	
	WifiNetwork *curr = 0;
	int count = 0;

	while ( iwlist.readln ( data, true ) >= 0 )
	{
		
		if ( data.contains ( "t support scanning" ) ){
			KMessageBox::sorry ( 0,	 "Your card does not support scanning. The results window will not contain any results." ,  "Scanning not possible"  );
			list.clear();
			break;
		}
		if ( data.contains ( "Scan completed" ) ) continue;

		

		if ( data.contains ( "Cell" ) ) {
			
			//networks->setNumRows ( networks->numRows() +1 );
			WifiNetwork *m_net = new WifiNetwork;
			list.insert(QString::number(count), m_net );
			curr = m_net;
			
			curr->index_pos=count;
			QString ap_mac = data.mid ( data.find ( "Address:" )+8 );
			curr->ap_mac=ap_mac.stripWhiteSpace();
			curr->full="";
			count++;
			
		}
		if (curr)curr->full+=data.stripWhiteSpace()+"\n";

		if (data.contains("Bit Rates:"))
		{
			QString bitrates = data.mid ( data.find ( ":" ) + 1);
			curr->bitrates=bitrates;
			while ( iwlist.readln ( data, true ) >= 0 )
			{
				if (data.contains(":") || data.contains("="))break;
				curr->bitrates+=data;
				curr->full+=data.stripWhiteSpace();
			}
			curr->full+="\n";
		}
		if ( data.contains ( "ESSID:" ) ) {
			QString ssid = data.mid ( data.find ( "\"" ) + 1, data.length (  ) - data.find ( "\"" ) - 2 );
			if ((ssid=="") || (ssid==" ")) ssid = "Hidden";
			curr->essid=ssid;
		}
		else if (data.contains ("Channel:")){
			QString channel = data.mid ( data.find ( ":" ) + 1);
			curr->channel=channel.toInt();
		}
		else if ( data.contains ( "Mode:" ) ) {
			QString mode_text = "Other";
			if ( data.contains ( "Master" ) ) mode_text = "Managed";
			if ( data.contains ( "Ad-Hoc" ) ) mode_text =  "Ad-Hoc";
			curr->mode = mode_text;	
		}
		else if ( data.contains ( "Encryption key:" ) ) {
			if ( data.contains ( "off" ) )curr->encryption=false;
			else curr->encryption=true;
		}
		else if (data.contains("Extra:")){
			QString extra = data.mid ( data.find ( ":" ) + 1);
			curr->extra=extra;
		}
		else if ( data.contains ( "Quality" ) ) {
			int spos = data.find ( "Quality" )+8;
			int epos = data.find ("/");
			int len = epos-spos;
			QString quality = data.mid ( spos, len );
			curr->quality=quality.toInt();
			QString level = data.mid(data.find("level")+1);
			curr->level=level;
		}
		else if ( data.contains ( "Protocol:" ) ) {
			QString proto = data.mid ( data.find ( ":" ) + 1 );
			curr->protocol=proto;
		}
		
	}
	
}


