/***************************************************************************
                       kcontainerimpl.h  -  description
                             -------------------
    begin                : Mon Oct 16 2000
    copyright            : (C) 2000 by Sergio Moretti
    email                : sermore@libero.it
    revision             : $Revision: 1.12 $
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KCONTAINERIMPL_H
#define KCONTAINERIMPL_H

#include <qlist.h>
#include <qintdict.h>
#include <qdir.h>
#include "kobjectimpl.h"

class KContainer;

typedef QIntDict<KObjectImpl> ObjDict;
typedef QIntDictIterator<KObjectImpl> ObjDictIterator;
typedef QValueList<int> IntList;
typedef QListIterator<KObjectImpl> ObjIterator;

/** 
 * @short Abstract base class for container implementation.
 * 
 * @author Sergio Moretti
 * 
 * @version $Id: kcontainerimpl.h,v 1.12 2001/03/18 16:31:03 sergio Exp $
 *
 * Childs are contained in a double linked list of @ref KObjectImpl objects, 
 * holds a dictionary containing one global object per type, on which is
 * maintained the global configuration for that type.
 */

class KContainerImpl : public KObjectImpl 
{
   Q_OBJECT

   friend class KContainer;

   static const char DOCID[];
   static const char DOCTYPE[];
   
protected:
   enum NewType {
      NEW_DEFAULT,
      NEW_GLOBAL,
      NEW_ITEM
   };

public:
   typedef QList<KObjectImpl> List;

   class Iterator : public QListIterator<KObjectImpl> 
   {
   public:
      Iterator(const KContainerImpl &c) 
	 : QListIterator<KObjectImpl>(c._items) {}
   };
   friend class Iterator;

public:
   virtual ~KContainerImpl();

   /** document name */
   virtual const char * docId() const { return DOCID; }

   /** dom document type */
   virtual const char * docType() const { return DOCTYPE; }

   /** return the global instance associate to this object */
   const KContainerImpl * global() const;

   // operations on globals

   /** dictionary containing global instances for childs */
   virtual const ObjDict & globals() const { return _glbDict; }

   /** find a global instance for childs of type "type" */
   virtual KObjectImpl * findGlobal(int type);

   virtual KObjectImpl * globalLoad(const QDomElement &e);
   
   virtual void initGlobals();

   // operations on this object

   /** start as no op */
   virtual bool start() { return false; }

   /** kill all active items */
   virtual bool kill(bool wait = false);

   virtual void runPeriodically();

   /** active if a child is active */
   virtual bool isRunning() const { return itemActiveCount() > 0; }

   // Container Operations

   /** initialize and load items */
   virtual void initItems();

   int itemType() const { return _itemType; }

   IntList itemTypes() const;

   /** create a new default or global object */
   virtual KObjectImpl * objectNew(int type, int newType);

   virtual bool isEmpty() const { return _items.isEmpty(); }

   virtual void clear(bool destroying = false);

   virtual KObjectImpl * itemFindId(int id);

   virtual int itemFind(const KObjectImpl *item) { return _items.find(item); }

   virtual KObjectImpl * itemAt(int index) { return _items.at(index); }

   virtual KObjectImpl * itemFirst() { return _items.first(); }

   virtual KObjectImpl * itemLast() { return _items.last(); }

   //virtual KObjectImpl * itemNext() { return _items.next(); }

   //virtual KObjectImpl * itemPrev() { return _items.prev(); }

   virtual int itemCount() const { return _items.count(); }

   //virtual int itemActiveCount() const { return _itemActiveCount; }

   /** add an item */
   virtual bool itemAdd(KObjectImpl *item);

   /** create an uninitialized item */
   //virtual KObjectImpl * itemCreate(int type);

   /** load an item from DOM element, and associate DOM element to object */
   virtual KObjectImpl* itemLoad(const QDomElement &e);

   virtual bool itemMove(KObjectImpl *item, KContainerImpl *to);

   //virtual KObjectImpl* itemNew(int type);

   /** call to check that item is insertable */
   virtual bool isItemInsertable(const KObjectImpl *) { return true; }

   /** activate/deactivate an item */
   virtual void incrItemActive(int v);

   /** copy an item from DOM element */
   virtual KObjectImpl * itemCopy(const QDomElement &e);

   /** remove an item */
   virtual void itemRemove(KObjectImpl *item);

   /** search an item in the container hierarchy */
   virtual KObjectImpl * itemSearch(int id);

   virtual int itemActiveCount() const { return _itemActiveCount; }

   // config entries

   PriorityPolicy getPriorityPolicy() const;
   void setPriorityPolicy(PriorityPolicy p);

protected:
   // Object Creation and Initialization
   KContainerImpl(int type);

   /** dom element containing global instances */
   virtual QDomElement domGlobals() const { return _domGlobals; }

   /** dom element containing item instances */
   virtual QDomElement domItems() const { return _domItems; }

   virtual void loadData();

private:
   /** items list */
   List _items;
   /** dom element containing items */
   QDomElement _domItems;
   /** dictionary of global items */
   ObjDict _glbDict;
   /** dom element containing global items */
   QDomElement _domGlobals;
   /** type of items */
   int _itemType;
   /** number of active items */
   int _itemActiveCount;
  
   // config entries

   /** priority policy */
   PriorityPolicy _cfg_priorityPolicy;
};

#endif // KCONTAINERIMPL_H

/* Local Variables: */
/* mode: c++ */
/* End: */
