#include <config.h>
/*
 * RPOT.c
 *
 * Contains the Resident Persistent Object Table.
 *
 */

#ifdef EVICTIONS

#ifndef GARBAGE
#error GARBAGE must be defined for TPOT.c
#endif

#ifdef PERSIST
/*#ifndef PERSIST
#error PERSIST must be defined for RPOT.c
#endif*/

#ifdef DEBUG
#include <assert.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <global.h>


#include "garbage.h"
#include "store.h"
#include "rpot.h"

#ifdef REVRPOT
tRPOTElement** TPOT_table = NULL;
tRPOTElement** TPOT_revtable = NULL;
#else
tRPOTElement* TPOT_table = NULL;
#endif
static int iNumItems;


/*
 * Initialises the TPOT_table.
 *
 */

void TPOT_Init
(
  void
)
{
  int i;

  #ifdef REVRPOT
  TPOT_revtable = (tRPOTElement**) malloc(sizeof(tRPOTElement*) * TPOT_TABLESIZE);
  TPOT_table = (tRPOTElement**) malloc(sizeof(tRPOTElement*) * TPOT_TABLESIZE);
  memset(TPOT_revtable, 0, sizeof(tRPOTElement*) * TPOT_TABLESIZE);
  memset(TPOT_table, 0, sizeof(tRPOTElement*) * TPOT_TABLESIZE);
  #else

  TPOT_table = (tRPOTElement*) malloc(sizeof(tRPOTElement) * TPOT_TABLESIZE);
  for (i = 0; i < TPOT_TABLESIZE; i++)
  {
    TPOT_table[i].pid = NULL;
    TPOT_table[i].la = NULL;
    TPOT_table[i].next = NULL;
  }
  #endif

  iNumItems = 0;
}

/*
 * Reclaims all space used by the TPOT_table.
 *
 */

void TPOT_Kill
(
  void
)
{
  int i;
  tRPOTElement* temp;
  tRPOTElement* next;

  #ifdef REVRPOT
  /* go through forward TPOT_table and delete all elements */
  /* we won't need to do this with the reverse TPOT_table
     because all the elements will have been killed
     by going through the forward TPOT_table */
  for (i = 0; i < TPOT_TABLESIZE; i++)
  {
    if (TPOT_table[i])
    {
      temp = TPOT_table[i]->next;
      free(TPOT_table[i]);
      while (temp)
      {
        next = temp->next;
        free(temp);
        temp = next;
      }
    }
  }
  free(TPOT_table);
  free(TPOT_revtable);
  #else
  for (i = 0; i < TPOT_TABLESIZE; i++)
  {
    temp = TPOT_table[i].next;
    while (temp)
    {
      next = temp->next;
      free(temp);
      temp = next;
    }
  }
  free(TPOT_table);
  #endif
}

/*
 * Returns the LA of the object if it is in the TPOT_table.
 *
 */

tObject* TPOT_GetLA
(
  PID pid
)
{
  tRPOTElement* temp;

  #ifdef REVRPOT
  if (TPOT_table[HASHFUNC(pid)] == NULL)
  {
    return NULL;
  }
  if ((temp = TPOT_table[HASHFUNC(pid)])->pid == pid)
  {
    return temp->la;
  }
  #else
  if ((temp = &(TPOT_table[HASHFUNC(pid)]))->pid == pid)
  {
    return temp->la;
  }
  #endif

  temp = temp->next;
  while (temp)
  {
    if (temp->pid == pid)
    {
      return temp->la;
    }
    temp = temp->next;
  }

  return NULL;
}

#ifdef REVRPOT
/*
 * Returns the LA of the object if it is in the TPOT_table.
 *
 */

PID TPOT_GetPID
(
  void* la
)
{
  tRPOTElement* temp;

  if (TPOT_revtable[REVHASHFUNC(la)])
  {
    if ((temp = TPOT_revtable[REVHASHFUNC(la)])->la == la)
    {
      return temp->pid;
    }

    temp = temp->revnext;
    while (temp)
    {
      if (temp->la == la)
      {
        return temp->pid;
      }
      temp = temp->revnext;
    }
  }

  return NULL;
}
#endif

/*
 * Adds a PID/LA pair to the TPOT_table. Does not check for duplicates because
 * this should only be called when a search for the entry has failed.
 *
 */

tRPOTElement* TPOT_Add
(
  PID pid,
  void* la
)
{
  tRPOTElement* temp;

  #ifdef REVRPOT
  int index;
  tRPOTElement* revtemp;

  index = HASHFUNC(pid);
  if (temp = TPOT_table[index])
  {
    while (temp->next)
    {
      temp = temp->next;
    }
    temp->next = (tRPOTElement*) malloc(sizeof(tRPOTElement));
    temp = temp->next;
    temp->pid = pid;
    temp->la = la;
    temp->next = NULL;
    temp->revnext = NULL;
  }
  else
  {
    TPOT_table[index] = (tRPOTElement*) malloc(sizeof(tRPOTElement));
    temp = TPOT_table[index];
    temp->pid = pid;
    temp->la = la;
    temp->next = NULL;
    temp->revnext = NULL;
  }

  index = REVHASHFUNC(temp->la);
  if (revtemp = TPOT_revtable[index])
  {
    while (revtemp->revnext)
    {
      revtemp = revtemp->revnext;
    }
    /* just point the next pointer to the existing element */
    revtemp->revnext = temp;
  }
  else
  {
    /* point straight to existing element */
    TPOT_revtable[index] = temp;
  }
  #else

  if ((temp = &(TPOT_table[HASHFUNC(pid)]))->pid)
  {
    while (temp->next)
    {
      temp = temp->next;
    }
    temp->next = (tRPOTElement*) malloc(sizeof(tRPOTElement));
    temp = temp->next;
    temp->pid = pid;
    temp->la = la;
    temp->next = NULL;
  }
  else
  {
    temp->pid = pid;
    temp->la = la;
  }
  #endif

  iNumItems += 1;

  return temp;
}

/*
 * Removes an entry from the TPOT_table. Indexed by PID.
 *
 */

void TPOT_Remove
(
  PID pid
)
{
  tRPOTElement* temp;
  tRPOTElement* prev;

  #ifdef REVRPOT
  int index;
  void* la;

  index = HASHFUNC(pid);
  /* find the element */
  for (prev = NULL, temp = TPOT_table[index]; temp->pid != pid; prev = temp, temp = temp->next);
  /* remember la for later */
  la = temp->la;
  if (prev)
  {
    prev->next = temp->next;
  }
  else
  {
    TPOT_table[index] = temp->next;
  }
  /* don't free it yet - have to do the reverse version first */

  index = REVHASHFUNC(la);
  for (prev = NULL, temp = TPOT_revtable[index]; temp->pid != pid; prev = temp, temp = temp->revnext);
  if (prev)
  {
    prev->revnext = temp->revnext;
  }
  else
  {
    TPOT_revtable[index] = temp->revnext;
  }
  /* now free it */
  free(temp);

  #else
  tRPOTElement* next;

  if ((temp = &(TPOT_table[HASHFUNC(pid)]))->pid == pid)
  {
    if (temp->next)
    {
      temp->pid = temp->next->pid;
      temp->la = temp->next->la;
      next = temp->next;
      temp->next = temp->next->next;
      free(next);
    }
    else
    {
      temp->pid = NULL;
      temp->la = NULL;
    }
  }
  else
  {
    prev = temp;
    temp = temp->next;
    #ifdef DEBUG
    assert(temp);
    #endif
    while (temp->pid != pid)
    {
      prev = temp;
      temp = temp->next;
      #ifdef DEBUG
      assert(temp);
      #endif
    }
    prev->next = temp->next;
    free(temp);
  }

  #endif

  iNumItems -= 1;
}

tOBREF TPOT_ChangeLA
(
  PID pid,
  void* newLA
)
{
  tRPOTElement* temp;

  #ifdef REVRPOT
  void* la;
  /* in reverse RPOT case we must remove and re-insert so that the reverse
     hashing can be fixed up */
  la = TPOT_Remove(pid);
  if (la == NULL)
  {
    fprintf(stderr, "TPOT tried to change LA of PID %d which doesn't exist!\n", pid);
    exit(1);
  }
  TPOT_Add(pid, newLA);
  return;
  #else

  if ((temp = &(TPOT_table[HASHFUNC(pid)]))->pid == pid)
  {
    temp->la = newLA;
    return (tOBREF) &(temp->la);
  }
  temp = temp->next;
  while (temp)
  {
    if (temp->pid == pid)
    {
      temp->la = newLA;
      return (tOBREF) &(temp->la);
    }
    temp = temp->next;
  }

  fprintf(stderr, "TPOT tried to change LA of PID %d which doesn't exist!\n", pid);
  exit(1);
  return NULL;
  #endif
}


#endif

#else
int asdlf;
#endif
