#include <config.h>
#include "thread.h"
#include "wrappers.h"
#include <assert.h>

#ifndef TEASEME
int pthread_mutexattr_setkind_np (pthread_mutexattr_t* kind, int tkind);
#endif

/* 8 March 2001 -- jewel
 *
 * I want to make sure that we keep proper track of what Java stack frames are alive at any given time.
 *
 * The problem is that certain methods call Interpret to do something like call a constructor method (Strings for eg)
 * and other's just call Interpret because that's what they're meant to do (called from JNI).
 *
 * So, what I want to do is maintain state for each thread. This has already been done with the tthreadnode structure, but
 * it fails to record what Java stack frames belong to that thread.
 * 
 * So now every tthreadnode should keep a list of intial stack frames. Every time Interpret is called, a new initial stack frame will
 * be added to the list for that thread, and when Interpret returns, that stack frame is removed.
 * 
 * This way the Garbage collector can get every single Java stack frame for a thread.
 *
 *
 */

extern volatile uint32 iLiveThreads;

/* Life of a thread 
 *
 * 1. Thread is created
 * 2. Immediately create a tThreadNode for that thread
 * 3. Add that node to the list of nodes
 * 4. Every time Interpret is called for at thread, add a new InitialStackFrame to the node
 * 5. When Interpret exits, remove that InitialStackFrame
 * 6. When a thread terminates, remove its node
 *
 * 7. If a thread object is allocated for that thread, set that info in the thread node
 */

tThreadNode* THREADINFO_CreateNewNodeForThread(int thread_id, void* nstackBottom)
{
  tThreadNode* nnode = (tThreadNode *) sys_malloc (sizeof(tThreadNode));

#ifndef TEASEME
  pthread_mutexattr_t kind;

  pthread_mutexattr_init (&kind);
  pthread_mutexattr_setkind_np (&kind,
				PTHREAD_MUTEX_RECURSIVE_NP);
#endif

  if(nnode == NULL)
    return NULL;

  nnode->thread_id = thread_id;
  nnode->object = NULL;
  nnode->next = NULL;
  nnode->pstInitialFrames = LLIST_Create_List();
  nnode->stackBottom = nstackBottom;
  nnode->stackTop = NULL;
  nnode->isDaemon = 0;
  nnode->isTerminated = 0;

  nnode->i32ContentionBit = 0;
  nnode->mutContentionLock = sys_mutex_create(1);
  nnode->pstWaitList = NULL;

  return nnode;
}

int THREADINFO_SetNodeInfo(tThreadNode* node, tOBREF obj, void* stackBottom)
{
 node->object = obj;
 node->stackBottom = stackBottom;
 if(node->stackTop == 0)
   {
     node->stackTop = stackBottom;
   }
 return 0;
}

int THREADINFO_SetNodeInfoForCurrentThread(tOBREF obj, void* stackBottom)
{
 tThreadNode* node;
 node = THREAD_FindThread();
 node->object = obj;
 node->stackBottom = stackBottom;
 return 0;
}

/* Called when a new thread is started, this must be kept on a per process basis */

int THREADINFO_IncrementThreadCount()
{
  sys_mutex_lock(THREAD_getThreadListMutex());
  iLiveThreads++;
  sys_mutex_unlock(THREAD_getThreadListMutex());
  return iLiveThreads;
}

/* Called when a thread dies */

int THREADINFO_DecrementThreadCount()
{
  sys_mutex_lock(THREAD_getThreadListMutex());
  iLiveThreads--;
  sys_mutex_unlock(THREAD_getThreadListMutex());
  return iLiveThreads;
}

int THREADINFO_addInitialStackFrameToNode(tThreadNode* node, tStackFrame* initial_frame)
{
  tList* list;
  sys_mutex_lock(THREAD_getThreadListMutex());
#ifdef TEASEME
  assert(initial_frame->pstHeap);
#endif
  list = node->pstInitialFrames;
  assert(LLIST_AddAtHead(list, initial_frame) == 0);
  sys_mutex_unlock(THREAD_getThreadListMutex());
  return 0;
}

int THREADINFO_removeInitialStackFrameFromNode(tThreadNode* node, tStackFrame* initial_frame)
{
  tList* list;
  sys_mutex_lock(THREAD_getThreadListMutex());
#ifdef TEASEME
  assert(initial_frame->pstHeap);
#endif
  list = node->pstInitialFrames;
  assert(LLIST_Remove(list, initial_frame) == 0);
  sys_mutex_unlock(THREAD_getThreadListMutex());
  return 0;
}

int THREADINFO_countInitialStackFramesFromNode(tThreadNode* node)
{
  int ret;
  tList* list;
  sys_mutex_lock(THREAD_getThreadListMutex());
  list = node->pstInitialFrames;
  ret = LLIST_Size(list);
  sys_mutex_unlock(THREAD_getThreadListMutex());
  return ret;
}

tStackFrame* THREADINFO_getInitialStackFrameFromNode(tThreadNode* node, int index)
{
  tList* list;
  tStackFrame* ret;
  sys_mutex_lock(THREAD_getThreadListMutex());
  list = node->pstInitialFrames;
  ret = LLIST_ElementAt(list, index);
  sys_mutex_unlock(THREAD_getThreadListMutex());
  return ret;
}

int THREADINFO_addThreadNodeToList(tThreadNode* node)
{
  tThreadNode* prev = NULL;
  tThreadNode* iter;

  sys_mutex_lock(THREAD_getThreadListMutex());

  iter = THREAD_getThreadList();
  if(iter == NULL)
    {
      //add to head
      THREAD_setThreadList(node);
      node->next = NULL;
      sys_mutex_unlock(THREAD_getThreadListMutex());
      return 0;
    }
  else
    {
     while(iter != NULL)
	 {
	     prev = iter;
	     iter = iter->next;
	 }

     assert(prev != NULL);
     prev->next = node;
     prev->next->next = NULL;
     sys_mutex_unlock(THREAD_getThreadListMutex());
      return 0;
    }
  //should never get here
  sys_mutex_unlock(THREAD_getThreadListMutex());
  return -1;
}

