#include <config.h>

#ifdef STACKMAP 


#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include "../interp.h"
#include "../disass.h"
#include "../sys_linux_host/wrappers.h"
#include "stackmap.h"
#include "mapencoder.h"

void buildHistogram();


inline struct tstackmap* STACKMAP_startEncoding() {

  struct tstackmap* retval = (struct tstackmap*) sys_malloc( sizeof(struct tstackmap));
  assert(retval);
  retval->i32NumEntries = 0;
  retval->data = NULL;
  retval->bitsAllocated = 0;
  retval->bitsUsed = 0;

  return retval;
}

/* Structure to represent the stack map at each point of a method's execution 
 *
 * The format is an integer containing the number of entries. We are basically storing a table:
 *
 * Instruction range			Map
 *
 *    0 - 1				
 *    2 - 5				I
 *    6 - 7				III
 *    etc
 *
 *  The number of entries is the number of rows in this table.
 *
 *  Then we encode each instruction range as a byte, with the high bit set to 0.
 *  The Map is encoded with the following 2-bit values:
 *
 *	00 - End of sequence
 *	01 - Literal value on stack
 *	10 - Address on stack
 *	11 - Unused
 *
 *  It is just a repetition of 01, 10 values until ended by 00. 
 *
 *  The information for the next row follows directly in the bit stream.
 *
 *  If the range of instructions is greater than 128, we use an integer with the high bit set to 1
 *
 */

inline int STACKMAP_newRange(struct tstackmap* entry, int items) {

  assert(items < 128);

  //We need 8 bits to add the new range
      if(entry->bitsAllocated - entry->bitsUsed < 8) {

	//We need to grow the amount of bits
	    STACKMAP_growBits( entry, 8 );
      } 
  
	if(entry->bitsUsed % 8 == 0) {
	  //We just append
	      entry->data[ entry->bitsUsed / 8 ] = (unsigned char) items;
	} else {
	  //We have to write into two different bytes
	      int numLeft = 8 - (entry->bitsUsed % 8);
	      int numRight;

	      numRight = 8 - numLeft;
	      
	      //Shift out the right bits, they go in the next byte
	      entry->data[ entry->bitsUsed / 8] |= ( items >> (numRight));
	      entry->data[ entry->bitsUsed / 8 + 1] = ( items << (8 - numRight));
	}

	entry->bitsUsed += 8;
	return 0;
}



inline void STACKMAP_growBits(struct tstackmap* entry, int bitsRequired)
{
  int i = 0;
  int newNumBits = entry->bitsAllocated * 2;
  unsigned char* oldData;

  if(newNumBits == 0)
    newNumBits = 32;

  assert( newNumBits > (bitsRequired + entry->bitsUsed));
  
  oldData = entry->data;
  entry->data = (unsigned char*) sys_malloc( (newNumBits / 8) + 1);
  
  assert(entry->data);
  if( entry->bitsUsed != 0)
  for(i = 0; i <= (entry->bitsUsed / 8);i++)
    {
      entry->data[i] = oldData[i];
    }
  entry->bitsAllocated = newNumBits;
  sys_free(oldData);
}

void inline STACKMAP_printEncodedMap( struct tstackmap* map) {

  int i = 0;
if( map->bitsAllocated )
  {
  fprintf(stderr, "Printing encoded map %p\n", map);
  for(i = 0; i <= map->bitsUsed / 8; i++) {
    
    int j;
    for(j = 7; j >= 0;j--)
      {
	if( map->data[i] & (1 << j))
	  {
	    fprintf(stderr, "1");
	  }
	else
	  {
	    fprintf(stderr, "0");
	  }
      }
    fprintf(stderr, " ");
  }
  }
  fprintf(stderr, "\n");
}
struct histEntry {

  char* map;
  int count;
};

#include "../hash.h"

HASH_tstHashTable pstTable;
struct histEntry* histogram[256];

void buildHistogram() {

  int i = 0;
  HASH_InitHashTable( &pstTable, HASH_STRING_KEYS);
  for(i = 0; i < 256; i++) {
    histogram[i] = sys_malloc(sizeof(struct histEntry));
    histogram[i]->map = NULL;
    histogram[i]->count = -1;
  }
}



void addToHighest(struct histEntry* item) {

  int i = 255;

  while(i >= 0) {

    if( histogram[i]->count <= item->count) {

      //Shift down the old ones (bumping off the bottom one)
	  int j;
          for(j = 0; j < (255 - i);j++)
	    histogram[i + j] = histogram[i + j - 1];
	  //Insert the new one
	  histogram[i] = item;
//	  fprintf(stderr, "%i is %p, %i\n", i, item, item->count);
	  return ;
    }
    i--;
  }
}

void addToHistogram(char* map) {

  int i32Dummy;
  struct histEntry* mapEntry;
  HASH_tstHashEntry* hashEntry;

  hashEntry = HASH_FindHashEntry( &pstTable, map );
  
  if(hashEntry == NULL)
    {
//    fprintf(stderr, "%s not found\n", map);
    
    mapEntry = sys_malloc( sizeof(struct histEntry ));
    mapEntry->map = map;
    mapEntry->count = 1;

    hashEntry = HASH_CreateHashEntry( &pstTable, map, &i32Dummy);
    HASH_SetHashValue( hashEntry, mapEntry) ;
//    fprintf(stderr, "stored %p in %p\n", mapEntry, hashEntry);
    }
  else
    {
     mapEntry = HASH_GetHashValue( hashEntry );
     mapEntry->count++;
//     fprintf(stderr, "%s found, with %i entries\n", map, mapEntry->count);
//     fprintf(stderr, "loaded %p in %p\n", mapEntry, hashEntry);
    }

  addToHighest(mapEntry);
  
}

void STACKMAP_printHistogram() {

  HASH_tstHashSearch search;
  HASH_tstHashEntry* hashEntry;
  int i = 0;

  hashEntry = HASH_FirstHashEntry(&pstTable, &search);
  
  while(hashEntry != NULL) {
    
    struct histEntry* mapEntry;
    mapEntry = HASH_GetHashValue( hashEntry );
    fprintf(stderr, "Dumping entry %i, map %s, count %i\n", i++, mapEntry->map, mapEntry->count);
    hashEntry = HASH_NextHashEntry(  &search);
  }

//  for(i = 0; i < 256;i++)
//    fprintf(stderr, "Entry %i map %s, count %i\n", i, histogram[i]->map, histogram[i]->count);
}
#endif





