/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>


#ifdef KISSME_LINUX_USER
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <signal.h>
#include <string.h>
#include <pthread.h>
#endif

#include "../lib/indigenous/java.lang/VMSystem.h"
#include "../lib/indigenous/java.lang/VMClassLoader.h"
#include "threads_native.h"
#include "interp.h"
#include "jni.h"
#include "interp_methods.h"
#include "classfile_methods.h"
#include "garbage.h"
#include "garbage2.h"
#include "newobject.h"
#include "methodstacks.h"
#include "threadinfo.h"

/* global variables ********************************************************/


extern int INTERP_InitialStackSize; //This can be set by the ProcessArgs routine 
extern int canStartOptimising; //For the run-time-optimiser
int canStartStackMaps = 0; //For the stack maps
int showLoading = 0; //Show the classes loaded by the interpreter
int showClasspath = 0; //Show our classpaths


//Decides whether we are allowed to put objects into the persistent store

int use_persistent_store = 0;

/* We preallocate an OutOfMemoryError object so that we don't have a problem allocating it in an OOM situation

 */

static int LoadLocalConfigFile();
static int LoadHomeConfigFile();
static int LoadSystemConfigFile();
static int ScanConfigFile(FILE* fp);

/* Looks for a ./.kissmerc file*/
static int LoadLocalConfigFile()
{
 FILE* fp = fopen("./.kissmerc", "r");
 if(fp == NULL)
 {
  return 1;
 }
 return ScanConfigFile(fp);
}
static int LoadHomeConfigFile()
{
 FILE* fp = fopen("~/.kissme/kissmerc", "r");
 if(fp == NULL)
 {
  return 1;
 }
 return ScanConfigFile(fp);
}
static int LoadSystemConfigFile()
{
 FILE* fp = fopen("/etc/kissme/kissmerc", "r");
 if(fp == NULL)
 {
  return 1;
 }
 return ScanConfigFile(fp);
}

static int ScanConfigFile(FILE* fp)
{
 char keybuf[128];
 char pathbuf[4096];
 char* result;
 int i;

 strcpy(keybuf, "");
 strcpy(pathbuf, "");
 
 result = fgets(keybuf, 11, fp); //length of "classpath="
 if(result == NULL)
   return 2;
 keybuf[11] = 0;
 result = fgets(pathbuf, 4096, fp); //length of "classpath="
 if(result == NULL)
   return 2;
 if(strncmp(keybuf, "classpath=", 11) == 0)
 {
	char* pathlist;
	char* pathitem = NULL;
	char* freeptr;

	freeptr = pathlist = (char*) malloc(4096);
	if(pathlist == NULL)
	  return -6;

	strcpy(pathlist, pathbuf);

	//trim off any newlines
	for(i = 0; pathlist[i] != 0;i++)
	  if(pathlist[i] == '\n')
	    pathlist[i] = 0;

	while((pathitem = strtok(pathlist, ":")) != NULL)
	    {
	      if(CLASSFILE_u16NumClassPaths == 19)
		{
		  eprintf("Too many entries in the classpath list (reading .kissmerc)\n");
		  return -1;
		}
		CLASSFILE_ppszClassPaths[CLASSFILE_u16NumClassPaths] = (char*) malloc(4096);
		strcpy(CLASSFILE_ppszClassPaths[CLASSFILE_u16NumClassPaths++], pathitem);

		//subsequent calls to strtok require a NULL first argument
		if(pathlist != NULL)
		  {
		    pathlist = NULL;
		  }
	    }
	sys_free(freeptr);
	    }
   return 1; //success
 }

static int PrintHelp(char** argv)
{
    fprintf(stderr, "kissme %s (%s) - This program is free software\n\n", KISSME_VERSION, KISSME_BUILD_DATE);
    fprintf(stderr, "Usage: %s [switches] <main class> [program arguments]\n", argv[0]);
    fprintf(stderr, "Where [switches] can be:\n");
    fprintf(stderr, "  -oss<number>          to set the maximum Java stack size for any thread\n");
    fprintf(stderr, "  -cp or -classpath     <path> to specify a class path\n");
    fprintf(stderr, "  -bootclasspath        <path> to specify a complete class path (overrides default system libraries)\n");
    fprintf(stderr, "  -v		     to turn on verbose mode (shows classes as they are loaded and where from)\n");
    fprintf(stderr, "  -vv		     to turn on very verbose mode (shows what paths are used for class loading and classes as they are loaded and where from)\n");
    fprintf(stderr, "  -nojit		     ignored\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Typical usage: %s org.myorg.MyClassFile\n", argv[0]);
    return 0;
}

/*
 * @doc FUNC
 * @func
 * This function scans the command line parameters and deals with them.
 * @rdesc Returns the number of valid command-line arguments found 
 *
 */

static int ProcessArgs
(
  int argc,               /* @parm Number of arguments */
  char** argv,            /* @parm Argument vector */
  char** env,             /* @parm Environment vector */
  char** ppszMainClass,   /* @parm Pointer to main class name */
#ifdef PJSLREAL
  int* makeNewStore
#else
  char** ppszStoreName    /* @parm Name of persistent store */
#endif
)
{
  int i;
  int num = 1; /* start at 1 for the program name */
  int handledSystemClasspath = 0; //1 if we have specified a path to the class libraries
  int handledClasspath = 0; //1 if we have specified a path to user classes

  CLASSFILE_ppszClassPaths = (char**) sys_malloc(sizeof(char*) * 20); //Better solution?
  if(CLASSFILE_ppszClassPaths == NULL)
    return -5; //fail
  CLASSFILE_u16NumClassPaths = 0;

  for (i = 1; i < argc; i++)
  {
    if (strncmp(argv[i], "--help", 6) == 0)
    {
      PrintHelp(argv);
      return -2;
    }
    else if (strncmp(argv[i], "-h", 2) == 0)
    {
      PrintHelp(argv);
      return -2;
    }
    else if ((strncmp(argv[i], "-cp", 3) == 0) || (strncmp(argv[i], "-classpath", 10) == 0))
    {
      //we will do this after everything else
      i++;
    }
    else if (strncmp(argv[i], "-oss", 4) == 0)
    {
	char sizebuf[256];

	//I think we must increment i here: 
	i++;
	strcpy(sizebuf, argv[i] + 4);
	INTERP_InitialStackSize = atoi(sizebuf);
	if(strstr(sizebuf, "k") || strstr(sizebuf, "K"))
	    INTERP_InitialStackSize *= 1024;
	else if(strstr(sizebuf, "m") || strstr( sizebuf, "M"))
	    INTERP_InitialStackSize *= 1024 * 1024;
	fprintf(stderr, "GOT OSS VAR %i\n", INTERP_InitialStackSize);
	num++;
    }
    else if ((strcmp(argv[i], "-bootclasspath") == 0) && (handledClasspath == 0))
    {
	char* pathlist;
	char* pathitem = NULL;
	char* freeptr;

	freeptr = pathlist = (char*) malloc(4096);
	if(pathlist == NULL)
	  return -6;

	strcpy(pathlist, argv[i + 1]);
	handledSystemClasspath = 1;
	while((pathitem = strtok(pathlist, ":")) != NULL)
	    {
	      if(CLASSFILE_u16NumClassPaths == 19)
		{
		  eprintf("Too many entries in the bootclasspath list\n");
		  return -1;
		}
		CLASSFILE_ppszClassPaths[CLASSFILE_u16NumClassPaths] = (char*) malloc(4096);
		strcpy(CLASSFILE_ppszClassPaths[CLASSFILE_u16NumClassPaths++], pathitem);

//		eprintf("Adding \"%s\", number %i\n", CLASSFILE_ppszClassPaths[ CLASSFILE_u16NumClassPaths - 1], CLASSFILE_u16NumClassPaths);
		//subsequent calls to strtok require a NULL first argument
		if(pathlist != NULL)
		  {
		    pathlist = NULL;
		  }
	    }
	sys_free(freeptr);

	i++;
	num += 2; 
    }


    else if (strcmp(argv[i], "-store") == 0)
    {
      use_persistent_store = 1;
      *ppszStoreName = argv[++i];
      num += 2;
    }
    else if (strcmp(argv[i], "-heap") == 0)
    {
      long int hsize;
      sscanf(argv[++i], "%ld", &hsize);
#ifdef ALLOCATOR
       ALLOCATOR_SetHeapSize(hsize);
#endif
      //GARBAGE_INITIALHANDLENUM = GARBAGE_INITIALHEAPSIZE / 100;
//      i++;
      num += 2;
    }
    #ifdef EVICTIONS
    else if (strcmp(argv[i], "-ef") == 0)
    {
      sscanf(argv[++i], "%ld/%ld", &GARBAGE_FREEMEM_NUMER, &GARBAGE_FREEMEM_DENOM);
      num += 2;
    }
    else if (strcmp(argv[i], "-sm") == 0)
    {
      sscanf(argv[++i], "%ld,%ld", &GARBAGE_NUMDEFFRAMES, &GARBAGE_NUMDEFREACH);
      num += 2;
    }
    #endif
    else if (strcmp(argv[i], "-nojit") == 0)
    {
       num++;
    }
    else if (strcmp(argv[i], "-v") == 0)
    {
       num++;
       showLoading = 1; //Show which and where classes are loaded from 
    }
    else if (strcmp(argv[i], "-vv") == 0)
    {
       num++;
       showLoading = 1; //Show which and where classes are loaded from 
       showClasspath = 1; //show what we're using for a classpath 		   
    }
    else
    {
      if (*ppszMainClass == NULL)
      {
        *ppszMainClass = argv[i];
        num += 1;
      }
    }
  }

//      eprintf("Got %i classpaths before \n", CLASSFILE_u16NumClassPaths);

  if (CLASSFILE_u16NumClassPaths == 0)
  {
    for (i = 0; env[i]; i++)
    {
      if (strncmp(env[i], "CLASSPATH", 9) == 0)  /* using 9 instead of strlen(CLASSPATH) */
      {
        CLASSFILE_ppszClassPaths[CLASSFILE_u16NumClassPaths++] = env[i] + 10; /* should point to after the '=' character */
      }
    }
  }

  /* we didn't have -classpath on the command line, so we look for a .kissmerc file in the current directory */
//      eprintf("Got %i classpaths before config\n", CLASSFILE_u16NumClassPaths);

  if (CLASSFILE_u16NumClassPaths == 0)
    {
      if(LoadLocalConfigFile() != 1)
	eprintf("Error occurred loading local config file\n");
    }
  if (CLASSFILE_u16NumClassPaths == 0)
    {
      if(LoadHomeConfigFile() != 1)
	eprintf("Error occurred loading home config file\n");
    }
  if (CLASSFILE_u16NumClassPaths == 0)
    {
      if(LoadSystemConfigFile() != 1)
	eprintf("Error occurred loading system config file\n");
    }
  if (CLASSFILE_u16NumClassPaths != 0)
    handledSystemClasspath = 1;
  else
    {
      eprintf("Got %i classpaths\n", CLASSFILE_u16NumClassPaths);
    }    
  /* we go back through the args and look for "-cp" */
  for (i = 1; i < argc; i++)
  {
    if((strcmp(argv[i], "-cp") == 0) || (strncmp(argv[i], "-classpath", 10) == 0))
      {
	char* pathlist;
	char* pathitem = NULL;
	char* freeptr;

	freeptr = pathlist = (char*) malloc(4096);
	if(pathlist == NULL)
	  return -6;

	strcpy(pathlist, argv[i + 1]);
	handledClasspath = 1;
	while((pathitem = strtok(pathlist, ":")) != NULL)
	    {
	      if(CLASSFILE_u16NumClassPaths == 19)
		{
		  eprintf("Too many entries in the classpath list\n");
		  return -1;
		}

		//subsequent calls to strtok require a NULL first argument
		if(pathlist != NULL)
		  {
		    pathlist = NULL;
		  }

		CLASSFILE_ppszClassPaths[CLASSFILE_u16NumClassPaths] = (char*) malloc(4096);
		strcpy(CLASSFILE_ppszClassPaths[CLASSFILE_u16NumClassPaths++], pathitem);
	    }

	sys_free(freeptr);
	num += 2;
	break;
      }
  }
  /* If no user path was specified, add the current directory */
  if(handledClasspath == 0)
    {
      CLASSFILE_ppszClassPaths[CLASSFILE_u16NumClassPaths] = (char*) malloc(4096);
      strcpy(CLASSFILE_ppszClassPaths[CLASSFILE_u16NumClassPaths++], ".");
      handledClasspath = 1;
    }
  assert(num <= argc);
  return num;
}


volatile int totalThreadsInSystem; //we count all threads that are created and destroyed

/* The main function processes command line arguments and then calls the startup routine */

#include "startup.h"

int main(int argc, char* argv[], char* c_env[])
{
  char* pszMainClass = NULL;  char* pszStoreName = "persistent_store";
  int argsused = -1;

  if (argc < 2)
  {
    PrintHelp(argv);
    exit(-1);
  }

  //And then check the args for a -classpath
  argsused = ProcessArgs(argc, argv, c_env, &pszMainClass, &pszStoreName);
  if(argsused < 0)
    {
      fprintf(stderr, "Error occurred processing arguments\n");
      return -2;
    }
  java_lang_VMSystem_setArgsPointer(argv, argc);

if(pszMainClass == NULL)
  {
    fprintf(stderr, "You didn't specify a class to run.\n");
    fprintf(stderr, "Usage: %s [switches] <main class> [program arguments]\n", argv[0]);
    fprintf(stderr, "Where [switches] can be:\n");
    fprintf(stderr, "    -cp <path>	      to specify a class path\n");
    fprintf(stderr, "or  -classpath <path>    to specify a complete class path\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Try --help for more info");
    return -3;
  }

 if(showClasspath == 1)
   {
     int m = 0; 
     fprintf(stderr, "Classpath:\n");
     for(m = 0; m < CLASSFILE_u16NumClassPaths;m++)
       {
	 fprintf(stderr, "%i: %s\n", m + 1, CLASSFILE_ppszClassPaths[m]);
       }
   }

  return STARTUP_startup(pszMainClass, pszStoreName, argc, argv, argsused);
}


