  
/* Next version of GC code

   I want to create specific GC points, instead of just collecting whenever needed.

   This implies making GC points at method calls, at backward branches and at the beginning and end of native calls.

   It will mean that native methods can't block, especially IO methods such as networking and file. But this is a small price to pay for more reliable GC. We'll have to make the IO calls asynchronous somehow.


 */


#include <config.h>

#ifdef GARBAGE2

extern int instructions_since_gc;
#ifdef KISSME_LINUX_USER
#include <unistd.h>
#include <pthread.h>
#include <assert.h>
#include <stdio.h>
#endif

#include "kinterface.h"
#include "garbage.h"
#include "garbage2.h"
#include "threads_native.h"

/* This method is called by every thread at the specified GC points. It checks whether any other thread wants to do a Garbage Collection
 */

extern volatile int   totalThreadsInSystem;

/* There may be two threads that call this method, and this is quite likely since both will be out of memory if they share the same heap
 */

int GARBAGE_NeedGC(tAllocatorHeap* pstHeap, int numWords)
{
  int waited = 0;
  tThreadNode* thisNode;
  sys_mutex_lock(pstHeap->gc_resume_mutex);
  if(pstHeap->garbageFinished)
    {
      sys_mutex_unlock(pstHeap->gc_resume_mutex);
      return 1;
    }

  thisNode = THREAD_FindThread();
  thisNode->stackTop = (void*) &waited;

  //First check whether a GC isn't pending already
  if(pstHeap->wantGC == 0)
    {
    pstHeap->wantGC = 1;
    //Tell the GC thread that we need more memory
    sys_condition_signal( pstHeap->gc_point_cond );

    //Wait until the GC thread is ready 
    //It will lock gc_resume_mutex while changing wantGC

    }
    sys_mutex_unlock(pstHeap->gc_resume_mutex);	  

    while(1)
    {
      waited++;
      sys_mutex_lock(pstHeap->gc_resume_mutex);
      if(pstHeap->wantGC == 2)
	{
	  sys_mutex_unlock(pstHeap->gc_resume_mutex);	  
	  break;
	}
	
      if(waited == 50)
	{
	  sys_condition_signal( pstHeap->gc_point_cond );
	  waited = 0;
	}
      
      sys_mutex_unlock(pstHeap->gc_resume_mutex);	  
#ifdef KISSME_LINUX_USER
      usleep(100);
#endif
    }

#ifdef GARBAGE_VERBOSE
  eprintf("GC originating thread is entering GC point\n");
#endif
  GARBAGE_EnterGCPoint( pstHeap, sys_current_thread() );
  return 0;
}

//This thread is the 'garbage collector thread'

#ifdef KISSME_LINUX_KERNEL
int Garbage_Thread(void* args)
#else
void* Garbage_Thread(void* args)
#endif
{
  int spincount = 0;
  tAllocatorHeap* pstHeap = (tAllocatorHeap*) args;
  pstHeap->threadsPresent = 0; //We're here
  //Create a pstJNIData for ourselves
  java_lang_Thread__createJNIData(NULL, NULL);

  while(pstHeap->garbageFinished == 0 )
    {
#ifdef GARBAGE_VERBOSE
      eprintf("Garbage thread running %i (tid %x %i)\n", pstHeap->garbageFinished, sys_current_thread(), sys_current_thread());
#endif
      //We wait for a signal to start the gc
      sys_mutex_lock(pstHeap->gc_wait_mutex);	 
      pstHeap->garbageThreadReady = 1;

#ifdef GARBAGE_VERBOSE
      eprintf("Garbage thread is sleeping %i (tid %x %i) on cond %p\n", pstHeap->garbageFinished, sys_current_thread(), sys_current_thread(), &pstHeap->gc_point_cond);
#endif
      #ifdef KISSME_RJK_KERNEL
      eprintf("Garbage thread returning for RJK\n");
      return NULL;
      #endif
      sys_condition_wait_nogc(pstHeap->gc_point_cond, pstHeap->gc_wait_mutex);			
      sys_mutex_unlock(pstHeap->gc_wait_mutex);	

#ifdef GARBAGE_VERBOSE
      eprintf("Garbage thread is woken %i (tid %x %i)\n", pstHeap->garbageFinished, sys_current_thread(), sys_current_thread());
#endif
      if(pstHeap->garbageFinished)
	{
	  if(katomic_compare( &pstHeap->garbageFinished, 0) || katomic_compare( &pstHeap->garbageFinished, 1))
	    {
	      pstHeap->garbageFinished = 2; //this thread has realised it must exit
	    }
	  totalThreadsInSystem--;
	  #ifdef GARBAGE_VERBOSE
	  eprintf("garbage thread %x %i exited (number threads %i)\n", (int) sys_current_thread(), (int) sys_current_thread(), totalThreadsInSystem);
#endif

#ifdef KISSME_LINUX_KERNEL
	  return 0;
#else
	  return NULL;
#endif
	}

      sys_mutex_lock(pstHeap->gc_resume_mutex);	
      pstHeap->wantGC = 2;

      pstHeap->threadsRequired = THREAD_GetNumberThreads(pstHeap); //hmm
      sys_mutex_unlock(pstHeap->gc_resume_mutex);	

      //Once we get that signal we wait for the other threads to reach the GC point
      while( pstHeap->threadsPresent < pstHeap->threadsRequired )
	{
//	  sleep(2);
#ifdef KISSME_LINUX_USER
	  usleep( 50 ); //Sleep for 50 microseconds
#endif
	  if(spincount++ == 2000)
	    {
	      eprintf("Threads present %i and req %i\n", pstHeap->threadsPresent, pstHeap->threadsRequired); 
	      spincount = 0;
	    }
#ifdef KISSME_LINUX_KERNEL
	  schedule();
#endif
	}

      //Now that everyone is here, we do the GC
#ifdef GARBAGE_VERBOSE
       eprintf("Garbage thread is doing gc (tid %x %i)\n", sys_current_thread(), sys_current_thread());
#endif
       GARBAGE_lock_and_gc(pstHeap);     
       pstHeap->threadsPresent = 0;
       pstHeap->wantGC = 0;
      //And let everyone carry on with their business
#ifdef GARBAGE_VERBOSE
       //      eprintf("Garbage thread is signalling other threads (tid %x %i)\n", sys_current_thread(), (int) sys_current_thread());
#endif
      sys_condition_broadcast( pstHeap->gc_resume_cond );
    }
	  if(katomic_compare( &pstHeap->garbageFinished, 0) || katomic_compare( &pstHeap->garbageFinished, 1))
	    {
	      pstHeap->garbageFinished = 2; //this thread has realised it must exit
	    }
  totalThreadsInSystem--;
#ifdef KISSME_LINUX_KERNEL
  return 0;
#else
  return NULL;
#endif
}

int GARBAGE2_Init(tAllocatorHeap* pstHeap)
{
  int success;

  int tid = 0;

  pstHeap->garbageFinished = 0;

  pstHeap->gc_point_cond = sys_condition_create();
  assert(pstHeap->gc_point_cond != NULL);

  pstHeap->gc_resume_cond = sys_condition_create();
  assert(pstHeap->gc_resume_cond != NULL);

  pstHeap->gc_wait_mutex = sys_mutex_create(0);
  pstHeap->gc_resume_mutex = sys_mutex_create(0);

  pstHeap->threadsPresent = 0;
  pstHeap->threadsRequired = 0;
#ifdef GARBAGE_VERBOSE
  eprintf("Thread %x %i is creating garbage thread\n", (int) sys_current_thread(), (int) sys_current_thread());
#endif
  //Start the garbage thread
#ifdef KISSME_LINUX_USER
  success = pthread_create(  ((pthread_t*) &tid), NULL,  (void* (*)(void*)) &Garbage_Thread, (void*) pstHeap );
  assert(success == 0);
#else
  #ifdef KISSME_LINUX_KERNEL
  tid = 22; //remove the warning
  success = 0;
  kernel_thread( &Garbage_Thread, (void*) pstHeap, 0);
  #else
  kinterface->kthread_create(&Garbage_Thread, (void*) pstHeap, 32, 0);
  #endif
#endif

  totalThreadsInSystem++;
  return 0;
}

int GARBAGE2_Finish(tAllocatorHeap* pstHeap)
{
#ifdef GARBAGE_VERBOSE
  eprintf("Setting garbage finished and waking gc thread (our tid %x %i)\n", sys_current_thread(), sys_current_thread());
#endif
  while(pstHeap->garbageThreadReady != 1)
    {
      #ifdef KISSME_LINUX_USER
      pthread_yield();
      #endif
      #ifdef KISSME_LINUX_KERNEL
      schedule();
      #endif
      //XXX RJK
      //      eprintf("thread (tid %x %i) is waiting for garbage thread to get ready\n", sys_current_thread(), sys_current_thread());
    }
  pstHeap->garbageFinished = 1;
  while( pstHeap->garbageFinished != 2)
    {
      //      eprintf("thread (tid %x %i) is signalling %p\n", (int) sys_current_thread(), (int) sys_current_thread(), &pstHeap->gc_point_cond);
      sys_condition_signal(pstHeap->gc_point_cond );
    }
  return 0;
}

int GARBAGE_EnterGCPoint(tAllocatorHeap* pstHeap, int thread_id)
{

//If a GC is requested, the calling method will call GARBAGE_WaitForGC where it waits to be notified that the GC is complete
  if(pstHeap->wantGC == 2)
    {
      int someVariable;
      //We record the bottom of the stack so that we can mark the C stack accurately
      tThreadNode* node = THREAD_FindThread();
      assert(node);

#ifdef GARBAGE_VERBOSE
      eprintf("(Thread %x %i) is in GC Point\n", (int) sys_current_thread(), sys_current_thread());
#endif
      node->stackTop = &someVariable; //This is the address of the variable 'someVariable' on the C stack
      someVariable = 3; //make sure compiler doesn't remove it					  
#ifdef GARBAGE_VERBOSE
      eprintf("Waiting for GC end broadcast, stackBottom is %p, stackTop is %p\n", node->stackBottom, node->stackTop);
#endif
      sys_mutex_lock(pstHeap->gc_resume_mutex);
      pstHeap->threadsPresent++;
      sys_condition_wait_nogc(pstHeap->gc_resume_cond, pstHeap->gc_resume_mutex);
#ifdef GARBAGE_VERBOSE
      eprintf("Got GC end broadcast\n");
#endif
#ifdef DEBUG_CLASS_FOR_GC
      instructions_since_gc = 0;
#endif
      sys_mutex_unlock(pstHeap->gc_resume_mutex);
      return 1;
    }

  return 0;
}

#endif














