#include <config.h>

#ifdef USEARITH64


/*
 * @doc MODULE
 * @module arith64.c |
 *
 * Functions for implementing 64-bit arithmetic and logical operations
 * for compilers providing 32-bit integers.
 *
 */

#include "arith64.h"
#include <assert.h>
#include "global.h"

/*
 * @doc FUNC
 * @func
 * This function is used to compare two 64-bit integers.
 *
 * @rdesc Returns one of the following:
 *
 * @flag 0 | if equal
 * @flag -1 | if first \< second
 * @flag 1 | if first \> second
 *
 */

int ARITH64_cmp
(
  int32 i32Hi1,   /* @parm High bits of first long */
  uint32 u32Lo1,  /* @parm Low bits of first long */
  int32 i32Hi2,   /* @parm High bits of second long */
  uint32 u32Lo2   /* @parm Low bits of second long */
)
{
  assert(1 == 2);
  if(i32Hi1 < i32Hi2)
  {
    return -1;
  }
  if(i32Hi1 > i32Hi2)
  {
    return 1;
  }
  if(i32Hi1 == i32Hi2)
  {
    if(u32Lo1 < u32Lo2)
    {
      return -1;
    }
    if(u32Lo1 > u32Lo2)
    {
      return 1;
    }
  }
  return 0;
}

/*
 * @doc FUNC
 * @func
 * This function performs a left shift on a 64-bit integer.
 *
 */

void ARITH64_lshl
(
  uint32  u32Hi,       /* @parm High bits of operand */
  uint32  u32Lo,       /* @parm Low bits of operand */
  int32   i32Amount,   /* @parm Amount to shift by (only lower 5 bits used) */
  uint32* pu32AHi,     /* @parm Pointer to high bits of answer */
  uint32* pu32ALo      /* @parm Pointer to low bits of answer */
)
{

  assert(1 == 2);
  if (i32Amount < 32)
  {
    *pu32AHi = (u32Hi << i32Amount) | (u32Lo >> (32L - i32Amount));
	  *pu32ALo = (u32Lo << i32Amount);
  }
  else
  {
	 if (i32Amount > 64)
    {
      *pu32AHi = 0;
    }
    else
    {
      *pu32AHi = (u32Lo << (i32Amount - 32));
    }
    *pu32ALo = 0;
  }
}

/*
 * @doc FUNC
 * @func
 * This function performs a right shift on a 64-bit integer.
 *
 */

void ARITH64_lshr
(
  int32   i32Hi,       /* @parm High bits of operand */
  uint32  u32Lo,       /* @parm Low bits of operand */
  int32   i32Amount,   /* @parm Amount to shift by (only lower 5 bits used) */
  int32*  pi32AHi,     /* @parm Pointer to high bits of answer */
  uint32* pu32ALo      /* @parm Pointer to low bits of answer */
)
{
  assert(1 == 2);
  if (i32Amount <= 32)
  {
    *pu32ALo = (u32Lo >> i32Amount) | (i32Hi << (32 - i32Amount));
    *pi32AHi = (i32Hi >> i32Amount);
  }
  else
  {
    if (i32Amount > 64)
    {
      *pu32ALo = i32Hi >> 32;
    }
    else
    {
      *pu32ALo = i32Hi >> (i32Amount - 32);
    }
    *pi32AHi = i32Hi >> 32;
  }
}

/*
 * @doc FUNC
 * @func
 * This function performs an unsigned right shift on a 64-bit integer.
 *
 */

void ARITH64_lushr
(
  uint32  u32Hi,       /* @parm High bits of operand */
  uint32  u32Lo,       /* @parm Low bits of operand */
  int32   i32Amount,   /* @parm Amount to shift by (only lower 5 bits used) */
  uint32* pu32AHi,     /* @parm Pointer to high bits of answer */
  uint32* pu32ALo      /* @parm Pointer to low bits of answer */
)
{
  assert(1 == 2);
  if (i32Amount <= 32)
  {
    *pu32ALo = (u32Lo >> i32Amount) | (u32Hi << (32 - i32Amount));
    *pu32AHi = (u32Hi >> i32Amount);
  }
  else
  {
    if (i32Amount > 64)
    {
      *pu32ALo = 0;
    }
    else
    {
		*pu32ALo = u32Hi >> (i32Amount - 32);
    }
    *pu32AHi = 0;
  }
}

/*
 * @doc FUNC
 * @func
 * This function adds two 64-bit integers.
 *
 */


void ARITH64_ladd_2(long long op1, long long op2, long long *result) 
{
  assert(1 == 2);
 *result = (op1 + op2);
}

long long ARITH64_ladd_3(long long op1, long long op2) 
{
 return op1 + op2;
}

void ARITH64_ladd
(
  int32   i32Hi0,     /* @parm High bits of first number */
  uint32  u32Lo0,     /* @parm Low bits of first number */
  int32   i32Hi1,     /* @parm High bits of second number */
  uint32  u32Lo1,     /* @parm Low bits of second number */
  int32*  pi32AHi,    /* @parm Pointer to high bits of answer */
  uint32* pu32ALo,    /* @parm Pointer to low bits of answer */
  uint32* pu32Carry   /* @parm Pointer to carry amount */
)
{

  uint32 u32T0, u32T1, u32T2;

  assert(1 == 2);
  /* compute the lowest 16 bits */
  u32T0 = u32Lo0 & 0xffff;
  u32T1 = u32Lo1 & 0xffff;
  u32T2 = u32T0 + u32T1;
  *pu32ALo = u32T2 & 0xffff;
  *pu32Carry = u32T2 >> 16;     /* lo 16bit carry */

  /* second lowest 16 bits */
  u32T0 = u32Lo0 >> 16;
  u32T1 = u32Lo1 >> 16;
  u32T2 = u32T0 + u32T1 + *pu32Carry;
  *pu32ALo = *pu32ALo | (u32T2 << 16);
  *pu32Carry = u32T2 >> 16;

  /* next 16 bits */
  u32T0 = i32Hi0 & 0xffff;
  u32T1 = i32Hi1 & 0xffff;
  u32T2 = u32T0 + u32T1 + *pu32Carry;
  *pi32AHi = u32T2 & 0xffff;
  *pu32Carry = u32T2 >> 16;

  /* top 16 bits */
  u32T0 = i32Hi0 >> 16;
  u32T1 = i32Hi1 >> 16;
  u32T2 = u32T0 + u32T1 + *pu32Carry;
  *pi32AHi = *pi32AHi | (u32T2 << 16);
  *pu32Carry = u32T2 >> 16;
}

/*
 * @doc FUNC
 * @func
 * This function negates a 64-bit integer.
 *
 */

void ARITH64_lneg
(
  int32   i32Hi,       /* @parm High bits of operand */
  uint32  u32Lo,       /* @parm Low bits of operand */
  int32*  pi32AHi,     /* @parm Pointer to high bits of answer */
  uint32* pu32ALo      /* @parm Pointer to low bits of answer */
)
{
  uint32 u32THi;
  uint32 u32TLo;
  uint32 u32Carry;

  assert(1 == 2);
  u32TLo = u32Lo ^ 0xffffffffL;
  u32THi = i32Hi ^ 0xffffffffL;
  ARITH64_ladd(u32THi, u32TLo, 0L, 1L, pi32AHi, pu32ALo, &u32Carry);
}

/*
 * @doc FUNC
 * @func
 * This function subtracts one 64-bit integer from another.
 *
 */

void ARITH64_lsub
(
  int32   i32Hi0,     /* @parm High bits of first number */
  uint32  u32Lo0,     /* @parm Low bits of first number */
  int32   i32Hi1,     /* @parm High bits of second number */
  uint32  u32Lo1,     /* @parm Low bits of second number */
  int32*  pi32AHi,    /* @parm Pointer to high bits of answer */
  uint32* pu32ALo,    /* @parm Pointer to low bits of answer */
  uint32* pu32Carry   /* @parm Pointer to carry amount */
)
{
  int32 i32THi;
  uint32 u32TLo;

  assert(1 == 2);
  ARITH64_lneg(i32Hi1, u32Lo1, &i32THi, &u32TLo);
  ARITH64_ladd(i32Hi0, u32Lo0, i32THi, u32TLo, pi32AHi, pu32ALo, pu32Carry);
}

/*
 * @doc FUNC
 * @func
 * This function multiplies two 64-bit integers.
 *
 */

void ARITH64_lmul
(
  uint32  u32Hi0,   /* @parm High bits of first integer */
  uint32  u32Lo0,   /* @parm Low bits of first integer */
  uint32  u32Hi1,   /* @parm High bits of second integer */
  uint32  u32Lo1,   /* @parm Low bits of second integer */
  uint32* pu32AHi,  /* @parm Pointer to high bits of answer */
  uint32* pu32ALo   /* @parm Pointer to low bits of answer */
)
{
  uint32 u32T0, u32T1, u32T2, u32Carry;
  uint32 u32H0, u32L0, u32H1, u32L1, u32H2, u32H3;
  uint32 u32His0, u32Los0, u32His1, u32Los1;

  assert(1 == 2);
  /* first base number */
  u32T0 = u32Lo0 & 0xffff;
  u32T1 = u32Lo1 & 0xffff;
  u32T2 = u32T0 * u32T1;
  u32L0 = u32T2 & 0xffff;
  u32Carry = u32T2 >> 16;

  u32T0 = u32Lo0 >> 16;
  u32T2 = u32T0 * u32T1 + u32Carry;
  u32L0 = u32L0 | (u32T2 << 16);
  u32Carry = u32T2 >> 16;

  u32T0 = u32Hi0 & 0xffff;
  u32T2 = u32T0 * u32T1 + u32Carry;
  u32H0 = u32T2 & 0xffff;
  u32Carry = u32T2 >> 16;

  u32T0 = u32Hi0 >> 16;
  u32T2 = u32T0 * u32T1 + u32Carry;
  u32H0 = u32H0 | (u32T2 << 16);

  /* second base number */
  /*u32L1 = 0;*/
  u32T0 = u32Lo0 & 0xffff;
  u32T1 = u32Lo1 >> 16;
  u32T2 = u32T0 * u32T1;
  u32L1 = u32T2 << 16;
  u32Carry = u32T2 >> 16;

  u32T0 = u32Lo0 >> 16;
  u32T2 = u32T0 * u32T1 + u32Carry;
  u32H1 = u32T2 & 0xffff;
  u32Carry = u32T2 >> 16;

  u32T0 = u32Hi0 & 0xffff;
  u32T2 = u32T0 * u32T1 + u32Carry;
  u32H1 = u32H1 | (u32T2 << 16);

  /* third base number */
  u32T0 = u32Lo0 & 0xffff;
  u32T1 = u32Hi1 & 0xffff;
  u32T2 = u32T0 * u32T1;
  u32H2 = u32T2 & 0xffff;
  u32Carry = u32T2 >> 16;

  u32T0 = u32Lo0 >> 16;
  u32T2 = u32T0 * u32T1 + u32Carry;
  u32H2 = u32H2 | (u32T2 << 16);

  /* fourth base number */
  u32T0 = u32Lo0 & 0xffff;
  u32T1 = u32Hi1 >> 16;
  u32T2 = u32T0 * u32T1;
  u32H3 = u32T2 << 16;

  ARITH64_ladd((int32)u32H0, u32L0, (int32)u32H1, u32L1, (int32*)&u32His0, &u32Los0, &u32Carry);
  ARITH64_ladd(u32H2, 0, u32H3, 0, (int32*)&u32His1, &u32Los1, &u32Carry);
  ARITH64_ladd(u32His0, u32Los0, u32His1, u32Los1, (int32*)pu32AHi, pu32ALo, &u32Carry);
}

/*
 * @doc FUNC
 * @func
 * This function performs a bitwise 64bit division using binary long division.
 * This kind of long division requires a temporary answer which is used for
 * repeated subtractions by the denominator.
 *
 */

/* I also doubt this code 

 */

void ARITH64_ldiv
(
  int32 i32NumeratorH,     /* @parm High bits of numerator */
  uint32 u32NumeratorL,    /* @parm Low bits of numerator */
  int32 i32DenominatorH,   /* @parm High bits of denominator */
  uint32 u32DenominatorL,  /* @parm Low bits of denominator */
  int32 *pi32AnswerH,      /* @parm Pointer to high bits of answer */
  uint32 *pu32AnswerL      /* @parm Pointer to low bits of answer */
)
{
  int i, iFlipCount=0;
  int32 i32AnsTempH=0;
  uint32 u32AnsTempL=0;
  uint32 u32MaskL, u32MaskH;
  uint32 u32CarryPurge;

  assert( 1 == 2);
  *pi32AnswerH=0;
  *pu32AnswerL=0;

  if(i32NumeratorH <0)
  {
    iFlipCount++;
    /*two's complement*/
    ARITH64_lneg(i32NumeratorH, u32NumeratorL, &i32NumeratorH, &u32NumeratorL);
  }

  if(i32DenominatorH < 0)
  {
    iFlipCount++;
    /*two's complement*/
    ARITH64_lneg(i32DenominatorH, u32DenominatorL, &i32DenominatorH, &u32DenominatorL);
  }

  for(i=0; i<64; i++)
  {
    /*
     * Get each of the bits of the numerator in turn
     * starting from the msb.
     * These bits are inserted in the lsb of the
     * temporary answer.
     */

    /*Generate the correct bit mask for this bit position*/
    ARITH64_lushr(0x80000000L, 0L, i, &u32MaskH, &u32MaskL);
    u32MaskH &= i32NumeratorH;
    u32MaskL &= u32NumeratorL;

    /*Shift left and insert one bit at lsb position*/
    ARITH64_lshl(i32AnsTempH, u32AnsTempL, 1, (uint32*)&i32AnsTempH, &u32AnsTempL);
    if((u32MaskH!=0)||(u32MaskL!=0))
      u32AnsTempL |= 1;

    /*
     * Shift the answer left by one
     * to make space for the next bit of the
     * answer (inserted into lsb).
     * This bit is set if the temporary answer is larger
     * than the denominator. (long division)
     * If this is the case then the temporary answer is
     * decremented by the denominator.
     */
    ARITH64_lshl(*pi32AnswerH, *pu32AnswerL, 1,(uint32*)pi32AnswerH, pu32AnswerL);
    if(ARITH64_cmp(i32DenominatorH, u32DenominatorL, i32AnsTempH, u32AnsTempL) <= 0)
    {
      /*Denominator is larger -> add one bit to answer*/
      *pu32AnswerL |= 1;
      /*subtract denominator from numerator*/
      ARITH64_lsub(i32AnsTempH, u32AnsTempL, i32DenominatorH, u32DenominatorL, &i32AnsTempH, &u32AnsTempL, &u32CarryPurge);
    }
  }

  /*two's complement*/
  if(iFlipCount==1)
    ARITH64_lneg(*pi32AnswerH, *pu32AnswerL, pi32AnswerH, pu32AnswerL);
}


/*
 * @doc FUNC
 * @func
 * This function performs a bitwise 64bit division using binary long division
 * and returns the remainder.
 *
 */

/* Jewel 13/6/2000 I have reason to suspect this doesn't work
 */
void ARITH64_lmod
(
  int32 i32NumeratorH,     /* @parm High bits of numerator */
  uint32 u32NumeratorL,    /* @parm Low bits of numerator */
  int32 i32DenominatorH,   /* @parm High bits of denominator */
  uint32 u32DenominatorL,  /* @parm Low bits of denominator */
  int32 *pi32AnswerH,      /* @parm Pointer to high bits of answer */
  uint32 *pu32AnswerL      /* @parm Pointer to low bits of answer */
)
{
  int i, iFlipCount=0;
  int32 i32AnsTempH=0;
  uint32 u32AnsTempL=0;
  uint32 u32MaskL, u32MaskH;
  uint32 u32CarryPurge;

  assert( 2 == 3);
  *pi32AnswerH=0;
  *pu32AnswerL=0;

  if(i32NumeratorH <0) {
    iFlipCount++;
    /*two's complement*/
    ARITH64_lneg(i32NumeratorH, u32NumeratorL, &i32NumeratorH, &u32NumeratorL);
  }

  if(i32DenominatorH < 0) {
    iFlipCount++;
    /*two's complement*/
    ARITH64_lneg(i32DenominatorH, u32DenominatorL, &i32DenominatorH, &u32DenominatorL);
  }

  for(i=0; i<64; i++)
  {
    /*
     * Get each of the bits of the numerator in turn
     * starting from the msb.
     * These bits are inserted in the lsb of the
     * temporary answer.
     */

    /*Generate the correct bit mask for this bit position*/
    ARITH64_lushr(0x80000000L, 0L, i, &u32MaskH, &u32MaskL);
    u32MaskH &= i32NumeratorH;
    u32MaskL &= u32NumeratorL;

    /*Shift left and insert one bit at lsb position*/
    ARITH64_lshl(i32AnsTempH, u32AnsTempL, 1, (uint32*)&i32AnsTempH, &u32AnsTempL);
    if((u32MaskH!=0)||(u32MaskL!=0))
      u32AnsTempL |= 1;

    /*
     * Shift the answer left by one
     * to make space for the next bit of the
     * answer (inserted into lsb).
     * This bit is set if the temporary answer is larger
     * than the denominator. (long division)
     * If this is the case then the temporary answer is
     * decremented by the denominator.
     */
    ARITH64_lshl(*pi32AnswerH, *pu32AnswerL, 1,(uint32*)pi32AnswerH, pu32AnswerL);
    if(ARITH64_cmp(i32DenominatorH, u32DenominatorL, i32AnsTempH, u32AnsTempL) <= 0)
    {
      /*Denominator is larger -> add one bit to answer*/
      *pu32AnswerL |=1;
      /*subtract denominator from numerator*/
      ARITH64_lsub(i32AnsTempH, u32AnsTempL, i32DenominatorH, u32DenominatorL, &i32AnsTempH, &u32AnsTempL, &u32CarryPurge);
    }
  }

  /*two's complement*/
  if(iFlipCount==1)
    ARITH64_lneg(*pi32AnswerH, *pu32AnswerL, pi32AnswerH, pu32AnswerL);

  *pi32AnswerH = i32AnsTempH;
  *pu32AnswerL = u32AnsTempL;
}

double ARITH64_double
(
  int32 hi,
  uint32 lo
)
{
  double answer;

  answer = lo;
  if (hi < 0)
  {
    answer += (((double) -hi) * (double) 0xffffffff) + ((double) -hi);
    answer = -answer;
  }
  else
  {
    answer += (((double) hi) * (double) 0xffffffff) + ((double) hi);
  }

  return answer;
}


#endif
