#include "packetracker.h"

Packetracker::Packetracker() {
    gps = NULL;

    num_networks = num_packets = num_dropped = num_noise =
        num_crypt = num_interesting = num_cisco = 0;

    errstr[0] = '\0';

}

void Packetracker::AddGPS(GPSD *in_gps) {
    gps = in_gps;
}

vector<wireless_network *> Packetracker::FetchNthRecent(unsigned int n) {
    vector<wireless_network *> return_vec = network_list;

    // XXX
    // This is much easier now that we use vectors.  We're already ordered
    // by time since we're inserted in order, so we can just erase...
    // XXX
    // We'll sort by the last time seen... this might become annoying, so
    // lets see how it behaves
    sort(return_vec.begin(), return_vec.end(), WirelessTimeCombo());

    int drop = return_vec.size() - n;

    if (drop > 0) {
        return_vec.erase(return_vec.begin(), return_vec.begin() + drop);
    }

    /*
    while (netlist.size() > n) {
        time_t killer = time(0);
        netlitr killitr;
        wireless_network *killme = NULL;

        for (netlitr i = netlist.begin(); i != netlist.end(); ++i) {
            wireless_network *net = *i;

//            if (net == exclude)
//                continue;

            if (net->last_time < killer) {
                killer = net->last_time;
                killme = net;
                killitr = i;
            }
        }

        if (killme == NULL) {
            killme = netlist.front();
            killitr = ++netlist.begin();
        }

        netlist.erase(killitr);
        }
        */

    return return_vec;
}

// Is a string blank?
bool Packetracker::IsBlank(const char *s) {
    int len, i;
    if (NULL == s) { return true; }
    if (0 == (len = strlen(s))) { return true; }
    for (i = 0; i < len; ++i) {
        if (' ' != s[i]) { return false; }
    }
    return true;
}

// Return the string literal
// This is a really inefficient way of doing this, but I'm tired right now.
string Packetracker::Mac2String(uint8_t *mac, char seperator) {
    char tempstr[MAC_STR_LEN];

    // There must be a better way to do this...
    if (seperator != '\0')
        snprintf(tempstr, MAC_STR_LEN, "%02X%c%02X%c%02X%c%02X%c%02X%c%02X",
                 mac[0], seperator, mac[1], seperator, mac[2], seperator,
                 mac[3], seperator, mac[4], seperator, mac[5]);
    else
        snprintf(tempstr, MAC_STR_LEN, "%02X%02X%02X%02X%02X%02X",
                 mac[0], mac[1], mac[2],
                 mac[3], mac[4], mac[5]);

    string temp = tempstr;
    return temp;
}

int Packetracker::ProcessPacket(packet_info info, char *in_status) {
    wireless_network *net;
    int ret = 0;
    string bssid_mac;

    num_packets++;

    // Junk unknown and noise packets
    if (info.type == packet_noise) {
        num_dropped++;
        num_noise++;
        return(0);
    } else if (info.type == packet_unknown) {
        // If we can't figure out what it is
        // or if FromDS and ToDS are set, we can't make much sense of it so don't
        // try to make a network out of it -- toss it.
        num_dropped++;
        return(0);
    }

    // If it's a broadcast (From and To DS == 1) try to match it to an existing
    // network
    if (info.type == packet_data_broadcast) {
        string ts_mac, fs_mac;
        ts_mac = Mac2String(info.source_mac, ':');
        fs_mac = Mac2String(info.dest_mac, ':');
        if (bssid_map.find(ts_mac) != bssid_map.end()) {
            memcpy(info.bssid_mac, info.source_mac, MAC_LEN);
        } else if (bssid_map.find(fs_mac) != bssid_map.end()) {
            memcpy(info.bssid_mac, info.dest_mac, MAC_LEN);
        } else {
            num_dropped++;
            return(0);
        }
    }

    bssid_mac = Mac2String(info.bssid_mac, ':');

    if (bssid_mac == "00:00:00:00:00:00") {
        num_dropped++;
        return(0);
    }

    // Attach them to a network if they're a probe req we know about
    if (info.type == packet_probe_req) {
        if (ssid_map.find(info.ssid) != ssid_map.end()) {
            bssid_mac = ssid_map[info.ssid]->bssid;
        } else {
            // We just don't handle probe req's right now that don't go
            // to someone we know about, so kill it off.
            return(0);
        }
    }


    // Find out if we have this network -- Every network that actuall
    // gets added has a bssid, so we'll use that to search
    if (bssid_map.find(bssid_mac) == bssid_map.end() && info.type != packet_unknown) {

        // Make a network for them
        net = new wireless_network;

        // They're just data until they get something interesting
        net->type = network_data;

        net->llc_packets = net->data_packets =
            net->crypt_packets = net->interesting_packets =
            net->octets = net->listed = 0;

        net->atype = address_none;

        memset(&net->range_ip, 0, 4);
        memset(&net->mask, 0, 4);
        memset(&net->gate_ip, 0, 4);

        if (strlen(info.ssid) == 0)
            net->ssid = NOSSID;
        else
            net->ssid = info.ssid;

        net->channel = info.channel;
        if (info.ap == 1)
            net->type = network_ap;
        net->wep = info.wep;

        net->bssid = bssid_mac;

        // Put us in the master list
        network_list.push_back(net);
        net->listed = 1;

        // And add us to all the maps
        bssid_map[net->bssid] = net;
        ssid_map[net->ssid] = net;

        net->first_time = time(0);

        snprintf(in_status, STATUS_MAX, "Detected new network \"%s\" bssid %s WEP %c Ch %d",
                 net->ssid.c_str(), net->bssid.c_str(), net->wep ? 'Y' : 'N',
                 net->channel);

        if (gps != NULL)
            gps->FetchLoc(&net->first_lat, &net->first_lon, &net->first_alt,
                          &net->first_spd, &net->first_mode);

//        gui->WriteStatus(status);
        num_networks++;

        // Return 1 if we make a new network entry
        ret = 1;

    } else {
        net = bssid_map[bssid_mac];
        if (net->listed == 0) {
            network_list.push_back(net);
            net->listed = 1;
            /*
            if (num_networks > LINES-3-statheight)
            NetListDelOldest(net);
            */
        }
    }

    net->last_time = time(0);

    if (gps != NULL)
        gps->FetchLoc(&net->gps_lat, &net->gps_lon, &net->gps_alt,
                      &net->gps_spd, &net->gps_mode);


    if (info.type != packet_data && info.type != packet_data_broadcast) {
        // Update the ssid record if we got a beacon for a data network
        if (info.type == packet_beacon) {
            if (net->ssid == NOSSID && strlen(info.ssid) > 0) {
                net->ssid = info.ssid;

                if (net->type == network_ap) {
                    snprintf(in_status, STATUS_MAX, "Found SSID \"%s\" for cloaked network BSSID %s",
                             net->ssid.c_str(), net->bssid.c_str());
                } else {
                    snprintf(in_status, STATUS_MAX, "Found SSID \"%s\" for network BSSID %s",
                             net->ssid.c_str(), net->bssid.c_str());
                }

                ret = 2;
            }

            net->channel = info.channel;
            net->wep = info.wep;
            net->type = network_ap;
        }

        // If this is a probe response and the ssid we have is blank, update it.
        // With "closed" networks, this is our chance to see the real ssid.
        // (Thanks to Jason Luder <jason@ixid.net> for this "closed network" detection)
        if (info.type == packet_probe_response && (strlen(info.ssid) > 0)) {
            if (net->ssid == NOSSID || IsBlank(net->ssid.c_str())) {
                net->ssid = info.ssid;
                net->channel = info.channel;
                net->wep = info.wep;
                snprintf(in_status, STATUS_MAX, "Found SSID \"%s\" for cloaked network BSSID %s",
                         net->ssid.c_str(), net->bssid.c_str());
                ret = 2;
            }
        }

        // Update what type
        if (net->type != network_ap && info.type == packet_beacon) {
            net->type = network_ap;
        }

        if (net->type != network_ap && info.type == packet_adhoc) {
            net->type = network_adhoc;
        }

        if (net->type != network_ap && net->type != network_probe &&
            info.type == packet_probe_req) {
            net->type = network_probe;
        }

        net->llc_packets++;

        // Catch adhocs and count them as LLC and Data simultaneously
        if (info.type == packet_adhoc)
            net->data_packets++;

    } else {
        if (info.encrypted) {
            net->crypt_packets++;
            num_crypt++;
        }

        if (info.interesting) {
            num_interesting++;
            net->interesting_packets++;
        }

        net->data_packets++;

        // Record a cisco device
        if (info.proto.type == proto_cdp) {
            net->cisco_equip[info.proto.cdp.dev_id] = info.proto.cdp;
            num_cisco++;
        }

        // If we're not aware of a dhcp server already, try to find one.
        if (info.proto.type == proto_dhcp_server && net->atype < address_dhcp) {
            // Jackpot, this tells us everything we need to know
            net->atype = address_dhcp;

            memcpy(net->range_ip, info.proto.misc_ip, 4);
            memcpy(net->mask, info.proto.mask, 4);
            memcpy(net->gate_ip, info.proto.gate_ip, 4);

            snprintf(in_status, STATUS_MAX, "Found IP range for \"%s\" via DHCP %d.%d.%d.%d mask %d.%d.%d.%d",
                     net->ssid.c_str(), net->range_ip[0], net->range_ip[1], net->range_ip[2], net->range_ip[3],
                     net->mask[0], net->mask[1], net->mask[2], net->mask[3]);

            //gui->WriteStatus(status);
            /*
            snprintf(status, 1024, "     mask %d.%d.%d.%d gw %d.%d.%d.%d",
                     net->mask[0], net->mask[1], net->mask[2], net->mask[3],
                     net->gate_ip[0], net->gate_ip[1], net->gate_ip[2], net->gate_ip[3]);

                     gui->WriteStatus(status);
                     */

            net->octets = 0;

            ret = 2;

        } else if (info.proto.type == proto_arp && net->atype < address_arp) {
            uint8_t new_range[4];

            memset(new_range, 0, 4);

            if (info.proto.source_ip[0] != 0x00 &&
                info.proto.misc_ip[0] != 0x00) {

                int oct;
                for (oct = 0; oct < 4; oct++) {
                    if (info.proto.source_ip[oct] != info.proto.misc_ip[oct])
                        break;

                    new_range[oct] = info.proto.source_ip[oct];
                }

                if (oct < net->octets || net->octets == 0) {
                    net->octets = oct;
                    memcpy(net->range_ip, new_range, 4);
                    snprintf(in_status, STATUS_MAX, "Found IP range for \"%s\" via ARP %d.%d.%d.%d",
                             net->ssid.c_str(), net->range_ip[0], net->range_ip[1], net->range_ip[2], net->range_ip[3]);
                    //gui->WriteStatus(status);

                    net->atype = address_arp;

                    ret = 2;
                }
            } // valid arp
        } else if (info.proto.type == proto_udp && net->atype <= address_udp) {
            uint8_t new_range[4];

            memset(new_range, 0, 4);

            // Not 0.x.x.x.  Not 255.x.x.x.  At least first octet must
            // match.
            if (info.proto.source_ip[0] != 0x00 &&
                info.proto.dest_ip[0] != 0x00 &&
                info.proto.dest_ip[0] != 0xFF &&
                info.proto.source_ip[0] == info.proto.dest_ip[0]) {

                int oct;
                for (oct = 0; oct < 4; oct++) {
                    if (info.proto.source_ip[oct] != info.proto.dest_ip[oct])
                        break;

                    new_range[oct] = info.proto.source_ip[oct];
                }

                if (oct < net->octets || net->octets == 0) {
                    net->octets = oct;
                    memcpy(net->range_ip, new_range, 4);
                    snprintf(in_status, STATUS_MAX, "Found IP range for \"%s\" via UDP %d.%d.%d.%d",
                             net->ssid.c_str(), net->range_ip[0], net->range_ip[1], net->range_ip[2], net->range_ip[3]);
//                    gui->WriteStatus(status);

                    net->atype = address_udp;

                    ret = 2;
                }
            } // valid arp
        } // arp

    } // data packet

    return ret;
}

int Packetracker::WriteNetworks(FILE *in_file) {
    int netnum = 1;
    vector<wireless_network *> bssid_vec;

    // Convert the map to a vector and sort it
    for (map<string, wireless_network *>::const_iterator i = bssid_map.begin();
         i != bssid_map.end(); ++i)
        bssid_vec.push_back(i->second);

    sort(bssid_vec.begin(), bssid_vec.end(), WirelessFirstTimeLT());

    for (unsigned int i = 0; i < bssid_vec.size(); i++) {
        wireless_network *net = bssid_vec[i];

        char lt[25];
        char ft[25];

        snprintf(lt, 25, "%s", ctime(&net->last_time));
        snprintf(ft, 25, "%s", ctime(&net->first_time));

        fprintf(in_file, "Network %d: \"%s\" BSSID: \"%s\"\n"
                "    Channel  : %02d\n"
                "    WEP      : \"%s\"\n"
                "    LLC      : %d\n"
                "    Data     : %d\n"
                "    Crypt    : %d\n"
                "    Weak     : %d\n"
                "    Total    : %d\n"
                "    First    : \"%s\"\n"
                "    Last     : \"%s\"\n",
                netnum,
                net->ssid.c_str(), net->bssid.c_str(),
                net->channel, net->wep ? "Yes" : "No",
                net->llc_packets, net->data_packets,
                net->crypt_packets, net->interesting_packets,
                (net->llc_packets + net->data_packets),
                ft, lt);

        if (net->first_mode > 1) {
            char fix1[16], fix2[16];

            if (net->gps_mode == -1)
                snprintf(fix1, 16, "No signal");
            else if (net->first_mode == 2)
                snprintf(fix1, 5, "2D");
            else if (net->first_mode == 3)
                snprintf(fix1, 5, "3D");
            else
                snprintf(fix1, 5, "NONE");


            if (net->gps_mode == -1)
                snprintf(fix2, 16, "No signal");
            else if (net->gps_mode == 2)
                snprintf(fix2, 5, "2D");
            else if (net->gps_mode == 3)
                snprintf(fix2, 5, "3D");
            else
                snprintf(fix2, 5, "NONE");

            fprintf(in_file,
                    "    First Loc: Lat %f Lon %f Alt %f Spd %f Fix: %s\n"
                    "    Last Loc : Lat %f Lon %f Alt %f Spd %f Fix: %s\n",
                    net->first_lat, net->first_lon, net->first_alt, net->first_spd, fix1,
                    net->gps_lat, net->gps_lon, net->gps_alt, net->gps_spd, fix2);
        }


        if (net->atype == address_dhcp)
            fprintf(in_file, "    Address found via DHCP %d.%d.%d.%d \n"
                    "      netmask %d.%d.%d.%d gw %d.%d.%d.%d\n",
                    net->range_ip[0], net->range_ip[1],
                    net->range_ip[2], net->range_ip[3],
                    net->mask[0], net->mask[1], net->mask[2], net->mask[3],
                    net->gate_ip[0], net->gate_ip[1], net->gate_ip[2], net->gate_ip[3]);
        else if (net->atype == address_arp)
            fprintf(in_file, "    Address found via ARP %d.%d.%d.%d\n",
                    net->range_ip[0], net->range_ip[1],
                    net->range_ip[2], net->range_ip[3]);
        else if (net->atype == address_udp)
            fprintf(in_file, "    Address found via UDP %d.%d.%d.%d\n",
                    net->range_ip[0], net->range_ip[1],
                    net->range_ip[2], net->range_ip[3]);
        fprintf(in_file, "\n");
        netnum++;
    }

    return 1;
}

// Write out the cisco information
int Packetracker::WriteCisco(FILE *in_file) {
    vector<wireless_network *> bssid_vec;

    // Convert the map to a vector and sort it
    for (map<string, wireless_network *>::const_iterator i = bssid_map.begin();
         i != bssid_map.end(); ++i)
        bssid_vec.push_back(i->second);

    sort(bssid_vec.begin(), bssid_vec.end(), WirelessFirstTimeLT());

    for (unsigned int i = 0; i < bssid_vec.size(); i++) {
        wireless_network *net = bssid_vec[i];

        if (net->cisco_equip.size() == 0)
            continue;


        fprintf(in_file, "Network: \"%s\" BSSID: \"%s\"\n",
                net->ssid.c_str(), net->bssid.c_str());

        int devnum = 1;
        for (map<string, cdp_packet>::const_iterator x = net->cisco_equip.begin();
             x != net->cisco_equip.end(); ++x) {
            cdp_packet cdp = x->second;

            fprintf(in_file, "CDP Broadcast Device %d\n", devnum);
            fprintf(in_file, "    Device ID : %s\n", cdp.dev_id);
            fprintf(in_file, "    Capability: %s%s%s%s%s%s%s\n",
                    cdp.cap.level1 ? "Level 1 " : "" ,
                    cdp.cap.igmp_forward ? "IGMP forwarding " : "",
                    cdp.cap.nlp ? "Network-layer protocols " : "",
                    cdp.cap.level2_switching ? "Level 2 switching " : "",
                    cdp.cap.level2_sourceroute ? "Level 2 source-route bridging " : "",
                    cdp.cap.level2_transparent ? "Level 2 transparent bridging " : "",
                    cdp.cap.level3 ? "Level 3 routing " : "");
            fprintf(in_file, "    Interface : %s\n", cdp.interface);
            fprintf(in_file, "    IP        : %d.%d.%d.%d\n",
                    cdp.ip[0], cdp.ip[1], cdp.ip[2], cdp.ip[3]);
            fprintf(in_file, "    Platform  : %s\n", cdp.platform);
            fprintf(in_file, "    Software  : %s\n", cdp.software);
            fprintf(in_file, "\n");
            devnum++;
        } // cdp
    } // net

    return 1;
}


