/*
 * $Id: shuttlepro.c,v 1.6 2002/07/18 04:48:10 ddennedy Exp $
 *
 *  Copyright (c) 2001 Tomoaki Hayasaka <hayasaka@postman.riken.go.jp>
 *     and Dan Dennedy <dan@dennedy.org>
 *
 *  Contour ShuttlePro support
 *
 */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/input.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/usb.h>

/*
 * Version Information
 */
#define DRIVER_VERSION "v0.2"
#define DRIVER_AUTHOR "Tomoaki Hayasaka <hayasaka@postman.riken.go.jp> and Dan Dennedy <dan@dennedy.org>"
#define DRIVER_DESC "Contour USB ShuttlePro driver"

MODULE_AUTHOR( DRIVER_AUTHOR );
MODULE_DESCRIPTION( DRIVER_DESC );
MODULE_LICENSE("GPL");

/*
 * Contour ShuttlePro packet:
 *
 * byte 0: shuttle (00, ff=-1, fe=-2, fd=-3, fc=-4, fb=-5, fa=-6, f9=-7, 1, 2, 3, 4, 5, 6, 7)
 * byte 1: jog (increasing = clockwise, decreasing = counter-clockwise)
 * byte 2: 
 * byte 3: buttons (top row = 01, 02, 04, 08; 2nd row = 10, 20, 40, 80)
 * byte 4: buttons (last 2nd row = 01, inner SW = 02, inner SE = 04, outer SW = 08, outer SE = 10)
 * bytes 5-7: unused?
 * all buttons reset to zero upon release
 *
 */

static unsigned char shuttlepro_keycode[2][256] = { {
	  0, 49, 24,  0, 25,  0,  0,  0, 16,  0,  0,  0,  0,  0,  0,  0,
	 47,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	 45,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	 21,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	 31,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
},
{
	  0, 44, 20,  0, 22,  0,  0,  0, 20,  0,  0,  0,  0,  0,  0,  0,
	 22,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
} };

static signed char shuttlepro_shuttle[256] = {
     30, 48, 46, 32, 18, 33, 34, 35,  0,  0,  0,  0,  0,  0,  0,  0,
	  2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0, 23, 43, 27, 13, 50, 38, 37, 36
};

#define USB_VENDOR_ID_CONTOUR	0x05f3
#define USB_PRODUCT_ID_SHUTTLEPRO	0x0240

struct shuttlepro_features {
	char *name;
	int pktlen;
	void (*irq)(struct urb *urb);
};

struct shuttlepro {
	unsigned char newdata[8];
	unsigned char olddata[8];
	struct input_dev dev;
	struct usb_device *usbdev;
	struct urb irq;
	struct shuttlepro_features *features;
	int open;
};

static void shuttlepro_irq(struct urb *urb)
{
	struct shuttlepro *shuttlepro = urb->context;
	unsigned char *newdata = shuttlepro->newdata;
	unsigned char *olddata = shuttlepro->olddata;
	struct input_dev *dev = &shuttlepro->dev;
	int i;

	if (urb->status)
		return;

#if 0
	printk(KERN_INFO "received: %02x %02x %02x %02x %02x %02x %02x %02x\n",
	    newdata[0], newdata[1], newdata[2], newdata[3],
	    newdata[4], newdata[5], newdata[6], newdata[7]);
#endif

	if (newdata[0] != olddata[0]) {
		input_report_key(dev, shuttlepro_shuttle[newdata[0]], 1);
        input_report_abs(dev, ABS_MISC, (short) newdata[0]);
		input_report_key(dev, shuttlepro_shuttle[newdata[0]], 0);
    }

	if ((newdata[1] < olddata[1]) || (newdata[1]==0xff && olddata[1]==0x00)) {
		input_report_key(dev, shuttlepro_shuttle[0x10], 1);
		input_report_rel(dev, REL_DIAL, -1);
		input_report_key(dev, shuttlepro_shuttle[0x10], 0);
	}
	if ((newdata[1] > olddata[1]) || (newdata[1]==0x00 && olddata[1]==0xff)) {
		input_report_key(dev, shuttlepro_shuttle[0x11], 1);
		input_report_rel(dev, REL_DIAL, 1);
		input_report_key(dev, shuttlepro_shuttle[0x11], 0);
    }

    /* TODO: key combinations */
	for (i = 3; i < 5; i++) {
		if (olddata[i] > 0 && newdata[i] == 0) {
			if (shuttlepro_keycode[i-3][olddata[i]])
				input_report_key(dev, shuttlepro_keycode[i-3][olddata[i]], 0);
			else
				info("Unknown key (scancode %#x) released.", olddata[i]);
		}

		if (newdata[i] > 0 && olddata[i] == 0) {
			if (shuttlepro_keycode[newdata[i]])
				input_report_key(dev, shuttlepro_keycode[i-3][newdata[i]], 1);
			else
				info("Unknown key (scancode %#x) pressed.", newdata[i]);
		}

	}

	memcpy(olddata, newdata, 8);
}

struct shuttlepro_features shuttlepro_features[] = {
	{ "Contour ShuttlePro", 8, shuttlepro_irq },
	{ NULL , 0 }
};

struct usb_device_id shuttlepro_ids[] = {
	{ USB_DEVICE(USB_VENDOR_ID_CONTOUR, USB_PRODUCT_ID_SHUTTLEPRO) },
	{ }
};

MODULE_DEVICE_TABLE(usb, shuttlepro_ids);

static int shuttlepro_open(struct input_dev *dev)
{
	struct shuttlepro *shuttlepro = dev->private;

	if (shuttlepro->open++)
		return 0;

	shuttlepro->irq.dev = shuttlepro->usbdev;
	if (usb_submit_urb(&shuttlepro->irq))
		return -EIO;

	return 0;
}

static void shuttlepro_close(struct input_dev *dev)
{
	struct shuttlepro *shuttlepro = dev->private;

	if (!--shuttlepro->open)
		usb_unlink_urb(&shuttlepro->irq);
}

static void *shuttlepro_probe(struct usb_device *dev, unsigned int ifnum, const struct usb_device_id *id)
{
	struct usb_endpoint_descriptor *endpoint;
	struct shuttlepro *shuttlepro;
	int i, j;

	if (!(shuttlepro = kmalloc(sizeof(struct shuttlepro), GFP_KERNEL)))
		return NULL;
	memset(shuttlepro, 0, sizeof(struct shuttlepro));

	shuttlepro->features = shuttlepro_features;

	shuttlepro->dev.evbit[0] |= BIT(EV_KEY) | BIT(EV_ABS) | BIT(EV_REL);
	shuttlepro->dev.absbit[0] |= BIT(ABS_MISC);
	shuttlepro->dev.relbit[0] |= BIT(REL_DIAL);
	for (j = 0; j < 2; j++)
    	for (i = 0; i < 255; i++)
    		set_bit(shuttlepro_keycode[j][i], shuttlepro->dev.keybit);
  	for (i = 0; i < 255; i++)
  		set_bit(shuttlepro_shuttle[i], shuttlepro->dev.keybit);
	clear_bit(0, shuttlepro->dev.keybit);

	shuttlepro->dev.absmin[ABS_MISC] = -8;
	shuttlepro->dev.absmax[ABS_MISC] = 7;
	shuttlepro->dev.absfuzz[ABS_MISC] = 0;

	shuttlepro->dev.private = shuttlepro;
	shuttlepro->dev.open = shuttlepro_open;
	shuttlepro->dev.close = shuttlepro_close;

	shuttlepro->dev.name = shuttlepro->features->name;
	shuttlepro->dev.idbus = BUS_USB;
	shuttlepro->dev.idvendor = dev->descriptor.idVendor;
	shuttlepro->dev.idproduct = dev->descriptor.idProduct;
	shuttlepro->dev.idversion = dev->descriptor.bcdDevice;
	shuttlepro->usbdev = dev;

	endpoint = dev->config[0].interface[ifnum].altsetting[0].endpoint + 0;

	FILL_INT_URB(&shuttlepro->irq, dev, usb_rcvintpipe(dev, endpoint->bEndpointAddress),
		     shuttlepro->newdata, shuttlepro->features->pktlen, shuttlepro->features->irq, shuttlepro, endpoint->bInterval);

	input_register_device(&shuttlepro->dev);

	printk(KERN_INFO "input%d: %s on usb%d:%d.%d\n",
		 shuttlepro->dev.number, shuttlepro->features->name, dev->bus->busnum, dev->devnum, ifnum);

	return shuttlepro;
}

static void shuttlepro_disconnect(struct usb_device *dev, void *ptr)
{
	struct shuttlepro *shuttlepro = ptr;
	usb_unlink_urb(&shuttlepro->irq);
	input_unregister_device(&shuttlepro->dev);
	kfree(shuttlepro);
}

static struct usb_driver shuttlepro_driver = {
	name:		"shuttlepro",
	probe:		shuttlepro_probe,
	disconnect:	shuttlepro_disconnect,
	id_table:	shuttlepro_ids,
};

static int __init shuttlepro_init(void)
{
	usb_register(&shuttlepro_driver);
	info(DRIVER_VERSION ":" DRIVER_DESC);
	return 0;
}

static void __exit shuttlepro_exit(void)
{
	usb_deregister(&shuttlepro_driver);
}

module_init(shuttlepro_init);
module_exit(shuttlepro_exit);
