/*
 * playlist.cc
 * Copyright (C) 2000 Arne Schirmacher <arne@schirmacher.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// C includes

#include <gnome.h>
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <pthread.h>

// C++ includes

#include <iostream>
#include <fstream>
#include <strstream>

using std::cerr;
using std::endl;
using std::ends;
using std::ofstream;
using std::strstream;

// local includes

#include "playlist.h"
#include "error.h"
#include "riff.h"
#include "avi.h"
#include "frame.h"

#include "progress_dialog.h"

typedef bool (*callback)(xmlNodePtr node, void *p);

typedef struct
{
    int		absFrame;
    int		absBegin;
    int		absEnd;
    int		clipFrame;
    int		clipBegin;
    int		clipEnd;
    int 	clipNumber;
    int 	clipLength;
    char	fileName[ 1024 ];
    xmlNodePtr	sequence;
    xmlNodePtr	video;
}
MovieInfo;


/** Finds the file corresponding to an absolute frame number.
 
    The absolute frame number is passed in MovieInfo.absFrame.
    If found, the MovieInfo struct is filled:
 
	absFrame:   some absolute frame number
	absBegin:   absolute frame number of movie start
	absEnd:     absolute frame number of movie end
        clipFrame:  corresponding relative frame number
        clipBegin:  corresponding relative frame number
        clipEnd:    corresponding relative frame number
	clipNumber: 
        clipLength: frame count in this clip
        fileName:   its file name
	sequence:   node ptr to its <seq> node
	video:      node ptr to its <video> node
 
    \param node a node
    \param p pointer to MovieInfo struct
    \result true if file has been found and xml tree walk is done */

static bool findFile(xmlNodePtr node, void *p)
{
    MovieInfo *data = (MovieInfo*)p;

    if (xmlStrcmp(node->name, (const xmlChar*)"seq") == 0) {
        data->sequence = node;
        data->clipNumber++;
    }

    // if this is a <video> node, calculate its absolute begin and end positions

    else if (xmlStrcmp(node->name, (const xmlChar*)"video") == 0) {

        data->video = node;

        // Check whether the required properties exist

        xmlChar *src = xmlGetProp(node, (const xmlChar*)"src");
        xmlChar *clipBegin = xmlGetProp(node, (const xmlChar*)"clipBegin");
        xmlChar *clipEnd = xmlGetProp(node, (const xmlChar*)"clipEnd");

        if ((src != NULL) && (clipBegin != NULL) && (clipEnd != NULL)) {

            data->clipBegin = atoi((const char*)clipBegin);
            data->clipEnd = atoi((const char*)clipEnd);

            data->absBegin += data->clipLength; // add length of previous clip
            data->clipLength = data->clipEnd - data->clipBegin + 1;
            data->absEnd = data->absBegin + data->clipLength - 1;

            // cerr << "Number: " << data->clipNumber << " starts " << data->absBegin << " ends " << data->absEnd << endl;

            // if absFrame is within this scene, we have found the corresponding file.
            // Otherwise, add frame count of this scene to absBegin

            if (data->absFrame <= data->absEnd) {
                strcpy( data->fileName, (char *)src );
                data->clipFrame = data->absFrame - data->absBegin + data->clipBegin;

				// Free memory used 
            	xmlFree(src);
            	xmlFree(clipEnd);
            	xmlFree(clipBegin);

                // cerr << "Obtaining frame " << data->clipFrame << " from " << data->clipNumber << endl;

                return true; // true means done traversing xml tree
            }
        }

        if (src)
            xmlFree(src);
        if (clipEnd)
            xmlFree(clipEnd);
        if (clipBegin)
            xmlFree(clipBegin);

    }
    return false;
}


/** Searches tree for file names not yet in our filemap
 
    \param node a node
    \param p pointer to filemap
    \result true if file has been found and xml tree walk is done */

static bool fillMap(xmlNodePtr node, void *p)
{
    if (xmlStrcmp(node->name, (const xmlChar*)"video") == 0) {

        // Check whether the required properties exist

        xmlChar *src = xmlGetProp(node, (const xmlChar*)"src");
        xmlChar *clipBegin = xmlGetProp(node, (const xmlChar*)"clipBegin");
        xmlChar *clipEnd = xmlGetProp(node, (const xmlChar*)"clipEnd");

        if ((src != NULL) && (clipBegin != NULL) && (clipEnd != NULL)) {

            // Check whether we already have this file in our map

            map<const char*, AVIFile, ltstr>* filemap = (map<const char*, AVIFile, ltstr>*)p;

            if (filemap->find((char*)src) == filemap->end()) {

                AVIFile *avi;
                avi = new AVIFile();
                if ( avi->Open((char*)src) ) {
                	avi->ParseRIFF();
                	avi->ReadIndex();
                	(*filemap)[ strdup( ( char *) src ) ] = *avi;
				}
				else {
					cerr << "Unable to open " << (char *)src
						 << " - removing from list" << endl;
					xmlUnlinkNode( node );
					xmlFreeNode( node );
				}
                delete avi;
            }
        }

		xmlFree( src );
        xmlFree(clipEnd);
        xmlFree(clipBegin);
    }
    return false;
}


/** Finds the start of a scene.
 
    \param node a node
    \param p pointer to some private data
    \result true if file has been found and xml tree walk is done 
 
    If the scene has been found in the playlist, the file name 
    and first frame number are returned in the private data. */

static bool findSceneStart(xmlNodePtr node, void *p)
{
    int         fileCount = 0;
    MovieInfo   *data = (MovieInfo*)p;

    // if this is a <seq> node process all of its <video> child nodes

    if (xmlStrcmp(node->name, (const xmlChar*)"seq") == 0) {

        data->sequence = node;

        node = node->xmlChildrenNode;
        while (node != NULL) {
            if (xmlStrcmp(node->name, (const xmlChar*)"video") == 0) {

                data->video = node;

                xmlChar *src = xmlGetProp(node, (const xmlChar*)"src");
                xmlChar *clipBegin = xmlGetProp(node, (const xmlChar*)"clipBegin");
                xmlChar *clipEnd = xmlGetProp(node, (const xmlChar*)"clipEnd");

                if ((src != NULL) && (clipBegin != NULL) && (clipEnd != NULL)) {

                    data->clipBegin = atoi((const char*)clipBegin);
                    data->clipEnd = atoi((const char*)clipEnd);

                    // if this is the first file remember its name and start

                    if (fileCount == 0) {
                        data->clipFrame = data->clipBegin;
                        strcpy( data->fileName, (char *)src );
                    } 

                    // if absFrame is within current scene we are done.
                    // fine name and relative frame number have been already found (see above)
                    // otherwise update absBegin to hold abs frame num of next file

                    if (data->absFrame <= data->absBegin + data->clipEnd - data->clipBegin) {
                        xmlFree(clipBegin);
                        xmlFree(clipEnd);
                        xmlFree(src);
                        return true;
					}
                    else {
                        data->absBegin += (data->clipEnd - data->clipBegin + 1);
					}
                    fileCount++;
                }
                if (src)
                    xmlFree(src);
                if (clipEnd)
                    xmlFree(clipEnd);
                if (clipBegin)
                    xmlFree(clipBegin);
            }
            node = node->next;
        }
    }
    data->clipFrame = 0;
	strcpy( data->fileName, "" );
    return false;
}


static bool findSceneEnd(xmlNodePtr node, void *p)
{
    bool found = false;
    xmlChar *src = NULL;
    MovieInfo *data = (MovieInfo*)p;

    // if this is a <seq> node process all of its <video> child nodes

    if (xmlStrcmp(node->name, (const xmlChar*)"seq") == 0) {

        data->sequence = node;

        node = node->xmlChildrenNode;
        while (node != NULL) {
            if (xmlStrcmp(node->name, (const xmlChar*)"video") == 0) {

                data->video = node;

                if (src)
                    xmlFree(src);

                src = xmlGetProp(node, (const xmlChar*)"src");
                xmlChar *clipBegin = xmlGetProp(node, (const xmlChar*)"clipBegin");
                xmlChar *clipEnd = xmlGetProp(node, (const xmlChar*)"clipEnd");

                if ((src != NULL) && (clipBegin != NULL) && (clipEnd != NULL)) {

                    data->clipBegin = atoi((const char*)clipBegin);
                    data->clipEnd = atoi((const char*)clipEnd);
                    data->clipFrame = data->clipEnd;

                    if (data->absFrame <= data->absBegin + data->clipEnd - data->clipBegin)
                        found = true;
                    data->absBegin += (data->clipEnd - data->clipBegin + 1);
                }

                if (clipEnd)
                    xmlFree(clipEnd);
                if (clipBegin)
                    xmlFree(clipBegin);
            }
            node = node->next;
        }

        if (found) {
            strcpy( data->fileName, (char *)src );
			xmlFree( src );
            data->absEnd = data->absBegin - 1;
            return true;
		}

        if (src)
            xmlFree(src);
    }
    data->clipFrame = 0;
    strcpy( data->fileName, "" );
    return false;
}

/** Count number of frames in one node
 
    \param node a node
    \param p pointer to some private data
 
    If this is a <video> node then calculate the number of frames
    in this clip. p is a ptr to an integer holding the total
    number of frames so far.
 
    The node must have this format:
    <video src="file.avi" clipBegin="200" clipEnd="800" \>
 
    \note This code requires a <video> node format that is not defined in smil. */

static bool countFrames(xmlNodePtr node, void *p)
{
    if (xmlStrcmp(node->name, (const xmlChar*)"video") == 0) {

        xmlChar *src = xmlGetProp(node, (const xmlChar*)"src");
        xmlChar *clipBegin = xmlGetProp(node, (const xmlChar*)"clipBegin");
        xmlChar *clipEnd = xmlGetProp(node, (const xmlChar*)"clipEnd");

        if ((src != NULL) && (clipBegin != NULL) && (clipEnd != NULL))
            *((int*)p) += atoi((const char*)clipEnd) - atoi((const char*)clipBegin) + 1;

        if (clipEnd)
            xmlFree(clipEnd);
        if (clipBegin)
            xmlFree(clipBegin);
        if (src)
            xmlFree(src);
    }
    return false;
}


/** Walk the xml tree
 
    \param node the root node
    \param func the function to execute on each node
    \param p storage to some private data for func 
 
    This function is usually called with the root node
    of an XML tree. It calls the user supplied callback
    function on each node and then visits recursively
    all child nodes.
 
    If the callback function returns true the xml
    tree walk is aborted. */

static bool parse(xmlNodePtr node, callback func, void *p)
{
    bool done = false;

    while (node != NULL && done == false) {
        done = (*func)(node, p);
        if (done == false)
            done = parse(node->xmlChildrenNode, func, p);
        node = node->next;
    }
    return done;
}


/** Default Constructor */

PlayList::PlayList()
{
    xmlNsPtr	ns;
    xmlNodePtr	root;

    // cerr << "*PlayList::PlayList()" << endl;

    doc = xmlNewDoc((const xmlChar*)"1.0");
    root = xmlNewNode(NULL, (const xmlChar*)"smil");
    ns = xmlNewNs(root,(const xmlChar*)"http://www.w3.org/2001/SMIL20/Language",(const xmlChar*)"smil2");
    xmlDocSetRootElement(doc, root);
}


/** Copy Constructor */

PlayList::PlayList(const PlayList& playList)
{
    // cerr << "*PlayList::PlayList(const PlayList& playList)" << endl;

    doc = xmlCopyDoc(playList.doc, true);
    filemap = playList.filemap;
}


/** Assignment Operator */

PlayList& PlayList::operator=(const PlayList& playList)
{
    // cerr << "*PlayList::operator=(const PlayList& playList)" << endl;

    if (doc != playList.doc) {
        xmlFreeDoc(doc);
        doc = xmlCopyDoc(playList.doc, true);
        filemap = playList.filemap;
    }
    return *this;
}


/** The PlayList Destructor
    Deletes the XML document if we have one */

PlayList::~PlayList()
{
    // cerr << "*PlayList::~PlayList()" << endl;

    if (doc != NULL) {
        xmlFreeDoc(doc);
        doc = NULL;
    }
}


/** Counts the number of frames in the playlist
 
    \return the number of frames in the playlist */

int PlayList::GetNumFrames() const
{
    int count = 0;

    // cerr << "int PlayList::GetNumFrames()";

    if (doc != NULL)
        parse(xmlDocGetRootElement(doc), countFrames, &count);

    // cerr << ": " << count << endl;

    return count;
}


char* PlayList::GetFileNameOfFrame(int frameNum) const
{
    // cerr << "char* PlayList::GetFileNameOfFrame(int frameNum)" << endl;

    return NULL;
}


/** Get one frame
 
    gets one frame of the playlist.
 
    \param absFrame the frame number to get
    \param frame the frame
    \return true if a frame could be copied, false otherwise
*/

bool PlayList::GetFrame(int frameNum, Frame &frame)
{
    MovieInfo data;

    // cerr << "bool PlayList::GetFrame(" << frameNum << ", Frame &frame)" << endl;

    memset(&data, 0, sizeof(MovieInfo));
    data.absBegin = 0;
    data.absEnd = 0;
    data.absFrame = frameNum;

    parse(xmlDocGetRootElement(doc), findFile, &data);

    if ( strcmp( data.fileName, "" ) ) {
        AVIFile &avi = filemap[(char*)data.fileName];
        avi.GetDVFrame(frame, data.clipFrame);
#if 0
        /* if the filemap fails try direct loading */
        AVIFile *avi = new AVIFile;
        fail_if(avi->Open((char*)data.fileName) == false);
        avi->ParseRIFF();
        avi->ReadIndex();
        avi->GetDVFrame(frame, data.clipFrame);
        avi->Close();
        delete avi;
#endif
        return true;
    }
	else {
		cerr << "Frame not found " << frameNum << endl;
	}

    return false;
}


int PlayList::FindStartOfScene(int frameNum) const
{
    MovieInfo data;

    // cerr << "int PlayList::FindStartOfScene(int frameNum)" << endl;

    memset(&data, 0, sizeof(MovieInfo));
    data.absBegin = 0;
    data.absEnd = 0;
    data.absFrame = frameNum;

    parse(xmlDocGetRootElement(doc), findSceneStart, &data);

    if ( strcmp( data.fileName, "" ) )
        return data.absBegin;
    else
        return 0;
}


int PlayList::FindEndOfScene(int frameNum) const
{
    MovieInfo data;

    // cerr << "int PlayList::FindEndOfScene(int frameNum)" << endl;

    data.absBegin = 0;
    data.absEnd = 0;
    data.absFrame = frameNum;

    parse(xmlDocGetRootElement(doc), findSceneEnd, &data);

    if ( strcmp( data.fileName, "" ) )
        return data.absEnd;
    else
        return 999999;
}


void  PlayList::AutoSplit(int start, int end)
{
    MovieInfo   firstFile;
    MovieInfo   lastFile;
    Frame	*frame;
    struct tm	recDate;
    time_t	startTime;
    time_t	endTime;

    // cerr << "PlayList::AutoSplit(int start=" << start << ", int end=" << end << ")" ;
    // cerr << endl;

    memset(&firstFile, 0, sizeof(MovieInfo));
    firstFile.absBegin = 0;
    firstFile.absEnd = 0;
    firstFile.absFrame = start;

    parse(xmlDocGetRootElement(doc), findFile, &firstFile);
    AVIFile &avi1 = filemap[(char*)firstFile.fileName];
    frame = new Frame;
    avi1.GetDVFrame(*frame, firstFile.clipFrame);
    frame->GetRecordingDate(recDate);
    startTime = mktime(&recDate);

    memset(&lastFile, 0, sizeof(MovieInfo));
    lastFile.absBegin = 0;
    lastFile.absEnd = 0;
    lastFile.absFrame = end;

    parse(xmlDocGetRootElement(doc), findFile, &lastFile);

    AVIFile &avi2 = filemap[(char*)lastFile.fileName];
    avi2.GetDVFrame(*frame, lastFile.clipFrame);
    frame->GetRecordingDate(recDate);
    endTime = mktime(&recDate);

    int fps = frame->IsPAL() ? 25 : 30;
    delete frame;
    // bail out on invalid recording date/time
    if (startTime < 0 || endTime < 0) return;

    if (((difftime(endTime, startTime) * fps) - (end - start)) > fps) {
        if ((end - start) > 1) {
            // cerr << " ! " << endl;
            AutoSplit (start, start + (end - start) / 2);
            AutoSplit (start + (end - start) / 2, end);
        } else {
            // cerr << endl << " *** split before " << end << endl;
            SplitSceneBefore(end);
        }
    } else {
        // cerr << " ok" << endl;
    }
}


void  PlayList::SplitSceneBefore(int frameNum)
{
    MovieInfo data;

    // cerr << "PlayList::SplitSceneBefore(int frameNum=" << frameNum << ")" << endl;

    memset(&data, 0, sizeof(MovieInfo));
    data.absBegin = 0;
    data.absEnd = 0;
    data.absFrame = frameNum;
    parse(xmlDocGetRootElement(doc), findSceneStart, &data);
    int begin = data.absBegin;

    memset(&data, 0, sizeof(MovieInfo));
    data.absBegin = 0;
    data.absEnd = 0;
    data.absFrame = frameNum;
    parse(xmlDocGetRootElement(doc), findSceneEnd, &data);
    int end = data.absEnd;

    if ( strcmp( data.fileName, "" ) ) {

        // duplicate the whole scene.

        xmlNode *firstSequence = data.sequence;
        xmlNode *secondSequence = xmlCopyNodeList(firstSequence);
        xmlAddNextSibling(firstSequence, secondSequence);

        // in the first sequence, delete from frameNum to end of scene

        Delete(frameNum, end);

        // in the second sequence, delete from start of scene to frameNum - 1

        Delete(frameNum, frameNum - begin + frameNum - 1);
    }
}


/** Get a playlist
 
    Returns a subset of the frames as a playlist. The parameters first and last
    must be within the available frames in the playlist.
 
    \param first number of first frame
    \param last number of last frame
    \return true if the frames could be copied, false otherwise
*/

bool PlayList::GetPlayList(int first, int last, PlayList &playlist) const
{
    MovieInfo firstFile, lastFile;
    bool copyFlag = false;

    // cerr << " bool PlayList::Copy(int first, int last, PlayList &playlist) " << endl;

    memset(&firstFile, 0, sizeof(MovieInfo));
    firstFile.absBegin = 0;
    firstFile.absEnd = 0;
    firstFile.absFrame = first;

    parse(xmlDocGetRootElement(doc), findFile, &firstFile);

    memset(&lastFile, 0, sizeof(MovieInfo));
    lastFile.absBegin = 0;
    lastFile.absEnd = 0;
    lastFile.absFrame = last;

    parse(xmlDocGetRootElement(doc), findFile, &lastFile);

    if ( strcmp( firstFile.fileName, "" )  && strcmp(lastFile.fileName, "" )) {

        xmlNodePtr srcNode = xmlDocGetRootElement(doc);
        xmlNodePtr dstNode = xmlDocGetRootElement(playlist.doc);

        for (xmlNodePtr srcSeq = srcNode->xmlChildrenNode; srcSeq != NULL; srcSeq = srcSeq->next) {
            if (xmlStrcmp(srcSeq->name, (const xmlChar*)"seq") == 0) {
                xmlNodePtr seq = xmlNewNode(NULL, (const xmlChar*)"seq");
                xmlAddChild(dstNode, seq);
                for (xmlNodePtr srcVideo = srcSeq->xmlChildrenNode; srcVideo != NULL; srcVideo = srcVideo->next) {
                    if (xmlStrcmp(srcVideo->name, (const xmlChar*)"video") == 0) {

                        // case 1: selection contains more than one file. This one is neither the first nor the last.

                        if (copyFlag && srcVideo != firstFile.video && srcVideo != lastFile.video) {
                            xmlAddChild(seq, xmlCopyNode(srcVideo,1));
                        }

                        // case 2: selection contains more than one file and this is the first file

                        else if (srcVideo == firstFile.video && srcVideo != lastFile.video) {

                            strstream sb1, sb2;

                            xmlNodePtr video = xmlNewNode(NULL, (const xmlChar*)"video");
                            xmlNewProp(video, (const xmlChar*)"src", (const xmlChar*)firstFile.fileName);
                            sb1 << firstFile.clipFrame << ends;
                            xmlNewProp(video, (const xmlChar*)"clipBegin", (const xmlChar*)sb1.str());
                            sb2 << firstFile.clipEnd << ends;
                            xmlNewProp(video, (const xmlChar*)"clipEnd", (const xmlChar*)sb2.str());
                            xmlAddChild(seq, video);
                            copyFlag = true;
                        }

                        // case 3: selection contains more than one file and this is the last file

                        else if (srcVideo != firstFile.video && srcVideo == lastFile.video) {

                            strstream sb1, sb2;

                            xmlNodePtr video = xmlNewNode(NULL, (const xmlChar*)"video");
                            xmlNewProp(video, (const xmlChar*)"src", (const xmlChar*)lastFile.fileName);
                            sb1 << lastFile.clipBegin << ends;
                            xmlNewProp(video, (const xmlChar*)"clipBegin", (const xmlChar*)sb1.str());
                            sb2 << lastFile.clipFrame << ends;
                            xmlNewProp(video, (const xmlChar*)"clipEnd", (const xmlChar*)sb2.str());
                            xmlAddChild(seq, video);
                            copyFlag = false;
                        }

                        // case 4: selection contains exactly one file

                        else if (srcVideo == firstFile.video && srcVideo == lastFile.video) {

                            strstream sb1, sb2;

                            xmlNodePtr video = xmlNewNode(NULL, (const xmlChar*)"video");
                            xmlNewProp(video, (const xmlChar*)"src", (const xmlChar*)firstFile.fileName);
                            sb1 << firstFile.clipFrame << ends;
                            xmlNewProp(video, (const xmlChar*)"clipBegin", (const xmlChar*)sb1.str());
                            sb2 << lastFile.clipFrame << ends;
                            xmlNewProp(video, (const xmlChar*)"clipEnd", (const xmlChar*)sb2.str());
                            xmlAddChild(seq, video);
                        }
                    }
                }

                // if this sequence does not have any video clips, remove it

                if (seq->xmlChildrenNode == NULL) {
                    xmlUnlinkNode(seq);
                    xmlFreeNode(seq);
                }
            }
        }
    }
    return true;
}


/** Insert a playlist
 
    Inserts all frames contained in the parameter playlist.
    To insert the frames at the start of the playlist, use
    a before value of 0. To append it at the end of the playlist.
    pass the first unused (= number of frames contained) index.
 
    \param playlist The playlist to insert
    \param before insert playlist before this frame
*/

bool PlayList::InsertPlayList(PlayList &playlist, int before)
{
    // cerr << "bool PlayList::Paste(PlayList &playlist, int before(=" << before << "))" << endl;

	parse(xmlDocGetRootElement(playlist.doc), fillMap, &filemap);

    MovieInfo file;

    memset(&file, 0, sizeof(MovieInfo));
    file.absBegin = 0;
    file.absEnd = 0;
    file.absFrame = before;
    file.sequence = NULL;
    file.video = NULL;

	// Fill the map with any new files now, before we change the doc
    parse(xmlDocGetRootElement(doc), findFile, &file);

    xmlNodePtr node = xmlDocGetRootElement(playlist.doc);

    bool first = true;
    xmlNodePtr next = NULL;
    xmlNodePtr sequence = file.sequence;

    for (xmlNodePtr ptr = node->children; ptr != NULL; ptr = next) {
        //cerr << endl << "Sibling" << endl;
        //xmlElemDump(stderr, NULL, ptr);
        //cerr << endl;

        // Get the next sibling before adding
        next = ptr->next;

        // If first and at start of scene insert, otherwise append
        // cerr << "Scene i'm pasting into starts at " << file.absBegin << " [" << file.absEnd << "]" << endl;

        if (first && sequence == NULL) {
            ptr = xmlCopyNode(ptr, -1);
            sequence = xmlAddChild(xmlDocGetRootElement(doc), ptr);
        } else if (first  && before == file.absBegin && before != (file.absEnd + 1)) {
            // cerr << "Inserting before " << before << endl;
            sequence = xmlAddPrevSibling(sequence, ptr);
        } else if (first && before != (file.absEnd + 1)) {
            // cerr << "Splitting scene that start at " << file.absBegin << " and ends at " << file.absEnd << " at " << before << endl;
            // cerr << endl << "Before Split" << endl;
            // xmlElemDump(stderr, NULL, sequence);
            // cerr << endl;

            // Split the current scene
            SplitSceneBefore(before);

            // Find our new position
            memset(&file, 0, sizeof(MovieInfo));
            file.absBegin = 0;
            file.absFrame = before;
            file.sequence = NULL;
            file.video = NULL;

            parse(xmlDocGetRootElement(doc), findFile, &file);

            // cerr << endl << "After Split" << endl;
            // xmlElemDump(stderr, NULL, sequence);
            // cerr << endl;

            // Add before the scene returned
            sequence = xmlAddPrevSibling(file.sequence, ptr);
        } else {
            // cerr << "Inserting after " << before << endl;
            sequence = xmlAddNextSibling(sequence, ptr);
        }

        // We're definitely no longer first
        first = false;
    }

    return true;
}


bool PlayList::Delete(int first, int last)
{
    int         absClipBegin;
    int         clipBegin;
    int         clipEnd;
    static int  firstCall = 0;

    // cerr << "bool PlayList::Delete(int first=" << first << ", int last=" << last << ")" << endl;

    // SplitSceneBefore calls Delete, avoid recursion

    if (firstCall == 0) {
        firstCall = 1;
        SplitSceneBefore(first);
        firstCall = 0;
    }

    xmlNodePtr srcNode = xmlDocGetRootElement(doc);
    absClipBegin = 0;
  	xmlNodePtr nextSequence = NULL;
    for (xmlNodePtr srcSeq = srcNode->xmlChildrenNode; srcSeq != NULL; srcSeq = nextSequence ) {

		// In case we need to delete this node, get the next pointer before starting
		nextSequence = srcSeq->next;

        if (xmlStrcmp(srcSeq->name, (const xmlChar*)"seq") == 0) {
    		xmlNodePtr nextVideo = NULL;

            for (xmlNodePtr srcVideo = srcSeq->xmlChildrenNode; srcVideo != NULL; srcVideo = nextVideo ) {

				// In case we have to delete this node
				nextVideo = srcVideo->next;

                if (xmlStrcmp(srcVideo->name, (const xmlChar*)"video") == 0) {

                    strstream sb1, sb2;
                    xmlChar *s;

                    sb1 << (s = xmlGetProp(srcVideo, (const xmlChar*)"clipBegin")) << ends;
                    sb1 >> clipBegin;
                    if (s)
                        xmlFree(s);
                    s = xmlGetProp(srcVideo, (const xmlChar*)"clipEnd");
					clipEnd = atoi( (char *)s );
                    sb2 << (s = xmlGetProp(srcVideo, (const xmlChar*)"clipEnd")) << ends;
                    sb2 >> clipEnd;
                    if (s)
                        xmlFree(s);

                    // case 1: selection covers this file completely. Remove this file from playlist.

                    if (first <= absClipBegin && last >= absClipBegin + clipEnd - clipBegin) {
                        xmlUnlinkNode(srcVideo);
                        xmlFreeNode(srcVideo);
                        // cerr << "case 1 " << endl;
                    }

                    // case 2: selection starts before or at start of file and ends somewhere in the file.
                    // New start of file is now end of selection + 1

                    else if (first <= absClipBegin && last >= absClipBegin && last <= absClipBegin + clipEnd - clipBegin) {

                        strstream sb;

                        sb << last - absClipBegin + clipBegin + 1 << ends;
                        xmlSetProp(srcVideo,(const xmlChar*)"clipBegin", (const xmlChar*)sb.str());
                        // cerr << "case 2 " << endl;
                    }

                    // case 3: selection starts somewhere in the file and ends at or after the file
                    // New end of file is now start of selection - 1

                    else if (first > absClipBegin && first <= absClipBegin + clipEnd - clipBegin && last >= absClipBegin + clipEnd - clipBegin) {

                        strstream sb;

                        sb << first - absClipBegin + clipBegin - 1 << ends;
                        xmlSetProp(srcVideo,(const xmlChar*)"clipEnd", (const xmlChar*)sb.str());
                        // cerr << "case 3 " << endl;
                    }

                    // case 4: selection starts somewhere in the file and ends in the file.
                    // We must split this node such that end of first file is start of selection - 1
                    // and start of second file is end of selection + 1

                    else if (first > absClipBegin && last < absClipBegin + clipEnd - clipBegin) {

                        strstream sb1,sb2;
                        xmlChar	  *s;

                        xmlNodePtr video = xmlNewNode(NULL, (const xmlChar*)"video");
                        xmlNewProp(video, (const xmlChar*)"src", (s = xmlGetProp(srcVideo, (const xmlChar*)"src")));
                        if (s)
                            xmlFree(s);
                        sb1 << last - absClipBegin + clipBegin + 1 << ends;
                        xmlNewProp(video, (const xmlChar*)"clipBegin", (const xmlChar*)sb1.str());
                        xmlNewProp(video, (const xmlChar*)"clipEnd",  (s = xmlGetProp(srcVideo, (const xmlChar*)"clipEnd")));
                        if (s)
                            xmlFree(s);
                        xmlAddNextSibling(srcVideo, video);
                        sb2 << first - absClipBegin + clipBegin - 1 << ends;
                        xmlSetProp(srcVideo,(const xmlChar*)"clipEnd", (const xmlChar*)sb2.str());
                        // cerr << "case 4 " << endl;
                    }

                    absClipBegin += clipEnd - clipBegin + 1;
                }
            }

            // if the node is now empty, delete it (can delete - see nextSequence above)

            if (srcSeq->xmlChildrenNode == NULL) {
        		xmlUnlinkNode( srcSeq );
        		xmlFreeNode( srcSeq );
            }
        }
    }

    return true;
}


void PlayList::LoadAVI(char *filename)
{
    // cerr << "bool PlayList::LoadAVI(" << filename << ")" << endl;

    xmlNodePtr	seq;
    xmlNodePtr	node;
    strstream	sb;
    AVIFile	*avi;
    int		existingFrames;
    int		framesInFile;

    avi = new AVIFile();
    fail_if(avi->Open(filename) == false);
    avi->ParseRIFF();
    avi->ReadIndex();
    framesInFile = avi->GetTotalFrames();
    existingFrames = GetNumFrames();
    filemap[filename] = *avi;
    delete avi;

    seq = xmlNewNode(NULL, (const xmlChar*)"seq");
    node = xmlDocGetRootElement(doc);
    xmlAddChild(node,seq);
    node = xmlNewNode(NULL, (const xmlChar*)"video");
    xmlNewProp(node, (const xmlChar*)"src", (const xmlChar*)filename);
    xmlNewProp(node, (const xmlChar*)"clipBegin", (const xmlChar*)"0");
    sb << framesInFile - 1 << ends;
    xmlNewProp(node, (const xmlChar*)"clipEnd", (const xmlChar*)sb.str());
    xmlAddChild(seq,node);

    AutoSplit(existingFrames, existingFrames + framesInFile - 1);
}


bool PlayList::LoadPlayList(char *filename)
{
    // cerr << "bool PlayList::LoadPlayList(" << filename << ")" << endl;

    xmlNsPtr    ns;
    xmlNodePtr  node;

    xmlFreeDoc(doc);
    fail_null(doc = xmlParseFile(filename));

    node = xmlDocGetRootElement(doc);
    if (node == NULL) {
        cerr << "empty document" << endl;
        xmlFreeDoc(doc);
        doc = NULL;
        return false;
    }
    ns = xmlSearchNsByHref(doc, node, (const xmlChar *) "http://www.w3.org/2001/SMIL20/Language");
    if (ns == NULL) {
        cerr << "document of the wrong type, Namespace not found" << endl;
        xmlFreeDoc(doc);
        doc = NULL;
        return false;
    }
    if (xmlStrcmp(node->name, (const xmlChar *) "smil")) {
        cerr << "document of the wrong type, root node != smil" << endl;
        xmlFreeDoc(doc);
        doc = NULL;
        return false;
    }
    CleanPlayList(node);

    parse(xmlDocGetRootElement(doc), fillMap, &filemap);

    return true;
}


bool PlayList::SavePlayList(char *filename)
{
    // cerr << "bool PlayList::SavePlayList(char *filename)" << endl;
    bool	success;

    try {
        xmlSaveFormatFile(filename,doc,1);
        success = true;
    } catch (string exc) {
        cerr << exc << endl;
        success = false;
    }
    return success;
}


/** Recursively deletes unnecessary items from our XML tree.
 
    We need only <smil>, <seq> and <video> nodes, delete
    everything else, in particular the text nodes too.
 
    \param node start deleting here
*/


void PlayList::CleanPlayList(xmlNodePtr node)
{
    while (node != NULL) {

        xmlNodePtr nodeToDelete = NULL;

        CleanPlayList(node->xmlChildrenNode);
        if (xmlStrcmp(node->name, (const xmlChar*)"smil") == 0) {
            //do nothing
        }
        else if (xmlStrcmp(node->name, (const xmlChar*)"seq") == 0) {
            if (node->xmlChildrenNode == NULL) {
                nodeToDelete = node;
            }
        } else if (xmlStrcmp(node->name, (const xmlChar*)"video") == 0) {
            // do nothing
        }
        else
            nodeToDelete = node;
        node = node->next;

        if (nodeToDelete != NULL) {
            xmlUnlinkNode(nodeToDelete);
            xmlFreeNode(nodeToDelete);
        }
    }
}


