/*
 * page_export.cc Notebook Firewire/AVI/Still Frame Export Page Object
 * Copyright (C) 2001 Dan Dennedy <dan@dennedy.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <iostream>
using std::cout;
using std::endl;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include <sys/time.h>

#include "page_export.h"
#include "page_export_1394.h"
#include "page_export_avi.h"
#include "page_export_stills.h"
#include "page_export_audio.h"
#include "page_editor.h"

extern "C" {
	#include "support.h"
	#include "callbacks.h"
	#include "interface.h"
	#include "commands.h"

	extern struct navigate_control g_nav_ctl;
	extern char cmd[16];
	char lastcmd_export[16];
}

/** Constructor for page.

  	\param common	common object to which this page belongs
*/

PageExport::PageExport( KinoCommon *common ) :  currentMode(-1), isExporting(false), exportMutex(false)
{
	cout << "> Creating Export Page" << endl;
	this->common = common;
	this->notebook = GTK_NOTEBOOK( lookup_widget( common->getWidget(), "notebook_export" ) );
	this->progressBar = GTK_PROGRESS_BAR( lookup_widget( common->getWidget(), "progressbar_export" ) );

	// Create PageExport objects
	this->export1394 = new Export1394(this, common);
	this->exportAVI = new ExportAVI(this, common);
	this->exportStills = new ExportStills(this, common);
	this->exportAudio = new ExportAudio(this, common);

	// Set to the first page
	this->currentMode = EXPORT_MODE_1394;

}

/** Destructor for page.
*/

PageExport::~PageExport() {
	cout << "> Destroying Export Page" << endl;
}

/** Start of page.
*/

void PageExport::start() {
	cout << ">> Starting Export" << endl;
	getCurrentPage()->start();
	common->activateWidgets();
	resetProgress();
}

/** Define active widgets.
*/

gulong PageExport::activate() {
	return this->getCurrentPage()->activate();
}

/** Leaving the page
*/

void PageExport::clean() {
	cout << ">> Leaving Export" << endl;
	getCurrentPage()->clean();
}


/** accessor method to the common widget

	For use by the export objects.
*/
GtkWidget *PageExport::getWidget() {
	return common->getWidget();
}


/**	Sends change export mode request to the gui.

	\param mode export mode to change to
*/

void PageExport::changeModeRequest( int mode ) {
	gtk_notebook_set_page( notebook, mode );
}

/**	Carries out a change mode - this should be triggered by changeModeRequest
	to ensure that the GUI stays in sync with the common structure.

	\param mode export mode to change to
*/
void PageExport::setCurrentMode( int mode ) {
	if ( currentMode != mode ) {
		if ( currentMode != -1 )
			clean();
		currentMode = mode;
		start();
	}
}


/** Returns the Page object associated to the specified notebook page.

	\param mode export mode of PageExport object to return
	\return the PageExport object associated to the given notebook page or NULL if invalid.
*/

Export *PageExport::getPage( int mode ) {
	Export *ret = NULL;

	switch( mode ) {
		case EXPORT_MODE_1394:
			ret = getPageExport1394();
			break;
		case EXPORT_MODE_AVI:
			ret = getPageExportAVI();
			break;
		case EXPORT_MODE_STILLS:
			ret = getPageExportStills();
			break;
		case EXPORT_MODE_AUDIO:
			ret = getPageExportAudio();
			break;
	}

	return ret;
}

/** Returns the Page object that corresponds to the current export mode.

	\return the current Page object (null if current page is invalid).
*/

Export *PageExport::getCurrentPage( ) {
	Export *ret = getPage( this->currentMode );
	return ret;
}

/** Returns the Page associated to the 1394 export mode.

	\return the Page object requested
*/

Export1394 *PageExport::getPageExport1394( ) {
	return export1394;
}

/** Returns the PageExport associated to the AVI export mode.

	\return the PageExport object requested
*/

ExportAVI *PageExport::getPageExportAVI( ) {
	return exportAVI;
}

/** Returns the PageExport associated to the still frames export mode.

	\return the PageExport object requested
*/

ExportStills *PageExport::getPageExportStills( ) {
	return exportStills;
}

/** Returns the PageExport associated to the audio export mode.

	\return the PageExport object requested
*/

ExportAudio *PageExport::getPageExportAudio( ) {
	return exportAudio;
}


/** Start export generically 
*/
void PageExport::startExport() {
	if ( !exportMutex && !isExporting ) {
		exportMutex = true;
		isExporting = true;
		
		getCurrentPage()->startExport();
	
		exportMutex = false;
    }
}


/** Stop the export generically
*/
void PageExport::stopExport() {
	if ( !exportMutex  && isExporting ) {
		exportMutex = true;
		
		getCurrentPage()->stopExport();
		
        isExporting = false;
        exportMutex = false;
    }
}


/** Set the export range to a particular scene.

    \param scene the numerical index referring to a scene in the playlist
     
*/
void PageExport::selectScene( int scene ) {
	getCurrentPage()->selectScene( scene );
}


/** Process a keyboard event. 

  	\param event	keyboard event
*/

gboolean PageExport::processKeyboard( GdkEventKey *event ) {
	gboolean ret = FALSE;

#if 0
		printf("send_event: %2.2x\n", event->send_event);
		printf("time  : %8.8x\n", event->time);
		printf("state : %8.8x\n", event->state);
		printf("keyval: %8.8x\n", event->keyval);
		printf("length: %8.8x\n", event->length);
		printf("string: %s\n", event->string);
		printf("(hex) : %2.2x\n", event->string[0]);
		printf("cmd   : %s\n", cmd);
		printf("(hex) : %8.8x\n", cmd[0]);
#endif

	// Only process while not escape mode
	if ( g_nav_ctl.escaped == FALSE ) {
		switch (event->keyval) {
			// Translate special keys to equivalent command
			case GDK_Return:
				common->keyboardFeedback(cmd, "Start Export");
				startExport();
				break;
			case GDK_Escape:
				if (this->isExporting) {
					common->keyboardFeedback(cmd, "Stop Export");
					stopExport( );
				} else {
					common->changePageRequest( PAGE_EDITOR );
				}
				cmd[0] = 0;
				return FALSE;
			case GDK_Alt_L:
			case GDK_Alt_R:
				strcpy( lastcmd_export, "alt");
				return FALSE;
			default:
				strcat(cmd, event->string);
				break;
		}
		
		if ( !strcmp( cmd, "." ) )
			strcpy( cmd, lastcmd_export );
	
		ret = processKeyboardCommand( cmd );

	} else if (event->keyval == GDK_Return || event->keyval == GDK_Escape) {
		g_nav_ctl.escaped = FALSE;
		gtk_widget_grab_focus( GTK_WIDGET(notebook) );
	}

	return ret;
}


/** Internal method for handling a complete keyboard scene.

  	\param cmd		command to be processed;
*/

gboolean PageExport::processKeyboardCommand( char *cmd ) {
	int count = 1;
	char real[ 256 ] = "";

	switch( sscanf( cmd, "%d%s", &count, real ) )
	{
		case 1:
			// Numeric value only - return immediately if the cmd is not "0"
			if ( strcmp( cmd, "0" ) ) {
				common->keyboardFeedback(cmd, "");
				return FALSE;
			}
			break;
		case 0:
			sscanf( cmd, "%s", real );
			count = 1;
			break;
	}

#if 0
	/* switch to file entry */
	if ( !strcmp( cmd, ":" ) ) 
	{
		gtk_entry_set_editable( fileEntry, TRUE );
		gtk_widget_grab_focus( GTK_WIDGET(fileEntry) );
		gtk_entry_set_text( fileEntry, "" );
		g_nav_ctl.escaped = TRUE;
		cmd[0] = 0;
	}
#endif

	/* switch modes */
	if ( strcmp( lastcmd_export, "alt") == 0 ) {
		if (strcmp(cmd, "i") == 0) 
		{
			common->keyboardFeedback(cmd, "Switch to IEEE 1394 export mode");
			changeModeRequest( EXPORT_MODE_1394 );
			cmd[0] = 0;
		}
		
		else if (strcmp(cmd, "v") == 0) 
		{
			common->keyboardFeedback(cmd, "Switch to AVI export mode");
			changeModeRequest( EXPORT_MODE_AVI );
			cmd[0] = 0;
		}
	
		else if ( strcmp(cmd, "s") == 0 ) 
		{
			common->keyboardFeedback(cmd, "Switch to Still Frames export mode");
			changeModeRequest( EXPORT_MODE_STILLS );
			cmd[0] = 0;
		}
	
		else if ( strcmp(cmd, "u") == 0 ) 
		{
			common->keyboardFeedback(cmd, "Switch to Audio export mode");
			changeModeRequest( EXPORT_MODE_AUDIO );
			cmd[0] = 0;
		}
	}

	strcpy( lastcmd_export, cmd );
	
	{
		// Check for invalid commands
		if ( strlen( real ) > 3 )
			cmd[ 0 ] = 0;
		else if ( strchr( "dgy ", real[ strlen( real ) - 1 ] ) == NULL ) 
			cmd[ 0 ] = 0;

		common->keyboardFeedback(cmd, "");
	}

	return FALSE;
}

/** set the preview drawable size to 50 percent frame size
*/
void PageExport::view50percent() {
	common->getPageEditor()->view50percent();
}

/** set the preview drawable size to 100 percent frame size
*/
void PageExport::view100percent() {
	common->getPageEditor()->view100percent();
}


/** update the progress bar for the export process
*/

void PageExport::updateProgress( gfloat val ) {
    struct timeval tv;
    long long now;
	static long long lastUpdateTime = 0;

    gettimeofday(&tv, 0);
    now = 1000000 * (long long) tv.tv_sec + (long long) tv.tv_usec;

    /* update every 0.3 second */

    if (val < 0.0) val = 0;
    if (val > 1.0) val = 1.0;

    if ( now > lastUpdateTime + 300000 ) {
        gtk_progress_bar_update( progressBar, val );
        lastUpdateTime = now;
    }

    while (gtk_events_pending()) {
        gtk_main_iteration();
	}
}

void PageExport::resetProgress() {
	gtk_progress_bar_update( progressBar, 0.0 );
}


extern "C" {
	void
	on_notebook_export_switch_page         (GtkNotebook     *notebook,
                                        GtkNotebookPage *page,
                                        gint             page_num,
                                        gpointer         user_data) {
		setExportMode(page_num);
    	
    }

	gboolean
	on_fileentry_export_avi_focus_in_event (GtkWidget       *widget,
	                                        GdkEventFocus   *event,
	                                        gpointer         user_data)
	{
	
		g_nav_ctl.escaped = TRUE;
		return FALSE;
	}
	
	
	gboolean
	on_fileentry_export_avi_focus_out_event
	                                        (GtkWidget       *widget,
	                                        GdkEventFocus   *event,
	                                        gpointer         user_data)
	{
		g_nav_ctl.escaped = FALSE;
		return FALSE;
	}


	gboolean
	on_fileentry_export_stills_focus_out_event
	                                        (GtkWidget       *widget,
	                                        GdkEventFocus   *event,
	                                        gpointer         user_data)
	{
		g_nav_ctl.escaped = FALSE;
		return FALSE;
	}
	
	
	gboolean
	on_fileentry_export_stills_focus_in_event
	                                        (GtkWidget       *widget,
	                                        GdkEventFocus   *event,
	                                        gpointer         user_data)
	{
		g_nav_ctl.escaped = TRUE;
		return FALSE;
	}

	gboolean
	on_fileentry_export_audio_focus_out_event
	                                        (GtkWidget       *widget,
	                                        GdkEventFocus   *event,
	                                        gpointer         user_data)
	{
		g_nav_ctl.escaped = TRUE;
		return FALSE;
	}
	
	
	gboolean
	on_fileentry_export_audio_focus_in_event
	                                        (GtkWidget       *widget,
	                                        GdkEventFocus   *event,
	                                        gpointer         user_data)
	{
		g_nav_ctl.escaped = TRUE;
		return FALSE;
	}

}
