/*
 * page_edit.cc Notebook Editor Page Object
 * Copyright (C) 2001 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <iostream>
using std::cout;
using std::endl;

#include "page_editor.h"
#include "frame.h"
#include "commands.h"
#include "sys/time.h"
#include "pthread.h"
#include "message.h"

// #define PLAY_WITH_STATS

extern KinoCommon *common;

extern "C" {
	#include "support.h"
	extern struct navigate_control g_nav_ctl;
	char cmd[16] = { 0 };
	char lastcmd[ 256 ] = { 0 };
	static int _getOneSecond(void);

	static gint videoIdler( void *info );

	static void *audioThread( void *info );
	static pthread_t thread;
	static pthread_mutex_t threadlock = PTHREAD_MUTEX_INITIALIZER;
}

/** Constructor for the page editor object.
	
  	\param common	KinCommon object to which this page belongs
*/

PageEditor::PageEditor( KinoCommon *common ) {
	cout << "> Creating page editor" << endl;
	this->common = common;
	this->bar = NULL;

	this->frameArea = GTK_DRAWING_AREA( lookup_widget( common->getWidget(), "drawingarea1" ) );
	this->barArea = GTK_DRAWING_AREA( lookup_widget( common->getWidget(), "drawingarea2" ) );
	this->frameScale = GTK_HSCALE( lookup_widget( common->getWidget(), "frame_pos" ) );
	this->positionLabel = GTK_LABEL( lookup_widget( common->getWidget(), "position_label" ) );
	this->lastFrameShown = -1;
	this->g_copiedPlayList = new PlayList();
}

/** Destructor for the page editor object.
*/

PageEditor::~PageEditor( ) {
	cout << "> Destroying page editor" << endl;
	delete this->g_copiedPlayList;
}

/** New File action.
*/

void PageEditor::newFile() {
	this->stopNavigator();
	this->ResetBar( );
}

/** Start action. Called when the page becomes current.
*/

void PageEditor::start() {
	static bool first = true;
	cout << ">> Starting Editor" << endl;
	this->displayer = new FrameDisplayer();
	this->lastFrameShown = common->g_currentFrame;
	RefreshBar( common->g_currentFrame );
	if ( !first )
		windowChanged();
	first = false;
}

/** Activate the returned widgets. 
*/

gulong PageEditor::activate() {
	return EDIT_MENU | 
		   COMMAND_MENU | 
		   SCENE_LIST |
		   VIDEO_START_OF_MOVIE |
		   VIDEO_START_OF_SCENE |
		   VIDEO_REWIND |
		   VIDEO_BACK |
		   VIDEO_PLAY |
		   VIDEO_PAUSE |
		   VIDEO_STOP |
		   VIDEO_FORWARD |
		   VIDEO_FAST_FORWARD |
		   VIDEO_NEXT_SCENE |
		   VIDEO_END_OF_MOVIE |
		   VIDEO_COMMAND |
		   VIDEO_SHUTTLE |
		   INFO_FRAME;
}

/** Clean action. Called when another page becomes current.
*/

void PageEditor::clean() {
	cout << ">> Leaving Editor" << endl;
	stopNavigator();
	//common->toggleComponents( VIDEO_STOP, true );
	delete this->displayer;
}

/** Starts the navigator thread. See comments on g_nav_ctl.
*/

void PageEditor::startNavigator() {
	if ( g_nav_ctl.active == FALSE ) {
		g_nav_ctl.active = TRUE;
		pthread_create( &thread, NULL, audioThread, &g_nav_ctl );
		this->idleCommand = gtk_idle_add_priority( GTK_PRIORITY_DEFAULT, (GtkFunction)videoIdler, (gpointer) &g_nav_ctl);
	}
}

/** Stops the navigator thread. See comments on g_nav_ctl.
*/

void PageEditor::stopNavigator() {
	if ( g_nav_ctl.active ) {
		g_nav_ctl.active = FALSE;
		gtk_idle_remove( this->idleCommand );
		gdk_threads_leave();
		pthread_join( thread, NULL );
		gdk_threads_enter();
		common->moveToFrame();
		g_nav_ctl.step = 1;
	}
}

/** Called when the current frame has changed through the common moveToFrame
	method and this page is the current page.

	\param	frame	frame moved to
*/

void PageEditor::movedToFrame( int frame ) {
	if ( g_nav_ctl.active == FALSE ) {
		showFrame( frame, frame == lastFrameShown );
		if ( common->hasListChanged ) {
			InitBar( common->g_currentFrame);
			common->hasListChanged = FALSE;
		}
	}
}

/** Show the frame requested

  	\param i		frame to be shown
	\param no_audio	indicate if audio is required or not
*/

void PageEditor::showFrame( int i, gboolean no_audio ) {
	if (common->getPlayList()->GetNumFrames() == 0) {
		common->loadSplash( frameArea );
		common->showFrameInfo( 0 );
		return;
	}
	static Frame *frame = new Frame;
	common->getPlayList()->GetFrame(i, *frame);
	common->showFrameInfo( i );
	getFrameDisplayer()->Put(*frame, GTK_WIDGET( frameArea ), TRUE);
	lastFrameShown = i;
	DrawBar( i );
}

/** Move to the start of the movie.
*/

void PageEditor::videoStartOfMovie() {
	common->moveToFrame( 0 );
	common->toggleComponents( VIDEO_START_OF_MOVIE, false);
}

/** Move to the start of the previous scene.
*/

void PageEditor::videoPreviousScene() {
	int frame = common->getPlayList()->FindStartOfScene(common->g_currentFrame);

	if ( g_nav_ctl.active == FALSE && frame == common->g_currentFrame )
		frame = common->getPlayList()->FindStartOfScene( frame - 1 );
	else if ( g_nav_ctl.active == TRUE && ( frame == common->g_currentFrame || ( common->g_currentFrame - frame ) <= 15 ) )
		frame = common->getPlayList()->FindStartOfScene( frame - 1 );

	common->moveToFrame( frame );
	common->toggleComponents( VIDEO_START_OF_SCENE, false);
}

/** Move to the start of the current scene.
*/

void PageEditor::videoStartOfScene() {
	int frame = common->getPlayList()->FindStartOfScene(common->g_currentFrame);
	common->moveToFrame( frame );
	common->toggleComponents( VIDEO_START_OF_SCENE, false);
}

/** Rewind.
*/

void PageEditor::videoRewind() {
	common->toggleComponents( common->getComponentState(), false );

	// Toggle Rewind state
	if ( g_nav_ctl.step != -10 ) {
		common->toggleComponents( VIDEO_REWIND, true );
		g_nav_ctl.step = -10;
		startNavigator();
	}
	else {
		stopNavigator();
		g_nav_ctl.step = 1;
		common->toggleComponents( VIDEO_REWIND, false );
		common->toggleComponents( VIDEO_STOP, true );
	}
}

/** Move one frame back. If the navigator is active, then this action toggles 
	between reverse and stop.
*/

void PageEditor::videoBack() {
	common->toggleComponents( common->getComponentState(), false );
	if ( g_nav_ctl.active ) {
		if ( g_nav_ctl.step != -1 ) {
			g_nav_ctl.step = -1;
			common->toggleComponents( VIDEO_BACK, true );
		}
		else {
			stopNavigator();
			g_nav_ctl.step = 1;
			common->toggleComponents( VIDEO_BACK, false );
			common->toggleComponents( VIDEO_STOP, true );
		}
	} 
	else {
		common->moveByFrames( -1 );
		common->toggleComponents( VIDEO_BACK, false );
		common->toggleComponents( VIDEO_STOP, true );
	}
}

/** Play.
*/

void PageEditor::videoPlay() {
	common->toggleComponents( common->getComponentState(), false );
	if ( g_nav_ctl.active == FALSE || g_nav_ctl.step != 1 ) {
		common->toggleComponents( VIDEO_PLAY, true );
		g_nav_ctl.step = 1;
		startNavigator();
	}
	else {
		stopNavigator();
		common->toggleComponents( VIDEO_PLAY, false );
		common->toggleComponents( VIDEO_STOP, true );
	}
}

/** Pause.
*/

void PageEditor::videoPause() {
	common->toggleComponents( common->getComponentState(), false );
	common->toggleComponents( VIDEO_PAUSE, true );
	stopNavigator();
}

/** Stop.
*/

void PageEditor::videoStop() {
	common->toggleComponents( common->getComponentState(), false );
	common->toggleComponents( VIDEO_STOP, true );
	stopNavigator();
}

/** Move one frame forward. If the navigator is active, then this action puts
	the video into normal play.
*/

void PageEditor::videoForward() {
	common->toggleComponents( common->getComponentState(), false );
	if ( g_nav_ctl.active && g_nav_ctl.step != 1 ) {
		g_nav_ctl.step = 1;
		common->toggleComponents( VIDEO_FORWARD, true );
	} else {
		stopNavigator();
		common->moveByFrames( 1 );
		common->toggleComponents( VIDEO_FORWARD, false);
		common->toggleComponents( VIDEO_STOP, true);
	}
}

/** Fast forward.
*/

void PageEditor::videoFastForward() {
	common->toggleComponents( common->getComponentState(), false );

	if ( g_nav_ctl.step != 10 ) {
		common->toggleComponents( VIDEO_FAST_FORWARD, true );
		g_nav_ctl.step = 10;
		startNavigator();
	}
	else {
		stopNavigator();
		g_nav_ctl.step = 1;
		common->toggleComponents( VIDEO_FAST_FORWARD, false );
		common->toggleComponents( VIDEO_STOP, true );
	}
}

/** Shuttle

	Bi-directionaly variable-speed playback.
	
	\param angle A number from -7 (fastest reverse) to 7 (fastest forward), 0=stop
*/

void PageEditor::videoShuttle( int angle ) {
	int speedTable[] = { 0, 20, 33, 100, 200, 500, 1000, (_getOneSecond() * 100) };

	stopNavigator();

	if (angle < -7) angle = -7;
	if (angle > 7) angle = 7;

	int speed = speedTable[(angle < 0) ? -angle : angle] * ((angle < 0) ? -1 : 1);
	if (speed == 0) {
		common->toggleComponents( VIDEO_PAUSE, true );
		return;
	}

	g_nav_ctl.step = speed / 100;
	g_nav_ctl.rate = 100 / speed;
	g_nav_ctl.subframe = 0;
	
	startNavigator();
}

/** Move to the start of the next scene.
*/

void PageEditor::videoNextScene() {
	int frame = common->getPlayList()->FindEndOfScene(common->g_currentFrame);
	common->moveToFrame( frame + 1 );
	common->toggleComponents( VIDEO_NEXT_SCENE, false);
}

/** Move to the end of the current scene.
*/

void PageEditor::videoEndOfScene() {
	int frame = common->getPlayList()->FindEndOfScene(common->g_currentFrame);
	common->moveToFrame( frame );
	common->toggleComponents( VIDEO_NEXT_SCENE, false);
}

/** Move to the end of the play list.
*/

void PageEditor::videoEndOfMovie() {
	common->toggleComponents( common->getComponentState(), false );
	common->toggleComponents( VIDEO_END_OF_MOVIE, true);
	stopNavigator();
	common->moveToFrame( common->getPlayList()->GetNumFrames() - 1 );
	common->toggleComponents( VIDEO_END_OF_MOVIE, false);
	common->toggleComponents( VIDEO_STOP, true);
}

/** Move to start of selected scene.

	\param i	scene to move to
*/

void PageEditor::selectScene( int i ) {
	vector <int> scene = GetScene();
	int value = i == 0 ? 0 : scene[ i - 1 ];
	common->moveToFrame( value );
}

/** Process a keyboard event. 

  	\param event	keyboard event
*/

gboolean PageEditor::processKeyboard( GdkEventKey *event ) {
	gboolean ret = FALSE;

	// Only process while not escape mode
	if ( g_nav_ctl.escaped == FALSE ) {

		// Translate special keys to equivalent command
		switch (event->keyval) {
			case GDK_BackSpace:
			case GDK_Left:
				strcat(cmd, "h");
				break;
			case GDK_Up:
				strcat(cmd, "k");
				break;
			case GDK_Right:
				strcat(cmd, "l");
				break;
			case GDK_Return:
			case GDK_Down:
				strcat(cmd, "j");
				break;
			case GDK_Delete:
				strcat( cmd, "x" );
				break;
			case GDK_Escape:
				common->keyboardFeedback(cmd, "Stop");
				common->videoStop( );
				cmd[0] = 0;
				return FALSE;
			default:
				if ( strcmp( event->string, "." ) )
					strcat(cmd, event->string);
				break;
		}
	
		bool commandLine = false;
		if ( !strcmp( event->string, "." ) ) {
			strcpy( cmd, lastcmd );
			commandLine = lastcmd[ 0 ] == ':' && strlen( lastcmd ) > 1;
		}
		else if (strcmp(cmd, "\006") == 0) // Ctrl+f
			strcpy( cmd, "5$" );
		else if (strcmp(cmd, "\002") == 0) // Ctrl+b
			strcpy( cmd, "5^" );
		else if ( cmd[0] == 0x0a ) { // Ctrl+j
			strcpy( cmd, ":split" );
			commandLine = true;
		}

#if 0
		printf("send_event: %2.2x\n", event->send_event);
		printf("time  : %8.8x\n", event->time);
		printf("state : %8.8x\n", event->state);
		printf("keyval: %8.8x\n", event->keyval);
		printf("length: %8.8x\n", event->length);
		printf("string: %s\n", event->string);
		printf("(hex) : %2.2x\n", event->string[0]);
		printf("cmd   : %s\n", cmd);
		printf("(hex) : %8.8x\n", cmd[0]);
#endif

		if ( !commandLine ) {
			ret = processKeyboardCommand( cmd );
		}
		else {
			ret = processCommandLine( cmd );
		}
	}
	return ret;
}

/** Process a menu command.

  	\param command	command to be processed
*/

gboolean PageEditor::processMenuCommand( char *command ) {
	strcpy( cmd, command );
	strcpy( lastcmd, command );
	gboolean ret = processKeyboardCommand( cmd );
	return ret;
}

/** Internal method for handling a complete keyboard scene.

  	\param cmd		command to be processed;
*/

gboolean PageEditor::processKeyboardCommand( char *cmd ) {
	int	start, end;
	GtkWidget *video_command = lookup_widget(common->getWidget(), "command");
	int count = 1;
	char real[ 256 ] = "";

	switch( sscanf( cmd, "%d%s", &count, real ) )
	{
		case 1:
			// Numeric value only - return immediately if the cmd is not "0"
			if ( strcmp( cmd, "0" ) ) {
				common->keyboardFeedback(cmd, "");
				return FALSE;
			}
			break;
		case 0:
			sscanf( cmd, "%s", real );
			count = 1;
			break;
	}

	if ( strcmp( cmd, "." ) && strcmp( cmd, ":" ) )
		strcpy( lastcmd, cmd );

	if ( !strcmp( cmd, ":" ) ) 
	{
		gtk_entry_set_editable( GTK_ENTRY( video_command ), TRUE );
		gtk_widget_grab_focus( video_command );
		gtk_entry_set_text( GTK_ENTRY( video_command ), "" );
		g_nav_ctl.escaped = TRUE;
		cmd[0] = 0;
	}
	/* Navigation */

	/* play, pause */
	
	else if (strcmp(cmd, " ") == 0) 
	{
		if ( g_nav_ctl.active == FALSE ) {
			common->keyboardFeedback(cmd, "Play");
			common->videoPlay( );
		} else {
			common->keyboardFeedback(cmd, "Pause");
			common->videoPause( );
		}
		cmd[0] = 0;
	}
	
	/* advance one frame */

	else if ( strcmp(real, "l") == 0 ) 
	{
		common->keyboardFeedback(cmd, "Move forward" );
		common->moveByFrames( count );
		cmd[0] = 0;
	}

	/* backspace one frame */

	else if ( strcmp(real, "h") == 0 ) 
	{
   		common->keyboardFeedback(cmd, "Move backward");
		common->moveByFrames( 0 - count );
		cmd[0] = 0;
	}

	/* advance one second */

	else if ( strcmp(real, "w") == 0 || strcmp(real, "W") == 0 ||
		  	strcmp(real, "e") == 0 || strcmp(real, "E") == 0 ) 
	{
		common->keyboardFeedback(cmd, "Move forward second");
		common->moveByFrames( count * _getOneSecond() );
		cmd[0] = 0;
	}

	/* backspace one second */

	else if ((strcmp(real, "b") == 0) || (strcmp(real, "B") == 0)) 
	{
		common->keyboardFeedback(cmd, "Move backwards one second");
		common->moveByFrames( 0 - count * _getOneSecond() );
		cmd[0] = 0;
	}

	/* start of scene */

	else if ((strcmp(cmd, "0") == 0) || (strcmp(real, "^") == 0)) 
	{
		common->videoStartOfScene( );
		for ( ; count > 1; count -- ) 
		{
			common->g_currentFrame --;
			videoStartOfScene( );
		}
		common->keyboardFeedback(cmd, "Move to start of scene");
		cmd[0] = 0;
	}

	/* end of scene */

	else if (strcmp(real, "$") == 0) 
	{
		common->videoEndOfScene( );
		for ( ; count > 1; count -- ) 
		{
			common->g_currentFrame ++;
			videoEndOfScene( );
		}
		common->keyboardFeedback(cmd, "Move to end of scene");
		cmd[0] = 0;
	}

	/* start of next scene */

	else if ((strcmp(real, "j") == 0) || strcmp(real, "+") == 0 ) 
	{
		for ( ; count >= 1; count -- )
			common->videoNextScene( );
		common->keyboardFeedback(cmd, "Move to start of next scene");
		cmd[0] = 0;
	}

	/* start of previous scene */

	else if ((strcmp(real, "k") == 0) || (strcmp(real, "-") == 0)) 
	{
		for ( ; count >= 1; count -- )
			common->videoPreviousScene( );
		common->keyboardFeedback(cmd, "Move to start of previous scene");
		cmd[0] = 0;
	}

	/* first frame */

	else if (strcmp(cmd, "gg") == 0) 
	{
		common->videoStartOfMovie( );
		common->keyboardFeedback(cmd, "Move to first frame");
		cmd[0] = 0;
	}

	/* last frame */

	else if (strcmp(cmd, "G") == 0) 
	{
		common->videoEndOfMovie( );
		common->keyboardFeedback(cmd, "Move to last frame");
		cmd[0] = 0;
	}

	/* delete current frame */

	else if ((strcmp(real, "x") == 0) || (strcmp(cmd, "d ") == 0) || (strcmp(real, "dl") == 0) ) 
	{
		CopyFrames(common->g_currentFrame, common->g_currentFrame + count - 1);
		DeleteFrames(common->g_currentFrame, common->g_currentFrame + count - 1);
		common->moveToFrame( );
		common->keyboardFeedback(cmd, "Cut current frame");
		RefreshBar( common->g_currentFrame );
		cmd[0] = 0;
	}

	/* delete one second */

	else if (strcmp(cmd, "dw") == 0) 
	{
		end = common->g_currentFrame + _getOneSecond() - 1;
		CopyFrames(common->g_currentFrame, end);
		DeleteFrames(common->g_currentFrame, end);
		common->moveToFrame( );
		common->keyboardFeedback(cmd, "Cut one second");
		RefreshBar( common->g_currentFrame );
		cmd[0] = 0;
	}

	/* delete current scene */

	else if (strcmp(real, "dd") == 0) 
	{
		start = common->getPlayList()->FindStartOfScene(common->g_currentFrame);
		end = start;
		for ( ; count >= 1; count -- ) {
			end = common->getPlayList()->FindEndOfScene( end );
			end ++;
		}
		CopyFrames(start, end - 1);
		DeleteFrames(start, end - 1);
		common->moveToFrame( start );
		common->keyboardFeedback(cmd, "Cut current scene");
		RefreshBar( common->g_currentFrame );
		cmd[0] = 0;
	}

	/* delete from current frame up to end of scene */

	else if (strcmp(cmd, "d$") == 0) 
	{
		end = common->getPlayList()->FindEndOfScene(common->g_currentFrame);
		CopyFrames(common->g_currentFrame, end);
		DeleteFrames(common->g_currentFrame, end);
		common->moveToFrame( );
		common->keyboardFeedback(cmd, "Cut to end of scene");
		RefreshBar( common->g_currentFrame );
		cmd[0] = 0;
	}

	/* delete from current frame up to end of file */

	else if (strcmp(cmd, "dG") == 0) 
	{
		end = common->getPlayList()->GetNumFrames();
		CopyFrames(common->g_currentFrame, end);
		DeleteFrames(common->g_currentFrame, end);
		common->moveByFrames( -1 );
		common->keyboardFeedback(cmd, "Cut to end of file");
		RefreshBar( common->g_currentFrame );
		cmd[0] = 0;
	}

	/* delete from start of scene just before current frame */

	else if ((strcmp(cmd, "d0") == 0) || strcmp(cmd, "d^") == 0) 
	{
		start = common->getPlayList()->FindStartOfScene(common->g_currentFrame);
		if (start < common->g_currentFrame) 
		{
			CopyFrames(start, common->g_currentFrame - 1);
			DeleteFrames(start, common->g_currentFrame - 1);
			common->moveToFrame( start );
		}
		common->keyboardFeedback(cmd, "Cut from start of scene");
		RefreshBar( common->g_currentFrame );
		cmd[0] = 0;
	}

	/* delete from start of file just before current frame */

	else if (strcmp(cmd, "dgg") == 0) 
	{
		CopyFrames(0, common->g_currentFrame - 1);
		DeleteFrames(0, common->g_currentFrame - 1);
		common->moveToFrame( 0 );
		common->keyboardFeedback(cmd, "Cut from start of file");
		RefreshBar( common->g_currentFrame );
		cmd[0] = 0;
	}

	/* copy current frame */

	else if ((strcmp(cmd, "y ") == 0) || (strcmp(real, "yl") == 0)) 
	{
		CopyFrames(common->g_currentFrame, common->g_currentFrame + count - 1);
		common->keyboardFeedback(cmd, "Copy current frame");
		cmd[0] = 0;
	}

	/* copy current scene */

	else if ((strcmp(real, "yy") == 0) || (strcmp(real, "Y") == 0)) 
	{
		start = common->getPlayList()->FindStartOfScene(common->g_currentFrame);
		end = start;
		for ( ; count >= 1; count -- ) {
			end = common->getPlayList()->FindEndOfScene( end );
			end ++;
		}
		CopyFrames(start, end - 1);
		common->keyboardFeedback(cmd, "Copy current scene");
		cmd[0] = 0;
	}

	/* copy from current frame up to end of scene */

	else if (strcmp(cmd, "y$") == 0) 
	{
		end = common->getPlayList()->FindEndOfScene(common->g_currentFrame);
		CopyFrames(common->g_currentFrame, end);
		common->keyboardFeedback(cmd, "Copy to end of scene");
		cmd[0] = 0;
	}

	/* copy from start of scene just before current frame */

	else if ((strcmp(cmd, "y0") == 0) || strcmp(cmd, "y^") == 0) 
	{
		start = common->getPlayList()->FindStartOfScene(common->g_currentFrame);
		if (start < common->g_currentFrame) 
		{
			CopyFrames(start, common->g_currentFrame - 1);
			common->moveToFrame( start );
		}
		common->keyboardFeedback(cmd, "Copy from start of scene");
		cmd[0] = 0;
	}

	/* paste after current frame */

	else if (strcmp(real, "p") == 0) 
	{
		start = common->g_currentFrame;
		for ( ; count >= 1; count -- ) 
			PasteFrames(common->g_currentFrame + 1);
		common->moveToFrame( start + 1 );
		common->keyboardFeedback(cmd, "Paste after current frame");
		RefreshBar( common->g_currentFrame );
		cmd[0] = 0;
	}

	/* paste before current frame */

	else if (strcmp(real, "P") == 0) 
	{
		for ( ; count >= 1; count -- ) 
			PasteFrames(common->g_currentFrame);
		end = common->getPlayList()->FindEndOfScene(common->g_currentFrame + 1);
		common->moveToFrame( );
		common->keyboardFeedback(cmd, "Paste before current frame");
		RefreshBar( common->g_currentFrame );
		cmd[0] = 0;
	}

	/* Switch to capture mode */

	else if ( strcmp( cmd, "i" ) == 0 ) 
	{
		common->keyboardFeedback(cmd, "Capture");
		common->changePageRequest( PAGE_CAPTURE );
		cmd[0] = 0;
	}

	else if ( strcmp( cmd, "a" ) == 0 ) 
	{
		common->keyboardFeedback(cmd, "Capture");
		end = common->getPlayList()->FindEndOfScene(common->g_currentFrame );
		common->moveToFrame( end );
		common->changePageRequest( PAGE_CAPTURE );
		cmd[0] = 0;
	}

	else if ( strcmp( cmd, "A" ) == 0 ) 
	{
		common->keyboardFeedback(cmd, "Capture");
		end = common->getPlayList()->GetNumFrames();
		common->moveToFrame( end );
		common->changePageRequest( PAGE_CAPTURE );
		cmd[0] = 0;
	}

	else 
	{
		// Check for invalid commands
		if ( strlen( real ) > 3 )
			cmd[ 0 ] = 0;
		else if ( strchr( "dgy ", real[ strlen( real ) - 1 ] ) == NULL ) 
			cmd[ 0 ] = 0;

		common->keyboardFeedback(cmd, "");

	}

	return FALSE;
}

/** Process command line.

  	\param command	command to be processed.
*/

gboolean PageEditor::processCommandLine( char *command ) {

	if ( !strcmp( command, "." ) )
		strcpy( command, lastcmd );

	/* read AVI or PlayList */

	if (strcmp(command, ":r") == 0) {
		common->keyboardFeedback(command, "Insert file");
		common->insertFile( );
	}

	else if ( strcmp( command, ":a" ) == 0 ) 
	{
		common->keyboardFeedback(command, "Append file to scene");
		int end = common->getPlayList()->FindEndOfScene(common->g_currentFrame );
		common->moveToFrame( end );
		common->appendFile( );
	}

	else if ( strcmp( command, ":A" ) == 0 ) 
	{
		common->keyboardFeedback(command, "Append file to movie");
		int end = common->getPlayList()->GetNumFrames();
		common->moveToFrame( end );
		common->appendFile( );
	}

	/* switch to export mode */

	else if (strcmp(command, ":W") == 0) {
		common->keyboardFeedback(command, "Export");
		common->changePageRequest( PAGE_EXPORT );
	}

	/* write PlayList */

	else if (strcmp(command, ":w") == 0) {
		common->keyboardFeedback(command, "Write playlist");
		common->savePlayList( );
	}

	/* quit */

	else if (strcmp(command, ":q") == 0) {
		common->keyboardFeedback(command, "quit");
		kinoDeactivate();
	} 
	
	/* split scene */

	else if (strcmp(command, ":split") == 0) {
		common->keyboardFeedback(command, "Split scene before frame");
		common->getPlayList()->SplitSceneBefore( common->g_currentFrame );
		common->hasListChanged = TRUE;
		common->moveToFrame();
		RefreshBar( common->g_currentFrame );
	} 
	
	/* goto a frame */
	else if ( strncmp( command, ":", 1 ) == 0 ) {
		int val = 0;
		char t[ 132 ] = "";
		if ( sscanf( command + 1, "%d", &val ) == 1 ) {
			common->moveToFrame( val - 1 );
			sprintf( t, "Move to frame %d", val );
			common->keyboardFeedback(command, t );
		}
	}

	/* cleanup the command entry */

	GtkWidget *video_command = lookup_widget(common->getWidget(), "command");
	g_nav_ctl.escaped = FALSE;
	gtk_entry_set_text( GTK_ENTRY(video_command), "");
	gtk_entry_set_editable( GTK_ENTRY(video_command), FALSE );
	gtk_widget_grab_focus( common->getWidget() );

	// Last command is now
	strcpy( lastcmd, command );

	return FALSE;
}

/** Copy the requested frames to the playlist buffer.

  	\param first	first frame
	\param last		last frame
*/

void PageEditor::CopyFrames(int first, int last) {

	char	msg[256];
	PlayList *playList = new PlayList;

	// Delete the old list
	delete g_copiedPlayList;
	// Create the one requested
	common->getPlayList()->GetPlayList(first, last, *playList);
	g_copiedPlayList = playList;
	sprintf(msg, "%d frames copied", g_copiedPlayList->GetNumFrames());
	//gtk_label_set_text (GTK_LABEL (result_label), msg);
}


/** Paste the playlist buffer before the specified frame.

	\param before		frame
*/

void PageEditor::PasteFrames(int before) {

	char	msg[256];
	PlayList temp( *g_copiedPlayList );

	sprintf(msg, "%d frames pasted", temp.GetNumFrames());
	common->getPlayList()->InsertPlayList(temp, before);
	//gtk_label_set_text (GTK_LABEL (result_label), msg);
	common->hasListChanged = TRUE;
}

/** Delete and copy the requested frames to the playlist buffer.

  	\param first	first frame
	\param last		last frame
*/

void PageEditor::DeleteFrames(int first, int last) {
	char	msg[256];
	int before, after;

	before = common->getPlayList()->GetNumFrames();
	common->getPlayList()->Delete(first, last);
	after = common->getPlayList()->GetNumFrames();
	sprintf(msg, "%d frames deleted", before - after);
	//gtk_label_set_text (GTK_LABEL (result_label), msg);
	common->hasListChanged = TRUE;
}

void PageEditor::showFrame( int position, Frame frame ) {
	if ( common->getPlayList()->GetNumFrames() == 0 ) {
		common->loadSplash( frameArea );
	}
	else {
		DrawBar( common->g_currentFrame );
		getFrameDisplayer()->Put( frame, GTK_WIDGET( frameArea ), TRUE );
		common->showFrameInfo( position );
	}
}


/** set the preview drawable size to 50 percent frame size
*/
void PageEditor::view50percent() {
	if ( common->g_currentFrame != -1 ) {
		static Frame frame;
		common->getPlayList()->GetFrame( common->g_currentFrame, frame );
		GtkWidget *container = lookup_widget( common->getWidget(), "drawingarea1_frame" );
		AspectRatioCalculator calc( frame.decoder->width/2 +4, frame.decoder->height/2 +4, 
			frame.decoder->width/2 +4, frame.decoder->height/2 +4, frame.IsPALfromHeader(),
			frame.IsWide() );
		gtk_widget_set_usize( container, calc.width, calc.height);
	} else {
		modal_message( "A video file must be loaded to set the preview size." );
	}
}


/** set the preview drawable size to 100 percent frame size
*/
void PageEditor::view100percent() {
	if ( common->g_currentFrame != -1 ) {
		static Frame frame;
		common->getPlayList()->GetFrame( common->g_currentFrame, frame );
		GtkWidget *container = lookup_widget( common->getWidget(), "drawingarea1_frame" );
		AspectRatioCalculator calc( frame.decoder->width +4, frame.decoder->height +4, 
			frame.decoder->width +4, frame.decoder->height +4, frame.IsPALfromHeader(),
			frame.IsWide() );
		gtk_widget_set_usize( container, calc.width, calc.height );
	} else {
		modal_message( "A video file must be loaded to set the preview size." );
	}
}

void PageEditor::windowChanged() {
	common->loadSplash( frameArea );
}

void PageEditor::windowMoved() {
	if ( common->getPlayList()->GetNumFrames() )
		showFrame( common->g_currentFrame, FALSE );
	else
		common->loadSplash( frameArea );
}


void PageEditor::showFrameInfo( int i ) {
	char msg[256];
#if 0
/* save this for when we do scene detail page */
	TimeCode tc;
	static Frame frame;
	common->getPlayList()->GetFrame( i, frame );
	frame.GetTimeCode(tc);
	sprintf(msg, "%d / %d\n%s\n%d:%d:%d:%d", 
			i + 1, common->getPlayList()->GetNumFrames(),
			frame.GetRecordingDate().c_str(),
			tc.hour, tc.min, tc.sec, tc.frame );
#endif
	static Frame frame;
	int hours, mins, secs;
	common->getPlayList()->GetFrame( i, frame );
	int cur = common->g_currentFrame;
	int fps = _getOneSecond();

	if (cur == -1) {
		i = -1;
		hours = 0;
		mins = 0;
		secs = 0;
		cur = 0;
	} else {
		hours = cur / (fps * 3600);
		cur -= hours * (fps * 3600);
		mins = cur / (fps * 60);
		cur -= mins * (fps * 60);
		secs = cur / fps;
		cur -= secs * fps;
	}
	
	
	sprintf(msg, "%d / %d\n%s\n%2.2d:%2.2d:%2.2d:%2.2d", 
			i + 1, common->getPlayList()->GetNumFrames(),
			frame.GetRecordingDate().c_str(),
			hours, mins, secs, cur );
	gtk_label_set_text( positionLabel, msg);
}

/**	Save the current frame as a jpg.
*/

void PageEditor::saveFrame( )
{
   	char *filename = common->getFileToSave( "Save Still Frame" );
	if ( strcmp( filename, "" ) ) 
	{
		common->save_cd( filename );
		common->saveFrame( common->g_currentFrame, filename );
	}
}

extern "C" {

	// Share the frames extracted from the audio and video thread - this
	// provides a smoother playback and is less intensive on the CPU. The
	// code assumes the video will never fall more than PLAYBACK_FRAMES behind 
	// the audio.
	//
	// 'position' stores last obtained index in the 'frameContent' array and the 
	// corresponding frame number is stored in the corresponding entry in the
	// frameNumber array.
	//
	// The audio thread is responsible for write access to these.

	#define PLAYBACK_FRAMES 25

	static int position = -1;
	static int frameNumber[ PLAYBACK_FRAMES ];
	static Frame frameContent[ PLAYBACK_FRAMES ];

#ifdef PLAY_WITH_STATS
	// Statistical analysis variables
	static int dropped = 0;
	static int count = 0;
	static int lastFrame = 0;
#endif

	/** This function carries out the audio playback and is responsible for obtaining
		the each frame and releasing it to the videoIdler function below. The videoIdler
		is responsible for rendering the image associated to each frame (or as many as
		it is possible to do). audioThread has no involvement with the GUI directly.
	*/

	void *audioThread( void *info ) {
		struct navigate_control *ctl = (struct navigate_control *)info;
		gint newFrame = 0;
		gint lastFrame = common->g_currentFrame;
		gint totalFrames = common->getPlayList()->GetNumFrames();
		gint countFrames = 0;
		int lastPos = -1;
		static Preferences &prefs = Preferences::getInstance();

#ifdef PLAY_WITH_STATS
		dropped = 0;
		count = 0;
		lastFrame = common->g_currentFrame;
#endif

		// We're starting again, so initialise the value
		position = -1;

		// Lock the current frame position
		pthread_mutex_lock( &threadlock );

		// Loop while active
		while( ctl->active ) {

			// Determine the frame to render
			newFrame = common->g_currentFrame + ctl->step;

			// determine new frame based upon jogshuttle rate
			if (ctl->step == 0) {
		  		ctl->subframe++;
		  		if (ctl->rate < 0) {
					if (ctl->subframe >= -ctl->rate) {
			  			newFrame --;
			  			ctl->subframe = 0;
					}
		  		} else {
					if (ctl->subframe >= ctl->rate) {
			  			newFrame ++;
			  			ctl->subframe = 0;
					}
		  		}
			}
	
			// Check the bounds and adjust as necessary
			if ( newFrame < 0 )
				newFrame = 0;
			else if ( newFrame >= totalFrames )
				newFrame = totalFrames - 1;

			// If we aren't on the same frame as the previous iteration, then continue
			if ( newFrame  != lastFrame ) {

				// Move the main frame position to our newly derived location
				common->g_currentFrame = newFrame;

				// Determine which locations in frameNumber and frameContent we need to use
				int current = ( position + 1 ) % PLAYBACK_FRAMES;
				int next = ( current + 1 ) % PLAYBACK_FRAMES;

				// If we're on the first iteration, get the frame but don't play it and don't 
				// release it. Subsequent iterations will play the audio from the previous 
				// iteration, release it and get the next frame.
				if ( countFrames == 0 ) {
					common->getPlayList()->GetFrame( newFrame, frameContent[ current ]);
					lastFrame = newFrame;
					lastPos = current;
					frameNumber[ current ] = newFrame;
				}
				else {
					// Release the previously obtained frame
					position = lastPos;

					// Toggle the mutex to allow any thread to sync if necessary
					pthread_mutex_unlock( &threadlock );
					pthread_mutex_lock( &threadlock );

					// Put the sound out and allow the video idler a chance to grab it
					common->getPageEditor()->getFrameDisplayer()->PutSound( frameContent[ lastPos ] );

					// Render all frames if requested
					if ( ( ! prefs.dropFrame || ! prefs.enableAudio ) && ctl->active ) {
						gdk_threads_enter();
						common->getPageEditor()->showFrame( frameNumber[ lastPos ], frameContent[ lastPos ] );
						gdk_flush();
						gdk_threads_leave();
					}

					// Grab the next frame while its playing (we don't care about stepping over bounds checks
					// at this point).
					frameNumber[ next ] = newFrame + ctl->step;
					common->getPlayList()->GetFrame( frameNumber[ next ], frameContent[ next ]);

					// update the lastFrame and lastPos variables
					lastFrame = newFrame;
					lastPos = next;
				}

				// Incrment the frame count
				countFrames ++;
			}
			else {
				// Release the lock momentarily (to allow the idler to pass through).
				pthread_mutex_unlock( &threadlock );
				pthread_mutex_lock( &threadlock );

				// Sleep for a little bit to avoid clocking up excessive CPU cycles
				struct timespec t;
				t.tv_sec = 0;
				t.tv_nsec = 10000;
				nanosleep( &t, NULL );
			}
		}

		// Just in case the video idler is waiting on the lock
		position = -1;

		// Unlock the current frame
		pthread_mutex_unlock( &threadlock );

#ifdef PLAY_WITH_STATS
		cout << "Video stopped: Dropped " << dropped << " in " << count << " frames - " << (double)( ( (double)dropped / (double)count ) * 100.0 ) << "%" << endl;
#endif

		return NULL;
	}

	gint videoIdler( void *info ) {

		static Preferences &prefs = Preferences::getInstance();
		struct navigate_control *ctl = (struct navigate_control *)info;

		// Only do this when drop frames is active (otherwise the audio thread takes care of it)

		if ( prefs.dropFrame && prefs.enableAudio ) {
			static int last = -1;

			// If we're here and the frame position hasn't changed, then we need to sync (unless
			// we're here before any frames have been delivered
			if ( position != -1 && last == position ) {
				pthread_mutex_lock( &threadlock );
				pthread_mutex_unlock( &threadlock );
			}

			// Obtain the position of the last released frame
			int current = position;

			// As long as its different to the previous one, then display it
			if ( current != -1 && last != current ) {

#ifdef PLAY_WITH_STATS
				// Statistical analysis of frame playback (useful for limited kinds of tests)
				if ( ctl->step == 1 ) {
					int skipped = frameNumber[ current ] - lastFrame - 1;
					count += skipped + 1;
					if ( skipped != 0 && lastFrame != 0 && count != 0 ) {
						dropped += skipped;
						cout << "Dropped " << dropped << " in " << count << " frames - " << (double)( ( (double)dropped / (double)count ) * 100.0 ) << "%" << endl;
					}
					lastFrame = frameNumber[ current ];
				}
#endif

				// Show this frame
				if ( ctl->active ) 
					common->getPageEditor()->showFrame( frameNumber[ current ], frameContent[ current ] );

				// Remember last position rendered
				last = current;
			}
		}
		else {
			// It is safe to cancel the idler here since a change to prefs necessitates a restart.
			return 0;
		}

		return 1;
	}

	static int _getOneSecond(void)
	{
		static Frame frame;
		common->getPlayList()->GetFrame(common->g_currentFrame, frame);
		return (frame.IsPALfromHeader() ? 25 : 30);
	}
	
}

