/*
 * ieee1394io.cc -- asynchronously grabbing DV data
 * Copyright (C) 2000 Arne Schirmacher <arne@schirmacher.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef _IEEE1394IO_H
#define _IEEE1394IO_H 1

#include <libraw1394/raw1394.h>
#include <libraw1394/csr.h>
#include <deque>
using std::deque;

class Frame;

class IEEE1394Reader
{
private:

    /** the interface card to use (typically == 0) */
    int			    port;

    /// the iso channel we listen to (typically == 63)
    int			    channel;

    /// contains information about our thread after calling StartThread
    pthread_t               thread;

    /// this mutex protects variables that could possibly accessed
    /// from two threads at the same time
    static pthread_mutex_t  mutex;

    /// the handle to the ieee1394 subsystem
    static raw1394handle_t  handle;
    static raw1394handle_t  avc_handle;

    /// the number of frames that had to be thrown away because
    /// our inFrames queue did not contain available frames
    static int              droppedFrames;

    /// a pointer to the frame which is currently been transmitted
    static Frame	    *currentFrame;

    /// a list of empty frames
    static deque < Frame* > inFrames;

    /// a list of already received frames
    static deque < Frame* > outFrames;

	int lastPhyId;
public:

    IEEE1394Reader(int crd = 0, int chn = 63, int frames = 50);
    ~IEEE1394Reader();

    static int AviHandler(raw1394handle_t handle, int channel, size_t length, quadlet_t *data);
    static int ResetHandler(raw1394handle_t handle);
    static int TagHandler(raw1394handle_t handle, unsigned long tag, int errcode);

    static void* Thread(void *arg);
    bool StartThread(void);
    void StopThread(void);

    bool Open(void);
    void Close(void);
    bool StartReceive(void);
    void StopReceive(void);
    void Flush(void);
    Frame* GetFrame(void);
    void DoneWithFrame(Frame*);

    int isPhyIDValid(int id);
    int AVCPlay(int id);
    int AVCPause(int id);
    int AVCStop(int id);
    int AVCFastForward(int id);
    int AVCRewind(int id);
    int AVCForward(int id);
    int AVCBack(int id);
    int AVCNextScene(int id);
    int AVCPreviousScene(int id);
    int AVCRecord(int id);
    int AVCShuttle(int id, int speed);
    quadlet_t AVCStatus(int id);
    char *AVCTimecode(int id);

};

/* ***************************************************************************
   dvsend part
   ***************************************************************************/

#define VIDEO1394_MAX_SIZE 0x4000000

enum {
    VIDEO1394_BUFFER_FREE = 0,
    VIDEO1394_BUFFER_QUEUED,
    VIDEO1394_BUFFER_READY
};

enum {
    VIDEO1394_LISTEN_CHANNEL = 0,
    VIDEO1394_UNLISTEN_CHANNEL,
    VIDEO1394_LISTEN_QUEUE_BUFFER,
    VIDEO1394_LISTEN_WAIT_BUFFER,
    VIDEO1394_TALK_CHANNEL,
    VIDEO1394_UNTALK_CHANNEL,
    VIDEO1394_TALK_QUEUE_BUFFER,
    VIDEO1394_TALK_WAIT_BUFFER
};

#define VIDEO1394_SYNC_FRAMES          0x00000001
#define VIDEO1394_INCLUDE_ISO_HEADERS  0x00000002
#define VIDEO1394_VARIABLE_PACKET_SIZE 0x00000004

struct video1394_mmap
{
    unsigned int channel;
    unsigned int sync_tag;
    unsigned int nb_buffers;
    unsigned int buf_size;
    unsigned int packet_size; /* For VARIABLE_PACKET_SIZE: Maximum packet size */
    unsigned int fps;
    unsigned int syt_offset;
    unsigned int flags;
};

/* For TALK_QUEUE_BUFFER with VIDEO1394_VARIABLE_PACKET_SIZE use */
struct video1394_queue_variable
{
    unsigned int channel;
    unsigned int buffer;
    unsigned int* packet_sizes; /* Buffer of size: buf_size / packet_size  */
};

struct video1394_wait
{
    unsigned int channel;
    unsigned int buffer;
};

#define CIP_N_NTSC 2436
#define CIP_D_NTSC 38400
#define CIP_N_PAL 1
#define CIP_D_PAL 16

class IEEE1394Writer
{
    unsigned char                       *send_buf;
    struct video1394_mmap               v;
    struct video1394_queue_variable	w;
    int                                 unused_buffers;
    int                                 got_frame;
    unsigned char                       continuity_counter;
    unsigned int                        packet_sizes[321];
    unsigned int								 cip_counter;
    unsigned int								 cip_n;
    unsigned int								 cip_d;
    unsigned char						f50_60;

public:
    int                                 viddev;

    IEEE1394Writer();
    ~IEEE1394Writer();
    int FillBuffer(unsigned char *data, unsigned char* targetbuf, unsigned long targetbufsize, unsigned int * packet_sizes, int video_packets_per_frame, int min_out_packet_size);
    void SendFrame(Frame &frame);
};

#endif
