#!/usr/bin/ksh


#  Copyright Mission Critical Linux, 2000

#  Kimberlite is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License as published by the
#  Free Software Foundation; either version 2, or (at your option) any
#  later version.

#  Kimberlite is distributed in the hope that it will be useful, but
#  WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  General Public License for more details.

#  You should have received a copy of the GNU General Public License
#  along with Kimberlite; see the file COPYING.  If not, write to the
#  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
#  MA 02139, USA.
#
# $Revision: 1.17 $
#
# Author: Gregory P. Myrdal <Myrdal@MissionCriticalLinux.Com>
#
#------------------------------------------------------------

#
# Service resource action script to be used as a template.
#

#---------------------------------------------------------------------------
# Global variables
#---------------------------------------------------------------------------

export PATH=/usr/sbin:/sbin:/bin:/usr/sbin:/usr/bin:$PATH

MYNAME=$(basename $0)
ERROR_FILE=/tmp/$(basename $MYNAME).$$.error	# temp error file
KSH_LIB=$(dirname $0)/../ksh_lib

LIBRARIES="\
	$KSH_LIB/globals.ksh \
	$KSH_LIB/svcdb.ksh \
	$KSH_LIB/log.ksh \
	$KSH_LIB/user.ksh \
	$KSH_LIB/nfs.ksh \
	$KSH_LIB/device.ksh \
	$KSH_LIB/ip.ksh \
"

for library in $LIBRARIES
do
	if [ -f $library ]; then
	  . $library
	else
	  logAndPrint $LOG_ERR "$MYNAME Error: Cannot find $library" | \
	              tee > /dev/console
	  return $FAIL
	fi
done

#
# Initialize args that might be passed in
#
typeset DEVICE=""
typeset MOUNT_POINT=""
typeset FILESYSTEM_TYPE=""
typeset MOUNT_OPTIONS=""
typeset PERMISSIONS""
typeset OWNER=""
typeset GROUP""
typeset FORCE_UMOUNT""

#---------------------------------------------------------------------------
# Functions
#---------------------------------------------------------------------------

#
# Print a usage message
#
print_usage() {
	logAndPrint $LOG_ERR "\
Usage: $MYNAME {start,stop} -d device
		[-m mount] [-t type] 
		[-o option1,option2, ... optionN] 
		[-p mode] [-u owner] [-g group]"
}

#
# Anything that needs to be done before we leave can be put here.
#
clean_up () {

	rm -f $ERROR_FILE
	exit $SUCCESS
}

#
# Check to see if either the device or mount point are in use anywhere on
# the system.  It is not required that the device be mounted on moint point,
# just if either are in use.
#
in_use () {
	typeset mp mp_arg
	typeset dev dev_arg
	typeset junk

	if [ $# -ne 2 ]; then
		logAndPrint $LOG_ERR "Usage: in_use device mount_point".
		return $FAIL
	fi

	dev_arg=$1
	mp_arg=$2

	(mount | awk '{print $1,$3}') |&
	while read -p dev mp junk ; do
		if [ -n "$dev" -a "$dev" = "$dev_arg" ]; then
			return $YES
		fi
		
		if [ -n "$mp" -a "$mp" = "$mp_arg" ]; then
			return $YES
		fi
	done

	return $NO
}

#
# Check to see if the device is mounted.  Print a warning if its not
# mounted on the directory we expect it to be mounted on.
#
is_mounted () {

	typeset mp mp_arg
	typeset dev dev_arg

	if [ $# -ne 2 ]; then
		logAndPrint $LOG_ERR "Usage: is_mounted mount_point"
		return $FAIL
	fi

	dev_arg=$1
	mp_arg=$2

	(mount | awk '{print $1,$3}') |&
	while read -p dev mp ; do
		if [ -n "$dev" -a "$dev" = "$dev_arg" ]; then
			#
			# Check to see if its mounted in the right
			# place
			#
			if [ -n "$mp"  -a "$mp"  != "$mp_arg" ]; then
				logAndPrint $LOG_WARNING "\
Device $dev is mounted on $mp instead of $mp_arg"
			fi
			return $YES
		fi
	done

	return $NO
}

#
# Run the start actions
#
mount_start() {
	typeset -i ret_val=$SUCCESS
	typeset dev=""
	typeset mp=""
	typeset device_in_use=""
	typeset already_mounted=""
	typeset mount_options_line=""

	#
	# Make sure the device special file exists
	#
	if [ ! -e $DEVICE ]; then
	  logAndPrint $LOG_ERR "\
Device special file '$DEVICE' does not exist"
	  return $FAIL
	fi

	#
	# See if we have already mounted this device.
	# 
	if [ -n "$MOUNT_POINT" ]; then
	  is_mounted $DEVICE $MOUNT_POINT
	  case $? in
	  $NO)	:
		;;
	  $YES)	logAndPrint $LOG_DEBUG "$DEVICE already mounted"
		already_mounted=$YES
		;;
	  $FAIL) return $FAIL
		;;
	  esac
	fi

	#
	# If we have a mount point and its not already mounted, or we
	# have a raw device, set up its permissions.
	#
	if [ "$already_mounted" != $YES ]; then

		#
		# If needed, created the mount point
		#
	        if [ ! -d $MOUNT_POINT ]; then
	                rm -f $MOUNT_POINT	# rm in case its a plain file
			mkdir -p $MOUNT_POINT
			ret_val=$?
			if [ $ret_val -ne 0 ]; then
				logAndPrint $LOG_ERR "\
'mkdir -p $MOUNT_POINT' failed, error=$ret_val"
				return $FAIL
			fi
	        fi

	        #
	        # Set the permissions.  If we do not have a mount point we
	        # must be setting the permissions on a raw device.
	        #
	        if [ -n "$MOUNT_POINT" ]; then
		        dev=$MOUNT_POINT 
	        else
		        dev=$DEVICE
	        fi

	        if [ -n "$PERMISSIONS" ]; then
		        logAndPrint $LOG_DEBUG "\
Setting permissions on $dev to $PERMISSIONS"
		        chmod $PERMISSIONS $dev
		        ret_val=$?
		        if [ $ret_val -ne 0 ]; then
			        logAndPrint $LOG_ERR "\
'chmod $PERMISSIONS $dev' failed, error=$ret_val"
				return $FAIL
		        fi
	        fi

	        if [ -n "$GROUP" ]; then
		        logAndPrint $LOG_DEBUG "\
Setting group of $dev to $GROUP"
		        chgrp $GROUP $dev
		        ret_val=$?
		        if [ $ret_val -ne 0 ]; then
			        logAndPrint $LOG_ERR "\
'chgrp $GROUP $dev' failed, error=$ret_val"
	                  return $FAIL
		        fi
	        fi

	        if [ -n "$OWNER" ]; then
		        logAndPrint $LOG_DEBUG "\
Setting owner of $dev to $OWNER"
		        chown $OWNER $dev
		        ret_val=$?
		        if [ $ret_val -ne 0 ]; then
			        logAndPrint $LOG_ERR "\
'chown $OWNER $dev' failed, error=$ret_val"
	                  return $FAIL
		        fi
	        fi
	fi

	#
	# If not mounted, make sure that neither the device nor the mount point
	# are mounted (i.e. they may be mounted in a different location).  The
	# 'in_use' function check to see if either the device or mount point 
	# are in use somewhere else on the system.
	#
	if [ -n "$MOUNT_POINT" -a "$already_mounted" != $YES ]; then
		in_use $DEVICE $MOUNT_POINT
		case $? in
		$NO)	:	# good, no one else is using it
			;;
		$YES)		# uh oh, someone is using device or mount
	                logAndPrint $LOG_ERR "\
Cannot mount $DEVICE on $MOUNT_POINT, the device or mount point is already in use!"
			return $FAIL
			;;
		$FAIL)	return $FAIL
		 	;;
		*) 	logAndPrint $LOG_ERR "Unknown return from in_use"
			return $FAIL
			;;
		esac

		#
		# Check to determine if we need to fsck the filesystem
		#
		# Note: this code should not indicate in any manner suggested
		# file systems to use in the cluster.  Known filesystems are
		# listed here for correct operation.
		#
                case "$FILESYSTEM_TYPE" in
                reiserfs)	typeset fsck_needed="" ;;
                ext3)		typeset fsck_needed="" ;;
                jfs)		typeset fsck_needed="" ;;
                xfs)		typeset fsck_needed="" ;;
                ext2)		typeset fsck_needed=yes ;;
                minix)		typeset fsck_needed=yes ;;
                vfat)		typeset fsck_needed=yes ;;
                msdos)		typeset fsck_needed=yes ;;
		"")		typeset fsck_needed=yes	;;	# assume fsck
		*)		typeset fsck_needed=yes		# assume fsck
				logAndPrint $LOG_NOTICE "\
Unknown file system type '$FILESYSTEM_TYPE' for device $DEVICE.  Assuming fsck is required."
				;;
		esac

		#
		# Fsck the device, if needed.
		#
		if [ -n "$fsck_needed" ]; then
			logAndPrint $LOG_DEBUG "Running fsck on $DEVICE"
			fsck -p $DEVICE \
	                      >> /tmp/$(basename $DEVICE).fsck.log 2>&1
			ret_val=$?
			if [ $ret_val -gt 1 ]; then
				logAndPrint $LOG_ERR "\
'fsck -p $DEVICE' failed, error=$ret_val"
				logAndPrint $LOG_DEBUG \
					"Invalidating buffers for $DEVICE"
				$INVALIDATEBUFFERS -f $DEVICE
	                	return $FAIL
			fi
		fi

	        #
	        # Get the mount options, if any
	        #
	        if [ -n "$MOUNT_OPTIONS" ]; then
		  logAndPrint $LOG_DEBUG "\
Setting mount options of $MOUNT_POINT to $MOUNT_OPTIONS"
	          mount_options_line="-o $MOUNT_OPTIONS"
	        fi

		#
		# Mount the device
		#
		logAndPrint $LOG_DEBUG "\
mount $mount_options_line $DEVICE $MOUNT_POINT"
		mount $mount_options_line $DEVICE $MOUNT_POINT
		ret_val=$?
		if [ $ret_val -ne 0 ]; then
			logAndPrint $LOG_ERR "\
'mount $mount_options_line $DEVICE $MOUNT_POINT' failed, error=$ret_val"
			return $FAIL
		fi
	fi
	
	return $ret_val
}

#
# Run the stop actions
#
mount_stop() {
	typeset -i ret_val=0
	typeset -i try=1
	typeset -i max_tries=5		# how many times to try umount
	typeset -i sleep_time=2		# time between each umount failure
	typeset done=""
	typeset umount_failed=

	#
	# Unmount the device.  
	#
	# If we fail, try to forcefully unmount the filesystem using lsof.
	# (The poor man's forced unmount).
	#
	while [ ! "$done" ]; do
		is_mounted $DEVICE $MOUNT_POINT
		case $? in
		$NO)	logAndPrint $LOG_INFO "$DEVICE is not mounted"
			umount_failed=
			done=$YES
			;;
		$FAIL)	return $FAIL
			;;
		$YES)
			sync; sync; sync
			logAndPrint $LOG_INFO "\
unmounting $DEVICE ($MOUNT_POINT)"

			umount $DEVICE
			((ret_val=$?))
			if  [ $ret_val -eq 0 ]; then
				#
				# The Linux kernel will not invalidate buffers
				# after an unmount.  Since this disk might come
				# back with different data on the disk we want
				# to make sure that the system buffers are
				# invalidated before it goes.  NOTE: this is
				# fixed in the 2.4 kernel and all calls to
				# $INVALIDATEBUFFERS can be removed when all
				# clusters are running this version of the
				# kernel.
				#
				logAndPrint $LOG_DEBUG \
					"Invalidating buffers for $DEVICE"
				$INVALIDATEBUFFERS -f $DEVICE

				umount_failed=
				done=$YES
				continue
			fi

			umount_failed=yes

	                if [ -z "$FORCE_UMOUNT" ]; then
				if [ $try -ge $max_tries ]; then
					done=$YES
				else
					sleep $sleep_time
					((try=try+1))
				fi
				continue	# do not force unmount
	                fi

			#
			# Force unmount code
			#

			if [ $try -eq 1 ]; then
				logAndPrint $LOG_INFO "\
forcefully unmounting $DEVICE ($MOUNT_POINT)"
			fi

			#
			# Use lsof to free up mount point
			#
			(lsof $DEVICE | \
				awk '{print $1,$2,$3,$9}' | \
	                        grep -v PID | \
	                        sort -u -k 1,3) |&
			while read -p command pid user name
			do
				if [ $try -eq 1 ]; then
					logAndPrint $LOG_INFO "\
killing process $pid ($user $command $name)"
				fi

				if [ $try -gt 1 ]; then
					kill -9 $pid
				else
					kill -TERM $pid
				fi
			done
			;;

		*)	return $FAIL
			;;
		esac

		if [ $try -ge $max_tries ]; then
			done=$YES
		else
			sleep $sleep_time
			((try=try+1))
		fi
	done

	if [ -n "$umount_failed" ]; then
		logAndPrint $LOG_ERR "\
'umount $DEVICE' failed ($MOUNT_POINT), error=$ret_val"
		return $FAIL
	else
		return $SUCCESS
	fi
}

#---------------------------------------------------------------------------
# Main
#---------------------------------------------------------------------------

#trap "clean_up" 0 1 2 3 4 5 6 7 8 9 10 11 15

if [ $# -lt 1 ]; then
	print_usage
	exit $FAIL
fi

typeset ACTION=$1; shift

while [ $# -ne 0 ]; do
	case $1 in
	-d)	DEVICE=$2		# device name (/dev/sda5)
		shift ;;
	-m)	MOUNT_POINT=$2		# mount point (/usr/projects/cluster)
		shift ;;
	-t)	FILESYSTEM_TYPE=$2	# filesystem type (ext2)
		shift ;;
	-o)	MOUNT_OPTIONS=$2	# mount options (rw,nosuid,nodev)
		shift ;;
	-p)	PERMISSIONS=$2		# device/mount permissions
		shift ;;
	-u)	OWNER=$2		# device/mount owner
		shift ;;
	-g)	GROUP=$2		# device/mount group
		shift ;;
	-f)	FORCE_UMOUNT=$YES	# force unmount
	        ;;
	*) 	logAndPrint $LOG_ERR "Unknown option: $1"
		print_usage
		exit $FAIL ;;
	esac
	shift
done

if [ -z $DEVICE ]; then		# A device name is required
	logAndPrint $LOG_ERR "device name is required"
	print_usage
	exit $FAIL
fi

case $ACTION in
'start'|'START'|'Start')
	mount_start
	ret_val=$?
	if [ $ret_val -eq $SUCCESS ]; then
		exit 0
	else
		exit 1
	fi
	;;

'stop'|'STOP'|'Stop')
	mount_stop
	ret_val=$?
	if [ $ret_val -eq $SUCCESS ]; then
		exit 0
	else
		exit 1
	fi
	;;

*)
	logAndPrint $LOG_ERR "\
Cannot decipher action argument \'$ACTION\'"
	print_usage
	exit 1
	;;
esac

