/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
/*
 * $Id: hb.c,v 1.15 2000/10/10 19:31:16 moyer Exp $
 *
 * Author: Brian Stevens <stevens@missioncriticallinux.com>
 * description: cluster heartbeat implementation.
 */

static const char *version __attribute__ ((unused)) = "$Revision: 1.15 $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <sys/wait.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/time.h>
#include <sys/param.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <arpa/inet.h>
#include <stdarg.h>
#include <termios.h>
#include <sys/syslog.h>

/*
 * Local include files
 */

#include <parseconf.h>
#include <msgsvc.h>
#include <clusterdefs.h>
#include <clucfg.h>
#include <logger.h>

#define PROC_NET	"/proc/net"
#define HB_PORT		1120
#define MSG_SIZE	6
#define DEF_INTERVAL	2
#define DEF_TKO_COUNT	5
#define CFG_HB_PORT	"heartbeat%port"
#define CFG_HB_INTERVAL	"heartbeat%interval"
#define CFG_HB_TKO	"heartbeat%tko_count"
#define CFG_HB_LOGLEVEL "heartbeat%logLevel"
#define MAX_MSG_CONNS	32

struct hb_chan {
    int			fd;
    int			skips;
    int			type;
    struct sockaddr_in	me;
    struct sockaddr_in	to;
    char		lip_name[MAXHOSTNAMELEN];
    char		rip_name[MAXHOSTNAMELEN];
    char		name[MAXPATHLEN];
};

int chan_ip_create(struct hb_chan *, char *, char *, int); 
int chan_serial_create(struct hb_chan *, char *);
static void setup_sig_handlers(void);
static void sighup_handler(int signo);
static void hb_read_params(void);

/*
 * Below taken from util/daemon_init.c
 */
extern void daemon_init(char *prog);

struct hb_chan		chan[MAX_CHANNELS];
char			msg[MSG_SIZE];
int			debug = 0;
int			interval = DEF_INTERVAL;
int			tko_count = DEF_TKO_COUNT;
int			msg_fds[MAX_MSG_CONNS];
int			hb_port = HB_PORT;

static const char       *hb_string = "CPING";

static void
sighup_handler(int signo)
{
	/*
	 * Re-read our part of the configuration file. First, we need to
	 * explicitly close the CFG library, so that we don't just reread
	 * parameters cached by the library.
	 */

	if ( signo != SIGHUP )
           return;
	CFG_Destroy();
	hb_read_params();
	return;
}


static void
setup_sig_handlers(void)
{
	sigset_t         set;
	struct sigaction act;

	sigemptyset(&set);
	sigaddset(&set, SIGHUP);
	/*
	 * Make sure the HUP signal is not blocked.
	 */
	sigprocmask(SIG_UNBLOCK, &set, NULL);

	act.sa_handler = sighup_handler;
	sigemptyset(&act.sa_mask);
	act.sa_flags |= SA_RESTART;
	sigaction(SIGHUP, &act, NULL);
}


static void
hb_read_params(void)
{
    char  *p;

    /*
     * Read in any override to the log level, heartbeat port, ping interval
     * or tko count
     */

    clulog(LOG_DEBUG, "hb: setting configuration parameters.\n");
    if (CFG_Get((char *) CFG_HB_LOGLEVEL, NULL, &p) == CFG_OK) {
        if (p) {
	    clulog(LOG_DEBUG, "hb: changing loglevel from %d to %d\n",
	           clu_set_loglevel(atoi(p)), atoi(p));
        }
    }

    if (CFG_Get((char *) CFG_HB_PORT, NULL, &p) == CFG_OK) {
        if (p) {
	    hb_port = atoi(p);
	    clulog(LOG_DEBUG, "hb: overriding hb port to be %d\n", hb_port);
        }
    }

    if (CFG_Get((char *) CFG_HB_INTERVAL, NULL, &p) == CFG_OK) {
        if (p) {
	    interval = atoi(p);
	    clulog(LOG_DEBUG, "hb: overriding interval to be %d\n", interval);
        }
    }
    
    if (CFG_Get((char *) CFG_HB_TKO, NULL, &p) == CFG_OK) {
        if (p) {
	    tko_count = atoi(p);
	    clulog(LOG_DEBUG, "hb: overriding tko count to be %d\n", tko_count);
        }
    }

    return;
}


int
chan_ip_create(
    struct hb_chan	*cp,
    char		*lif,
    char		*rif,
    int			port)
{
    struct hostent	*hp;
    int			sopt;

    /*
     * Initialize the state of the channel
     */

    cp->skips = tko_count;
    cp->type = CHAN_NET;
    strcpy(cp->name, lif);
    strcpy(cp->lip_name, lif);
    strcpy(cp->rip_name, rif);

    /*
     * Create the UDP endpoint
     */

    cp->fd = socket(AF_INET, SOCK_DGRAM, 0);
    if (cp->fd < 0) {
	clulog(LOG_ERR, "socket: %s\n", strerror(errno));
        return(1);
    }

    /*
     * Get the address of the local interface
     */

    if ((hp = gethostbyname(lif)) == NULL) {
	clulog(LOG_WARNING, "hb: unable to get address for host %s\n", lif);
	return(1);
    }

    /*
     * Bind the socket to the local interface and port
     */

    memset((char *)&cp->me, 0, sizeof(cp->me));
    cp->me.sin_family = AF_INET;
    cp->me.sin_port = htons((unsigned short)port);
    cp->me.sin_addr = *(struct in_addr *)hp->h_addr;

    if (bind(cp->fd, (struct sockaddr *)&cp->me,
	sizeof(struct sockaddr)) < 0) {
	clulog(LOG_ERR, "chan_ip_create: %s\n", strerror(errno));
        return(1);
    }

    sopt = 1;
    if (setsockopt(cp->fd, SOL_SOCKET, SO_DONTROUTE, &sopt, sizeof(sopt)) < 0) {
	clulog(LOG_ERR, "chan_ip_create: %s\n", strerror(errno));
        return(1);
    }

    /*
     * Mark that we do not want to block when performing io on the socket.
     */

    if (fcntl(cp->fd, F_SETFL, O_NONBLOCK)) {
	clulog(LOG_WARNING, "chan_ip_create: %s\n", strerror(errno));
	clulog(LOG_WARNING, "chan_ip_create: unable to enable non-blocking io\n");
	return(1);
    }

#ifdef notdef
    /*
     * Get the interface name
     */

    bcopy(&cp->me, &ifquery.ifr_ifru, sizeof(struct sockaddr));
    if (ioctl(cp->fd, SIOCGIFNAME, (void *)&ifquery)) {
	clulog(LOG_ERR, "chan_ip_create: %s\n", strerror(errno));
	clulog(LOG_ERR, "chan_ip_create: unable to get the interface name "
	                "for %s\n", lif);
	return(1);
    }  
    strcpy(cp->if_name, ifquery.ifr_name);
#endif

    /*
     * Get the address of the remote interface
     */

    if ((hp = gethostbyname(rif)) == NULL) {
	clulog(LOG_ERR, "chan_ip_create: unable to get address for host %s\n",
	       rif);
	return(1);
    }

    /*
     * Record the address of the interface we heartbeat to on this channel
     */

    memset((char *)&cp->to, 0, sizeof(cp->to));
    cp->to.sin_family = AF_INET;
    cp->to.sin_port = htons((unsigned short)port);
    cp->to.sin_addr = *(struct in_addr *)hp->h_addr;

    return(0);
}


int
chan_serial_create(
    struct hb_chan	*cp,
    char		*line)
{
    struct termios  ti;

    /*
     * Initialize the state of the channel
     */

    strcpy(cp->name, line);
    cp->skips = tko_count;
    cp->type = CHAN_SERIAL;

    /*
     * Open the serial line
     */

    cp->fd = open(line, O_RDWR | O_EXCL);
    if (cp->fd < 0) {
	clulog(LOG_ERR, "chan_serial_create: open of serial device failed.\n");
	return(1);
    }

    /*
     * Discard any queued data
     */

    (void) tcflush(cp->fd, TCIOFLUSH);

    /*
     * Put the serial line in raw mode and select the baud rate
     */

    if (tcgetattr(cp->fd, &ti) < 0) {
	clulog(LOG_ERR, "chan_serial_create: tcgetattr on serial device failed\n");
	return(1);
    }

    cfmakeraw(&ti);
    cfsetospeed(&ti, B19200);
    cfsetispeed(&ti, B19200);

    if (tcsetattr(cp->fd, TCSAFLUSH, &ti) < 0) {
	clulog(LOG_ERR, "chan_serial_create: tcsetattr on serial device failed\n");
	return(1);
    }                             

    /*
     * Mark that we do not want to block when performing io
     */

    if (fcntl(cp->fd, F_SETFL, O_NONBLOCK)) {
	clulog(LOG_ERR, "chan_serial_create: unable to enable non-blocking io\n");
	return(1);
    }

    return(0);
}

int
main(
    int			argc,
    char		**argv)
{
    struct sockaddr_in	from;
    int			addrlen;
    char		opt;
    int			nbytes;
    int			cid;
    int			i;
    int			lid, rid;
    struct timeval	tv;
    msg_handle_t        msg_listen_fd;
    fd_set		readfds;
    int                 ipcmsg;
    int                 auth=0;
    CluCfg		*cinfo;
    int			nfds, ret=0;

    while ((opt = getopt(argc, argv, "d")) != EOF) {
        switch (opt) {
	case 'd':
	    debug = 1;
	    break;
	default:
	    break;
        }
    }

    if (!debug) {
        (void)daemon_init(argv[0]);
        (void)clu_set_loglevel(LOG_INFO);
    } else {
        (void)clu_set_loglevel(LOG_DEBUG);
    }

    /*
     * We want to catch sighup
     */
    setup_sig_handlers();

    /*
     * Get the cluster's static configuration
     */

    cinfo = get_clu_cfg(NULL);
    if (cinfo == NULL) {
	clulog(LOG_CRIT, "hb: could not read cluster configuration\n"); 
	exit(1);
    }

    /*
     * Read in any overrides from the cluster database.
     */
    hb_read_params();

    clulog(LOG_DEBUG, "hb: startup, cluster \"%s\"\n", cinfo->name);

    /*
     * Initialize the Message Service
     */
    msg_listen_fd = msg_listen(PROCID_HEARTBEAT);
    if (msg_listen_fd < 0) {
	clulog(LOG_CRIT, "hb: Unable to initialize messaging subsystem.\n");
	exit(1);
    }

    lid = cinfo->lid;
    rid = lid ? 0 : 1;

    addrlen = sizeof(struct sockaddr);

    clulog(LOG_DEBUG, "hb: local member is %s\n", cinfo->nodes[lid].name);
    clulog(LOG_DEBUG, "hb: remote member is %s\n", cinfo->nodes[rid].name);

    /*
     * See if networking is configured
     */

    if (access(PROC_NET, R_OK)) {
	if (errno == ENOENT) {
		clulog(LOG_CRIT, "networking not configured");
	} else {
		clulog(LOG_CRIT, "%s", strerror(errno));
	}
	exit(1);
    }

    /*
     * Create the channels
     */

    for (i = 0; i < cinfo->num_chans; i++) {

        clulog(LOG_DEBUG, "hb: chan %d (%s) is %s - %s\n", i,
		 cinfo->nodes[lid].chans[i].type == CHAN_NET ? "net" : "serial",
	       /*
		*  The unions both have a first member of 'name'.  Thus,
		*  it doesn't matter whether this is a net device or a 
		*  serial device, the offset is the same.  Sorry for the
		*  confusion.
		*/
	       cinfo->nodes[lid].chans[i].dev.serial.name,
	       cinfo->nodes[rid].chans[i].dev.serial.name);


	switch (cinfo->nodes[lid].chans[i].type) {
	case CHAN_NET:
            if (chan_ip_create(&chan[i], 
			       cinfo->nodes[lid].chans[i].dev.net.name,
	    		       cinfo->nodes[rid].chans[i].dev.net.name, 
			       hb_port)) {
		clulog(LOG_ERR,  "channel creation failed\n");
	    } else {
		clulog(LOG_DEBUG, "IP chan created on fd %d, %s:%d - ",
			 chan[i].fd, inet_ntoa(chan[i].me.sin_addr), hb_port);
		clulog(LOG_DEBUG, "%s:%d\n",
		 	 inet_ntoa(chan[i].to.sin_addr), hb_port);
	    }
	    break;
	case CHAN_SERIAL:
            if (chan_serial_create(&chan[i], 
				 cinfo->nodes[lid].chans[i].dev.serial.name)) {
		clulog(LOG_ERR, "channel creation failed\n");
	    } else {
		clulog(LOG_DEBUG, "Serial channel created on fd %d\n", chan[i].fd);
	    }
	    break;
	default:
	    break;
	}
    }

    for (i = 0; i < MAX_MSG_CONNS; i++)
    	msg_fds[i] = 0;

    clulog(LOG_DEBUG, "listening ...\n");

    while (1) {

	/*
	 * Delay for the interval time. We need to be able to be
	 * interrupted in order to respond to messages while we are
	 * waiting.
	 */

        tv.tv_sec = interval;
        tv.tv_usec = 0;

	clulog(LOG_DEBUG, "hb: delaying for %d seconds\n", interval);

	while (tv.tv_sec || tv.tv_usec) {

	    FD_ZERO(&readfds);
            FD_SET(msg_listen_fd, &readfds);
    	    for (i = 0; i < MAX_MSG_CONNS; i++) {
	    	if (msg_fds[i])
		    FD_SET(msg_fds[i], &readfds);
	    }

	    nfds = select(64, &readfds, NULL, NULL, &tv);
	    if (nfds <= 0)
	        continue;

	    /*
	     * Respond to any messages
	     */

	    if (FD_ISSET(msg_listen_fd, &readfds)) {

		FD_CLR(msg_listen_fd, &readfds);

    	        for (i = 0; i < MAX_MSG_CONNS; i++)
		    if (!msg_fds[i])
		        break;

		if (i == MAX_MSG_CONNS) {
		    clulog(LOG_INFO, "hb: too many concurrent msg contexts\n");
		    continue;
		}

		if ((msg_fds[i] = msg_accept(msg_listen_fd)) < 0) {
		    clulog(LOG_INFO, "hb: failed accepting msg connection\n");
		    msg_fds[i] = 0;
		}
		
		clulog(LOG_DEBUG, "hb: established msg channel %d\n",
		    msg_fds[i]);

		continue;
	    }

    	    for (i = 0; i < MAX_MSG_CONNS; i++) {

	        if (!FD_ISSET(msg_fds[i], &readfds))
		    continue;

		/*
		 * msg_receive_timeout will never return a partial message.
		 */

		if ((msg_receive(msg_fds[i], &ipcmsg, sizeof(ipcmsg),&auth)) !=
			sizeof(ipcmsg)) {
		    clulog(LOG_DEBUG, "hb: shutting down msg channel %d\n",
		    	   msg_fds[i]);
		    msg_close(msg_fds[i]);
		    msg_fds[i] = 0;
		    continue;
		}

		switch (ipcmsg) {
		case HB_QUERY_NETUP:
		    clulog(LOG_INFO, "hb: received HB_QUERY_NETUP message\n");

		    ipcmsg = 0;

		    /*
		     * Walk interfaces.  We return a bitmask of the interfaces
		     * which are online.
		     */

		    for (cid = 0; cid < cinfo->num_chans; cid++) {
			    if (chan[cid].skips < tko_count) {
				    ipcmsg |= (1 << cid);;
			    }
		    }

		    clulog(LOG_INFO, "hb: replying with %d\n", ipcmsg);

		    ret = msg_send(msg_fds[i], &ipcmsg, sizeof(ipcmsg));
		    if (ret != sizeof(ipcmsg))
			    clulog(LOG_ERR, "hb: only sent %d bytes.\n", ret);

		    msg_close(msg_fds[i]);
		    msg_fds[i] = 0;

		    break;

		case HB_TERM: /* Terminate HeartBeat */
		    if (!auth) {
			clulog(LOG_ERR, "Warning: HB_TERM request received "
			       "from unauthenticated source!\n");
			msg_close(msg_fds[i]);
			break;
		    }
		    clulog(LOG_INFO, "hb: received HB_TERM message\n");
		    msg_close(msg_fds[i]);
		    msg_close(msg_listen_fd);

		    for (i = 0; i < MAX_MSG_CONNS; i++) {
			if (msg_fds[i])
			    msg_close(msg_fds[i]);
		    }

		    goto done;

		default:
		    clulog(LOG_INFO, "hb: received unknown message %d\n", ipcmsg);
		    msg_close(msg_fds[i]);
		    break;

		} /* switch */

	    } /* for (i = 0; i < MAX_CONNECTIONS... */

	} /* while (tv.tv_sec|tv.tv_usec) */

	for (cid = 0; cid < cinfo->num_chans; cid++) {
	    chan[cid].skips++;

	    clulog(LOG_DEBUG, "hb: reading on chan %d\n", cid);

	    switch (chan[cid].type) {
	    case CHAN_SERIAL:

    		while ((nbytes = read(chan[cid].fd, msg, MSG_SIZE)) > 0) {
		    clulog(LOG_DEBUG, "hb: received msg on serial %s\n", chan[cid].name);

		    /*
		     * Ignore potential noise
		     */
    
		    if (strcmp(msg, hb_string)) {
			/*
			 * Check to see if other node is panicking
			 */
			if ((unsigned long)msg == 0x1abacdeb) {
			    /* Don't shoot partner */
			}
		 	continue;
		    }

		    /*
		     * Valid message
		     */

		    if (chan[cid].skips >= tko_count) {
			clulog(LOG_INFO, "peer ONLINE over channel %s\n",
		    	        chan[cid].name);
		    }
		    chan[cid].skips = 0;
	        }
		break;

	    case CHAN_NET:
                while ((nbytes = recvfrom(chan[cid].fd, msg, MSG_SIZE, 0,
	    			     (struct sockaddr *)&from, &addrlen)) > 0) {
		    /*
		     * Ignore messages from anyone but the other cluster member.
		     */
    
		    clulog(LOG_DEBUG, "hb: received msg of %d bytes from %s\n",
			   nbytes, inet_ntoa(from.sin_addr));

	            if (*(int *)&chan[cid].to.sin_addr !=
		        *(int *)&from.sin_addr) {
			clulog(LOG_ERR, "hb: received bogus msg\n");
		        continue;
		    }
    	
		    if (chan[cid].skips >= tko_count) {
			clulog(LOG_INFO, "peer ONLINE over channel %s\n",
		    	        chan[cid].name);
		    }
		    chan[cid].skips = 0;
	        }
	        break;
	    }

	    if (chan[cid].skips == tko_count) {
		clulog(LOG_INFO,  "peer OFFLINE over channel %s\n",
		    	chan[cid].name);
	    }

	    clulog(LOG_DEBUG, "hb: sending on chan %d\n", cid);
	    switch (chan[cid].type) {
	    case CHAN_SERIAL:
    	        nbytes = write(chan[cid].fd, hb_string, MSG_SIZE);
		clulog(LOG_DEBUG, "hb: wrote %d bytes on serial\n", nbytes);
		break;
	    case CHAN_NET:
	        nbytes = sendto(chan[cid].fd, hb_string, MSG_SIZE, 0,
			        (struct sockaddr *)&chan[cid].to,
			        sizeof(struct sockaddr));
		clulog(LOG_DEBUG, "hb: wrote %d bytes on net\n", nbytes);
	        break;
	    }
	}
    }
    
done:
    if (cinfo != NULL)
	free(cinfo);
    clulog(LOG_INFO, "exiting\n");
    exit(0);
}
